// Copyright (C)  2001 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/jar_zip_utils/Jar_Entry.cpp,v 1.1 2001/09/10 01:58:06 gwu2 Exp $
//

#include "stdio.h"
#include "string.h"
#include "Jar_Entry.h"


/************************************************************************************/
/*Constuctors*/
Jar_Entry::Jar_Entry()
{
	p_JarFile = NULL;
	p_JarFile_Info = NULL;

	m_Entry_Info.is_file_ok = false;
	m_Entry_Info.file_name = NULL;
	m_Is_Reading_Info_OK = false;
}

Jar_Entry::Jar_Entry(const char* jarflname,const char* entryflname)
{
	p_JarFile = new JarFile(jarflname);				//First,open the zip or jar file
	p_JarFile_Info = p_JarFile->Get_JarFile_Info();	//Return NULL if not successful

	m_Entry_Info.is_file_ok = false;
	m_Entry_Info.file_name = NULL;
	Open(entryflname);				//m_Entry_Info.is_file_ok will be true if successful
	m_Is_Reading_Info_OK = false;
}

Jar_Entry::Jar_Entry(JarFile* jarfile,const char* entryflname)
{
	p_JarFile = jarfile;			//Important
	p_JarFile_Info = p_JarFile->Get_JarFile_Info();

	m_Entry_Info.is_file_ok = false;
	m_Entry_Info.file_name = NULL;
	Open(entryflname);				//m_Entry_Info.is_file_ok will be true if successful
	p_JarFile = NULL;				//Important
	m_Is_Reading_Info_OK = false;
}

/************************************************************************************/
Jar_Entry::~Jar_Entry()
{
	if(p_JarFile)
		delete p_JarFile;

	Close();
}

/************************************************************************************/
/************************************************************************************/
/*Return true if the entry is opened successfully.*/
bool Jar_Entry::Is_Open_OK()
{
	return m_Entry_Info.is_file_ok;
}

/*Return the length of the entry, or -1 if the entry is not opened successfully*/
int Jar_Entry::Get_Entry_File_Length()
{
	if(m_Entry_Info.is_file_ok)
		return m_Entry_Info.length_uncompressed;
	return -1;
}

/************************************************************************************/
/************************************************************************************/
/*Get the current entry's infomation,including the followings:
  flags, compression method, CRC, compressed length, uncompressed length, file'sname
*/
int Jar_Entry::Get_Current_File_Info()
{
	uLong temp_data;
	int err = UNJAR_OK;

	if(fseek(p_JarFile_Info->file_handle,
		m_Entry_Info.pos_in_central_dir ,
		SEEK_SET)!=0)
		return UNJAR_ERROR;

	/*Get the tag 504b0102,which is the begin of the file info in the central directory*/
	if(unjar_get_long(p_JarFile_Info->file_handle,&temp_data) != UNJAR_OK)
		err = UNJAR_ERROR;
	else if(temp_data!=0x02014b50)
		err = UNJAR_ERROR;

	/*Look over the next 4 bytes : version,version needed*/
	if(fseek(p_JarFile_Info->file_handle,4,SEEK_CUR)!=0)
		err = UNJAR_ERROR;

	/*Get the flags of the current file*/
	if(unjar_get_short(p_JarFile_Info->file_handle,&(m_Entry_Info.flags))!=UNJAR_OK)
		err = UNJAR_ERROR;

	/*Get the compression method of the current file*/
	if(unjar_get_short(p_JarFile_Info->file_handle,&(m_Entry_Info.compression_method))!=UNJAR_OK)
		err = UNJAR_ERROR;

	/*Look over the next 4 bytes : Dos date*/
	if(fseek(p_JarFile_Info->file_handle,4,SEEK_CUR)!=0)
		err = UNJAR_ERROR;

	/*Get the CRC of the current file*/
	if(unjar_get_long(p_JarFile_Info->file_handle,&(m_Entry_Info.crc))!=UNJAR_OK)
		err = UNJAR_ERROR;

	/*Get the length of the compressed current file*/
	if(unjar_get_long(p_JarFile_Info->file_handle,&(m_Entry_Info.length_compressed))!=UNJAR_OK)
		err = UNJAR_ERROR;

	/*Get the length of the uncompressed current file*/
	if(unjar_get_long(p_JarFile_Info->file_handle,&(m_Entry_Info.length_uncompressed))!=UNJAR_OK)
		err = UNJAR_ERROR;

	/*Get the length of the file name of the current file*/
	if(unjar_get_short(p_JarFile_Info->file_handle,&(m_Entry_Info.length_filename))!=UNJAR_OK)
		err = UNJAR_ERROR;

	/*Get the size of the file extra*/
	if(unjar_get_short(p_JarFile_Info->file_handle,&(m_Entry_Info.length_restdata))!=UNJAR_OK)
		err = UNJAR_ERROR;

	/*Get the size of the file comment*/
	if(unjar_get_short(p_JarFile_Info->file_handle,&temp_data)!=UNJAR_OK)
		err = UNJAR_ERROR;
	if(UNJAR_OK == err)
		m_Entry_Info.length_restdata += temp_data;

	/*Look over the next 8 bytes :	start disk number(2 bytes),
	internal_fa(2 bytes) , external_fa(4 bytes)*/
	if(fseek(p_JarFile_Info->file_handle,8,SEEK_CUR)!=0)
		err = UNJAR_ERROR;

	/*Get the offset of the current file in the zip or jar file*/
	if(unjar_get_long(p_JarFile_Info->file_handle,&(m_Entry_Info.offset))!=UNJAR_OK)
		err = UNJAR_ERROR;

	/*Get the file name of the current file*/
	if((UNJAR_OK == err )&&(m_Entry_Info.length_filename > 0))
	{
		if(m_Entry_Info.file_name != NULL)
			free(m_Entry_Info.file_name);

		m_Entry_Info.file_name = (char*)malloc(m_Entry_Info.length_filename + 1);
		*(m_Entry_Info.file_name + m_Entry_Info.length_filename) = '\0';
		if(fread(m_Entry_Info.file_name,(uInt)m_Entry_Info.length_filename,1,p_JarFile_Info->file_handle)!=1)
			err = UNJAR_ERROR;
	}

	return err;
}
/************************************************************************************/
int Jar_Entry::Open(const char* entryflname)
{
    if (strlen(entryflname)>=UNJAR_MAX_FILENAME_IN_JAR)
        return UNJAR_ERROR;

	if (NULL == p_JarFile_Info)
		return UNJAR_ERROR;


/************Look for the given entry in the zip or jar file************************/
	m_Entry_Info.pos_in_central_dir = p_JarFile->Get_Entry_Pos_CD(entryflname);
	if(-1 == m_Entry_Info.pos_in_central_dir)
		return UNJAR_ERROR;

	int err = Get_Current_File_Info();

	m_Entry_Info.is_file_ok = (UNJAR_OK == err);

	return err;
}
/************************************************************************************/
/************************************************************************************/
/*
There is a copy of a specific file's info in the zip or jar file.It is at the begin of 
this file.We check it with the info contained in the central directory for coherency.
*/
int Jar_Entry::Check_Header_Coherency(uLong* length_filename_extra)
{
	int err=UNJAR_OK;
	uLong temp_data,flags;

	if(fseek(p_JarFile_Info->file_handle,
		m_Entry_Info.offset + p_JarFile_Info->byte_before_file,
		SEEK_SET)!=0)
		return UNJAR_ERROR;

	if(unjar_get_long(p_JarFile_Info->file_handle,&temp_data)!=UNJAR_OK)
		err = UNJAR_ERROR;
	else if(temp_data != 0x04034b50)
		err = UNJAR_BADJARFILE;

	/*Look over the next 2 bytes : version*/
	if(fseek(p_JarFile_Info->file_handle,2,SEEK_CUR)!=0)
		err |= UNJAR_ERROR;

	/*Get the flags of the current file*/
	if(unjar_get_short(p_JarFile_Info->file_handle,&flags)!=UNJAR_OK)
		err |= UNJAR_BADJARFILE;

	/*Get the compression method of the current file*/
	if(unjar_get_short(p_JarFile_Info->file_handle,&temp_data)!=UNJAR_OK)
		err |= UNJAR_ERROR;
	else if(UNJAR_OK == err)
	{
		if(temp_data != m_Entry_Info.compression_method)
			err = UNJAR_BADJARFILE;
		else if((temp_data != 0)&&(temp_data != Z_DEFLATED))
			err = UNJAR_BADJARFILE;
	}

	/*Look over the next 4 bytes : Dos date*/
	if(fseek(p_JarFile_Info->file_handle,4,SEEK_CUR)!=0)
		err |= UNJAR_ERROR;

	if(unjar_get_long(p_JarFile_Info->file_handle,&temp_data)!=UNJAR_OK)
		err |= UNJAR_ERROR;
	else if((UNJAR_OK == err)&&(temp_data != m_Entry_Info.crc)
		&&(flags & 8==0))
		err = UNJAR_BADJARFILE;


	if(unjar_get_long(p_JarFile_Info->file_handle,&temp_data)!=UNJAR_OK)
		err |= UNJAR_ERROR;
	else if((UNJAR_OK == err)&&(temp_data != m_Entry_Info.length_compressed)
		&&(flags & 8==0))
		err = UNJAR_BADJARFILE;

	if(unjar_get_long(p_JarFile_Info->file_handle,&temp_data)!=UNJAR_OK)
		err |= UNJAR_ERROR;
	else if((UNJAR_OK == err)&&(temp_data != m_Entry_Info.length_uncompressed)
		&&(flags & 8==0))
		err = UNJAR_BADJARFILE;

	if(unjar_get_short(p_JarFile_Info->file_handle,&temp_data)!=UNJAR_OK)
		err |= UNJAR_ERROR;
	else if((UNJAR_OK == err)&&(temp_data != m_Entry_Info.length_filename))
		err = UNJAR_BADJARFILE;


	if(unjar_get_short(p_JarFile_Info->file_handle,length_filename_extra)!=UNJAR_OK)
		err |= UNJAR_ERROR;

	if(UNJAR_OK == err)
		*length_filename_extra+=temp_data;

	return err;

}


/************************************************************************************/
/*
  Close the file in zip opened with unjaripOpenCurrentFile
  Return UNJAR_BADJARFILE if all the file was read but the CRC is not good
*/
int Jar_Entry::Close ()
{
	if((!m_Is_Reading_Info_OK)||(!m_Entry_Info.is_file_ok))
		return UNJAR_OK;

	if(m_Entry_Info.crc!=m_Reading_Info.crc)
		return UNJAR_BADJARFILE;

	if(m_Reading_Info.read_buffer!=NULL)
	{
		free(m_Reading_Info.read_buffer);
		m_Reading_Info.read_buffer = NULL;
	}

	if(m_Reading_Info.is_stream_initialized)
		inflateEnd(&(m_Reading_Info.stream));

	m_Is_Reading_Info_OK=false;

	if(m_Entry_Info.file_name!=NULL)
	{
		free(m_Entry_Info.file_name);
		m_Entry_Info.file_name = NULL;
	}

	return UNJAR_OK;
}
/************************************************************************************/
/*
  Open for reading data the current entry in the zip or jar file.
  If there is no error and the file is opened, the return value is UNJAR_OK.
*/
int Jar_Entry::Open_Current_File ()
{
	uLong length_filename_extra;
	int err=UNJAR_OK;

	if(!m_Entry_Info.is_file_ok)
		return UNJAR_PARAMETER;

	if(m_Is_Reading_Info_OK)
		Close();

	if((err=Check_Header_Coherency(&length_filename_extra))!=UNJAR_OK)
		return err;

	m_Reading_Info.read_buffer=(char*)malloc(m_Entry_Info.length_compressed);
	if(NULL == m_Reading_Info.read_buffer)
		return UNJAR_ERROR;

	/*Some initializations*/

	m_Is_Reading_Info_OK=true;

	m_Reading_Info.is_stream_initialized = 0;
	m_Reading_Info.crc = 0;
	m_Reading_Info.stream.total_out = 0;
	
	if(m_Entry_Info.compression_method != 0)
	{
		m_Reading_Info.stream.zalloc = (alloc_func)0;
		m_Reading_Info.stream.zfree = (free_func)0;
		m_Reading_Info.stream.opaque = (voidpf)0; 
		err = inflateInit2(&(m_Reading_Info.stream),-MAX_WBITS);
		if(Z_OK == err)
			m_Reading_Info.is_stream_initialized = 1;
	}

//	m_Reading_Info.length_rest_compressed=m_Entry_Info.length_compressed;
//	m_Reading_Info.length_rest_uncompressed=m_Entry_Info.length_uncompressed;

	m_Reading_Info.stream.avail_in = 0;

	m_Reading_Info.pos_in_jarfile=m_Entry_Info.offset
										+ SIZE_JAR_LOCAL_HEADER
										+ length_filename_extra;

    return UNJAR_OK;
}

/************************************************************************************/
/*
Read the entry file into buf.
Notice:this function read all the content of the entry file by one time.
return:a buffer;NULL if meet errors.
*/
Bytef* Jar_Entry::Read()
{
	int err=UNJAR_OK;
	if((NULL == p_JarFile_Info)||(!m_Entry_Info.is_file_ok))
		return NULL;

	err=Open_Current_File();
	if(err != UNJAR_OK)
		return NULL;

	Bytef* buf=new Bytef[m_Entry_Info.length_uncompressed+1];
	buf[m_Entry_Info.length_uncompressed]='\0';

	m_Reading_Info.stream.next_out=buf;

	if(fseek(p_JarFile_Info->file_handle,
		m_Reading_Info.pos_in_jarfile+p_JarFile_Info->byte_before_file,SEEK_SET)!=0)
	{
		Close();
		delete[] buf;
		return NULL;
	}

	if(fread(m_Reading_Info.read_buffer,m_Entry_Info.length_compressed,
			1,p_JarFile_Info->file_handle)!=1)
	{
		Close();
		delete[] buf;
		return NULL;
	}

	m_Reading_Info.pos_in_jarfile += m_Entry_Info.length_compressed;
	m_Reading_Info.stream.avail_in = m_Entry_Info.length_compressed;
	m_Reading_Info.stream.next_in=(Bytef*)m_Reading_Info.read_buffer;
	m_Reading_Info.stream.avail_out = m_Entry_Info.length_uncompressed;

	if(0 == m_Entry_Info.compression_method)
	{
		int i;
		int num_copy=m_Reading_Info.stream.avail_in<m_Reading_Info.stream.avail_out?
			m_Reading_Info.stream.avail_in:m_Reading_Info.stream.avail_out;

		for(i=0;i<num_copy;i++)
			*(buf+i)=*(m_Reading_Info.read_buffer+i);

		m_Reading_Info.crc=crc32(m_Reading_Info.crc,buf,num_copy);
	}
	else
	{
		int flush=Z_SYNC_FLUSH;
		err=inflate(&(m_Reading_Info.stream),flush);

		m_Reading_Info.crc=crc32(m_Reading_Info.crc,buf,m_Entry_Info.length_uncompressed);
		if (err==Z_STREAM_END)
		{
			err=Close();
			if(UNJAR_OK == err)
				return buf;
			return NULL;
		}

	}
	err=Close();

	if(UNJAR_OK == err)
		return buf;

	delete[] buf;
	return NULL;

}
