// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc_v2/nursery_step_gen.cpp,v 1.10 2002/01/11 15:47:38 weldon Exp $
// 

#include "nursery_step_gen.h"
#include "gc_space.h"
#include "card_table.h"
#include "gc_hooks.h"
#include "gc_plan.h"
#include "gc_consts.h"
#include "orp_threads.h"
#include "orp_synch.h"
#include "mrl_gc_v1.h"

#include "gc_header.h"


int clear_nursery_hint = 0;
// Returns a nursery with nursery_status == thread_clearing_nursery or NULL if none need clearing.
block_info *get_me_next_nursery_to_zero ()
{
    block_info *result = NULL;
    // Starting at the hint, loop through the nurseries looking for one that is free.
    int i;
    for (i = clear_nursery_hint; i < allocated_nurseries; i++) {
        if (nurseries[i]->nursery_status == free_uncleared_nursery) {
            clear_nursery_hint = i;
            result = nurseries[i];
            if (InterlockedCompareExchange((PVOID *)&(result->nursery_status), (void *)thread_clearing_nursery, (void *)free_uncleared_nursery) == (void *)free_uncleared_nursery) {
                // We win this is now our nursery, it isn't free but it also isn't spent or active.
                // If verifier runs in other thread we can have problems here...
                return result; // We are done.
            } else {
                result = NULL; // We lose race keep looking.
            }
        }
    }
    for (i = 0; i < clear_nursery_hint; i++) {
        if (nurseries[i]->nursery_status == free_uncleared_nursery) {
            clear_nursery_hint = i;
            result = nurseries[i];
           if (InterlockedCompareExchange((PVOID *)&(result->nursery_status), (void *)thread_clearing_nursery, (void *)free_uncleared_nursery) == (void *)free_uncleared_nursery) {
                // We win this is now our nursery, it isn't free but it also isn't spent or active.
                // If verifier runs in other thread we can have problems here...
                return result; // We are done.
            } else {
                result = NULL; // We lose race keep looking.
            }
        }
    }
    return result;
}

// Global things I need for the nursery zeroing logic.

HANDLE Zero_Thread_Handle;

// Boolean indicating if Zero thread has suspended itself.
bool zero_thread_is_sleeping;

// The Zero thread will die when this boolean is set...(it is set to true by the step destructor)
bool zero_thread_die_signal;
#ifdef ORP_POSIX
DWORD zero_thread_func(LPVOID ignore)
{
  orp_cout << "-gc xero_thread not available (yet) on Linux" << endl;
  orp_exit(37);
}

void start_nursery_zeroing_thread()
{
  orp_cout << "-gc xero_thread not available (yet) on Linux" << endl;
  orp_exit(37);
}
void stop_nursery_zeroing_thread ()
{
  orp_cout << "-gc xero_thread not available (yet) on Linux" << endl;
  orp_exit(37);

}

#else
DWORD WINAPI 
zero_thread_func(LPVOID ignore)
{
    int num_zeroed = 0;
    int previous_nursery_zeroed = 0;

    // The main loop. Keep asking for nurseries and keep zeroing them.

    while(true) {

        // Check if I must die...
        if (zero_thread_die_signal) { 
            orp_cout << "zero thread dying...\n";
            ExitThread(0);
        }

        block_info *nursery_to_zero = 
             get_me_next_nursery_to_zero();

        if (nursery_to_zero) { 
            assert (nursery_to_zero->nursery_status == thread_clearing_nursery);
            // Clear the nursery, as fast as possible.
            memset (GC_BLOCK_ALLOC_START(nursery_to_zero), 0, GC_BLOCK_ALLOC_SIZE);
            nursery_to_zero->nursery_status = free_nursery;
            num_zeroed++;
        } // Just clear one nursery and then sleep. cycle nursery will wake you up.
          else {
            // suspend_thread_and_continue
            assert(!zero_thread_is_sleeping);            
            zero_thread_is_sleeping = true;
            SuspendThread(Zero_Thread_Handle);
            zero_thread_is_sleeping = false;
//            orp_cout << "zero thread just awoke...." << endl;
            num_zeroed = 0;
        }
    }

    return 1;
}


void start_nursery_zeroing_thread ()
{
    // This is a good place to start the GC zero thread for this particular step.
    
    DWORD ThreadId;

#ifdef ORP_POSIX

    assert (!use_zeroing_thread);
    // zeroing thread only implemented for NT.

#endif

    // Boolean indicating if Zero thread has suspended itself.
    zero_thread_is_sleeping = true;
 
    // The Zero thread will die when this boolean is set...(it is set to true by the step destructor)
    zero_thread_die_signal = false;

    Zero_Thread_Handle = CreateThread(
                                 NULL, 
                                 0, 
                                 zero_thread_func, 
                                 (LPVOID) NULL,     // arg passed to zero_thread_func.
                                 CREATE_SUSPENDED, 
                                 &ThreadId
                                 );
    
    if (Zero_Thread_Handle == NULL) { 
            orp_cout << "Step_Plus_Nursery_Generation: CreateThread() failed...exiting...\n";
            orp_exit(-1);
    }

    // Start the thread.
    if (ResumeThread(Zero_Thread_Handle) == 0xFFFFFFFF) { 
        orp_cout << "Step_Plus_Nursery_Generation: ResumeThread() failed...exiting...\n";
        orp_exit(-1);
    }

    zero_thread_is_sleeping = false;

}


void stop_nursery_zeroing_thread ()
{

    // Bring down the zero thread....

    // Set the die signal 
    zero_thread_die_signal = true;
  
    if (zero_thread_is_sleeping) { 
        
        // Restart the sleeping thread so that it can kill itself.
        if (ResumeThread(Zero_Thread_Handle) == 0xFFFFFFFF) { 
            orp_cout << "~Step_Plus_Nursery_Generation: ResumeThread() failed...exiting...\n";
            orp_exit(-1);
        }
    }
}
#endif
block_info *get_new_nursery()
{
    block_info *this_nursery = p_get_new_block(true, false, true);
    this_nursery->next = NULL;
//    this_nursery->is_in_young_object_space = true;
    assert (this_nursery->train_birthday == 0);
    assert (this_nursery->car_birthday == 0);
    assert (gc_block_status_table[this_nursery->block_status_table_index] == block_in_free);
    this_nursery->in_nursery_p = true; 
    gc_block_status_table[this_nursery->block_status_table_index] = block_in_nursery;
    this_nursery->in_free_p = false;
    this_nursery->nursery_status = free_uncleared_nursery;
    return this_nursery;
}


// we have race conditions here with the verifier if the verify does not have
// all the threads stopped. A nursery can go from free to not free to active and
// a verify can happen before the active (or later) the spent flags can
// be set.

block_info *get_free_nursery ()
{
    block_info *result = NULL;
    int hint = free_nursery_hint;
    // Starting at the hint, loop through the nurseries looking for one that is free.
    int i;
    for (i = hint; i < allocated_nurseries; i++) {
        if (nurseries[i]->nursery_status == free_nursery) {
            free_nursery_hint = i;
            result = nurseries[i];
            if (InterlockedCompareExchange((PVOID *)&(result->nursery_status), (void *)active_nursery, (void *)free_nursery) == (void *)free_nursery) {
                // We win this is now our nursery, it isn't free but it also isn't spent or active.
                // If verifier runs in other thread we can have problems here...        
                assert (result->nursery_status == active_nursery);
                break;  // and just return
            } else {
                result = NULL; // We lose race keep looking.
            }
        }
    }
    if (!result) {
        for (i = 0; i < hint; i++) {
            if (nurseries[i]->nursery_status == free_nursery) {
                free_nursery_hint = i;
                result = nurseries[i];
                if (InterlockedCompareExchange((PVOID *)&(result->nursery_status), (void *)active_nursery, (void *)free_nursery) == (void *)free_nursery) {
                    // We win this is now our nursery, it isn't free but it also isn't spent or active.
                    // If verifier runs in other thread we can have problems here...
                    assert (result->nursery_status == active_nursery);
                    break; // and just return
                } else {
                    result = NULL; // We lose race keep looking.
                }
            }
        }
    }
    
    if (!result) {
        result = get_me_next_nursery_to_zero();
        
        if (result) {
            assert (result->nursery_status == thread_clearing_nursery);
            // Clear the nursery, as fast as possible.
            memset (GC_BLOCK_ALLOC_START(result), 0, GC_BLOCK_ALLOC_SIZE);        
            result->nursery_status = active_nursery;
        }
    }
    if (result) {
        assert (result->nursery_status == active_nursery);
    }
    return result;
}

//
// We probably ran out of nurseries because there were more threads
// than nurseries. In this case, we need to try to add more.
//
void
add_nurseries(int how_many)
{
    int i;
    for (i = allocated_nurseries; i < allocated_nurseries + how_many; i++) {
        assert (nurseries[i] == NULL);
        nurseries[i] = get_new_nursery();
    }
    free_nursery_hint = allocated_nurseries;
    allocated_nurseries = allocated_nurseries + how_many;
}

void notify_out_of_space(unsigned int size);
//
// The thread just ran out of another nursery. Give it a fresh one.
//
block_info *p_cycle_nursery(block_info *p_old_nursery, bool returnNullOnFail)
{
    block_info *result = NULL;
    
    if (p_old_nursery) {
        assert (p_old_nursery->nursery_status == active_nursery);
        p_old_nursery->nursery_status = spent_nursery;
    }
    
    result = get_free_nursery(); // Null means no free nurseries are available.

    if (result == NULL) {
        if (returnNullOnFail) {
            return NULL;
		}
        orp_enable_gc();
        bool stat = p_gc_lock->_lock_enum_or_null(returnNullOnFail);
        orp_disable_gc();
        assert (stat);
        //
	    // Notify GC I'm (still) out of space
		// 
        result = get_free_nursery(); // Check to see if someone has beaten me to a GC and freed up some nurseries.

        while (result == NULL) { 
            // This thread holds lock to notify GC
            notify_out_of_space(0);   // 0 signifies that you need 
                                                    // nurseries and not
                                                    // LOS space.
            result = get_free_nursery();            
        }
        p_gc_lock->_unlock_enum_or_null ();
    }   // or here.
    assert(result->nursery_status == active_nursery);
    // This is where we should clear out the nursery.........
    result->free = GC_BLOCK_ALLOC_START(result);
    result->ceiling = (void *)((POINTER_SIZE_INT)GC_BLOCK_ALLOC_START(result) + (POINTER_SIZE_INT)GC_BLOCK_ALLOC_SIZE - 1);
    result->scan = result->free;
    result->in_nursery_p = true; 
    gc_block_status_table[result->block_status_table_index] = block_in_nursery;
    result->in_los_p = false;
    result->train_birthday = 0;
    result->car_birthday = 0;
    result->number_of_blocks = 1;
    
    if (use_zeroing_thread) {

        // Wake up zeroing thread to clear the next nursery.
#ifdef ORP_POSIX
      orp_cout << "-gc zero_thread not available" << endl;
      orp_exit(37);
#else    
        if (ResumeThread(Zero_Thread_Handle) == 0x0FFffFFff) {
            exit (911);
        }
#endif
    }

    return result;
}

// 
// Move the blocks in the spent nursery into the step and refill the
// nurseries with empty blocks. 
// Return true if you were successful and a full collection is not needed.
// otherwise return false. 
//

// GC_FALSE_COLLECTION

bool move_spent_nurseries_into_step()
{
    int i;
    for (i = 0; i < allocated_nurseries; i++) {
        block_info *this_nursery = nurseries[i];
        if (this_nursery->nursery_status == spent_nursery) {
            assert(this_nursery->list_info == NULL);
            assert(this_nursery->train_birthday == 0);
            assert(this_nursery->car_birthday == 0);
            assert(this_nursery->in_nursery_p);
            assert(!this_nursery->in_step_p);
            block_info *new_nursery = get_new_nursery();
            if (new_nursery == NULL ) {
                // we ran out of blocks.
                return false;
            }
            nurseries[i] = new_nursery;
            this_nursery->c_area_p = false;
            this_nursery->in_step_p = true;

            assert (gc_block_status_table[this_nursery->block_status_table_index] == block_in_nursery);
            this_nursery->in_nursery_p = false;
            gc_block_status_table[this_nursery->block_status_table_index] = block_in_step;
            
            this_nursery->list_info = step;
            this_nursery->next = step->blocks;
            this_nursery->scan = this_nursery->free; // scanning not needed up to free.
            step->blocks = this_nursery;
            this_nursery->nursery_status = bogus_nursery;
            // This is not locally thread safe but we should hold the gc lock.
        }
    }
    return true;
}

Step_Plus_Nursery_Generation::
    Step_Plus_Nursery_Generation(unsigned char generation_number,
                                 Gc_Fast_Hooks *p_gc_hooks,
                                 Gc_Plan       *p_gc_plan,
					             Gc_Interface  *p_container,
								 Generation    *p_superior,
                                 Card_Table    *p_card_table,
								 Block_Store   *p_bs)

                : Step_Generation(generation_number,
                                  p_gc_hooks,
                                  p_gc_plan,
				                  p_container,
				  			      p_superior,
                                  p_card_table,
								  p_bs) 
{
    _number_of_nurseries = 0;
	//
	// Find out how many nurseries we plan to have from the GC plan.
	//
	unsigned long number_of_large_nurseries =
		_p_gc_plan->number_of_large_nurseries();

    add_nurseries(number_of_large_nurseries);
    step = add_new_step();

    // Nursery list such as the free list and the spent are updated atomically using
    // InterlockedCompareExchangePointer so there is no need for critical sections.

    if (use_zeroing_thread) {   
        start_nursery_zeroing_thread();
    }
}

Step_Plus_Nursery_Generation::~Step_Plus_Nursery_Generation() 
{

    if (use_zeroing_thread) {
        stop_nursery_zeroing_thread();
    }

}

#ifdef GC_SAPPHIRE
//
// sapphire_move_spent_nurseries_into_focus_car
//
// Move the blocks in the spent nurseries into the focus car and refill
// the nurseries with empty blocks. Link the free nurseries onto the free_list.
//
bool Step_Plus_Nursery_Generation::sapphire_evict_spent_nurseries_into_focus_car(Car *focus_car)
{
    // First get the spent nurseries, removing them from the _p_spent_nurseries list.
    Nursery  *spent_nursery = (Nursery *)_p_spent_nurseries;
    while (spent_nursery != (Nursery *)InterlockedCompareExchangePointer((PVOID *)&_p_spent_nurseries, 
                                                            (PVOID)NULL, 
                                                            (PVOID)spent_nursery)) {
        // Compare Exchange failed, some other thread changed _p_spent_nurseries, try again.
        spent_nursery = (Nursery *)_p_spent_nurseries;
    }
//    assert (nurseries_are_orphans (spent_nursery));

    Nursery  *first_spent_nursery = spent_nursery;
    while (spent_nursery) {
        // try to evict blocks and and reload with fresh heap space.
        if (! spent_nursery->sapphire_evict_blocks(focus_car)) {
            assert (0); // stop and debug..... We have a lot of orphaned nurseries we need to deal with.
            return false;
       }
       spent_nursery = spent_nursery->p_next_nursery;
    }
    // Move the nursery from the spent to the available list.
    _unlink_spent_nurseries (first_spent_nursery);
    return true;
}


#endif // GC_SAPPHIRE


//
// After a minor stop-the-world collection, Mrl_Gc_Vxx gives us
// an opportunity to do any cleanups. This is done before the
// incremental collection of mature space.
//
void
Step_Plus_Nursery_Generation::cleanup()
{
    int spent_count = 0;
    int active_count = 0;
    int free_count = 0;
    int i;
    for (i = 0; i < allocated_nurseries; i++) {
        block_info *this_nursery = nurseries[i];
        if (this_nursery->nursery_status == spent_nursery) {
            this_nursery->c_area_p = FALSE;
            this_nursery->nursery_status = free_uncleared_nursery;
            spent_count++;
        } else {
            // If a thread is using a nursery clear only the part that has been written to and reset the free pointer.        
            if (this_nursery->nursery_status == active_nursery) {
                int bytes_to_free = (POINTER_SIZE_INT)(this_nursery->free) - (POINTER_SIZE_INT)GC_BLOCK_ALLOC_START(this_nursery);
                if (bytes_to_free) {
                    memset (GC_BLOCK_ALLOC_START(this_nursery), 0, bytes_to_free);
               }
                this_nursery->c_area_p = false;
                this_nursery->free = GC_BLOCK_ALLOC_START(this_nursery);
                active_count++;
            } else {
                if ((this_nursery->nursery_status == free_nursery) 
                    || (this_nursery->nursery_status == free_uncleared_nursery)
                    || (this_nursery->nursery_status == thread_clearing_nursery)) {
                    free_count++;
                } else {
                    assert (0); // We have a nursery that is not active spent or free bummer. 
                }
            }
        }
    }

    if (use_zeroing_thread) {
        // Restart the sleeping GC zero thread.

        if (zero_thread_is_sleeping) { 
	  // Wake up zeroing thread to clear the next nursery.
#ifdef ORP_POSIX
	  orp_cout << "-gc zero_thread not available" << endl;
	  orp_exit(37);
#else    
    //        orp_cout << "cleanup(): restarting zero thread" << endl;

            if (ResumeThread(Zero_Thread_Handle) == 0xFFFFFFFF) { 
                orp_cout << "move_spent_nurseries_into_step: ResumeThread() failed...exiting...\n";
                orp_exit(-1);
            }
#endif
        }
    }

    return;

}


#ifdef GC_SAPPHIRE

Nursery *
Step_Plus_Nursery_Generation::p_sapphire_release_nursery(Nursery *p_old_nursery,
                                                         bool returnNullOnFail)
{
    REDO...
} // p_sapphire_release_nursery

#endif // GC_SAPPHIRE


bool orp_enable_gc();
bool orp_disable_gc();

// This is not thread safe by any means....
bool 
Step_Plus_Nursery_Generation::verify_nurseries ()
{    
    return true;
}

// end file nursery_step_gen.cpp
