// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc_v2/include/gc_hooks.h,v 1.5 2001/11/05 13:36:35 rlhudson Exp $
//


#ifndef _gc_hooks_H_
#define _gc_hooks_H_

#include "platform.h"
#include "object_layout.h"
#include "gc_globals.h"
#include "gc_perf.h"

class Car;
class Gc_Interface;
class Nursery;
class Step;
class Step_Generation;
class Train;
class Train_Generation;
//
// Here is a summary of the GC_DEBUG levels:
//
// GC_DEBUG == 0   : run full speed without any hooks.
//
// GC_DEBUG == 1   : run full speed with minor sanity checks.
//
// GC_DEBUG == 2   : run with minimal performance instrumentation and
//                   minor sanity checks.
//
// GC_DEBUG == 3   : run with full performance instrumentation.
//
// GC_DEBUG == 4   : run with full verification checks.
//                   (And full performance instrumentation.
//                    Unnecessary, but simpler to implement.)
//

//
// Write barrier hooks: these had better be inlined.
//
inline void write_barrier_hook(Java_java_lang_Object *p_base_of_object)
{
#ifdef GC_COPY_V2
    //
    // Verify that no write barriers are generated for GC_COPY_V2
    //
    assert(0);
    cout << "Error: JIT generated an unsolicited write barrier" << endl;
    return; // The simple copy version doesn't use write barriers.
#endif

    return;
}

inline void memcopy_write_barrier_hook(Java_java_lang_Object **pp_src,
                                       Java_java_lang_Object **pp_tgt,
                                       unsigned int       reference_count)
{
#ifdef GC_COPY_V2
    //
    // Verify that no write barriers are generated for GC_COPY_V2
    //
    assert(0);
    cout << "Error: JIT generated an unsolicited memcopy write barrier" << endl;
    return; // The simple copy version doesn't use write barriers.
#endif

    return;
}

//
// Live reference hooks
//
inline void root_set_entry_hook(Java_java_lang_Object **ref)
{
    return;
}

inline void weak_reference_hook(Java_java_lang_Object **ref)
{
    return;
}


//
// Top level hooks: gc subsystem being started or terminated
//

//
// The first time the GC gets control. Currently called when the
// GC subsystem is initialized. Need to move this to the top of the
// startup routine to capture the ORP startup time.
//
void gc_system_start_hook();

//
// The ORP exit routine has just called the GC exit routine to cleanup,
// and any finalizers have already been run.
//
void gc_system_end_hook();


//
// Main stop-the-world entry/exit hooks
//

void gc_start_hook(Gc_Interface *p_gc);

void gc_end_hook(Gc_Interface *p_gc);

//
// Hooks to help monitor enumeration interaction with the ORP
//
inline void enumeration_start_hook(Remembered_Set *p_yos_rs, 
                                   Remembered_Set *p_car_rs,
                                   Remembered_Set *p_weak_refs)
{
    //
    // We are called with the remembered set from the previous
    // collection, before it is zeroed out and passed to the
    // ORP to start this collection cycle.
    //
	QueryPerformanceCounter(&live_reference_enumeration_start_time);
}

inline void enumeration_end_hook(Remembered_Set *p_yos_rs, 
                                 Remembered_Set *p_car_rs,
                                 Remembered_Set *p_weak_refs)
{
    //
    // We are called immediately upon return from the ORP
    // with the remembered set that the ORP just filled up.
    //
	LARGE_INTEGER live_reference_enumeration_end_time;

	QueryPerformanceCounter(&live_reference_enumeration_end_time);

	enumeration_time = 
		get_time_milliseconds(live_reference_enumeration_start_time,
                              live_reference_enumeration_end_time);

    if (stats_gc) {
		orp_cout << "<root enumeration time = " << (int)enumeration_time;
		orp_cout << " ms>" << endl;
	}
}

void incremental_collection_start_hook(Train_Generation *p_tg);

inline void incremental_collection_end_hook(Train_Generation *p_tg)
{
    //
    // We are called just after mature space collection ends.
    // 
#if (GC_DEBUG>0)
    LARGE_INTEGER current_time;

    QueryPerformanceCounter(&current_time);

    incremental_collection_time = 
		get_time_milliseconds(current_time,
		                      incremental_collection_start_time);

    cumulative_incremental_collection_time += incremental_collection_time;

    if (incremental_collection_time < best_case_incremental_collection_time) {
        best_case_incremental_collection_time = incremental_collection_time;
    }

    if (incremental_collection_time > worst_case_incremental_collection_time) {
        worst_case_incremental_collection_time = incremental_collection_time;
    }

#endif
}

//
// Other collection hooks
//
inline void car_reclaim_hook(Car *p_car)
{
    //
    // We are called just before the specified car is reclaimed.
    //
}

inline void root_reference_hook(Java_java_lang_Object **pp_obj_ref)
{
    //
    // We are called just before a reference (either from the ORP
    // live references or due to a cheney scan) is processed for
    // looking for reachable objects.
    //
}

inline void object_tenure_hook(Java_java_lang_Object *p_source, 
                        Java_java_lang_Object *p_target)
{
    //
    // We are called just after an object is tenured.
    //
}
inline void scavenge_hook(Java_java_lang_Object *p_obj,
                          Java_java_lang_Object *p_return_obj)
{
    //
    // We are called just before a scavenge operation is initiated.
    //
}
inline void evict_hook(Java_java_lang_Object **pp_obj_ref)
{
    //
    // We are called just before an evict operation is initiated.
    //
}

inline void large_object_evict_hook(Java_java_lang_Object **pp_obj_ref)
{
    //
    // We are called just before an evict operation on
    // a large (fixed) object.
    //
}


//
// Hooks notifying of space creations/deletions
//

inline void nursery_creation_hook(Nursery *p_nursery)
{
    //
    // We are called just after a new nursery is created and
    // initialized.
    //
}

inline void nursery_deletion_hook(Nursery *p_nursery)
{
    //
    // We are called just before a nursery is deleted.
    //
}

inline void step_creation_hook(Step *p_step)
{
    //
    // We are called just after a new step is created and
    // initialized.
    //
}

inline void step_deletion_hook(Step *p_step)
{
    //
    // We are called just before a step is deleted.
    //
}

inline void car_creation_hook(Car *p_car)
{
    //
    // We are called just after a new car is created and
    // initialized.
    //
}

inline void car_deletion_hook(Car *p_car)
{
    //
    // We are called just before a car is deleted.
    //
}

inline void train_creation_hook(Train *p_train)
{
    //
    // We are called just after a new train is created and
    // initialized.
    //
}

inline void train_deletion_hook(Train *p_train)
{
    //
    // We are called just before a train is deleted.
    //
}

inline void step_generation_creation_hook(Step_Generation *p_sg)
{
    //
    // We are called just after the mature space is created and
    // initialized.
    //
}

inline void step_generation_deletion_hook(Step_Generation *p_sg)
{
    //
    // We are called just before the mature space is deleted.
    //
}

inline void train_generation_creation_hook(Train_Generation *p_tg)
{
    //
    // We are called just after the mature space is created and
    // initialized.
    //
}

inline void train_generation_deletion_hook(Train_Generation *p_tg)
{
    //
    // We are called just before the mature space is deleted.
    //
}

//
// Allocation time hooks
//

inline void movable_object_alloc_hook(void *p_obj)
{
    //
    // We are called just before returning a newly allocated
    // movable object.
    //
}

inline void movable_finalizable_object_alloc_hook(void *p_obj)
{
    //
    // We are called just before returning a newly allocated
    // movable and finalizable object.
    //
}

inline void fixed_object_alloc_hook(void *p_obj)
{
    //
    // We are called just before returning a newly allocated
    // fixed object.
    //

}

inline void fixed_finalizable_object_alloc_hook(void *p_obj)
{
    //
    // We are called just before returning a newly allocated
    // fixed object.
    //
}

//
// Other ORP activity hooks
//
inline void before_enumerate_live_refs_hook()
{
    //
    // We are called just before the ORP is asked by the GC
    // to enumerate all live references.
    //
}

inline void after_enumerate_live_refs_hook()
{
    //
    // We are called just after the ORP returns all the 
    // live references.
    //
}

#endif // _gc_hooks_H_
