// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/include/mrl_gc_v1.h,v 1.2 2001/08/03 03:13:44 xli18 Exp $
//


#ifndef _mrl_gc_v1_H_
#define _mrl_gc_v1_H_

//
// Header for the Implementation of the version 1.0 MRL GC with multiple
// generations, large object space, and incremental collection of mature
// object spaces.
//

#include "nursery.h"

#include "gc_for_orp.h"

#include "Block_Store.h"
#include "card_table.h"
#include "gc_interface.h"
#include "generation.h"
#include "remembered_set.h"
#include "gc_space.h"
#include "los_container.h"

#ifdef ORP_POSIX
#include "linux_api.h"
#endif

#ifdef ORP_NT
#include "Win32_api.h"
#endif

#include "nursery_step_gen.h"
#include "car.h"
#include "train.h"

#define MAXIMUM_GENERATIONS 8

class Car;
//class Directory;
class Gc_Fast_Hooks;
class Gc_Plan;
class Gc_Space;
class LOS_Container;
class Step_Plus_Nursery_Generation;
class Train;

extern bool _collection_in_progress;

#ifdef ORP_POSIX
class Mrl_Gc_V1 : public Gc_Interface, Linux_Api {
#else
class Mrl_Gc_V1 : public Gc_Interface, Win32_Api {
#endif

public:
    Mrl_Gc_V1(ORP_Control *p_orp_control,
              char *p_plan_file_name);

    virtual ~Mrl_Gc_V1();


    //
    // A contained generation (the youngest) informs us that
    // it is out of space (nurseries exhausted)
    //

    //
    // By convention this routine is called with the gc_lock held by
    // the caller.
    //
    void notify_out_of_space(unsigned int size) {
        reclaim_heap(size, false);
    }

    //
    // The ORP is forcing a barrier collection:
    //
    void reclaim_heap(unsigned int size, bool forced);

    // The thread is telling us that the nursery is exhausted.
    // Add this to the used list and give the thread a new one.
    // If all nurseries are exhausted, trigger a collection.
    virtual Nursery *
    p_cycle_nursery(Nursery *p_nursery,
                    bool returnNullOnFail);

    //
    // The ORP is enumerating a weak reference.
    //
    virtual void gc_add_weak_reference(Java_java_lang_Object **ref) {
#if (GC_DEBUG>2)
        assert(_collection_in_progress);
#endif
        assert (0); // This is code for a better weak ref idea that died when
                    // Java 2 introduced their notion of weak reference. Hayes was right
                    // about weak references, I should have paid attention.
                    // Break and debug, we might want to treat this as a strong reference.
        _p_weak_references->add_entry((Java_java_lang_Object **)ref);
    }
    //
    // The ORP is providing us with yet another live reference.
    // We add it to our root set. This happens at the beginning
    // of a stop-the-world phase, when we call into the ORP
    // to enumerate all live references.
    //
//    inline
    void gc_add_root_set_entry(Java_java_lang_Object **ref);
#if 0 // do not inline for now.
    {
#if (GC_DEBUG>2)
        assert(_collection_in_progress);
#endif
        
        // Split the remembered set entry into one of four different
        // catagories.

        // Is is NULL.
        if (*ref == NULL) {
            return;
        }

        // Is it a pointer into the youngest generation?
        if (_p_young_generation->is_address_in_my_generation (*ref)) {
           _p_refs_to_young_object_space->add_entry ((Java_java_lang_Object **)ref);
        }
        
        // We know it isn't null or a pointer into YOS so it is a pointer into
        // MOS. Is it to the target car?

        if (_p_focus_car->is_address_in_this_car (*ref)) {
            _p_refs_to_focus_car->add_entry ((Java_java_lang_Object **)ref);
            ref_to_focus_train_exists = true;
        } else {
        
            // Is it into the target train?
            if (!ref_to_focus_train_exists) {
                if (_p_focus_train->is_my_address(*ref)) {
                    ref_to_focus_train_exists = true;
                }
            }
        }
//      _p_root_set->add_entry((Java_java_lang_Object **)ref);
    
    }
#endif // 0 moved to the cpp file.

    
    virtual Java_java_lang_Object *gc_pinned_malloc(long size,
                                   Partial_Reveal_VTable *p_VTable,
                                   bool return_null_on_fail,
                                   bool double_align
                                   ) {
        return _p_los_container->gc_pinned_malloc(size, 
                                                  p_VTable,
                                                  return_null_on_fail,
                                                  double_align                                            
                                                  );
    }

    virtual Java_java_lang_Object *gc_pinned_malloc_noclass(long size) {
        return _p_los_container->gc_pinned_malloc_noclass(size);
    }
 
    
    void *gc_large_malloc(long size,
                          Partial_Reveal_VTable *p_VTable) {
        //
        // Currently we just call pinned blindly. 
        // 'Large' doesn't imply 'pinned'.
        // May want to add heuristics here.
        //
        return _p_los_container->gc_pinned_malloc(size, 
                                                  p_VTable,
                                                  false,
                                                  false
                                                  );
    }

 

    void free(Java_java_lang_Object *p_obj);

    void freeze();

#if (GC_DEBUG>3)
    //
    // Debug time routine for inspecting the state of the collector.
    //
    virtual void inspect(unsigned int level);
#endif // _DEBUG

    //
    // Check that there are no pending cheney scans in any generation.
    //
    bool no_pending_cheney_scans();
    //
    // An older generation is telling us that it has discovered a
    // reference to a younger generation in the process of importing
    // an object. 
    //
    void notify_new_old_to_young_reference(Java_java_lang_Object **pp_obj_ref);

    //
    // This method is called during system exit if the application
    // requested this feature.
    //
    void run_all_finalizers();


    //
    // The ORP is passing on to the GC the verbosity level that
    // the user has specified on the command line.
    //
    void set_verbose_level(unsigned int level);

    //
    // Take a spent nursery and store it away till 
    // the next reclaim.
    //
    virtual void store_spent_nursery(Nursery *p_nursery);

    //
    // Subsequent to a freeze, the mutator is now 
    // telling us that all is clear.
    //
    void thaw() {
        orp_resume_threads_after(false);
    }

    //
    // ORP notification to GC to clean up.
    //
    virtual void wrapup_gc();

#if (GC_DEBUG>3)
    //
    // A debug-time interface to tell the ORP if a reference
    // has already been enumerated. It only works when the
    // ORP has control and is in the process of enumeration.
    //
    inline bool is_recorded_live_reference(Java_java_lang_Object **pp_obj_ref) { 
        
        if (_p_refs_to_focus_car->is_present(pp_obj_ref)) {
            return true;
        }
        
        if (_p_refs_to_young_object_space->is_present(pp_obj_ref)) {
            return true;
        }

        return false;
    }
#endif

    // The Mrl_Gc_V1 sets up and maintains a focus car which is usually nil or the
    // oldest car in the oldest train... If Nil then no collection of a car is going on.
    Car *p_get_focus_car();

    Train *p_get_focus_train();

#ifdef OBJECT_SPLITTING
	int get_large_nursery_size()
	{
		// Return the size of the large nursery.
		return _p_gc_plan->large_nursery_block_size_bytes();
	}
#endif // OBJECT_SPLITTING

private:

    Mrl_Gc_V1();

    //
    // The block store manages free blocks which are the basic
    // units of storage used by cars, nurseries, steps, etc...
    //
    Block_Store *_p_block_store;

    //
    // The card table is used for card marking write barriers.
    //
    Card_Table  *_p_card_table;

//#if (GC_DEBUG>2)
    //
    // This is a fast verification guard to ensure that the
    // JIT is not enumerating at an inappropriate time.
    // This could result in stale references.
    //
//  extern bool _collection_in_progress;
//#endif

    //
    // The plan object contains the various GC size parameters.
    //
    Gc_Plan            *_p_gc_plan;

    //
    // Routine to ask the ORP for all live references.
    //
    void _get_orp_live_references();
    void enumerate_weak_reference_queues ();

    //
    // The LOS holds objects too big (hence expensive) to move,
    // or which can't move due to their being position dependent.
    //
    LOS_Container *_p_los_container;

    //
    // The mature generation which is implemented using a train.
    //
    Train_Generation *_p_mature_generation;
    //
    // The remembered set used to store all the live
    // references enumerated by the ORP.
    //
//    Remembered_Set *_p_root_set;

    //
    // The remembered set used to store all the weak
    // references enumerated by the ORP.
    //
    Remembered_Set *_p_weak_references;

    // There are three Remembered Sets that need to be maintained for
    // the train algorithm. Since the other algorithms are just subsets
    // of the train algorithm these sets also work for them.

    Remembered_Set *_p_refs_to_young_object_space;
    Remembered_Set *_p_refs_to_focus_car;
//    Remembered_Set *_p_refs_to_focus_car_from_focus_train;
    bool ref_to_focus_train_exists;

    Car *_p_focus_car;
    Train *_p_focus_train;

 //   unsigned int _car_size_car_blocks;    
    unsigned long _card_size_bytes;

    unsigned long _initial_heap_size_bytes;
    unsigned long _current_heap_size_bytes;
    unsigned long _final_heap_size_bytes;
    //
    // After a stop-the-world collection, give all
    // generations an opportunity to clean up and
    // then re-start the ORP.
    //
    void _clean_up_and_resume_orp();

    //
    // Execute a minor stop-the-world collection.
    //
    Remembered_Set *_execute_minor_collection(Remembered_Set *_p_refs_to_young_object_space,
                                              Remembered_Set *p_weak_refs,
                                              bool *p_is_train_alive);
  
    //
    // Execute one unit of incremental collection of mature space.
    //
    void _execute_incremental_collection_of_mature_space(Remembered_Set *p_y_to_o_rs,
                                                         Remembered_Set *p_weak_refs,
                                                         bool is_train_alive);
    //
    // This private method is called during the creation of the
    // Mrl_Gc_V1 object in order to set up the hooks and 
    // create the plan object.
    //
    void _initialize_plan_and_hooks(char *p_plan_file_name);

    //
    // The hooks object is used for performance measurement and
    // debugging.
    //
    Gc_Fast_Hooks *_p_gc_hooks;

    unsigned short _maximum_number_of_generations;

    //
    // Number of generations in this GC implementation.
    //
    unsigned int _number_of_generations;

};

#endif // _MRL_GC_V1_H_
