// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/include/gc_plan.h,v 1.1.1.1 2001/07/23 07:25:39 xli18 Exp $
//


//
// A "plan" sets various parameters in the moving collector,
// such as the number of steps, the sizes of nurseries, etc.
// There is a default plan that is hardwired into the source
// code (this file.) In addition, if the user places a ".plan"
// file in the same directory as the ORP executable, that file
// is read, and the values replace the defaults.
//


#ifndef _gc_plan_h_
#define _gc_plan_h_

//
// Define how many bits we shift an address to get its card index. So for
// a 256 byte card we shift by 8 bits. This number should be between 
// 8 and 11 to correspond with the 256 to 1024 byte card size.
//
// This needs to be a compile time constant for speed.
//
 
#include "gc_for_orp.h"
 

extern bool global_verbose_gc;

class Gc_Plan {
public:
    Gc_Plan(char *p_plan_file_name) {
		_set_defaults();
		_read_plan_file(p_plan_file_name);
	}

	Gc_Plan() {
		_set_defaults();
	}


    virtual ~Gc_Plan() {

    }

	//
	// The number of bytes to extend the heap by when
	// we run out of space (or fragment sufficiently).
	//
	int heap_extension_increment_bytes() {
		return _heap_extension_increment_bytes;
	}
    //
    // The size in bytes of the blocks - the basic unit of
    // allocation in the moving collector.
    //
    unsigned int sub_block_size_bytes() {
        return _sub_block_size_bytes;
    }
	//
	// The size of blocks used by cars.
	//
	unsigned int car_block_size_bytes() {
		return _car_block_size_sub_blocks * _sub_block_size_bytes;
	}

	//
	// The size of blocks used by nurseries.
	//
	unsigned int nursery_block_size_bytes() {
		return _nursery_size_sub_blocks * _sub_block_size_bytes;
	}
	//
	// The size of blocks used by nurseries.
	//
	unsigned int large_nursery_block_size_bytes() {
		return _large_nursery_size_sub_blocks * _sub_block_size_bytes;
	}

	//
	// The size of blocks allocated to the steps. If blocks are promoted into
    // a step then they can be of different sizes.
	//
	unsigned int step_block_size_bytes() {
		return _step_block_size_sub_blocks * _sub_block_size_bytes;
	}

    //
    // The number of (non-contiguous) blocks in a car.
    //
    unsigned int car_size_blocks() {
        return _car_size_car_blocks;
    }
    //
    // The card size for card marking schemes.
    //
    unsigned long card_size_bytes() {
        return _card_size_bytes;
    }

	unsigned long initial_heap_size_bytes()
	{
		return _initial_heap_size_bytes;
	}

	unsigned long final_heap_size_bytes()
	{
		return _final_heap_size_bytes;
	}

    //
    // The size of the large object space.
    //
    unsigned long los_block_size_bytes() {
        return _los_block_size_bytes;
    }

    //
    // Sometimes we need to modify the plan by increasing the size of los spaces
    //
    void set_los_block_size_bytes (unsigned long size) {
        _los_block_size_bytes = size;
    }

    //
    // The size of the nurseries in bytes
    //
    unsigned long total_nurseries_size_bytes () {
        return _number_of_nurseries * _nursery_size_sub_blocks * _sub_block_size_bytes;
    }

    //
    // The threshold above which an object is deemed "large"
    // (has to be at least smaller than a block.)
    //
    unsigned long los_threshold_bytes() {
        return _los_threshold_bytes;
    }
    //
    // The number of nurseries in the pool.
    //
    unsigned long number_of_nurseries() {
        return _number_of_nurseries;
    }
    //
    // The number of large nurseries in the pool.
    //
    unsigned long number_of_large_nurseries() {
        return _number_of_large_nurseries;
    }
    //
    // The number of blocks in a large nursery
    //
    unsigned long large_nursery_size_sub_blocks() {
        return _large_nursery_size_sub_blocks;
    }
    //
    // The number of steps in the young space.
    // (not including the nurseries, which are not 
    // considered to be "steps" in our terminology.)
    //
    unsigned long number_of_steps() {
        return _number_of_steps;
    }
    //
    // The number of (contiguous) blocks in any nursery.
    // WARNING: THIS NEEDST TO BE UPDATED TO REFLECT VARIABLE SIZE NURSERIES.
	//
    unsigned long nursery_size_blocks() {
        return _nursery_size_blocks;
    }

    // 
    // The number of trains to maintain.
    //
    unsigned long minimum_number_of_trains() {
        return _minimum_number_of_trains;
    }

    //
    // The number of large object stores, each with its own lock
    //
    unsigned long number_of_large_object_stores() {
        return _number_of_large_object_stores;
    }

	unsigned int verbose_gc;

private:

	int _heap_extension_increment_bytes;

    int  _sub_block_size_bytes;
 
    int  _nursery_size_sub_blocks;

    int  _large_nursery_size_sub_blocks;

    int  _step_block_size_sub_blocks;
 
	int  _car_block_size_sub_blocks;

	int _car_size_car_blocks;

    int _los_block_size_bytes;

    unsigned long  _card_size_bytes;
 
	unsigned long _initial_heap_size_bytes;

	unsigned long _final_heap_size_bytes;

    unsigned long _los_threshold_bytes;

    unsigned long _number_of_nurseries;

	//
	// These first N nurseries get really large blocks.
	//
	unsigned long _number_of_large_nurseries;

    unsigned long _number_of_steps;

	unsigned long _nursery_size_blocks;

    int _nursery_size_bytes;
	//
	// The first N nurseries have this many 
	//
	int _large_nursery_size_bytes;

    unsigned long _minimum_number_of_trains;

    unsigned long _number_of_large_object_stores;

	void _read_plan_file(char *p_plan_file_name) {
		// TBD
	}

	void _set_defaults();
};

#endif // _gc_plan_h_
