// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/gc_perf.cpp,v 1.1.1.1 2001/07/23 07:25:39 xli18 Exp $
//

 
#include "gc_for_orp.h"
 
#include "platform.h"
#include "gc_perf.h"
#include "gc_globals.h"
#include "train_generation.h"


#ifdef CLASS_DEMOGRAPHICS
void record_for_class_demographics(VTable *p_vt)
{
    p_gc_lock->_lock_enum_or_null(false);
    p_loaded_vtable_directory->increment_entry((void *)p_vt);
    p_gc_lock->_unlock_enum_or_null();
}
void delete_for_class_demographics(VTable *p_vt)
{
    p_gc_lock->_lock_enum_or_null(false);
    p_loaded_vtable_directory->decrement_entry((void *)p_vt);
    p_gc_lock->_unlock_enum_or_null();
}
bool walk_report_dead_object(Object_Gc_Header *p_hdr, Remembered_Set   *p_rs)
{

    Java_java_lang_Object *p_obj = get_object_from_gc_header(p_hdr);

    if (is_object_unmarked(p_obj)) {
        delete_for_class_demographics(p_obj->vt);
    }

    return true;
}

#endif // CLASS_DEMOGRAPHICS

#ifdef CLASS_DEMOGRAPHICS
void dump_class_demographics()
{
    VTable *p_vtable;

    cout << "Dumping Class Demographics:" << endl;

    int count;

    p_loaded_vtable_directory->rewind();

    while ((p_vtable = 
             (VTable *)p_loaded_vtable_directory->next(&count)) != NULL) {

#if (GC_DEBUG>0)
        assert(count >= 0);
        assert(p_vtable);
#endif
        if (count > 0) {
            cout << p_vtable->clss->name->bytes << " " << count << endl;
        }
    }

    cout << "END Class Demographics" << endl;
}

#endif


//
// The performance frequency of the high-resolution performance counter.
//
LARGE_INTEGER performance_frequency;

//
// The instant the GC subsystem started initialization.
//
LARGE_INTEGER gc_system_init_time;

//
// The instant the most recent stop-the-world collection started.
//
LARGE_INTEGER gc_start_time;

//
// The time spent in the most recent stop-the-world collection.
//
unsigned long gc_time;

//
// The cumulative time spent in all past stop-the-world collections.
//
unsigned long cumulative_gc_time;

//
// The best case gc pause time so far.
//
unsigned long best_case_gc_time;

//
// The worst case gc pause time so far.
//
unsigned long worst_case_gc_time;

//
// The instant the most recent minor collection started.
//
LARGE_INTEGER minor_collection_start_time;

//
// The time spent in the most recent minor collection.
//
unsigned long minor_collection_time;

//
// The cumulative time spent in minor collections so far.
//
unsigned long cumulative_minor_collection_time;

//
// The worst case minor collection time.
//
unsigned long worst_case_minor_collection_time;

//
// The best case minor collection time.
//
unsigned long best_case_minor_collection_time;

//
// The instant the most recent incremental mature collection started.
//
LARGE_INTEGER incremental_collection_start_time;

//
// The time spent in the most recent mature incremental collection.
//
unsigned long incremental_collection_time;

//
// The total time spent in all mature incremental collections.
//
unsigned long cumulative_incremental_collection_time;

//
// The worst case incremental collection time.
//
unsigned long worst_case_incremental_collection_time;

//
// The best case incremental collection time.
//
unsigned long best_case_incremental_collection_time;

//
// The instant the most recent request to the ORP for 
// enumerating live references was made.
//
LARGE_INTEGER live_reference_enumeration_start_time;

//
// The time taken to do the most recent enumeration of live refs.
//
unsigned long enumeration_time;

//
// Thread enumeration timers for sapphire timings.
//

unsigned long thread_enum_time;

//
// The cumulative time spent in enumerating threads.
//
unsigned long cumulative_thread_enum_time;

//
// The best case thread enumeration pause time so far.
//
unsigned long best_case_thread_enum_time;

//
// The worst case thread enumeration pause time so far.
//
unsigned long worst_case_thread_enum_time;

//
// The instant the most recent thread enumeration started.
//
LARGE_INTEGER thread_enum_start_time;

unsigned long number_of_thread_enums = 0;

void initialize_gc_performance_counters()
{
    QueryPerformanceFrequency(&performance_frequency);

    cumulative_gc_time                     = 0;
    cumulative_minor_collection_time       = 0;
    cumulative_incremental_collection_time = 0;

    worst_case_minor_collection_time       = 0;
    worst_case_incremental_collection_time = 0;
    worst_case_gc_time                     = 0;

    best_case_minor_collection_time       = 99999999;
    best_case_incremental_collection_time = 99999999;
    best_case_gc_time                     = 99999999;

    cumulative_thread_enum_time             = 0;
    worst_case_thread_enum_time                  = 0;
    number_of_thread_enums                  = 0;

}

//
// Top level hooks: gc subsystem being started or terminated
//

void gc_system_start_hook()
{
    //
    // The ORP is telling the GC to initialize.
    // We are called just after the GC initializes itself.
    //

    initialize_gc_performance_counters();
    QueryPerformanceCounter(&gc_system_init_time);
}

void gc_system_end_hook()
{
    //
    // The ORP is telling the GC to clean up before exiting.
    // We are called just before the GC cleans up and returns.
    // WE ARE CALLED BEFORE FINALIZERS ARE RUN (IF NEEDED).
    //

    // if counters (e.g., QueryPerformanceCounter) aren't supported
    // divide by 0 may occur

#if 1 // (GC_DEBUG>5)
    LARGE_INTEGER gc_system_end_time;

    QueryPerformanceCounter(&gc_system_end_time);

    //
    // We calculate the ORP runtime from the instant the GC
    // subsystem was initialized. This is not completely accurate,
    // and we need the ORP to start the counter the instant it 
    // becomes conscious.
    //
    unsigned long orp_runtime = get_time_milliseconds(gc_system_init_time,
        gc_system_end_time);
    //
    // The GC overhead is the percentage of the ORP runtime
    // that was spent in the garbage collector while the ORP
    // Java threads were frozen. Currently the following is an
    // approximation which needs to be refined.
    //
    if (orp_runtime == 0) {
//        orp_cout << "GC Runtime not available." << endl;
    } else {
        assert (orp_runtime);
        unsigned long a = cumulative_gc_time + orp_runtime;
	assert(a);
        unsigned long b = cumulative_gc_time / a;
        double gc_overhead_double = 
            ((double)cumulative_gc_time / 
            (double)(cumulative_gc_time + orp_runtime))
            * 100.0;
        unsigned long gc_overhead = (unsigned long)gc_overhead_double;
        
        if (global_verbose_gc) {
            //
            // Display ORP and GC runtime statistics.
            //
            cout << "<END runtime = " << orp_runtime << " ms,";
            cout << " GC overhead  = " << gc_overhead_double << " %" << endl;
        }
    }
#endif
}
//
// Main stop-the-world entry/exit hooks
//

void gc_start_hook(Gc_Interface *p_gc)
{
    //
    // We are called before any stop-the-world activity is initiated
    //

	if (stats_gc) {
#ifdef GC_SAPPHIRE
	    orp_cout << endl;
#endif
		orp_cout << "<starting gc #:" << incremental_collection_count++;

        //
        // Debugging help for the JIT:
        //
        orp_cout << ", base = " << p_gc_base << ", ceiling = " << p_gc_ceiling << ">" << endl;
	}

#if 1 // (GC_DEBUG>5)
    QueryPerformanceCounter(&gc_start_time);
#endif

    LARGE_INTEGER current_time;

    QueryPerformanceCounter(&current_time);
#ifndef GC_SAPPHIRE
    gc_time = get_time_milliseconds(gc_start_time,
                                    current_time);
#endif
//    orp_cout << "Empty latency = " << gc_time << endl;
}

void gc_end_hook(Gc_Interface *p_gc)
{
    //
    // We are called just before the ORP threads are thawed.
    //

    LARGE_INTEGER current_time;

    QueryPerformanceCounter(&current_time);

    gc_time = get_time_milliseconds(gc_start_time,
                                    current_time);

    cumulative_gc_time += gc_time;

    if (gc_time < best_case_gc_time) {
        best_case_gc_time = gc_time;
    }

    if (gc_time > worst_case_gc_time) {
        worst_case_gc_time = gc_time;
    }



	if (stats_gc) {
#ifdef GC_SAPPHIRE

		orp_cout << endl << "<ending gc #: "<< incremental_collection_count << " , latency = " << gc_time;
		orp_cout << " ms, best = " << best_case_gc_time;
		orp_cout << " ms, worst = " << worst_case_gc_time << " ms>" << endl;
#else
		orp_cout << "<ending gc, latency = " << (int)gc_time;
		orp_cout << " ms, best = " << (int)best_case_gc_time;
		orp_cout << " ms, worst = " << (int)worst_case_gc_time << " ms>" << endl;
#endif
    }

#ifdef CLASS_DEMOGRAPHICS
    dump_class_demographics();
#endif
}

//
// Main thread enumeration entry/exit hooks
//

void thread_enum_start_hook()
{
    // Increment the number of enumerations.

     number_of_thread_enums++;
    //
    // We are called before a thread enumeration activity is initiated
    //

#if 1 // (GC_DEBUG>5)
    QueryPerformanceCounter(&thread_enum_start_time);
#endif

    LARGE_INTEGER current_time;

    QueryPerformanceCounter(&current_time);

    thread_enum_time = get_time_microseconds(thread_enum_start_time,
                                    current_time);

//    orp_cout << "Empty latency = " << thread_enum_time << endl;
}
#ifdef GC_SAPPHIRE

int unsafe_counter = 0;

int total_unsafe_counter = 0;

void thread_enum_end_hook(thread_suspend_state how_suspended)
#else
void thread_enum_end_hook()
#endif
{
    //
    // We are called just before the ORP threads are thawed.
    //

    LARGE_INTEGER current_time;

    QueryPerformanceCounter(&current_time);

    thread_enum_time = get_time_microseconds(thread_enum_start_time,
                                    current_time);

    cumulative_thread_enum_time += thread_enum_time;

    if (thread_enum_time < best_case_thread_enum_time) {
#ifdef GC_SAPPHIRE
        best_case_thread_enum_time = thread_enum_time;
#else
        best_case_thread_enum_time = gc_time;
#endif
    }

    if (thread_enum_time > worst_case_thread_enum_time) {
        worst_case_thread_enum_time = thread_enum_time;
    }

//	if (stats_gc) {
#ifdef GC_SAPPHIRE
        if (thread_enum_time >= 1) {
    		orp_cout << endl << "<thread enum number " << (int)number_of_thread_enums ;
            orp_cout << " has latency = " << (int)thread_enum_time;
		    orp_cout << " Ms, worst = " << (int)worst_case_thread_enum_time;
            orp_cout << " Ms, cumulative is " << cumulative_thread_enum_time << " state is " ;
            if (how_suspended == SUSPENDED_AT_SAFE_POINT) {
                orp_cout << " safe after trying " << unsafe_counter << " times.>" << endl;
                unsafe_counter = 0;
            }
            if (how_suspended == SUSPENDED_AT_UNSAFE_POINT) {
                unsafe_counter++;
                total_unsafe_counter++;
                orp_cout << " unsafe time " << total_unsafe_counter << ">" << endl;
            }
            if (how_suspended == SUSPENDED_BY_ENABLED_WILL_BLOCK) {
                orp_cout << " enabled after trying " << unsafe_counter << " times.>" << endl;
                unsafe_counter = 0;
            }
        } else {
            number_of_thread_enums--;
        }
#else
        if (thread_enum_time >= 1) {
    		orp_cout << "<thread enum number " << (int)number_of_thread_enums << " has latency = " << (int)thread_enum_time;
		    orp_cout << " Ms, worst = " << (int)worst_case_thread_enum_time << " Ms>" << endl;
        } else {
            number_of_thread_enums--;
        }
#endif
//    }

#ifdef CLASS_DEMOGRAPHICS
    dump_class_demographics();
#endif
}

void incremental_collection_start_hook(Train_Generation *p_tg)
{
    //
    // We are called just before mature space collection starts.
    //

#if (GC_DEBUG>0)
    QueryPerformanceCounter(&incremental_collection_start_time);
#endif

#if 0 //(GC_DEBUG>3) // TEMPORARILY TURNED THIS OFF: RDS 10/5/98
	//
	// At debug time we do a painfully slow scan of all of mature
	// space to determine if there are any mature to young pointers
	// that didn't get recorded by the write barriers.
	//
	p_tg->verify_all_mature_to_young_references_are_recorded();
#endif
}
// end file gc\exit.cpp
