// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/block_store.cpp,v 1.4 2001/09/04 01:04:58 xli18 Exp $
//
           
#include "gc_for_orp.h"
 
#include "Block_Store.h"
#include "generation.h"
#include "gc_space.h"
#include "obsolete_space.h"
#include "car.h"
#include "train.h"
 
#include "gc_plan.h"
#include "gc_hooks.h"

#ifdef GC_PT_WB
#include <windows.h>
#include <winbase.h>
#include <winnt.h>
#include <stdio.h>    
#endif

extern Card_Table *p_global_card_table_manager;

//
// There are five tables of meta information which are used by the block
// store to manage blocks of varying sizes.
// 
// The _p_block_container_table contains a NULL for an unallocated sub or
// super block. If allocated, an entry for a super block contains a pointer
// to the container owning it. An entry for any sub block in the super block
// is 0xFFFF FFFF.
//
// The _p_block_size_blocks_table contains the number of contiguous sub-blocks
// if it is the first in a cluster of sub-blocks. Otherwise it is -1.
//
// The _p_block_start_id_table contains the index of the super block
// of this sub block, for allocated or free blocks.
//
// The _p_block_end_table is indexed by the sub block it and contains the 
// last byte in this super block.
// The last byte in the sub block can be calculated directly from an address.
//
// The _p_block_generation_table contains -1 for unallocated sub blocks,
// 0 for sub blocks that are a part of the young generation, or an
// encoding of the car and train if they are in mature space.
//
void
Block_Store::_allocate_block_tables()
{
	//
	// Take some space off the top of the GC heap for the block container table.
	//

    _p_block_container_table = (Gc_Space **)_gc_free_list;

    //
	// Bump the free list beyond the block container table.
	//
	_gc_free_list  = (void *)((Byte *)_gc_free_list + _table_size_bytes);

    //
	// Take some space off the top of the GC heap for the block size table.
	//
	_p_block_size_blocks_table = (int *)_gc_free_list;
	//
	// Bump the free list beyond the block size table.
	//
	_gc_free_list  = (void *)((Byte *)_gc_free_list + _table_size_bytes);

	//
	// Take some space off the top of the GC heap for the block start table.
	//
	_p_block_start_id_table = (int *)_gc_free_list;
	//
	// Bump the free list beyond the block offset table.
	//
	_gc_free_list  = (void *)((Byte *)_gc_free_list + _table_size_bytes);

   	//
	// Take some space off the top of the GC heap for the block end table.
	//
    // This is not used currently but if we need to speed up locating the end of a block
    // we will take an address and shift it the size of a sub block and index
    // this table.
    //
	_p_block_end_table = (POINTER_SIZE_INT *)_gc_free_list;
	//
	// Bump the free list beyond the block offset table.
	//
	_gc_free_list  = (void *)((Byte *)_gc_free_list + _table_size_bytes);

	//
	// Take some space off the top of the GC heap for the generation table.
	//
	_p_block_generation_table = (int *)_gc_free_list;
	//
	// Bump the free list beyond the block offset table.
	//
	_gc_free_list  = (void *)((Byte *)_gc_free_list + _table_size_bytes);

    //
    // Be sure to look at _number_of_tables below if we add another table here.
    //

	//
	// Make sure it is aligned on a super block boundary, this is needed
    // so that we can do shifts and clears to quickly figure out several things
    // such as block starts and block ends as well as comparing source and destination
    // of pointers to see if they are in the same super block.
	//
    
    int pages_in_sub_block = _sub_block_size_bytes / _machine_page_size_bytes;

    //
    // Make sure this all works 
    // ie sub_block_size_bytes is a multiple of _machine_page_size_bytes
    //
    assert ((_sub_block_size_bytes / pages_in_sub_block) == _machine_page_size_bytes);


    _gc_free_list  = _p_page_align_up(_gc_free_list, pages_in_sub_block);

//    _gc_free_list = _p_super_block_align_up (_gc_free_list, 1);

	//
	// At higher debug levels, the following variable is used for doing
	// bounds checking of the tables created above.
	//
	_maximum_table_index = _table_size_bytes / sizeof(void *);

	return;
}

//
// The following routine is called when the block store is created.
// It is used to create one large unallocated super-block. 
// The fact that it is unallocated is indicated by the fact that the
// first sub-block has a NULL container, and all the block sizes ,
// except the first, are -1.
//
//
void
Block_Store::_initialize_block_tables(void *p_start_address,
									  void *p_last_block)
{
	int start_block_id = get_sub_block_id_of_address(p_start_address);
	int end_block_id   = get_sub_block_id_of_address(p_last_block);

#if (GC_DEBUG>2)
	assert((start_block_id >= 0) && (start_block_id < _maximum_table_index));
	assert((end_block_id >= 0) && (end_block_id < _maximum_table_index));
#endif

	_total_current_sub_blocks = 0;	
	
	for (int block_idx = start_block_id;
	         block_idx <= end_block_id;
			 block_idx++) {
	    _p_block_container_table[block_idx]   = (Gc_Space *)POINTER_SIZE_MINUS_ONE;
		_p_block_size_blocks_table[block_idx] = -1;
		_p_block_start_id_table[block_idx]    = 0;
        _p_block_generation_table[block_idx] = -1;
        _p_block_end_table[block_idx] = (POINTER_SIZE_INT)((Byte *)p_last_block + _sub_block_size_bytes - 1);
		_total_current_sub_blocks++;
	}
#if (GC_DEBUG>2)
	assert(_total_current_sub_blocks>0);
#endif
	_p_block_container_table[0]   = NULL;
	_p_block_size_blocks_table[0] = _total_current_sub_blocks;

	_free_sub_blocks = _total_current_sub_blocks;

    return;
}

//
// This routine is called when the heap is extended, so that the block tables
// can be appropriately updated.  The input arguments are the start of the
// new heap, and the address of the last sub block.
// The routine returns the total number of sub blocks after the heap has been
// extended.
//
int
Block_Store::_extend_block_tables(void *p_start_address,
								  void *p_last_block)
{
	int new_first_block_id   = get_sub_block_id_of_address(p_start_address);
	int new_last_block_id    = get_sub_block_id_of_address(p_last_block);
	int number_of_new_blocks = new_last_block_id - new_first_block_id + 1;
    int last_super_block_id = _p_block_start_id_table[_total_current_sub_blocks - 1];
    // _p_block_end_table is only updated for allocated blocks.
    
	//
	// See if the last super block in the current table is unallocated.
	// (If so, need to merge with the new super block.)
	//
#if (GC_DEBUG>2)
	// ensure block tables are initialized and the heap is not empty.
	assert(_total_current_sub_blocks > 0);
	// ensure we are adding a legal number of new sub blocks.
	assert(number_of_new_blocks > 0);
	// ensure last_super_block_id is a valid block index.
	assert((last_super_block_id >= 0) && (last_super_block_id < _maximum_table_index));
	// ensure last_super_block_id has a valid super block (allocated or otherwise)
#ifdef POINTER64
    assert(_p_block_container_table[last_super_block_id] != (Gc_Space *)0xFFFFFFFFffffffff);
#else
    assert(_p_block_container_table[last_super_block_id] != (Gc_Space *)0xFFFFFFFF);
#endif
	// another check for the same thing.
	assert(_p_block_start_id_table[last_super_block_id] == last_super_block_id);
#endif

	// Get the container corresponding to this last super block.
	Gc_Space *p_space = _p_block_container_table[last_super_block_id];

	if (p_space == NULL) {
		//
		// The last super block in the existing heap is unallocated.
		// Hence we need to coalesce it with the new super block.
		//
#if (GC_DEBUG>2)
		assert(_p_block_size_blocks_table[last_super_block_id] > 0);
#endif
		_p_block_size_blocks_table[last_super_block_id] += number_of_new_blocks;
		new_first_block_id    = last_super_block_id;
	} else {
		//
		// Last existing super block is allocated: don't need to coalesce.
		//
		_p_block_size_blocks_table[new_first_block_id] = number_of_new_blocks;
		_p_block_start_id_table[new_first_block_id]    = new_first_block_id;
        _p_block_end_table[new_first_block_id] = (POINTER_SIZE_INT)((Byte *)p_last_block + _sub_block_size_bytes - 1);
		_p_block_container_table[new_first_block_id]   = NULL;
        _p_block_generation_table[new_first_block_id] = -1;
	}
	//
	// Now craft a new unallocated super block, possibly coalesing a previous
	// unallocated super block.
	//
	for (int block_idx =  new_first_block_id + 1;
		     block_idx <= new_last_block_id;
		     block_idx++) {
		_p_block_container_table[block_idx]   = (Gc_Space *)POINTER_SIZE_MINUS_ONE;
		_p_block_size_blocks_table[block_idx] = -1;
		_p_block_start_id_table[block_idx]    = new_first_block_id;
        _p_block_end_table[block_idx] = (POINTER_SIZE_INT)((Byte *)p_last_block + _sub_block_size_bytes - 1);
        _p_block_generation_table[block_idx] = -1;     
        }
#if (GC_DEBUG>2)
	// another redundant check: new_first_block_id should point to a valid super block
	assert(_p_block_start_id_table[new_first_block_id] == new_first_block_id);
	// another similar check
	assert(_p_block_container_table[new_first_block_id] != (Gc_Space *)POINTER_SIZE_MINUS_ONE;
	// a related check
	assert(_p_block_size_blocks_table[new_first_block_id] > 0);
#endif
	_total_current_sub_blocks += number_of_new_blocks;
	_free_sub_blocks  += number_of_new_blocks;
//    orp_cout << "_total_current_sub_blocks is " << _total_current_sub_blocks << endl;
	return _total_current_sub_blocks;
}

Block_Store::Block_Store(Gc_Fast_Hooks *p_gc_hooks,
                         Gc_Plan       *p_gc_plan,
                         unsigned long initial_heap_size_bytes, 
						 unsigned long final_heap_size_bytes,
                         unsigned int  sub_block_size_bytes)
{
#if (GC_DEBUG>2)
	assert((p_gc_hooks) && (p_gc_plan));
#endif
	_p_gc_plan = p_gc_plan;

    //
    // Create an obsolete space for debugging purposes.
    //
    _p_global_obsolete_space = new Obsolete_Space(NULL);

    //
    // Initialize the lock.
    //
    InitializeCriticalSection(&_lock);

	//
	// Get information about the current system. (Page size, num processors, etc.)
	//
	_get_system_info();
	//
	// Initialize the Block Store parameters.
	//
    _initial_heap_size_bytes = initial_heap_size_bytes;
	_current_heap_size_bytes = initial_heap_size_bytes;
	_final_heap_size_bytes   = final_heap_size_bytes;
    _sub_block_size_bytes    = sub_block_size_bytes;

#ifdef GC_NONE_V0
    _table_size_entries = 0;
#else
	_table_size_entries      = _final_heap_size_bytes / _sub_block_size_bytes;
#endif

	_table_size_bytes        = _table_size_entries * sizeof(void *);

    _number_of_tables = 5; // count of the tables above.

	//
	// Reserve enough space for the GC'd heap and the Block Store's table.
	// 

#ifdef POINTER64
    assert ((POINTER_SIZE_INT)(_final_heap_size_bytes + _table_size_bytes * _number_of_tables)
                <= (POINTER_SIZE_INT) 0xFFFFFFFF);
#endif

	unsigned long maximum_size_to_reserve = 
		_final_heap_size_bytes + _table_size_bytes * _number_of_tables;
    
    //
	// Reserve enough contiguous space to be able to grow the heap.
	//
    //    orp_cout << " VirtualAlloc of " << maximum_size_to_reserve << endl;


#ifdef GC_NONE_V0
    // No Block store concept in GC_NONE_V0..just return!!!
	return;
#endif

#ifdef ORP_POSIX

    // DO RAW malloc() for now...
    _gc_free_list = malloc(maximum_size_to_reserve + 0xFFFF);

    // Save gc_free_list value so that it can be used to "free" the memory returned by malloc.
    _p_real_heap_base = _gc_free_list;
	
    // To make sure that _gc_free_list is 64KB aligned.
	_gc_free_list = (void *) (((POINTER_SIZE_INT)_gc_free_list + 0xFFFF) & (0xFFFF0000));


	if (_gc_free_list == NULL) { 
    
#else
		
		// This is needed but is not available from the winnt.h along with things like MEM_RESERVE
		// and MEM_COMMIT. I am guessing at the value...
#define MEM_WRITE_WATCH 0x200000

		if ((_gc_free_list = VirtualAlloc(NULL,
		                              maximum_size_to_reserve,
#ifdef GC_PT_WB // garbage collector page table write barrier
                                      // Let the OS handle card marking a page at a time
                                      MEM_WRITE_WATCH |
#endif  // GC_PT_WB                              
                                      MEM_RESERVE,
								      PAGE_NOACCESS)) == NULL) {

#endif
		DWORD error_code = GetLastError();
		orp_cout << "Error: Garbage Collector failed to reserve ";
		orp_cout << _final_heap_size_bytes;
		orp_cout << " bytes of virtual address space. Error code = ";
		orp_cout << error_code << endl;
		assert(0);
		orp_exit(error_code);
	}

	//
	// Populate the initial heap portion with real pages.
	//
	unsigned long initial_size_to_commit =
		_initial_heap_size_bytes + _table_size_bytes;


#ifdef ORP_POSIX

    // No need to commit specifically for LINUX yet...

#else
    // Commit of NT (and W2K
	if ((_gc_free_list = VirtualAlloc(_gc_free_list,
		                              initial_size_to_commit,
                                      MEM_COMMIT,
								      PAGE_READWRITE)) == NULL) {
		DWORD error_code = GetLastError();
		orp_cout << "Error: GC failed to commit initial heap of size ";
		orp_cout << _initial_heap_size_bytes << " bytes. Error code = " ;
		orp_cout << error_code << endl;
        assert(0);
		exit(error_code);
	}
#endif

	if (stats_gc) {
		orp_cout << "Garbage collected heap size is " << initial_size_to_commit << " bytes" << endl;
	}

#ifdef ORP_NT

    //
    // Save it away for later use (ex: deallocation)
    //
    // This differs from _p_real_heap_base since it include the block tables and other meta data.
    //
    _p_real_heap_base = _gc_free_list;

#endif


    //
    // Save global pointers for use in JIT debugging.
    //
    p_gc_base    = _gc_free_list;
    void *_p_gc_limit   =  (void *)((char *)p_gc_base + initial_size_to_commit);
    p_gc_ceiling =  (void *)((char *)p_gc_base + maximum_size_to_reserve);
#if (GC_DEBUG>0)
	if (stats_gc) {
		orp_cout << "Start = " << p_gc_base << ", ceiling = " << _p_gc_limit << endl;
	}
#endif
	//
	// Allocate the block table, which maps blocks to containers.
	// (Containers include nurseries, steps, cars, and LOS.)
	//
	_allocate_block_tables();

	//
	// Now that we have allocated all the meta data, find out how big
	// the effective heap is.
	//
	if( _p_gc_limit <= _gc_free_list ){ 
		orp_cout << "Initial heap size is too small, please increase it." << endl;
		assert(0);
		exit(2);
	}
	
	_p_effective_heap_base = _gc_free_list;
  
#ifdef POINTER64
    assert((POINTER_SIZE_INT)((Byte *)_p_gc_limit - (Byte *)_gc_free_list) 
        <= (POINTER_SIZE_INT)0xFFFFFFFF);
#endif
    _effective_heap_size_bytes = (unsigned long)((Byte *)_p_gc_limit - (Byte *)_gc_free_list);

    assert (GC_CARD_SHIFT_COUNT == GC_BLOCK_SHIFT_COUNT);

	//
	// Record the last block in the currently committed space.
	//
	_p_last_block =
		(void *)((((_effective_heap_size_bytes / _sub_block_size_bytes) - 1) * 
		          _sub_block_size_bytes) 
		         + (Byte *)_p_effective_heap_base);
    assert (_sub_block_size_bytes == (1 << GC_BLOCK_SHIFT_COUNT));
    //
	// Link all the free blocks into a free list.
	//

    _initialize_block_tables(_gc_free_list, _p_last_block);

    return;
}


Block_Store::~Block_Store()
{
#if (GC_DEBUG>3)
    orp_cout << "Deleting the block store" << endl;
#endif 

    // NOTE NOTE NOTE: replace this with RTTI
	// Gc_Space *p_space;
    //
    // Clean up any obsolete containers still hanging around.
    //
    for (unsigned long idx = 0; idx < _table_size_entries; idx++) {
#if 0 // FIX
	    if (p_space = 
            dynamic_cast<Obsolete_Space*>(_p_block_table[idx])) {
		        delete p_space;
	    }
#endif
    }
    //
    // Release the memory we obtained for the GC heap, including
	// the ancilliary table(s).
    //

#ifdef ORP_POSIX
	
	free(_p_real_heap_base);
	
#else
    if (!VirtualFree(_p_real_heap_base, 0, MEM_RELEASE)) {
        cerr << "Error: failed to deallocate GC heap.";
        cerr << " error code = " << GetLastError() << endl;
        assert(0);
        orp_exit(GetLastError());
    }
#endif

//    orp_cout << "VirtualFree of " << _p_real_heap_base << endl;

}

//
// The Block Store is being asked to extend the heap by the specified
// number of bytes.
//
bool
Block_Store::_extend_heap()
{
	int delta_bytes = _p_gc_plan->heap_extension_increment_bytes();

    if (stats_gc) {
    	orp_cout << "Extending the heap by " << delta_bytes << " bytes" << endl;
    }

	if (_current_heap_size_bytes + delta_bytes >= _final_heap_size_bytes) {
		orp_cout << "Error: attempt to grow the heap past the maximum size." << endl;
		orp_exit(1);
		return false; // why exit? (caller may be able to recover.)
	}

	//
	// Commit the specified number of bytes 
	//
	void *p_new_region = 
		(void *)((Byte *)_p_effective_heap_base + _effective_heap_size_bytes);
	
#ifdef ORP_NT

    if ((p_new_region = VirtualAlloc(p_new_region,
		                             delta_bytes,
                                     MEM_COMMIT,
								     PAGE_READWRITE)) == NULL) {
		DWORD error_code = GetLastError();
		orp_cout << "Error: GC failed to extend heap by ";
		orp_cout << _initial_heap_size_bytes << " bytes. Error code = " ;
		orp_cout << error_code << endl;

		orp_exit(error_code);

		return false; // why exit? (caller may be able to recover.)
	}
#else
    // For Linux, malloc() is equivalent to MEM_RESERVE and MEM_COMMIT. 
    // So this step a NOP.

#endif // ORP_NT

	//
	// Update the current heap size. BUG: NEED TO ROUND UP delta_bytes to
	// THE VirtualAlloc GRAIN OF ALLOCATION (FIRST FIGURE THIS OUT IN 
	// _get_system_info()).
	//
	_current_heap_size_bytes   += delta_bytes;
	_effective_heap_size_bytes += delta_bytes;
	//
	// Record the last block in the currently committed space.
	//
	_p_last_block = (void *)(((_effective_heap_size_bytes / _sub_block_size_bytes - 1) * 
		                      _sub_block_size_bytes) 
		+ (Byte *)_p_effective_heap_base);

	//
	// Now thread these blocks together and add to the free list.
	//
	unsigned long new_sub_blocks = 
		_extend_block_tables(p_new_region, _p_last_block);

	//
	// Finally graft these new blocks into the free list.
	//
	*((void **)_p_last_block) = _gc_free_list;
	_gc_free_list = p_new_region;

#ifdef GC_GEN_V3
    // 
    // Clear some card tables and some last object tables.
    //
    if (stats_gc) {
        orp_cout << "Extending card table by " << delta_bytes << endl;
    }
    p_global_card_table_manager->extend_card_table (delta_bytes); // extend related card structures

#endif // GC_GEN_V3

#ifdef GC_PT_WB
    UINT foo = dll_ResetWriteWatch (p_new_region, delta_bytes);
    if (foo != 0) {
        orp_cout << "Bad ResetWriteWatch." << endl;
        exit (99);
    }
#endif // GC_PT_WB
	return true;
}



//
// set_address_container takes an address and a container and assigns the appropriate
// values to various tables so that calculations such as address to container is fast. 
//
void 
Block_Store::set_address_container(void *p_addr,
						           Gc_Space *p_container) 
{
	unsigned long super_block_id = _get_super_block_id_of_address(p_addr);

#if (GC_DEBUG>0)
	assert((super_block_id >= 0) && (super_block_id < _table_size_entries) && (_is_super_block(super_block_id)));
#endif
	_p_block_container_table[super_block_id] = p_container;
	int block_size = _p_block_size_blocks_table[super_block_id];
	int block_idx_end = super_block_id + block_size;
	for (int block_idx = super_block_id; block_idx < block_idx_end; block_idx++) {
		_p_block_generation_table[block_idx] = p_container->p_container->get_generation_number();
	}
}


bool 
Block_Store::_is_super_block(int super_block_id)
{
#if (GC_DEBUG>2)
	assert((super_block_id >= 0) && (super_block_id < _maximum_table_index));
#endif

	return (_p_block_start_id_table[super_block_id] == super_block_id);

}

//
// This routine checks if the specified sub block is allocated
// out to some GC container. 
//
bool
Block_Store::_is_sub_block_allocated(int sub_block_id)
{

#if (GC_DEBUG>2)
	//
	// If debugging, check the array bounds.
	//
	assert((sub_block_id >= 0) && (sub_block_id < _maximum_table_index));
#endif

	// NOTE NOTE NOTE:
	// Currently we flag unallocated blocks by putting a -1 in the
	// start ID array. This needs to change if we retain super-blocks
	// that are released, and insert obsolete objects.
	//
	int super_block_id = _p_block_start_id_table[sub_block_id];

#if (GC_DEBUG>2)
	//
	// Again, if debugging, check the array bounds.
	//
	assert((super_block_id >= 0) && (sub_block_id < _maximum_table_index));
#endif

	return (_p_block_container_table[super_block_id] != NULL);
}


//
// Card table logic needs to know where heap starts to calculate
// card.

Byte * 
Block_Store::get_heap_base() 
{
	return (Byte *)_p_effective_heap_base;
}

void 
Block_Store::_get_system_info()
{
	SYSTEM_INFO system_info;

	GetSystemInfo(&system_info);
	_machine_page_size_bytes     = system_info.dwPageSize;
	_number_of_active_processors = system_info.dwNumberOfProcessors;
}




//
// Given the size of a super block, return the corresponding
// number of sub blocks required.
//
unsigned long
Block_Store::_get_sub_block_count_from_bytes(unsigned int number_of_bytes)
{
	if (number_of_bytes <= _sub_block_size_bytes) {
		//
		// A single sub block is the minimum we can use.
		//
		return 1;
	}

	//
	// WE HAVE A PACT WITH THE AUTHORS OF PLAN FILES THAT SUPER
	// BLOCKS WILL BE MULTIPLES OF SUB BLOCKS.
	//

	return number_of_bytes / _sub_block_size_bytes;
}

// 
// Given a number of bytes size up to the next number of blocks.
//
unsigned int 
Block_Store::size_up_to_block_size (unsigned int number_of_bytes)
{
	if (((number_of_bytes / _sub_block_size_bytes) * _sub_block_size_bytes) ==
		number_of_bytes) {
		return number_of_bytes;
	} else {
		return ((number_of_bytes / _sub_block_size_bytes) + 1) * _sub_block_size_bytes;
	}
}

// 
// Given a number of bytes size up to the next number of blocks.
//
unsigned int 
Block_Store::size_up_to_los_block_size (unsigned int number_of_bytes)
{
	if (((number_of_bytes / _p_gc_plan->los_block_size_bytes()) * _p_gc_plan->los_block_size_bytes()) ==
		number_of_bytes) {
		return number_of_bytes;
	} else {
		return ((number_of_bytes / _p_gc_plan->los_block_size_bytes()) + 1) * _p_gc_plan->los_block_size_bytes();
	}
}

void *
Block_Store::_p_page_align_down(void *p_addr,
							   unsigned int num_pages)
{
	POINTER_SIZE_INT page_space = (POINTER_SIZE_INT)(num_pages * _machine_page_size_bytes);
	return (void *)((POINTER_SIZE_INT)p_addr & ~(page_space - 1));
}

void *
Block_Store::_p_page_align_up(void *p_addr,
							 unsigned int num_pages)
{

	POINTER_SIZE_INT page_space = num_pages * _machine_page_size_bytes;
	POINTER_SIZE_INT temp1 = (POINTER_SIZE_INT)p_addr + page_space - 1;
	POINTER_SIZE_INT temp2 = page_space - 1;
	POINTER_SIZE_INT temp3 = ~temp2;
	POINTER_SIZE_INT temp4 = temp1 & temp3;
	POINTER_SIZE_INT temp5 = (((POINTER_SIZE_INT)p_addr + page_space - 1) & ~(page_space - 1));
	void *temp6 = (void *)temp5;

	return (void *)(((POINTER_SIZE_INT)p_addr + page_space - 1) & ~(page_space - 1));
}

//
// A super-block is being returned to the block store. May need to coalesce
// with adjacent block(s).
//
// A super-block consists of one or more sub blocks and is variable in size.
//
void 
Block_Store::release_super_block(void *p_block_address)
{
	int current_super_block_id  = _get_super_block_id_of_address(p_block_address);
	int current_sub_block_count = _p_block_size_blocks_table[current_super_block_id];

	int new_super_block_id      = current_super_block_id;
	int new_sub_block_count     = current_sub_block_count;
    POINTER_SIZE_INT super_block_end_address = 
        (POINTER_SIZE_INT)p_get_address_super_block_end_address (p_block_address);
    int previous_super_block_id;
#if (GC_DEBUG>0)
	assert(_p_block_container_table[current_super_block_id] != (Gc_Space *)POINTER_SIZE_MINUS_ONE);
	assert(_p_block_container_table[current_super_block_id] != NULL);
	assert(_p_block_start_id_table[current_super_block_id] == current_super_block_id);
	assert(_p_block_size_blocks_table[current_super_block_id] > 0);
#endif

    //
	// Bump up the number of free blocks in the store.
	//
	_free_sub_blocks += current_sub_block_count;

	//
	// See if the previous super_block is unallocated.
	//
	if (current_super_block_id > 0) {
		previous_super_block_id = _p_block_start_id_table[current_super_block_id - 1];
#if (GC_DEBUG>0)
		assert(_p_block_start_id_table[previous_super_block_id]==previous_super_block_id);
		assert(_p_block_container_table[previous_super_block_id] != (Gc_Space *)POINTER_SIZE_MINUS_ONE);
		assert(_p_block_size_blocks_table[previous_super_block_id]>0);
#endif
		if (_p_block_container_table[previous_super_block_id] == NULL) {
			//
			// Yes, the previous block is empty: need to coalesce.
			//
			new_super_block_id  = previous_super_block_id;
			new_sub_block_count = current_sub_block_count +
				               _p_block_size_blocks_table[previous_super_block_id];
		}
	}

	//
	// See if the next super_block is unallocated.
	//
	// Find the offset of the following super block.
	int next_super_block_id = current_super_block_id + current_sub_block_count;
	if (next_super_block_id < _total_current_sub_blocks) {

#if (GC_DEBUG>0)
        if (!(_p_block_start_id_table[next_super_block_id]==next_super_block_id)) {
            orp_cout << " _total_current_sub_blocks is " << _total_current_sub_blocks << endl;
            orp_cout << "next_super_block_id is " << next_super_block_id << endl;
            orp_cout << "_total_current_sub_blocks is " << _total_current_sub_blocks << endl;
            orp_cout << "_maximum_table_index is " << _maximum_table_index << endl;
            orp_cout << "current_super_block_id is " << current_super_block_id << endl;
        }
		assert(_p_block_start_id_table[next_super_block_id]==next_super_block_id);
		assert(_p_block_container_table[next_super_block_id] != (Gc_Space *)POINTER_SIZE_MINUS_ONE);
		assert(_p_block_size_blocks_table[next_super_block_id]>0);
#endif
		if (_p_block_container_table[next_super_block_id] == NULL) {
			//
			// Yes, the next super-block is NULL.
			//
			new_sub_block_count += _p_block_size_blocks_table[next_super_block_id];
            super_block_end_address = (POINTER_SIZE_INT)_p_block_end_table[next_super_block_id];
		}
	}
#if (GC_DEBUG>0)
	assert((new_super_block_id >= 0) && (new_super_block_id < _maximum_table_index));
#endif

	_p_block_container_table[new_super_block_id]   = NULL;
	_p_block_size_blocks_table[new_super_block_id] = new_sub_block_count;
	_p_block_start_id_table[new_super_block_id]    = new_super_block_id;
    _p_block_end_table[new_super_block_id] = super_block_end_address;
    _p_block_generation_table[new_super_block_id]  = -1;

	for (int block_idx = new_super_block_id + 1;
	         block_idx < new_super_block_id + new_sub_block_count;
			 block_idx++) {
		_p_block_container_table[block_idx]   = (Gc_Space *)POINTER_SIZE_MINUS_ONE;
		_p_block_size_blocks_table[block_idx] = -1;
		_p_block_start_id_table[block_idx]    = new_super_block_id;
        _p_block_end_table[block_idx]         = super_block_end_address;
		_p_block_generation_table[block_idx]  = -1;
	}

#if 0 // (GC_DEBUG>1)
    orp_cout << "Freed block obliterated" << endl;
    // Fill the released blocks with a non zero byte field, with the low bit set to 
    // trip bogus monitor calls on bad objects.
    long bytes_to_clear = new_sub_block_count * _p_gc_plan->sub_block_size_bytes();
    void *base_addr = _p_get_sub_block_start_address(new_super_block_id);
    memset (_p_get_sub_block_start_address(new_super_block_id), 0xab, bytes_to_clear);

    assert (_p_get_sub_block_start_address(new_super_block_id) == 
                 (void *)(super_block_end_address - bytes_to_clear + 1)) ;
#endif // (GC_DEBUG>1)

    return;
}

//
// Get a fresh block from the store and update the Block Table with the
// container that is requesting it.
//
void *
Block_Store::p_get_new_super_block(Gc_Space *p_container,
							       int block_size_bytes,
								   int generation_code,
                                   bool return_null_on_fail)
{
	int number_of_sub_blocks =
		_get_sub_block_count_from_bytes(block_size_bytes);

retry:
	//
	// For the time being, we do an address order first-fit scheme.
	// Need to graft in the splay tree access here.
	//
	int new_super_block_id = 
		_address_order_first_fit(p_container,
		                         generation_code,
		                         number_of_sub_blocks);

	if (new_super_block_id >= 0) {
		return _p_get_sub_block_start_address(new_super_block_id);

	} else {
        if (return_null_on_fail) {
            // Return null if asked to do so.
            return NULL;
        }
		//
		// Couldn't find a segment of contiguous sub-blocks.
		//
		if (_extend_heap()) {
		    goto retry;
		} else {
			// Throw an exception here....
			cerr << "Heap exhausted" << endl;
			assert(0);
			orp_exit(1);
			return NULL;
		}
	}
}

int
Block_Store::_address_order_first_fit(Gc_Space *p_container,
									  int generation_code,
									  int number_of_sub_blocks)
{
	//
	// Start at the front of the list. (Address order next fit would 
	// make this persistent and continue at the next iteration.)
	//
	int block_idx = 0;

	assert (number_of_sub_blocks > 0);

    if ((generation_code != 0) && (generation_code != -1)) {
        generation_code = generation_code;
    }

	while (true) {
		if (block_idx == _total_current_sub_blocks) {
			//
			// We reached the end of the list.
			// Couldn't find a suitable sized chunk.
			//
			return -1;
		}
#if (GC_DEBUG>0)
		assert(_p_block_start_id_table[block_idx]==block_idx);
#endif
		if (_p_block_container_table[block_idx] != NULL) {
			//
			// This super block is in use.
			// Skip to the next super-block.
			//
			block_idx += _p_block_size_blocks_table[block_idx];
		} else {
			//
			// This super block is unused. Let's see if it is large enough.
			//
#if (GC_DEBUG>1)
			assert(_p_block_size_blocks_table[block_idx] > 0);
#endif
			if (_p_block_size_blocks_table[block_idx] < number_of_sub_blocks) {
				//
				// Nope - sorry.
				//
#if (GC_DEBUG>1)
				assert(_p_block_size_blocks_table[block_idx]>0);
#endif				
				block_idx += _p_block_size_blocks_table[block_idx];

			} else if (_p_block_size_blocks_table[block_idx] >= number_of_sub_blocks) {

				if (_p_block_size_blocks_table[block_idx] > number_of_sub_blocks) {
					//
					// We have more than we need.
					// Craft a new free super block out of the remainder.
					//
					int new_super_block_idx = block_idx + number_of_sub_blocks;
#if (GC_DEBUG>1)
					assert(_p_block_size_blocks_table[block_idx]>0);
#endif
					int new_super_block_size_blocks = _p_block_size_blocks_table[block_idx]
						- number_of_sub_blocks;

					for (int new_block_idx = new_super_block_idx;
						     new_block_idx < new_super_block_idx + 
								              new_super_block_size_blocks;
						     new_block_idx++) {
					
						_p_block_container_table[new_block_idx] = NULL;
						_p_block_start_id_table[new_block_idx]  = new_super_block_idx;
                        // The _p_block_end_table end stays the same since we are removing 
                        // from the front of the block.
//						_p_block_size_blocks_table[new_block_idx] = -1;
                        assert (_p_block_size_blocks_table[new_block_idx] == -1);
                             }

					_p_block_size_blocks_table[new_super_block_idx] = 
						new_super_block_size_blocks;
					_p_block_generation_table[new_super_block_idx] = -1;
				}

				//
				// A perfect fit.
				//

                POINTER_SIZE_INT block_end = (POINTER_SIZE_INT)((((number_of_sub_blocks * _sub_block_size_bytes) - 1)
                                    + (Byte *)_p_get_sub_block_start_address(block_idx))); 

                int quick_check_idx = get_sub_block_id_of_address ((void *)block_end);
                assert ((block_idx + number_of_sub_blocks - 1) == quick_check_idx);

                for (int allocated_block_idx = block_idx; 
                         allocated_block_idx < block_idx + number_of_sub_blocks; 
                         allocated_block_idx++) {                            
                     _p_block_end_table[allocated_block_idx] = block_end; 
                     _p_block_generation_table[allocated_block_idx] = generation_code;
					 assert (_p_block_start_id_table[allocated_block_idx] ==  block_idx);
                }
				_p_block_container_table[block_idx]   = p_container;
                _p_block_size_blocks_table[block_idx] = number_of_sub_blocks;
                _free_sub_blocks -= number_of_sub_blocks;
				return block_idx;

			}
		}
	}
}
 


 
Large_Object_Store *
Block_Store::p_get_address_los(void *p_addr) 
{
    Gc_Space *p_container = p_get_address_container(p_addr);
#if (GC_DEBUG>2)
    assert(p_container->is_large_object_space());
#endif // _DEBUG
    return (Large_Object_Store *)p_container;
}
 

//
// Return the Nursery that this address is contained in.
//
Nursery *
Block_Store::p_get_address_nursery(void *p_addr) 
{
    Gc_Space *p_container = p_get_address_container(p_addr);
#if (GC_DEBUG>2)
    assert(p_container->is_nursery());
#endif // _DEBUG
    return (Nursery *)p_container;
}
//
// Return the Step that this address is contained in.
//
Step *
Block_Store::p_get_address_step(void *p_addr) 
{
    Gc_Space *p_container = p_get_address_container(p_addr);
#if (GC_DEBUG>2)
    assert(p_container->is_step());
#endif // _DEBUG
    return (Step *)p_container;
}
//
// Return the Train this address is contained in.
//
Train *
Block_Store::p_get_address_train(void *p_addr) 
{
    Gc_Space *p_gc_space = p_get_address_container(p_addr);

#if (GC_DEBUG>1)
    assert(p_gc_space->is_car());
#endif

    Train *p_train = ((Car *)p_gc_space)->p_get_train();

#if (GC_DEBUG>1)
    assert(p_train->is_train());
#endif

    return p_train;
}

#if (GC_DEBUG>3)
//
// A block is being freed or marked as unused.
//
void
Block_Store::set_address_obsolete_container(void *p_addr)
{
	unsigned long block_id = 
		get_sub_block_id_of_address(p_addr);

	Gc_Space *p_current = p_get_block_container(block_id);

#if 1 // Slow things down til we debug this.
	Obsolete_Space *p_space = new Obsolete_Space(p_current);

	set_address_container(p_addr, p_space);
 
#else
    //
    // Speed things up - we aren't debugging in Kansas any more.
    //
    //set_address_container(p_addr, _p_global_obsolete_space);
    set_address_container(p_addr, NULL);
#endif
}
#endif // GC_DEBUG>3

bool
Block_Store::is_address_in_young_generation(void *p_addr)
{
	if (is_address_outside_heap(p_addr)) {
		return false;
	}
    bool result = get_address_generation (p_addr) == YOS_GEN_NUMBER;
#if (GC_DEBUG>0)
    // double check for non release code.
	Generation *p_gen = p_get_address_generation(p_addr);
	if (p_gen->is_step_generation()) {
        assert (result);
		return true;
	}
    assert (!result);
	return false;
#else
    return result;
#endif
}


bool
Block_Store::is_address_in_car(void *p_addr)
{
	Gc_Space *p_container = p_get_address_container(p_addr);
	if (p_container->is_car()) {
		return true;
	}

	return false;
}	




bool 
Block_Store::is_object_in_moving_area(Java_java_lang_Object *p_obj)
{
	if (is_address_in_large_object_space((void *)p_obj)) {
		return false;
	}

	return true;
}

bool 
Block_Store::is_reference_in_moving_area(Java_java_lang_Object **pp_obj_ref)
{
	if (is_address_in_large_object_space((void *)pp_obj_ref)) {
		return false;
	}

	return true;
}


bool 
Block_Store::is_address_in_moving_area(void *p_address)
{
	if (is_address_in_large_object_space(p_address)) {
		return false;
	}

	return true;
}


bool 
Block_Store::is_object_in_fixed_area(Java_java_lang_Object *p_obj)
{
	if (is_object_in_moving_area(p_obj)) {
		return false;
	}
	return true;
}

bool 
Block_Store::is_obsolete_object(Java_java_lang_Object *p_obj) 
{
	Gc_Space *p_space = p_get_object_container(p_obj);

	if (p_space->is_obsolete()) {
		return true;

	}

	return false;
}


bool 
Block_Store::is_reference_in_fixed_area(Java_java_lang_Object **pp_obj_ref)
{
	if (is_reference_in_moving_area(pp_obj_ref)) {
		return false;
	}
	return true;
}


bool 
Block_Store::is_address_in_fixed_area(void *p_address)
{
	if (is_address_in_moving_area(p_address)) {
		return false;
	}
	return true;
}



bool
Block_Store::is_address_in_nursery(void *p_addr)
{
	Gc_Space *p_container = p_get_address_container(p_addr);
	if (p_container->is_nursery()) {
		return true;
	}

	return false;
}

bool
Block_Store::is_address_in_step(void *p_addr)
{
	Gc_Space *p_container = p_get_address_container(p_addr);
	if (p_container->is_step()) {
		return true;
	}

	return false;
}

//
// dump_block_store_info
//
// Comment - This routine dumps our the current state of the block store. It can be used
// to debug things like fragmentation. It is only meant for debugging and for understanding
// how the system is working when doing performance tuning. Someday I'll but a snazzy front end 
// on it and call it a development tool, but for now....
//
void
Block_Store::dump_block_store_info ()
{
	// Dump the high level stuff first.
//	orp_cout << "----------------vvv Start of block store information vvv--------------------" << endl;
//	orp_cout << "Total number of blocks is " << _maximum_table_index << endl;
//	orp_cout << "Block \tGenId\tTrain/Step\tCar" << endl;
	for (int indx = 0; indx <_maximum_table_index; indx++) {
		if ((indx == 0) || (_p_block_generation_table[indx] != _p_block_generation_table[indx-1])) {
//			orp_cout << indx << "\t" << _p_block_generation_table[indx] << "\t" << endl;
//			_p_block_end_table[sub_block_id];
			if (_p_block_generation_table[indx] == 0) {
				// it is in 
			}
		}
	}
		
//	orp_cout << "----------------^^^ End of block store information   ^^^--------------------" << endl;
}

// end file gc\block_store.cpp
