// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/base_natives/gnu_classpath/java_io_File.cpp,v 1.15 2001/12/10 09:42:10 gwu2 Exp $
//



#include "platform.h"

#include <sys/types.h>
#include <sys/stat.h>


#ifdef ORP_NT
#include <io.h>
#endif

#include "native_utils.h"
#include "jni_direct.h"
#include "jni_utils.h"
#include "object_layout.h"

#include <fcntl.h>

#ifdef ORP_NT
#include <direct.h>
#include <sys/utime.h>
#elif defined (ORP_POSIX)
#include <unistd.h>
#endif

#include <errno.h>


#include "java_io_File.h"
#include "gnu_classpath_jni_utils.h"

#ifdef OBJECT_LOCK_V2
#include "platform_utils_olv2.h"
#else
#include "platform_utils.h"
#endif

// Function to use the correct directory separator for the platform in the file path.
// Remove " static inline " here to let other file use it.
// e.g, java_io_FileInputStream.cpp used this file
// static inline char *get_corrected_path(const char * path_name) 
char *get_corrected_path(const char * path_name) 
{
// get HOME environment variable to replace ~ in path
#ifdef ORP_POSIX
    char *home = getenv("HOME");
    int path_length = strlen(path_name) + strlen(home) + 3;
#else
    int path_length = strlen(path_name) + 1;
#endif
    
    assert(path_length > 0);

    char *ca = (char *)malloc(path_length);
    assert(ca);

    int path_cursor = 0;

    int xx;	
    for (xx = 0; xx < path_length; xx++)
    {
        char c = path_name[path_cursor];
        path_cursor++;
// deal with ~ in path
#ifdef ORP_POSIX    
        if(c == '~'){
            strcpy(ca+xx, home);
            xx += strlen(home) - 1; //here -1 means to count in the effect of xx++
            if( home[strlen(home)-1] == DIR_SEPARATOR )
                xx--;
            continue;
        }
#endif
        if ((c == '/') || (c == '\\'))
            c = DIR_SEPARATOR;
        ca[xx] = c;
        if (path_name[path_cursor] == 0x0) {
            ca[xx + 1] = 0x0;
            break;
        }
    }
    if(ca[xx] == DIR_SEPARATOR)
        ca[xx] = '\0'; 
    return ca;
}


/*
 * Class:     File
 * Method:    canReadInternal
 * Signature: (Ljava/lang/String;)Z
 */
JNIEXPORT jboolean JNICALL Java_java_io_File_canReadInternal
  (JNIEnv *jenv, jobject jobj, jstring jpath)
{
	// Do we assert here or return false???
	if (!jpath)
		return false;
	
	// We need to check access permissions for this file embedded in jstr.
	jboolean is_copy;
	const char *path_name = jenv->GetStringUTFChars(jpath, &is_copy);
    char *path = get_corrected_path(path_name);

#ifdef ORP_NT
    // Make the call to _access.
	int result = _access(path, 04);
#elif defined (ORP_POSIX)
	int result = access(path, R_OK );
#endif

	bool ret = true;

	if (result == -1) {
		// Error in call or invalid access
		if ((errno == EACCES) || (errno == ENOENT)){
			// if return here, mem will not be reclaimed
			// return ret = false;
			ret = false;
		} else {
			// wgs: I'll change some of these assert(0) to false return, because on Linux,
		    // some test of path will cause some file routines return weird values of , 
		    // e.g, search file in each classpath directory, we must allow these return values.
			// assert(0); // Invalid value for errno on this call.
			ret = false;
		}
	} else if (result != 0) {
		assert(0); // Undefined return value from access.
	} 

	jenv->ReleaseStringUTFChars(jpath, path_name);	
    free(path);
	return (jboolean) ret;

} // Java_java_io_File_canReadInternal


/*
 * Class:     File
 * Method:    canWriteInternal
 * Signature: (Ljava/lang/String;)Z
 */

JNIEXPORT jboolean JNICALL Java_java_io_File_canWriteInternal
  (JNIEnv *jenv, jobject jobj, jstring jpath)
{
	// Do we assert here or return false???
	if (!jpath)
		return false;
	
	// We need to check access permissions for this file embedded in jstr.
	jboolean is_copy;
	const char *path_name = jenv->GetStringUTFChars(jpath, &is_copy);
    char *path = get_corrected_path(path_name);

#ifdef ORP_NT
	// Make the call to _access.
	int result = _access(path, 02 );
#elif defined (ORP_POSIX) 
	int result = access(path, W_OK ); // change from access(path_name...) to access(path...)
#endif

	bool ret = true;

	if (result == -1) {
		// Error in call or invalid access
		if ((errno == EACCES) || (errno == ENOENT)) {
			// return ret = false;
			ret = false;
		} else {
			// assert(0); // Invalid value for errno on this call.
			ret = false;
		}
	} else if (result != 0) {
		assert(0); // Undefined return value.
	} 

	jenv->ReleaseStringUTFChars(jpath, path_name);	
    free(path);
	return (jboolean) ret;

} // Java_java_io_File_canWriteInternal


/*
 * Class:     File
 * Method:    createInternal
 * Signature: (Ljava/lang/String;)Z
 */

JNIEXPORT jboolean JNICALL Java_java_io_File_createInternal
  (JNIEnv *jenv, jclass clazz, jstring jpath)
{
    if (!jpath)
		return false;
	
	// We need to check access permissions for this file embedded in jstr.
	jboolean is_copy;
	const char *path_name = jenv->GetStringUTFChars(jpath, &is_copy);
    char *path = get_corrected_path(path_name);

	int fmode = O_CREAT;
	
    // Actually open the file; use default permissions

	int file_descr = open(path, fmode, 0755);

	if (file_descr == -1) {
		char *msg = (char *) malloc(300);
		sprintf(msg, "Creating %s failed, cause: %s", path, strerror (errno));
		throw_exception_from_jni(jenv, "java/io/IOException", msg);
		free(msg);
	}

	close(file_descr);

	return true;
}


/*
 * Class:     File
 * Method:    deleteInternal
 * Signature: (Ljava/lang/String;)Z
 */
JNIEXPORT jboolean JNICALL Java_java_io_File_deleteInternal
  (JNIEnv *jenv, jobject jobj, jstring jpath)
{
    if (!jpath)
		return false;
	
	// We need to check access permissions for this file embedded in jstr.
	jboolean is_copy;
	const char *path_name = jenv->GetStringUTFChars(jpath, &is_copy);
    char *path = get_corrected_path(path_name);

	int ret = unlink(path);
	if(ret == -1){
        ret = rmdir(path); //if it's a directory wgs%
        int errCode = GetLastError();
        if(errCode == 32){ //if want to delete current directory
            chdir(".."); 
            ret = rmdir(path);
        }
    }

    jenv->ReleaseStringUTFChars(jpath, path_name);	
	free(path);
    
    return (ret == 0);
}



/*
 * Class:     File
 * Method:    existsInternal
 * Signature: (Ljava/lang/String;)Z
 */

JNIEXPORT jboolean JNICALL Java_java_io_File_existsInternal
  (JNIEnv *jenv, jobject jobj, jstring jpath)
{
	// Do we assert here or return false???
	if (!jpath)
		return false;
	
	// We need to check access permissions for this file embedded in jstr.
	jboolean is_copy;
	const char *path_name = jenv->GetStringUTFChars(jpath, &is_copy);
    char *path = get_corrected_path(path_name);

#ifdef ORP_NT
	// Make the call to _access.
	int result = _access(path, 00 );
#elif defined (ORP_POSIX)
	int result = access(path, F_OK ); //change from access(path_name...) to access(path...)
#endif

    bool ret = true;

	if (result == -1) {
		// Error in call or invalid access
		if ((errno == EACCES) || (errno == ENOENT)){
			// return ret = false;
			ret = false;
		} else {
            // assert(0); // Invalid value for errno on this call.
            ret = false;
		}
	} else if (result != 0) {
		assert(0); // Undefined return value.
	} 

	jenv->ReleaseStringUTFChars(jpath, path_name);	
	free(path);
    return (jboolean) ret;

} // Java_java_io_File_existsInternal


/*
 * Class:     File
 * Method:    isDirectoryInternal
 * Signature: (Ljava/lang/String;)Z
 */


JNIEXPORT jboolean JNICALL Java_java_io_File_isDirectoryInternal
  (JNIEnv *jenv, jobject jobj, jstring jpath)
{
	// Do we assert here or return false???
	if (!jpath)
		return false;
	
	// We need to check access permissions for this file embedded in jstr.
	jboolean is_copy;
	const char *path_name = jenv->GetStringUTFChars(jpath, &is_copy);
    char *path = get_corrected_path(path_name);
	int slen = strlen(path_name);
	//In windows platform, _stat("c:\\dir\\", &buf) will fail even 
	// c:\\dir do exist, you should remove the tail DIR_SEPARATOR
	if(path[slen-1] == DIR_SEPARATOR) 
		path[slen-1] = '\0';

	struct _stat buf;
	int result;
	bool worked, is_dir;
	
	// Get data associated with file   
#ifdef ORP_NT
    result = _stat(path, &buf);
#elif defined (ORP_POSIX)
	result = stat(path, &buf);
#endif

	if (result == -1) { 
		// Error
		if (errno == ENOENT) { 
			worked = false;
		} else { 
			// assert(0); // Invalid value for errno;
			worked = false;
		}
	} else if (result == 0) { 
		// Success
		worked = true;
#ifdef ORP_NT
        is_dir = (buf.st_mode & _S_IFDIR) ? true : false;
#elif defined (ORP_POSIX)
		is_dir = (buf.st_mode & S_IFDIR) ? true : false;
#endif
	} else { 
		assert(0); // Invalid return value from _stat.
	}
	jenv->ReleaseStringUTFChars(jpath, path_name);	
    free(path);

	if (!worked) 
		return (jboolean) false;

	return (jboolean) is_dir;

} // Java_java_io_File_isDirectoryInternal

/*
 * Class:     File
 * Method:    isFileInternal
 * Signature: (Ljava/lang/String;)Z
 */
JNIEXPORT jboolean JNICALL Java_java_io_File_isFileInternal
  (JNIEnv *jenv, jobject jobj, jstring jpath)
{
	// Do we assert here or return false???
	if (!jpath)
		return false;
	
	// We need to check access permissions for this file embedded in jstr.
	jboolean is_copy;
	const char *path_name = jenv->GetStringUTFChars(jpath, &is_copy);
    char *path = get_corrected_path(path_name);

	struct _stat buf;
	int result;
	bool worked, is_file;
	
#ifdef ORP_NT
	// Get data associated with file   
    result = _stat(path, &buf);
#elif defined (ORP_POSIX)
    // Get data associated with file   
    result = stat(path, &buf);
#endif

	if (result == -1) { 
		// Error
		if (errno == ENOENT) { 
			worked = false;
		} else { 
			// assert(0); // Invalid value for errno;
			worked = false;
		}
	} else if (result == 0) { 
		// Success
		worked = true;
#ifdef ORP_NT
        is_file = (buf.st_mode & _S_IFREG) ? true : false;
#elif defined (ORP_POSIX)
		is_file = (buf.st_mode & S_IFREG) ? true : false;
#endif
	} else { 
		assert(0); // Invalid return value from _stat.
	}
	jenv->ReleaseStringUTFChars(jpath, path_name);	
    free(path);

	if (!worked) 
		return (jboolean) false;

	return (jboolean) is_file;

} // Java_java_io_File_isFileInternal


/*
 * Class:     File
 * Method:    lastModifiedInternal
 * Signature: (Ljava/lang/String;)J
 */
JNIEXPORT jlong JNICALL Java_java_io_File_lastModifiedInternal
  (JNIEnv *jenv, jobject jobj, jstring jpath)
{
	if (!jpath)
		return false;
	
	jboolean is_copy;
	const char *path_name = jenv->GetStringUTFChars(jpath, &is_copy);
	assert(path_name);
    char *path = get_corrected_path(path_name);

	struct stat buf;
	int result;
	jlong ret;
    
#ifdef ORP_NT
	// Get data associated with file   
	result = stat(path, &buf);
#elif defined (ORP_POSIX)
	// Get data associated with file   
	result = stat(path, &buf);
#endif

	// Check if statistics are valid
	if (result != 0) {
        ret = 0;
    } else {
		ret = buf.st_mtime;
	}
	jenv->ReleaseStringUTFChars(jpath, path_name);
    free(path);
	return ret;

} // Java_java_io_File_lastModifiedInternal


/*
 * Class:     File
 * Method:    lengthInternal
 * Signature: (Ljava/lang/String;)J
 */
JNIEXPORT jlong JNICALL Java_java_io_File_lengthInternal
  (JNIEnv *jenv, jobject jobj, jstring jpath)
{
	if (!jpath)
		return false;
	
	jboolean is_copy;
	const char *path_name = jenv->GetStringUTFChars(jpath, &is_copy);
	assert(path_name);
    char *path = get_corrected_path(path_name);
    
	struct stat buf;
	int result;
	jlong ret;
 
#ifdef ORP_NT
    // Get data associated with file 
	result = stat(path, &buf );
#elif defined (ORP_POSIX)
	// Get data associated with file 
	result = stat(path, &buf );
#endif

	// Check if statistics are valid
    ret = (result != 0) ? 0 : buf.st_size;
	jenv->ReleaseStringUTFChars(jpath, path_name);
    free(path);
    return ret;

} // Java_java_io_File_lengthInternal


/*
 * Class:     File
 * Method:    listInternal
 * Signature: (Ljava/lang/String;)[Ljava/lang/String;
 */
JNIEXPORT jobjectArray JNICALL Java_java_io_File_listInternal
  (JNIEnv *jenv, jobject jobj, jstring jpath)
{
	// Used to list all the files in a given directory that 
	if (!jpath) 
		return NULL;

	// Get the path first.
	jboolean is_copy;
	const char *path_name = jenv->GetStringUTFChars(jpath, &is_copy);
	assert(path_name);
    char *path = get_corrected_path(path_name);

	// Get the number of entries in the directory.
	int num_entries = generic_get_num_dir_entries(path);
	// assert(num_entries >= 0);
	// A more elegant way is to throw an exception
	if(num_entries < 0){
		throw_exception_from_jni(jenv, "java/io/IOException", "The direcoty doesn't exist");
		return NULL;
	}
	
	// Get the first directory entry.
	char *name = NULL;
	__int64 handle = generic_first_dir_ent(path, &name);
	
	jarray final_array = 0;

	if ((handle == -1) || (num_entries <= 0)) {
		jenv->ReleaseStringUTFChars(jpath, path_name);
        free(path);
        free(name); 
		return (jobjectArray) 0;
	}

	// Valid value -- 
	assert(name);
	
	char **file_names = (char **) malloc(sizeof(char *) * num_entries);
	assert(file_names);
			
	jvalue jargs[2];
	jargs[0].l = jobj;

	// loop through the entries.
	int index = 0;
	// We will not list '.' and '..'
	bool dot_done = false, dot_dot_done = false;

	for(;;) { 
		if (name == NULL) 
			break;		// End of all the entries;
		if (!dot_done) { 
			if (!strcmp(name, ".")) {
				dot_done = true;
				free(name);
				name = (char *) generic_get_next_dir_entry(handle, path);			
				continue;
			}
		}
		if (!dot_dot_done) { 
			if (!strcmp(name, "..")) {
				dot_dot_done = true;
				free(name);
				name = (char *) generic_get_next_dir_entry(handle, path);			
				continue;
			}
		}
		
		file_names[index++] = name;

		name = (char *) generic_get_next_dir_entry(handle, path);
	}

	// Now create a whole new array of JNI strings, and copy exactly 
	// "index" entries into the new array.
	jclass strclazz = jenv->FindClass("java/lang/String");	
	assert(strclazz);
	
	final_array = jenv->NewObjectArray(index, strclazz, NULL);
	assert(final_array);
	jenv->DeleteLocalRef(strclazz);

	int slot = 0;
	while(slot < index) { 
		char *cstr = file_names[slot];
		assert(cstr);
		jstring jstr = jenv->NewStringUTF(cstr);	
		assert(jstr);
		jenv->SetObjectArrayElement(final_array, slot++, jstr);
		// Free the slot.
		free(cstr);
	}

	free(file_names);
	jenv->ReleaseStringUTFChars(jpath, path_name);
    free(path);
    return (jobjectArray) final_array;

} // Java_java_io_File_listInternal


/*
 * Class:     File
 * Method:    mkdirInternal
 * Signature: (Ljava/lang/String;)Z
 */
JNIEXPORT jboolean JNICALL Java_java_io_File_mkdirInternal
  (JNIEnv *jenv, jobject jobj, jstring jpath)
{
	// Need to delete a file based on path in name
	// This is a private method which needs implementation.
	jboolean is_copy;

	const char *path_name = jenv->GetStringUTFChars(jpath, &is_copy);
	assert(path_name);
    char *path = get_corrected_path(path_name);

    // Now use "mkdir" to create the directory.
#ifdef ORP_NT
	int ret = _mkdir(path);
#elif defined (ORP_POSIX)
	int ret = mkdir(path, 0777);
#endif
	
	jenv->ReleaseStringUTFChars(jpath, path_name);
    free(path);
	return (jboolean) (!ret);

} // Java_java_io_File_mkdirInternal


/*
 * Class:     File
 * Method:    renameToInternal
 * Signature: (Ljava/lang/String;Ljava/lang/String;)Z
 */
JNIEXPORT jboolean JNICALL Java_java_io_File_renameToInternal
  (JNIEnv *jenv, jobject jfile, jstring jfrom, jstring jto)
{
	// Rename files.
	assert(jfile);
	assert(jfrom);
    assert(jto);

	jboolean is_copy;
	const char *from_name = jenv->GetStringUTFChars(jfrom, &is_copy);
	assert(from_name);
	const char *to_name = jenv->GetStringUTFChars(jto, &is_copy);
	assert(to_name);
    char *from = get_corrected_path(from_name);
    char *to = get_corrected_path(to_name);

	// Now, do the actual "rename"
	int ret = rename(from, to);

	jenv->ReleaseStringUTFChars(jfrom, from_name);
	jenv->ReleaseStringUTFChars(jto, to_name);
    free(from);
    free(to);
    return (jboolean) (!ret);
} // Java_java_io_File_renameToInternal


/*
 * Class:     File
 * Method:    setLastModifiedInternal
 * Signature: (Ljava/lang/String;J)Z
 */
JNIEXPORT jboolean JNICALL Java_java_io_File_setLastModifiedInternal
  (JNIEnv *jenv, jobject jobj, jstring jpath, jlong jtime)
{

#ifdef ORP_POSIX
    
    orp_cout << "Java_java_io_File_setLastModifiedInternal() not implemented on linux" << endl;
    assert(0);
    orp_exit(-1);

#elif defined (ORP_NT)

	if (!jpath)
		return false;
	
	jboolean is_copy;
	const char *path_name = jenv->GetStringUTFChars(jpath, &is_copy);
	assert(path_name);
    char *path = get_corrected_path(path_name);

	struct stat buf;
	int result;
	jlong ret = 0;

    result = stat(path, &buf);

	// Check if statistics are valid
	if (result == 0) {
        // Valid
        struct utimbuf utbuf;
        utbuf.actime = buf.st_atime;
        utbuf.modtime = buf.st_mtime;

        result = utime(path, &utbuf);
    } else { 
        // Invalid
        result = -1;
    }
	jenv->ReleaseStringUTFChars(jpath, path_name);
    free(path);
    return (result == -1) ? 0 : 1; 

#endif

} // Java_java_io_File_setLastModifiedInternal


/*
 * Class:     File
 * Method:    setReadOnlyInternal
 * Signature: (Ljava/lang/String;)Z
 */
JNIEXPORT jboolean JNICALL Java_java_io_File_setReadOnlyInternal
  (JNIEnv *jenv, jobject jobj, jstring jpath)
{
	if (!jpath)
		return false;
	
	jboolean is_copy;
	int result;

    const char *path_name = jenv->GetStringUTFChars(jpath, &is_copy);
	assert(path_name);
    char *path = get_corrected_path(path_name);

    // Get data associated with file   
#ifdef ORP_NT
	result = _chmod(path, _S_IREAD);
#elif defined (ORP_POSIX)
	result = chmod(path, S_IREAD);
#endif

	jenv->ReleaseStringUTFChars(jpath, path_name);
    free(path);
	return (jboolean) !result;
} // Java_java_io_File_setReadOnlyInternal












