// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o3_jit/global_reg_alloc.cpp,v 1.2 2001/08/13 09:54:55 xhshi Exp $
//


#include "defines.h"
#include <assert.h>
#include <iostream.h>
#include "global_reg_alloc.h"
#include "Mem_Manager.h"
#include "../x86/x86.h"
#include "ir.h"
#include "expression.h"
#include "flow_graph.h"
#include "bit_vector_group.h"
#include "fpspill.h"

//#define DEBUG_SPILL


static int is_byte_reg(Operand *opnd)
{
    unsigned regno = opnd->bv_position();
    unsigned mask = 1u << regno;
    return (mask & ALL_X86_BYTE_REGS);
}

struct ref_type_count
{
    int cnt;
    GCTrack_Operand *opnd;
};

class Ref_Counting : public Closure
{
public:
    Ref_Counting(ref_type_count *array,char *def_cnt) 
        : refcount(array), defcount(def_cnt), was_call(false) {}
    ref_type_count *refcount;
    char *defcount;
    bool was_call;  // whether a call instruction was ever seen
};

class Liveness_Update : public Closure
{
public:
    Liveness_Update(GCTrack_Operand **a, unsigned s) :
      array(a), size(s){}
    GCTrack_Operand **array;
    unsigned size;
};

static void count_ref_opnd(Operand *opnd, ref_type_count *refcount,
                           char *defcount, bool is_def, Cfg_Node *node)
{
    if (opnd == NULL)
        return;
    if (opnd->kind == Operand::GCTrack)
    {
        unsigned varno = opnd->bv_position();
        refcount[varno].cnt ++;
#if 0
        refcount[varno].cnt += 64 * node->loop_depth();
#endif // 1
        if (is_def && defcount[varno] <= 1) defcount[varno]++;
        assert(varno < n_reg || refcount[varno].opnd == NULL || refcount[varno].opnd == opnd);
        refcount[varno].opnd = (GCTrack_Operand *)opnd;
    }
    else
    {
        count_ref_opnd(opnd->base(),  refcount, defcount, false, node);
        count_ref_opnd(opnd->index(), refcount, defcount, false, node);
    }
}

static void count_refs(Cfg_Node *node, Closure *c)
{
    Ref_Counting *cl = (Ref_Counting *)c;
    Inst *last_inst = node->IR_instruction_list();
    Inst *inst = last_inst->next();
    for (;inst!=last_inst; inst=inst->next())
    {
        if (inst->is_call() && !inst->exits_method())
            cl->was_call = true;
        if (!node->is_cold())
        {
            count_ref_opnd(inst->dst(), cl->refcount, cl->defcount, true, node);
            unsigned i;
            for (i=0; i<inst->n_srcs; i++)
                count_ref_opnd(inst->src(i), cl->refcount, cl->defcount, false, node);
        }
    }
}

static unsigned var_with_highest_ref_count(ref_type_count *refcount, unsigned numvars)
{
    unsigned result = 0;
    int refs = -1;
    unsigned var;
    for (var=n_reg; var<numvars; var++)
    {
        if (refcount[var].opnd != NULL &&
            refcount[var].opnd->global_reg_alloc_cand() &&
            !IS_FP_DBL_TYPE(refcount[var].opnd->type) &&
            refcount[var].cnt > refs)
        {
            refs = refcount[var].cnt;
            result = var;
        }
    }

    return result;
}

static void update_liveness(Cfg_Node *node, Closure *cl)
{
    Liveness_Update *c = (Liveness_Update *)cl;
    unsigned i;
    unsigned size = c->size;
    GCTrack_Operand **array = c->array;
    for (i=0; i<size; i++)
    {
        unsigned bvp = array[i]->bv_position();
        assert(bvp < n_reg);
        if (node->live->is_live_var(array[i]->id))
        {
            node->live->mark_live(bvp, array[i]->type);
            // XXX- maybe we should mark the old value as dead.
        }
    }
}

static bool live_in_exc_handler(unsigned varno, Flow_Graph *fg)
{
    Eh_Node *node;
    for (node=fg->handlers()->next(); node != fg->handlers(); node = node->next())
    {
        if (node->live->is_live_var(varno))
            return true;
    }
    return false;
}

unsigned find_available_register(unsigned mask)
{
    unsigned i;
    for (i=0; i<n_reg; i++)
    {
        if (mask & (1u << i))
            break;
    }
    return i;
}

void keep_return_values_in_regs(Cfg_Node *epilog,
                                unsigned reg_usage,
                                Method_Handle mh)
{
    assert(epilog != NULL);
    if (reg_usage == 0) return;
    if (IS_FP_DBL_TYPE(method_get_return_type(mh))) return;
    Inst *cand[2];
    cand[0] = cand[1] = NULL;
    Inst *last_inst = epilog->IR_instruction_list();
    Inst *i = last_inst->next();
    for (; !i->is_call() && i != last_inst; i = i->next()) {
        Operand *dst = i->dst();
        if (dst != NULL && dst->is_single_def_temp_reg() &&
            dst->assigned_preg() == n_reg) {
            if (cand[0] == NULL) 
                cand[0] = i;
            else if (cand[1] == NULL)
                cand[1] = i;
        }
        //
        // mask out any registers that are used
        //
        if (dst != NULL && dst->assigned_preg() != n_reg)
            reg_usage &= ~(1u << dst->assigned_preg());
        unsigned j;
        for (j=0; j < i->n_srcs; j++) {
            if (i->src(j)->assigned_preg() != n_reg)
                reg_usage &= ~(1u << i->src(j)->assigned_preg());
        }
    }
    assert(i != last_inst &&
        (((Call_Inst*)i)->kind == Call_Inst::monexit_call ||
         ((Call_Inst*)i)->kind == Call_Inst::monexit_static_call));
    //
    // mask out all caller regs
    //
    reg_usage &= ~(ALL_X86_CALLER_REGS);
    //
    // assign available callee regs to candidates
    //
    int j;
    for (j = 0; j < 2; j++) {
        if (cand[j] == NULL) continue;
        unsigned avail = find_available_register(reg_usage);
        if (avail != n_reg) {
            reg_usage &= ~(1u << avail); // remove avail from reg_usage
            ((Temp_Reg*)cand[0]->dst())->set_assigned_preg((X86_Reg_No)avail);
        }
    }
}

// Assign the 4 callee-saved registers to the 4 variables with the
// highest static reference counts.  Future enhancement: weight the
// static count according to its depth in a loop nest.
static unsigned simple_reg_alloc(Flow_Graph *fg, Expressions &exprs,
                                 unsigned scratch_regs_used,
                                 char *defcount)
{
    unsigned result = 0;
    GCTrack_Operand *vararray[n_reg];  // the variables that were register-allocated
    unsigned n_reg_allocated = 0;
    // Collect reference counts on all GCTrack_Operands.
    unsigned num_vars = exprs.reg_map.curr_tmp_reg_id();
    Mem_Manager ref_mm(num_vars * sizeof(ref_type_count));
    ref_type_count *refcount = 
        (ref_type_count *)ref_mm.alloc(num_vars * sizeof(*refcount));
    unsigned i;
    for (i=0; i<num_vars; i++)
    {
        refcount[i].cnt = 0;
        refcount[i].opnd = NULL;
        defcount[i] = 0;
    }
    Ref_Counting closure(refcount,defcount);
    fg->apply(count_refs, &closure);
    // Loop through all registers.  If the register is available,
    // find the variable with the highest reference count.  If there
    // are none, quit early.  Be sure to mark which registers were
    // used, for saving/restoring in the prolog/epilog, as well as for
    // generating spill code.
    unsigned regno;
    unsigned candidate;
    unsigned regmask1, regmask2;
    // Use caller-saved registers for global register allocation only
    // if there were no calls (except possibly for exception throws)
    // and no exception handlers.  I.e., places where the caller-saved
    // registers get killed.
    //
    // New model: don't unconditionally disable the use of scratch
    // registers when there are exception handlers.  Instead, disable
    // on a case-by-case basis when the candidate variable is live in
    // some exception handler.
    if (closure.was_call/* || fg->handlers()->next() != fg->handlers()*/)
    {
        regmask1 = regmask2 = ALL_X86_CALLEE_REGS;
    }
    else
    {
        regmask1 = ALL_X86_CALLEE_REGS | (~scratch_regs_used & ALL_X86_CALLER_REGS);
        regmask2 = ALL_X86_CALLEE_REGS;
    }
    unsigned r;
    for (r=0; r<n_reg; r++)
    {
        //if (regmask & (1u << regno))
        {
            candidate = var_with_highest_ref_count(refcount, num_vars);
            if (candidate == 0)
                break;  // all vars are register-allocated
            int oldrefcount = refcount[candidate].cnt;
            if (oldrefcount <= 2)
                break;
            // Now we have a candidate, so try to get a register.
            regno = n_reg;
            if (!live_in_exc_handler(candidate, fg))
                regno = find_available_register(regmask1);
            if (regno == n_reg)
                regno = find_available_register(regmask2);
            if (regno == n_reg)
                break;  // no registers left
            // Update the reg masks
            regmask1 &= ~(1u << regno);
            regmask2 &= ~(1u << regno);
            refcount[candidate].cnt = -1; // make sure we don't process it again
            Operand *opnd = refcount[candidate].opnd;
            GCTrack_Operand *vopnd = (GCTrack_Operand *)opnd;
            vararray[n_reg_allocated++] = vopnd;
            assert(vopnd->assigned_preg() == n_reg);
            vopnd->set_assigned_preg((X86_Reg_No)regno);
            result |= (1u << regno);
#ifdef TRACE_O3xxx
            cout << "simple_reg_alloc: assigning " << X86_Reg_Str[regno] << " to variable " << vopnd->id
                << " (" << oldrefcount << " references)" << endl;
#endif // TRACE_O3
        }
    }
    if (n_reg_allocated > 0)
    {
        Liveness_Update lu(vararray, n_reg_allocated);
        fg->apply(update_liveness, &lu);
    }
    // Returns a bit mask representing the callee-saved registers that were assigned.
    result = result & ALL_X86_CALLEE_REGS;
    //
    // keep return values in callee-save registers to avoid spilling across
    // monitorexit calls
    //
    if (method_is_synchronized(fg->m_handle())) {
        keep_return_values_in_regs(fg->epilog(), result, fg->m_handle());
    }
    return result;
}


#define ISPREG(opnd) ((opnd)->assigned_preg() != n_reg || (opnd)->is_fp_stk())
#define ISIMM(opnd) ((opnd)->kind == Operand::Immediate)
#define LR_ENDS(inst, bitpos) ((inst)->live_ranges_ended & (1u << (bitpos)))
#define AVAIL(opnd, avail) ((1u << (opnd)->bv_position()) & (avail))

static void SET_BC(Inst *new_inst, Inst *old_inst, bool is_prepended)
{
#ifdef PRINTABLE_O3
    new_inst->bc_index = old_inst->bc_index;
    new_inst->is_spill = (is_prepended ? -1 : 1);
#endif // PRINTABLE_O3
}

// We choose a register to spill by looking at the spill candidates.  We can spill
// caller-saved registers, and any callee-saved registers saved in the prolog.
// We can't spill anything in the "dont_spill" set, or anything that's already
// been spilled.  Also, if a byte register is required, we have to restrict ourselves
// to those registers.
//
// When we spill the register, we look up both the register and the spill location
// with an "unknown" type.  We generate an assignment of the register to the spill
// location.  We then look up another copy of the register, with the correct input
// type.
static Operand *spill_register(unsigned &spilled, unsigned dont_spill,
                               Inst *inst_head, Spill_Closure *sc, O3_Jit_Type ty,
                               int bytereg)
{
    unsigned avail_to_spill = (ALL_X86_CALLER_REGS | sc->callee) & ~dont_spill & ~spilled;
    if (bytereg)
        avail_to_spill &= ALL_X86_BYTE_REGS;
    assert(avail_to_spill);
    unsigned regno;
    for (regno=0; regno<n_reg; regno++)
    {
        if (avail_to_spill & (1u << regno))
        {
            Operand *assn_reg = sc->exprs.lookup_reg_exp(regno, JIT_TYPE_UNKNOWN, 0)->opnd;
            Physical_Reg *assn_preg = (Physical_Reg *)assn_reg;
            if (assn_preg->spill == NULL)
            {
                // Assign the register a spill location if we haven't already.
                Virtual_Reg *vreg = new (sc->mm)
                    Virtual_Reg(sc->exprs.reg_map.get_tmp_reg_id(0), JIT_TYPE_UNKNOWN);
                assn_preg->spill = vreg;
            }
            SET_BC(new(sc->mm)
                Assign_Inst(assn_preg->spill, assn_reg, inst_head->exp, inst_head),
                inst_head, true);
            sc->spill_manager.set_entry(assn_preg->spill, assn_reg);
            spilled |= (1u << regno);
            // Save a pointer to the register so that we can restore it later.
            sc->preg_spill_location[regno] = assn_reg;
            Operand *ret_reg = sc->exprs.lookup_reg_exp(regno, ty, 0)->opnd;
            return ret_reg;
        }
    }
    assert(0);
    return NULL;
}

static void prepend_assn(Operand *lhs, Operand *rhs, Inst *inst, Spill_Closure *sc)
{
    if (lhs != rhs)
    {
        if ( rhs->is_single_def_temp_reg() && 
            ((Temp_Reg*)rhs)->inst()->is_imm_assignment())
            rhs = ((Temp_Reg*)rhs)->inst()->src(0); // imm
        Assign_Inst *new_inst = new(sc->mm) Assign_Inst(lhs, rhs, inst->exp, inst);
        // prevent carry bit being destroyed by spill code (long operations)
        if (lhs->type == JIT_TYPE_LONG || rhs->type == JIT_TYPE_LONG)
            new_inst->reset_ok_with_carry();
        SET_BC(new_inst, inst, true);
    }
}

static Operand *get_available_register(Spill_Closure *sc, unsigned &old_avail, O3_Jit_Type ty,
                                       Operand *positive_pref, Operand *negative_pref,
                                       unsigned &spilled, Inst *inst_head,
                                       Operand *dont_spill_1, Operand *dont_spill_2, Operand *dont_spill_3,
                                       int use_fpstk_for_double, int bytereg)
{
    if (use_fpstk_for_double && ty == JIT_TYPE_DOUBLE)
        return sc->exprs.fp_stk_opnd(0);
    unsigned avail = old_avail;
    if (bytereg)
        avail &= ALL_X86_BYTE_REGS;
    if (!avail)
    {
        unsigned dont_spill = 0;
        if (dont_spill_1 != NULL && ISPREG(dont_spill_1))
            dont_spill |= (1u << dont_spill_1->bv_position());
        if (dont_spill_2 != NULL && ISPREG(dont_spill_2))
            dont_spill |= (1u << dont_spill_2->bv_position());
        if (dont_spill_3 != NULL && ISPREG(dont_spill_3))
            dont_spill |= (1u << dont_spill_3->bv_position());
        return spill_register(spilled, dont_spill, inst_head, sc, ty, bytereg);
    }
    unsigned positive = (positive_pref == NULL ? 0 : (1u << positive_pref->bv_position()));
    unsigned negative = (negative_pref == NULL ? 0 : (1u << negative_pref->bv_position()));
    if (avail & positive)
        avail &= positive;
    if (avail & ~negative)
        avail &= ~negative;
    unsigned i = sc->spill_manager.get_register(avail);
    old_avail &= ~(1u << i);
    return sc->exprs.lookup_reg_exp(i, ty, 0)->opnd;
    assert(0);
    return NULL;
}

// Find a register, spilling if necessary.  Emits an assignment to the register,
// and an assignment to a spill location if necessary.  Returns the register obtained.
static Operand *assign_register_any(Operand *rhs,           // what to assign to register
                                    Spill_Closure *sc,      // used for exprs and mm
                                    unsigned &avail,        // currently available registers
                                    Operand *positive_pref, // take this if it's an available register
                                    Operand *negative_pref, // don't take this if possible
                                    unsigned &spilled,      // registers spilled so far
                                    Inst *inst_head,        // current instruction
                                    Operand *dont_spill_1,  // don't spill this register
                                    Operand *dont_spill_2,  // don't spill this register either
                                    Operand *dont_spill_3,  // don't spill this register either
                                    int bytereg             // require one of ALL_X86_BYTE_REGS
                                    )
{
    // lookup if any register still holds rhs
    Operand *result = sc->spill_manager.lookup(rhs);
    if (result != NULL) {
        avail &= ~(1u << result->assigned_preg());
        return result;
    }

    result = get_available_register(sc, avail, rhs->type, positive_pref, negative_pref,
        spilled, inst_head, dont_spill_1, dont_spill_2, dont_spill_3, 0, bytereg);
    prepend_assn(result, rhs, inst_head, sc);
    sc->spill_manager.set_entry(result,rhs);
    return result;
}

// This is like assign_register_any(), except for 3 differences.  First, it uses "rhs"
// if rhs is already a physical register (even if it's not listed as available).
// Second, if the "immediate_ok" flag is set, and rhs is an immediate, it doesn't actually
// obtain a new register.  Third, if the "fp_ok" flag is set, and "rhs" is a double,
// it takes fp_stk(0) unconditionally.
static Operand *assign_register_ispreg(Operand *rhs, Spill_Closure *sc, unsigned &avail,
                                       Operand *positive_pref, Operand *negative_pref,
                                       unsigned &spilled, Inst *inst_head,
                                       Operand *dont_spill_1, Operand *dont_spill_2, Operand *dont_spill_3,
                                       int immediate_ok, int fp_ok, int bytereg)
{
    // lookup if any register still holds rhs
    Operand *result = sc->spill_manager.lookup(rhs);
    if (result != NULL && (!bytereg || is_byte_reg(result))) {
        avail &= ~(1u << result->assigned_preg());
        return result;
    }

#ifndef DEBUG_SPILL
    if ((ISPREG(rhs) && (!bytereg || is_byte_reg(rhs)) && !(spilled & (1u << rhs->bv_position())))
        || (immediate_ok && ISIMM(rhs)))
        result = rhs;
    else
#endif // DEBUG_SPILL
    {
        result = get_available_register(sc, avail, rhs->type, positive_pref, negative_pref,
            spilled, inst_head, dont_spill_1, dont_spill_2, dont_spill_3, fp_ok, bytereg);
        prepend_assn(result, rhs, inst_head, sc);
        sc->spill_manager.set_entry(result,rhs);
    }
    return result;
}

static Operand *get_register_ispreg(Operand *rhs, Spill_Closure *sc, unsigned &avail,
                                    Operand *positive_pref, Operand *negative_pref,
                                    unsigned &spilled, Inst *inst_head,
                                    Operand *dont_spill_1, Operand *dont_spill_2, Operand *dont_spill_3,
                                    int immediate_ok, int fp_ok, int bytereg)
{
    // lookup if any register still holds rhs
    Operand *result = sc->spill_manager.lookup(rhs);
    if (result != NULL && (!bytereg || is_byte_reg(result))) {
        avail &= ~(1u << result->assigned_preg());
        return result;
    }

#ifndef DEBUG_SPILL
    if ((ISPREG(rhs) && !bytereg && !(spilled & (1u << rhs->bv_position()))) || (immediate_ok && ISIMM(rhs)))
        result = rhs;
    else
#endif // DEBUG_SPILL
    {
        result = get_available_register(sc, avail, rhs->type, positive_pref, negative_pref,
            spilled, inst_head, dont_spill_1, dont_spill_2, dont_spill_3, fp_ok, bytereg);
        //prepend_assn(result, rhs, inst_head, sc);
    }
    sc->spill_manager.set_entry(result,rhs);
    return result;
}

static Operand *get_register_ispreg_lrends(Operand *rhs, unsigned pos, Spill_Closure *sc, unsigned &avail,
                                           Operand *positive_pref, Operand *negative_pref,
                                           unsigned &spilled, Inst *inst_head,
                                           Operand *dont_spill_1, Operand *dont_spill_2, Operand *dont_spill_3,
                                           int bytereg)
{
    // lookup if any register still holds rhs
    Operand *result = sc->spill_manager.lookup(rhs);
    if (result != NULL && (!bytereg || is_byte_reg(result))) {
        avail &= ~(1u << result->assigned_preg());
        return result;
    }

#ifndef DEBUG_SPILL
    if (ISPREG(rhs) && !bytereg && LR_ENDS(inst_head, pos) && !(spilled & (1u << rhs->bv_position())))
        result = rhs;
    else
#endif // DEBUG_SPILL
    {
        result = get_available_register(sc, avail, rhs->type, positive_pref, negative_pref,
            spilled, inst_head, dont_spill_1, dont_spill_2, dont_spill_3, 0, bytereg);
    }
    sc->spill_manager.set_entry(result,rhs);
    return result;
}

// This is like assign_register_ispreg(), except that rhs can only be used if it is a
// physical register and its live range ends in this instruction.  "pos" gives the bit
// position to check in inst_head->live_ranges_ended.
static Operand *assign_register_ispreg_lrends(Operand *rhs, unsigned pos, Spill_Closure *sc, unsigned &avail,
                                              Operand *positive_pref, Operand *negative_pref,
                                              unsigned &spilled, Inst *inst_head,
                                              Operand *dont_spill_1, Operand *dont_spill_2, Operand *dont_spill_3,
                                              int bytereg)
{
    // lookup if any register still holds rhs
    Operand *result = sc->spill_manager.lookup(rhs);
    if (result != NULL && (!bytereg || is_byte_reg(result))) {
        avail &= ~(1u << result->assigned_preg());
        return result;
    }

#ifndef DEBUG_SPILL
    if (ISPREG(rhs) && (!bytereg || ((1u << rhs->bv_position()) & ALL_X86_BYTE_REGS)) &&
        (LR_ENDS(inst_head, pos) && !(spilled & (1u << rhs->bv_position()))) || rhs->is_fp_stk())
        result = rhs;
    else
#endif // DEBUG_SPILL
    {
        result = get_available_register(sc, avail, rhs->type, positive_pref, negative_pref,
            spilled, inst_head, dont_spill_1, dont_spill_2, dont_spill_3, 0, bytereg);
        prepend_assn(result, rhs, inst_head, sc);
    }
    sc->spill_manager.set_entry(result,rhs);
    return result;
}

// Insert the restore *before* inst_head.
static void restore_registers(unsigned spilled, Spill_Closure *sc, Inst *inst_head)
{
    if (spilled == 0)
        return;
    unsigned regno;
    for (regno=0; regno<n_reg; regno++)
    {
        if (spilled & (1u << regno))
        {
            Operand *reg = sc->preg_spill_location[regno];
            Physical_Reg *preg = (Physical_Reg *)reg;
            Operand *spill = preg->spill;
            SET_BC(new(sc->mm) Assign_Inst(reg, spill, inst_head->prev()->exp, inst_head),
                inst_head->prev(), false);
            sc->spill_manager.reset_entry(regno);
        }
    }
}

static void append_assn(Operand *lhs, Operand *rhs, Inst *inst, Spill_Closure *sc)
{
//    if (lhs != rhs)
        SET_BC(new(sc->mm) Assign_Inst(lhs, rhs, inst->exp, inst->next()), inst, false);
}

//
// generate array_operand [b + i*arry->shift + arry->offset]
//
Operand *clone_array_operand(Mem_Manager& mm, Operand *b, 
                                    Operand *i, Operand *ary) {
    assert(ary->is_array());
    Array_Operand *a = (Array_Operand*)ary;
    Operand *result = new (mm) Array_Operand((Reg_Operand*)b,(Reg_Operand*)i,
        a->bound(),a->shift(),a->offset(),a->type);
    if (ary->hi_opnd() != NULL)
        result->set_hi_opnd(clone_array_operand(mm, b, i, ary->hi_opnd()));
    return result;
}

//
// generate field_operand [b + fld->offset]
//
Operand *clone_field_operand(Mem_Manager& mm, Operand *b, Operand *fld) {
    assert(fld->kind == Operand::Field && b->is_reg());
    Field_Operand *f = (Field_Operand*)fld;
    bool excp = f->may_throw_null_ptr_excp();
    Field_Operand *new_f = new (mm) Field_Operand((Reg_Operand*)b,f->offset(),f->type,f->fid,excp);
    if (f->was_originally_array_opnd())
        new_f->set_was_originally_array_opnd();
    if (fld->hi_opnd() != NULL)
        new_f->set_hi_opnd(clone_field_operand(mm, b, fld->hi_opnd()));
    return new_f;
}

static void replace_mem_access(Spill_Reg_Manager& spill_manager, Inst *inst, 
                               unsigned ith, Operand*& src, unsigned& avail) {
#if 0
    bool is_fp_dbl = IS_FP_DBL_TYPE(inst->type());
    if (!((is_fp_dbl && IS_FP_DBL_TYPE(src->type)) || (!is_fp_dbl && !IS_FP_DBL_TYPE(src->type))))
        cout << "here" << endl;
#endif // 0
    bool is_fp_dbl = IS_FP_DBL_TYPE(src->type);
    if (!is_fp_dbl && src->assigned_preg() == n_reg) {
        Operand *new_src = spill_manager.lookup(src);
        if (new_src != NULL) {
            src = new_src;
            unsigned preg = new_src->assigned_preg();
            unsigned old_avail = (avail & (1u << preg));
            avail &= ~(1u << preg);
            inst->replace_src(ith,new_src);
#if 1
            // Mark in the instruction that the live range does NOT end,
            // so that assign_register_ispreg_lrends() doesn't accidentally
            // reuse the register when it shouldn't.
            if (!old_avail)
                inst->live_ranges_ended &= ~(1u << (2*ith + 2));
            else // XXX- Is this safe???
                inst->live_ranges_ended |= (1u << (2*ith + 2));
#endif // 1
        }
    }
}

static void replace_mem_access_field(Spill_Reg_Manager& spill_manager, Inst *inst, 
                                     unsigned ith, unsigned& avail,
                                     Mem_Manager &mm) {
    if (ith == -1)
    {
        Operand *src = inst->dst();
        assert(src->kind == Operand::Field);
        Operand *base = src->base();
        if (base->assigned_preg() == n_reg) {
            Operand *new_base = spill_manager.lookup(base);
            if (new_base != NULL) {
                src = clone_field_operand(mm, new_base, src);
                unsigned preg = new_base->assigned_preg();
                unsigned old_avail = (avail & (1u << preg));
                avail &= ~(1u << preg);
                inst->set_dst(src);
#if 1
                // Mark in the instruction that the live range does NOT end,
                // so that assign_register_ispreg_lrends() doesn't accidentally
                // reuse the register when it shouldn't.
                if (!old_avail)
                    inst->live_ranges_ended &= ~(1u << 0);
#endif // 1
            }
        }
        return;
    }
    Operand *src = inst->src(ith);
    assert(src->kind == Operand::Field);
    Operand *base = src->base();
    if (base->assigned_preg() == n_reg) {
        Operand *new_base = spill_manager.lookup(base);
        if (new_base != NULL) {
            src = clone_field_operand(mm, new_base, src);
            //src = new_src;
            unsigned preg = new_base->assigned_preg();
            unsigned old_avail = (avail & (1u << preg));
            avail &= ~(1u << preg);
            inst->replace_src(ith,src);
#if 1
            // Mark in the instruction that the live range does NOT end,
            // so that assign_register_ispreg_lrends() doesn't accidentally
            // reuse the register when it shouldn't.
            if (!old_avail)
                inst->live_ranges_ended &= ~(1u << (2*ith + 2));
#endif // 1
        }
    }
}

static void replace_mem_access_array(Spill_Reg_Manager& spill_manager, Inst *inst, 
                                     unsigned ith, unsigned& avail,
                                     Mem_Manager &mm) {
    if (ith == -1)
    {
        Operand *src = inst->dst();
        assert(src->kind == Operand::Array);
        Operand *base = src->base();
        if (base->assigned_preg() == n_reg) {
            Operand *new_base = spill_manager.lookup(base);
            if (new_base != NULL) {
                src = clone_array_operand(mm, new_base, src->index(), src);
                //src = new_src;
                unsigned preg = new_base->assigned_preg();
                unsigned old_avail = (avail & (1u << preg));
                avail &= ~(1u << preg);
                inst->set_dst(src);
#if 1
                // Mark in the instruction that the live range does NOT end,
                // so that assign_register_ispreg_lrends() doesn't accidentally
                // reuse the register when it shouldn't.
                if (!old_avail)
                    inst->live_ranges_ended &= ~(1u << 0);
#endif // 1
            }
        }
        Operand *index = src->index();
        if (index->assigned_preg() == n_reg) {
            Operand *new_index = spill_manager.lookup(index);
            if (new_index != NULL) {
                src = clone_array_operand(mm, src->base(), new_index, src);
                unsigned preg = new_index->assigned_preg();
                unsigned old_avail = (avail & (1u << preg));
                avail &= ~(1u << preg);
                inst->set_dst(src);
#if 1
                // Mark in the instruction that the live range does NOT end,
                // so that assign_register_ispreg_lrends() doesn't accidentally
                // reuse the register when it shouldn't.
                if (!old_avail)
                    inst->live_ranges_ended &= ~(1u << 1);
#endif // 1
            }
        }
        return;
    }
    Operand *src = inst->src(ith);
    assert(src->kind == Operand::Array);
    Operand *base = src->base();
    if (base->assigned_preg() == n_reg) {
        Operand *new_base = spill_manager.lookup(base);
        if (new_base != NULL) {
            src = clone_array_operand(mm, new_base, src->index(), src);
            //src = new_src;
            unsigned preg = new_base->assigned_preg();
            unsigned old_avail = (avail & (1u << preg));
            avail &= ~(1u << preg);
            inst->replace_src(ith,src);
#if 1
            // Mark in the instruction that the live range does NOT end,
            // so that assign_register_ispreg_lrends() doesn't accidentally
            // reuse the register when it shouldn't.
            if (!old_avail)
                inst->live_ranges_ended &= ~(1u << (2*ith + 2));
#endif // 1
        }
    }
    Operand *index = src->index();
    if (index->assigned_preg() == n_reg) {
        Operand *new_index = spill_manager.lookup(index);
        if (new_index != NULL) {
            src = clone_array_operand(mm, src->base(), new_index, src);
            unsigned preg = new_index->assigned_preg();
            unsigned old_avail = (avail & (1u << preg));
            avail &= ~(1u << preg);
            inst->replace_src(ith,src);
#if 1
            // Mark in the instruction that the live range does NOT end,
            // so that assign_register_ispreg_lrends() doesn't accidentally
            // reuse the register when it shouldn't.
            if (!old_avail)
                inst->live_ranges_ended &= ~(1u << (2*ith + 3));
#endif // 1
        }
    }
}

// Case 1: a=b+c (no memory operands)
// Do nothing if a==b && (ispreg(a) || ispreg(c))
// Get register r1.  Preferences:
//   1. b, if ispreg(b) && lr_ends(b)
//   2. a, if ispreg(a) && avail(a)
//   3. Commute and select c, if ispreg(c)
//   4. any available register
//   5. spill something that's not c
// r1=b; r1+=c; a=r1;
//
// status=b+c
// Do nothing if ispreg(b) || ispreg(c)
// Get register r1.  Preferences:
//   1. any available register
//   2. spill something
// r1=b; status=r1+c
int is_power2(int v);
static void case1(Inst *inst, unsigned avail, Spill_Closure *sc, unsigned &spilled)
{
    Operand *a = inst->dst();
    Operand *b = inst->src(0);
    Operand *c = inst->src(1);
    if (IS_FP_DBL_TYPE(b->type))
        return;
    Operand *new_rhs;
    if (ISIMM(b) && ISIMM(c) && inst->is_copy_prop_assignment(sc->exprs, b, c, NULL, new_rhs))
    {
        append_assn(a, new_rhs, inst, sc);
        sc->spill_manager.reset_entry(a);
        inst->unlink();
        return;
    }
    replace_mem_access(sc->spill_manager,inst,0,b,avail);
    replace_mem_access(sc->spill_manager,inst,1,c,avail);
    Operand *r1;
    if (a->kind == Operand::Status)
    {
        if (ISPREG(b) || ISPREG(c))
            return;
        r1 = assign_register_any(b, sc, avail, NULL, NULL, spilled, inst, NULL, NULL, NULL, 0); // r1=b
        // status=r1+c
        inst->replace_src(0, r1);
        return;
    }
    // Do nothing if a==b && (ispreg(a) || ispreg(c))
    if ((a == b || (b->kind == Operand::GCTrack && a->bv_position() == b->bv_position())) &&
        (ISPREG(a) || ISPREG(c) || ISIMM(c)))
    {
        sc->spill_manager.reset_entry(a);
        return;
    }
    // Try to use the lea instruction.
    if (inst->can_use_lea())
    {
        sc->spill_manager.reset_entry(a);
        return;
    }
    if ((a == c || (ISPREG(a) && a->assigned_preg() == c->assigned_preg()))
        && inst->is_commutable())
    {
        r1 = c;
        Operand *tmp = b;
        b = c;
        c = tmp;
        inst->replace_src(0, b);
        inst->replace_src(1, c);
        sc->spill_manager.reset_entry(a);
        // Swap the portions of live_ranges_ended.
        unsigned char lre = inst->live_ranges_ended;
        lre = (lre & 0xc3) | ((lre & 0x0c) << 2) | ((lre & 0x30) >> 2);
        inst->live_ranges_ended = lre;
    }
    /*else*/ if (inst->is_mul() && ((Mul_Inst*)inst)->kind == Mul_Inst::mul && 
        ISPREG(a) && ISIMM(c) && is_power2(((Imm_Operand *)c)->imm()) == -1)
    {
        sc->spill_manager.reset_entry(a);
        return;
    }
    else if (ISPREG(b) && LR_ENDS(inst, 2))
        r1 = b;
    else if (ISPREG(a) && AVAIL(a, avail))
        r1 = a;
    else
        r1 = get_available_register(sc, avail, b->type, NULL, NULL, spilled, inst, c, NULL, NULL, 0, 0);
    prepend_assn(r1, b, inst, sc); // r1=b
    // r1=r1+c
    inst->set_dst(r1);
    if (!inst->can_use_lea())
        inst->replace_src(0, r1);
    inst->reset_marker();
    sc->spill_manager.reset_entry(r1);
    append_assn(a, r1, inst, sc); // a=r1
    sc->spill_manager.set_entry(a, r1);
}

// Case 2: a=b+[c,d]
// Set use_lea=(num_avail-(!ispreg(b)||!lr_ends(b))-!ispreg(c)-!ispreg(d) < 0)
// (i.e., use the "lea" instruction to avoid spilling more than 2)
// Get register r1.  Preferences:
//   1. c, if ispreg(c) && (!use_lea || lr_ends(c))
//   2. a, if ispreg(a) && avail(a) && use_lea
//   3. any available register, preferably NOT a if !use_lea
//   4. spill something that's not b or d
// Mark r1 as no longer available.
// Get register r2.  Preferences:
//   1. d, if ispreg(d) && (!use_lea || lr_ends(d))
//   2. any available register, preferably NOT a if !use_lea
//   3. spill something that's not r1 or b
// Mark r2 as no longer available.
// If !use_lea, get register r3.  Preferences:
//   1. b, if ispreg(b) && lr_ends(b)
//   2. any available register, preferably a.  assert(some_reg_avail)
// r1=c; r2=d; r2=lea[r1,r2]; r1=b; r1+=[r2]; a=r1;  (if use_lea)
// r1=c; r2=d; r3=b; r3+=[r1,r2]; a=r3;  (if !use_lea)
//
// status=b+[c,d]
// Get register r1.  Preferences:
//   1. b, if ispreg(b)
//   2. any available register
//   3. spill something that's not c or d
// Get register r2.  Preferences:
//   1. c, if ispreg(c)
//   2. any available register
//   3. spill something that's not r1 or d
// Get register r3.  Preferences:
//   1. d, if ispreg(d)
//   2. any available register
//   3. spill something that's not r1 or r2
// r1=b; r2=c; r3=d; status=r1+[r2,r3]
static void case2(Inst *inst, unsigned avail, Spill_Closure *sc, unsigned &spilled)
{
    Operand *a = inst->dst();
    Operand *b = inst->src(0);
    replace_mem_access(sc->spill_manager,inst,0,b,avail);
    replace_mem_access_array(sc->spill_manager, inst, 1, avail, sc->mm);
    Operand *c = inst->src(1)->base();
    Operand *d = inst->src(1)->index();
    Operand *r1, *r2, *r3;
    if (a->kind == Operand::Status)
    {
        r1 = assign_register_ispreg(b, sc, avail, NULL, NULL, spilled, inst, c, d, NULL, 0, 0, 0); // r1=b
        r2 = assign_register_ispreg(c, sc, avail, NULL, NULL, spilled, inst, r1, d, NULL, 0, 0, 0); // r2=c
        r3 = assign_register_ispreg(d, sc, avail, NULL, NULL, spilled, inst, r1, r2, NULL, 0, 0, 0); // r3=d
        // status=r1+[r2,r3]
        inst->replace_src(0, r1);
        inst->replace_src(1, clone_array_operand(sc->mm,r2,r3,inst->src(1)));
        return;
    }
    r1 = assign_register_ispreg(c, sc, avail, NULL, a, spilled, inst, a, b, d, 0, 0, 0); // r1=c
    r2 = assign_register_ispreg(d, sc, avail, NULL, a, spilled, inst, a, r1, b, 0, 0, 0); // r2=d
    r3 = assign_register_ispreg_lrends(b, 2, sc, avail, a, NULL, spilled, inst, r1, r2, NULL, 0); // r3=b
    // r3=r3+[r1,r2]
    inst->set_dst(r3);
    inst->replace_src(0, r3);
    inst->replace_src(1, clone_array_operand(sc->mm,r1,r2,inst->src(1)));
    inst->reset_marker();
    sc->spill_manager.reset_entry(r3);
    append_assn(a, r3, inst, sc); // a=r3
    sc->spill_manager.set_entry(a, r3);
}

// Case 5: a=b+[c]
// Get register r1.  Preferences:
//   1. c, if ispreg(c)
//   2. any available register, preferably not a
//   3. spill something that's not a or b
// Get register r2.  Preferences:
//   1. b, if ispreg(b) && lr_ends(b)
//   2. any available register, preferably a
//   3. spill something that's not r1
// r1=c; r2=b; r2+=[r1]; a=r2;
//
// status=b+[c]
// Do nothing if ispreg(b) && ispreg(c)
// Get register r1.  Preferences:
//   1. c, if ispreg(c)
//   2. any available register
//   3. spill something that's not b
// Get register r2.  Preferences:
//   1. b, if ispreg(b)
//   2. any available register
//   3. spill something that's not r1
// r1=c; r2=b; status=r2+[r1]
static void case5(Inst *inst, unsigned avail, Spill_Closure *sc, unsigned &spilled)
{
    Operand *a = inst->dst();
    Operand *b = inst->src(0);
    replace_mem_access(sc->spill_manager,inst,0,b,avail);
    replace_mem_access_field(sc->spill_manager, inst, 1, avail, sc->mm);
    Operand *c = inst->src(1)->base();
    Operand *r1, *r2;
    if (a->kind == Operand::Status)
    {
        if (ISPREG(b) && ISPREG(c))
            return;
        r1 = assign_register_ispreg(c, sc, avail, NULL, NULL, spilled, inst, b, NULL, NULL, 0, 0, 0); // r1=c
        r2 = assign_register_ispreg(b, sc, avail, NULL, NULL, spilled, inst, r1, NULL, NULL, 0, 0, 0); // r2=b
        // status=r2+[r1]
        inst->replace_src(0, r2);
        inst->replace_src(1, clone_field_operand(sc->mm,r1,inst->src(1)));
        return;
    }
    if (ISPREG(a) && ISPREG(b) && ISPREG(c) && a->bv_position() == b->bv_position())
    {
        sc->spill_manager.reset_entry(a);
        return;
    }
    r1 = assign_register_ispreg(c, sc, avail, NULL, a, spilled, inst, a, b, NULL, 0, 0, 0); // r1=c
    r2 = assign_register_ispreg_lrends(b, 2, sc, avail, a, NULL, spilled, inst, r1, NULL, NULL, 0); // r2=b
    // r2=r2+[r1]
    inst->set_dst(r2);
    inst->replace_src(0, r2);
    inst->replace_src(1, clone_field_operand(sc->mm,r1,inst->src(1)));
    inst->reset_marker();
    sc->spill_manager.reset_entry(r2);
    append_assn(a, r2, inst, sc); // a=r2
    sc->spill_manager.set_entry(a, r2);
}

// Case 8: a=b
// Do nothing if ispreg(a) || ispreg(b).
// But require ispreg(a) if inst->is_widen().
// Get register r1.  Preferences:
//   1. b, if ispreg(b)
//   2. any available register
//   3. spill something
// r1=b; a=r1;
//
// status=b
// Do nothing if ispreg(b)
// Get register r1.  Preferences:
//   1. any available register
//   2. spill something
// status=r1
static void case8(Inst *inst, unsigned avail, Spill_Closure *sc, unsigned &spilled)
{
    Operand *a = inst->dst();
    Operand *b = inst->src(0);
    replace_mem_access(sc->spill_manager,inst,0,b,avail);
    Operand *r1;
    if (a->kind == Operand::Status)
    {
        if (ISPREG(b))
            return;
        r1 = assign_register_any(b, sc, avail, NULL, NULL, spilled, inst, NULL, NULL, NULL, 0);
        // status=r1
        inst->replace_src(0, r1);
        return;
    }
    if (a == b && !inst->is_widen())
    {
        if (!inst->is_assignment())
            sc->spill_manager.reset_entry(a);
        return;
    }
    // For a=neg(b), do:  r1=b; r1=neg(r1); a=r1;
    if (inst->info()->is_unary)
    {
        r1 = assign_register_ispreg_lrends(b, 2, sc, avail, NULL, NULL, spilled, inst, NULL, NULL, NULL, 0); // r1=b
        inst->set_dst(r1);
        inst->replace_src(0, r1); // r1=op(r1)
        inst->reset_marker();
        sc->spill_manager.reset_entry(r1);
        append_assn(a, r1, inst, sc);  // a=r1
        sc->spill_manager.set_entry(a, r1);
        return;
    }
    // Do nothing if ispreg(a) || ispreg(b).
//    if ((ISPREG(a) || ISPREG(b)) && !inst->is_widen())
//        return;
    if ((ISPREG(a) || ISPREG(b) || ISIMM(b) ||
        (b->kind == Operand::Const && !((Const_Operand *)b)->need_data_space())) 
        && !inst->is_widen())
    {
        if (inst->is_assignment())
        {
            if (ISPREG(a) && ISPREG(b) && sc->spill_manager.is_redundant_assn(a, b))
            {
                // Change the statement in a way that will cause it to be killed later.
                inst->set_dst(b);
                return;
            }
            if (ISPREG(a) || ISPREG(b))
                sc->spill_manager.set_entry(a, b);
            else if (a->kind == Operand::GCTrack && !ISPREG(a) && ISIMM(b))
            {
                // This is something like "mov [esp+8], 1".  We aren't forced to
                // create spill code for this assignment.  However, the resulting
                // instruction is 8 bytes wide, causing a decode stall on the PPro.
                // So we try to split the instruction using an available physical
                // register.  If there are no completely free registers, we just
                // pay the penalty.
                unsigned free_reg = sc->spill_manager.find_avail_reg_containing(a, avail);
                if (free_reg == n_reg)
                    free_reg = sc->spill_manager.find_free_reg(avail);
                if (free_reg == n_reg) // none available
                    sc->spill_manager.reset_entry(a);
                else
                {
                    // Generate "reg=b; a=reg;"
                    Operand *reg = sc->exprs.lookup_reg_exp(free_reg, b->type, 0)->opnd;
                    inst->set_dst(reg);  // reg=b
                    append_assn(a, reg, inst, sc);  // a=reg
                    sc->spill_manager.reset_entry(reg);
                    sc->spill_manager.set_entry(a, reg);
                }
            }
            else
                sc->spill_manager.reset_entry(a);
            // If it has the form "t77 =.L Ret:eax", try to shadow it in a
            // completely unused callee-saved register.
            if (b->is_ret() && !ISPREG(a))
            {
                unsigned callee_reg = sc->spill_manager.find_free_reg(avail & ALL_X86_CALLEE_REGS);
                if (callee_reg != n_reg)
                {
                    Operand *callee = sc->exprs.lookup_reg_exp(callee_reg, b->type, 0)->opnd;
                    append_assn(callee, b, inst, sc);  // reg=b
                    sc->spill_manager.set_entry(callee, b);
                }
            }
        }
        else
        {
            sc->spill_manager.reset_entry(a);
        }
        return;
    }
#if 1
    if (!inst->is_widen())
    {
        assert(inst->is_assignment());
        if (ISPREG(a) || ISPREG(b)) {
            sc->spill_manager.set_entry(a, b);
            return;
        }
        r1 = assign_register_ispreg(b, sc, avail, NULL, NULL, spilled, inst, NULL, NULL, NULL, 1, 1, 0); // r1=b
        // a=r1
        inst->replace_src(0, r1);
    }
    else
    {
        if (ISPREG(a) && ISPREG(b) && !IS_BYTE_SIZE(a->type) && !IS_BYTE_SIZE(b->type))
//        if (ISPREG(a) && (!ISPREG(b) || is_byte_reg(b) || !IS_BYTE_SIZE(a->type)))
        {
            sc->spill_manager.reset_entry(a);
            return;
        }
        // The above doesn't work for "widen", because destination MUST be a physical register.
        // The version below does.
        // r1=b; r1=widen(r1); a=r1;   [r1 must be a byte register]
        r1 = assign_register_ispreg_lrends(b, 2, sc, avail, a, NULL, spilled, inst, NULL, NULL, NULL,
            IS_BYTE_SIZE(b->type) || IS_BYTE_SIZE(a->type));
        inst->replace_src(0, r1); // r1=widen(b)
        inst->set_dst(r1);
        sc->spill_manager.reset_entry(r1);
        // a=r1
        inst->reset_marker();
        append_assn(a, r1, inst, sc);
    }
    sc->spill_manager.set_entry(a, r1);
#else
    int need_byte_reg = (inst->is_widen() &&
        (IS_BYTE_SIZE(b->type) || IS_BYTE_SIZE(b->type)));
    if (need_byte_reg)
        need_byte_reg ++;
    r1 = assign_register_ispreg(b, sc, avail, NULL, NULL, spilled, inst, NULL, NULL, NULL, 1, 1,
        need_byte_reg); // r1=b
    // a=r1
    inst->replace_src(0, r1);
#endif
}

// Case 9: a=[b,c]
// Get register r1.  Preferences:
//   1. b, if ispreg(b) && lr_ends(b)
//   2. any available register, preferably a
//   3. spill something that's not a or c
// Get register r2.  Preferences:
//   1. c, if ispreg(c)
//   2. any available register
//   3. spill something that's not r1 or a
// r1=b; r2=c; r1=[r1,r2]; a=r1;
// NOTE: This is tricky for GC, because "r1=b" makes r1 a reference,
// and "r1=[r1,r2]" may make r1 a non-reference.
// NOTE: nothing special to do for Widen_Inst, I think.
//
// status=[b,c]
// Get register r1.  Preferences:
//   1. b, if ispreg(b)
//   2. any available register
//   3. spill something that's not c
// Get register r2.  Preferences:
//   1. c, if ispreg(c)
//   2. any available register
//   3. spill something that's not r1
// r1=b; r2=c; status=[r1,r2]
static void case9(Inst *inst, unsigned avail, Spill_Closure *sc, unsigned &spilled)
{
    replace_mem_access_array(sc->spill_manager, inst, 0, avail, sc->mm);
    Operand *a = inst->dst();
    Operand *b = inst->src(0)->base();
    Operand *c = inst->src(0)->index();
    Operand *r1a, *r1b, *r2;
    if (a == NULL || a->kind == Operand::Status)
    {
        if (ISPREG(b) && ISPREG(c))
            return;
        Operand *r1;
        r1 = assign_register_ispreg(b, sc, avail, NULL, NULL, spilled, inst, c, NULL, NULL, 0, 0, 0); // r1=b
        r2 = assign_register_ispreg(c, sc, avail, NULL, NULL, spilled, inst, r1, NULL, NULL, 0, 0, 0); // r2=c
        // status=[r1,r2]
        inst->replace_src(0, clone_array_operand(sc->mm,r1,r2,inst->src(0)));
        return;
    }
    if (ISPREG(a) && ISPREG(b) && ISPREG(c))
    {
        sc->spill_manager.reset_entry(a);
        return;
    }
    r1a = assign_register_ispreg_lrends(b, 2, sc, avail, a, NULL, spilled, inst, a, c, NULL, 0); // r1a=b
    r2 = assign_register_ispreg(c, sc, avail, NULL, NULL, spilled, inst, a, r1a, NULL, 0, 0/*XXX*/, 0); // r2=c
    // r1=[r1,r2]
    if (IS_FP_DBL_TYPE(inst->type()))
    {
        if (ISPREG(a))
            r1b = a;
        else
            r1b = sc->exprs.fp_stk_opnd(0);
    }
    else
        r1b = sc->exprs.lookup_reg_exp(r1a->bv_position(), a->type, 0)->opnd;
    inst->set_dst(r1b);
    Operand *newbase = r1a;
    if (ISPREG(b) && b->assigned_preg() != r2->assigned_preg())
        newbase = b;
    inst->replace_src(0, clone_array_operand(sc->mm,newbase,r2,inst->src(0)));
    inst->reset_marker();
    sc->spill_manager.reset_entry(r1b);
    append_assn(a, r1b, inst, sc); // a=r1b
    sc->spill_manager.set_entry(a, r1b);
}

// Case 10: [a,b]=c
// [I'm too lazy and weary to write the lea case for now.]
// Get register r1.  Preferences:
//   1. a, if ispreg(a)
//   2. any available register
//   3. spill something that's not b or c
// Get register r2.  Preferences:
//   1. b, if ispreg(b)
//   2. any available register
//   3. spill something that's not r1 or c
// Get register r3.  Preferences:
//   1. c, if ispreg(c)
//   2. any available register
//   3. spill something that's not r1 or r2
// r1=a; r2=b; r3=c; [r1,r2]=r3
static void case10(Inst *inst, unsigned avail, Spill_Closure *sc, unsigned &spilled)
{
    replace_mem_access_array(sc->spill_manager, inst, -1, avail, sc->mm);
    Operand *a = inst->dst()->base();
    Operand *b = inst->dst()->index();
    Operand *c = inst->src(0);
    Operand *r1, *r2, *r3;
    r1 = assign_register_ispreg(a, sc, avail, NULL, NULL, spilled, inst, b, c, NULL, 0, 0, 0); // r1=a
    r2 = assign_register_ispreg(b, sc, avail, NULL, NULL, spilled, inst, r1, c, NULL, 0, 0, 0); // r2=b
    r3 = assign_register_ispreg(c, sc, avail, NULL, NULL, spilled, inst, r1, r2, NULL, 1, 1,
        IS_BYTE_SIZE(inst->type())); // r3=c
    //sc->spill_manager.set_entry(r3,c, false);
    // [r1,r2]=r3
    inst->set_dst(clone_array_operand(sc->mm,r1,r2,inst->dst()));
    inst->replace_src(0, r3);
}

// Case 11: a=[b]
// Do nothing if they're both physical registers.
// Get register r1.  Preferences:
//   1. b, if ispreg(b) && lr_ends(b)
//   2. any available register, preferably a
//   3. spill something
// r1=b; r1=[r1]; a=r1;
// NOTE: This is tricky for GC, because "r1=b" makes r1 a reference,
// and "r1=[r1]" may make r1 a non-reference.
// NOTE: nothing special to do for Widen_Inst, I think.
//
// status=[b]
// Do nothing if ispreg(b)
// Get register r1,  Preferences:
//   1. any available register
//   2. spill something
// r1=b; status=[r1]
static void case11(Inst *inst, unsigned avail, Spill_Closure *sc, unsigned &spilled)
{
    replace_mem_access_field(sc->spill_manager, inst, 0, avail, sc->mm);
    Operand *a = inst->dst();
    Operand *b = inst->src(0)->base();
    Operand *r1a, *r1b;
    if (a == NULL || a->kind == Operand::Status)
    {
        Operand *r1;
        if (ISPREG(b))
            return;
        r1 = assign_register_any(b, sc, avail, NULL, NULL, spilled, inst, NULL, NULL, NULL, 0); // r1=b
        // status=[r1]
        inst->replace_src(0, clone_field_operand(sc->mm,r1,inst->src(0)));
        return;
    }
    if (ISPREG(a) && ISPREG(b))
    {
        sc->spill_manager.reset_entry(a);
        return;
    }
    if (IS_FP_DBL_TYPE(inst->type()))
    {
        // r2=b; r1=[r2]; a=r1;
        Operand *r1, *r2;
        r2 = assign_register_ispreg(b, sc, avail, NULL, NULL, spilled, inst, NULL, NULL, NULL, 0, 0, 0); // r2=b
        r1 = get_register_ispreg(a, sc, avail, NULL, NULL, spilled, inst, r2, NULL, NULL, 0, 1, 0);
        inst->set_dst(r1);
        inst->replace_src(0, clone_field_operand(sc->mm,r2,inst->src(0)));
        inst->reset_marker();
        append_assn(a, r1, inst, sc); // a=r1
        return;
    }
    r1a = assign_register_ispreg_lrends(b, 2, sc, avail, a, NULL, spilled, inst, NULL, NULL, NULL, 0); // r1a=b
    if (ISPREG(b)) {
        // r1=[b]
        r1b = sc->exprs.lookup_reg_exp(r1a->bv_position(), a->type, 0)->opnd;
        inst->set_dst(r1b);
        inst->reset_marker();
        sc->spill_manager.reset_entry(r1b);
        append_assn(a,r1b,inst,sc);
        sc->spill_manager.set_entry(a, r1b);
    } else {
#if 1
        // r2 = [r1]
        r1b = get_register_ispreg(a, sc, avail, NULL, NULL, spilled, inst, r1a, NULL, NULL, 0, 0, 0);
#else
        // r1=[r1]
        r1b = sc->exprs.lookup_reg_exp(r1a->bv_position(), a->type, 0)->opnd;
#endif
        inst->set_dst(r1b);
        inst->replace_src(0, clone_field_operand(sc->mm,r1a,inst->src(0)));
        inst->reset_marker();
        sc->spill_manager.reset_entry(r1b);
        append_assn(a, r1b, inst, sc); // a=r1b
        sc->spill_manager.set_entry(a, r1b);
    }
}

// Case 12: [a]=b
// Get register r1.  Preferences:
//   1. a, if ispreg(a)
//   2. any available register
//   3. spill something that's not b
// Get register r2.  Preferences:
//   1. b, if ispreg(b)
//   2. any available register
//   3. spill something that's not r1
// r1=a; r2=b; [r1]=r2;
static void case12(Inst *inst, unsigned avail, Spill_Closure *sc, unsigned &spilled)
{
    replace_mem_access_field(sc->spill_manager, inst, -1, avail, sc->mm);
    Operand *a = inst->dst()->base();
    Operand *b = inst->src(0);
    Operand *r1, *r2;
    if (ISPREG(a) && ISPREG(b) && (!IS_BYTE_SIZE(inst->type()) || is_byte_reg(b)))
        return;
    r1 = assign_register_ispreg(a, sc, avail, NULL, NULL, spilled, inst, b, NULL, NULL, 0, 0, 0); // r1=a
    r2 = assign_register_ispreg(b, sc, avail, NULL, NULL, spilled, inst, r1, NULL, NULL, 1, 1,
        IS_BYTE_SIZE(inst->type())); // r2=b
    // [r1]=r2
    inst->set_dst(clone_field_operand(sc->mm,r1,inst->dst()));
    inst->replace_src(0, r2);
}

// Case 13: deref b
// The source operand should be either an immediate or a register.
// If it's a register, make sure it's a physical register.
static void case13(Inst *inst, unsigned avail, Spill_Closure *sc, unsigned &spilled)
{
    Operand *b = inst->src(0);
    if (ISIMM(b))
        return;
    if (ISPREG(b))
        return;
    assert(b->is_reg());
    Operand *r1;
    r1 = assign_register_any(b, sc, avail, NULL, NULL, spilled, inst, NULL, NULL, NULL, 0); // r1=b
    inst->replace_src(0, r1);
}

static void remember_operand_pair(Bit_Vector_Group *live, Operand *opnd,
                                  unsigned ends1, unsigned ends2, Operand *&slot1, Operand *&slot2)
{
    if (ends1)
    {
        // opnd is either GCTrack, Array, or Field.
        if (opnd->kind == Operand::GCTrack)
            slot1 = opnd;
        else
            slot1 = opnd->base();
    }
    else
        slot1 = NULL;
    if (ends2)
    {
        // opnd must be Array
        slot2 = opnd->index();
    }
    else
        slot2 = NULL;
}

static void add_to_bvpmap(bvp_homeloc_mapping *bvpmap, GCTrack_Operand *gopnd)
{
    if ((gopnd->type == JIT_TYPE_CLASS || gopnd->type == JIT_TYPE_ARRAY) &&
        !gopnd->home_location_assigned())
    {
        RESIZE_ARRAY(unsigned, bvpmap->bvps, bvpmap->b_capacity, bvpmap->b_size,
            4, bvpmap->mem);
        RESIZE_ARRAY(GCTrack_Operand *, bvpmap->opnds, bvpmap->o_capacity, bvpmap->o_size,
            4, bvpmap->mem);
        bvpmap->bvps[bvpmap->b_size++] = gopnd->bv_position();
        bvpmap->opnds[bvpmap->o_size++] = gopnd;
        gopnd->set_home_location_assigned();
    }
}

static void assign_home_location(Operand *opnd, int &id,
                                 bvp_homeloc_mapping *bvpmap)
{
    if (opnd == NULL)
        return;
    if (opnd->is_arg())
    {
        add_to_bvpmap(bvpmap, (GCTrack_Operand *)opnd);
        return;
    }
    if (opnd->kind == Operand::GCTrack)
    {
        GCTrack_Operand *gopnd = (GCTrack_Operand *)opnd;
        if (!ISPREG(gopnd) && gopnd->home_location() == -1)
        {
            // Allocate 2 slots for a double.  Don't do anything for
            // a long, because it's already been expanded.  Return the
            // lower-numbered of the pair, because of the way the
            // spill_offset of the frame works.
            if (gopnd->type == JIT_TYPE_DOUBLE)
            {
                // If it's a spill operand and a high operand, don't assign anything yet.
                // If it's a spill operand and a low operand, assign for both parts.
                // If it's not a spill operand, allocate two slots.
                if (true/*gopnd->is_spill() || gopnd->is_vreg() || gopnd->is_temp_reg()*/)
                {
                    if (!gopnd->is_hi())
                    {
                        gopnd->set_home_location(id++);
                        ((GCTrack_Operand *)gopnd->hi_opnd())->set_home_location(id++);
                    }
                }
                else
                {
                    gopnd->set_home_location(id++);
                    id++;
                }
            }
            else
            {
                gopnd->set_home_location(id++);
            }
        }
        add_to_bvpmap(bvpmap, gopnd);
    }
    else
    {
        assign_home_location(opnd->base(), id, bvpmap);
        assign_home_location(opnd->index(), id, bvpmap);
    }
}

// Returns false if the instruction is an assignment from an Arg_Operand to a
// Reg_Operand.  In this case, the home locations are coalesced and the instruction
// can be unlinked.
static bool assign_home_locations(Inst *inst, int &id, bvp_homeloc_mapping *bvpmap,char *defcount)
{
    // KKK
    // the code should be merged into assign_home_location
    //
    if (inst->is_assignment() &&
        inst->dst()->is_reg() && inst->src(0)->is_reg()) {
        Reg_Operand *d = (Reg_Operand*)inst->dst();
        Reg_Operand *s = (Reg_Operand*)inst->src(0);
        if (s->assigned_preg() != n_reg ||!s->is_arg() ||
            d->assigned_preg() != n_reg || d->is_fp_stk())
            return true;
        if (1/*defcount[d->bv_position()] <= 1 && defcount[s->bv_position()] <= 1*/) {
            // The above is a hack by JMS to help out the argument coalescing.
            d->set_home_location(s->home_location());
            if (s->use_arg_home_loc()) d->set_use_arg_home_loc();
            d->id = s->id;
            d->bvp = s->bvp;
            if (d->type == JIT_TYPE_DOUBLE)
            {
                assert(s->type == JIT_TYPE_DOUBLE);
                assert(s->hi_opnd() != NULL);
                assert(d->hi_opnd() != NULL);
                assert(s->hi_opnd()->is_reg());
                assert(d->hi_opnd()->is_reg());
                Reg_Operand *s_hi = (Reg_Operand *) s->hi_opnd();
                Reg_Operand *d_hi = (Reg_Operand *) d->hi_opnd();
                if (s->is_arg())
                {
                    d->set_home_location(s_hi->home_location());
                    d_hi->set_home_location(s->home_location());
                }
                else
                {
                    d_hi->set_home_location(s_hi->home_location());
                }
                if (s_hi->use_arg_home_loc()) d_hi->set_use_arg_home_loc();
                d_hi->id = s_hi->id;
                d_hi->bvp = s_hi->bvp;
            }
            return false;
        }
    }
    return true;
}

static void update_liveness_bv(Inst *inst, Operand *dst, Operand *vars_to_kill[],
                               Bit_Vector_Group *live, Spill_Closure *sc)
{
    // Apply the instruction's liveness information to "live".
    // Find operands that are killed.
    unsigned i;
    unsigned n_opnds = 2*(inst->n_srcs+1);
    for (i=0; i<n_opnds; i++)
    {
        if (vars_to_kill[i] != NULL)
            live->mark_dead(vars_to_kill[i]->bv_position());
    }
    // The destination's live range begins.
    if (dst != NULL && dst->kind == Operand::GCTrack)
        live->mark_live(dst->bv_position(), dst->type);
    // For a call, the live ranges of the arguments and the scratch
    // registers end, and the return value begins.
    if (inst->is_call())
    {
        Call_Inst *cinst = (Call_Inst *)inst;
        //
        // a = b + c   ====>   r = b + c
        //                     a = r    <--- newly created
        // Beforp inserting spill code, arg array of cinst points to "a=b+c".
        // After inserting spill code, arg array must be updated so that arg
        // array points to "a = r" instead of "r = b + c".  Otherwise, "a = r"
        // will be eliminated by dead code elimination.
        //
        for (i = 0; i < cinst->n_args(); i++) {
            if (cinst->get_arg(i)->is_marked()) continue;
            Inst * n;
            for (n = cinst->get_arg(i)->next(); !n->is_marked(); n = n->next());
            cinst->replace_arg(i,n);
        }
        live->mark_dead((unsigned)eax_reg);
        live->mark_dead((unsigned)ecx_reg);
        live->mark_dead((unsigned)edx_reg);
        Inst *retinst = cinst->get_ret();
        if (retinst != NULL)
        {
            // Use src(0) instead of dst().  See comment in Flow_Graph_Eliminate.cpp.
            Operand *ret = retinst->src(0);
            assert(ret != NULL);
            live->mark_live(ret->bv_position(), ret->type);
            ret = ret->hi_opnd();
            if (ret != NULL)
                live->mark_live(ret->bv_position(), ret->type);
        }
        sc->spill_manager.reset_after_call();
    } else if (inst->is_branch() && !((Branch_Inst *)inst)->is_bound_branch())
        sc->spill_manager.freeze_spill_manager();
}

#ifdef _DEBUG
static void check_fpstk_consistency(Cfg_Node *node, Fp_Mimic_Stack &fpstk, Cfg_Node *epilog,
                                    bool has_fp)
{
    // If there's a single out-edge to the epilog, then there should be a single entry
    // on the fp stack, otherwise it should be empty.  This is the end of the BB, so
    // we can feel free to destroy the fpstk, e.g. by popping.
    //
    // WRONG -- we must NOT destroy the FP stack, because its state still needs to be
    // copied out!
    if (!has_fp)
    {
        assert(fpstk.is_empty());
        return;
    }
    bool contains_ret_val = false;
    if (node->out_edge_size() == 1 && node->out_edges(0) == epilog &&
        !node->is_empty_block() &&
        !node->IR_instruction_list()->prev()->exits_method() &&
        IS_FP_DBL_TYPE(method_get_return_type(node->flowgraph->m_handle()))
        )
    {
        assert(!fpstk.is_empty());
        assert(fpstk.stack_loc(0)->is_fp_stk());
        assert(fpstk.num_pushed() == 1);
        contains_ret_val = true;
    }
    Cfg_Node *succ = NULL;
    if (is_artificially_split_edge(node, succ) ||
        is_edge_into_hot_inlined(node, succ) ||
        is_edge_outof_hot_inlined(node, succ) ||
        is_edge_into_cold_inlined(node, succ) ||
        is_edge_outof_cold_inlined(node, succ))
    {}
    else
        assert(contains_ret_val || fpstk./*is_empty*/contains_only_globals() ||
        (node == epilog && fpstk.num_pushed() == 1));
}
#endif // _DEBUG

static void turn_array_opnd_into_field(Inst *inst, Operand *opnd, unsigned srcno, Mem_Manager &mm)
{
    if (opnd == NULL || opnd->kind != Operand::Array)
        return;
    Operand *index = opnd->index();
    if (index->kind != Operand::Immediate)
        return;
    Operand *base = opnd->base();
    assert(base->is_reg());
    Array_Operand *aopnd = (Array_Operand *) opnd;
    unsigned new_offset = aopnd->offset() + (((Imm_Operand *)index)->imm() << aopnd->shift());
    Field_Operand *new_opnd = new(mm) Field_Operand((Reg_Operand *)base, new_offset, opnd->type, 0,false);
    new_opnd->set_was_originally_array_opnd();
    if (srcno == -1)
        inst->set_dst(new_opnd);
    else
        inst->replace_src(srcno, new_opnd);
}

static void add_integer_compensation_code(Cfg_Node *node, Spill_Closure *sc)
{
    Cfg_Node *succ = NULL;
    if (!is_edge_outof_cold_inlined(node, succ))
        return;
    Cfg_Node *hot = NULL;
    is_edge_outof_hot_inlined(hot, succ);
    assert(hot != NULL);
    Spill_Hash_Entry *hotspill = sc->spill_regs + (NUM_SPILL_ENTRIES * hot->label);
    Inst *first_inst = node->IR_instruction_list()->next();
    Inst *last_inst = node->IR_instruction_list();
    unsigned i;

    // Add code to the end of the block to restore registers.
    for (i=0; i<NUM_SPILL_ENTRIES; i++)
    {
        Operand *opnd = hotspill[i].opnd;
        assert(opnd == NULL || opnd->kind == Operand::GCTrack);
        if (opnd != NULL &&
            succ->live->is_live_var(/*opnd->bv_position()*/((GCTrack_Operand *)opnd)->orig_id))
        {
            unsigned regmask = hotspill[i].regmask;
            assert(regmask);
            unsigned regno;
            for (regno=0; regno<n_reg; regno++)
            {
                if ((regmask & (1u << regno)) && !succ->live->is_live_var(regno))
                {
                    Operand *reg_opnd = sc->exprs.lookup_reg_exp(regno, opnd->type, 0)->opnd;
                    // XXX- inst->exp is probably the wrong thing to use.
                    new(sc->mm) Assign_Inst(reg_opnd, opnd, first_inst->exp, last_inst);
#if 0
                    cout << "BB" << node->label << ": adding "
                        << X86_Reg_Str[regno]
                        << "=v" << opnd->bv_position() << endl;
#endif // 0
                }
            }
        }
    }
}

static Cfg_Node *find_best_fpstk_config(Cfg_Node *node)
{
    Cfg_Node *result = NULL;
    Cfg_Int i;
    for (i=0; i<node->in_edge_size(); i++)
    {
        Cfg_Node *pred = node->in_edges(i);
        if (pred->did_fp_spilling())
        {
            // Just take the first available.
            result = pred;
            break;
        }
    }
    assert(result);
    return result;
}

static void remove_dead_vars_from_fp_stack(Cfg_Node *node, Fp_Mimic_Stack &fpstk)
{
    Reg_Operand *opnds[Fp_Mimic_Stack::size];
    unsigned num = 0;
    int i;
    for (i=fpstk.num_pushed()-1; i>=0; i--)
    {
        Reg_Operand *opnd = fpstk.stack_loc(i);
        if (!node->live->is_live_var(opnd->bv_position()))
            opnds[num++] = opnd;
    }
    fpstk.remove_operands(opnds, num);
}

static void fp_split_in_edges(Cfg_Node *node, Spill_Closure *sc)
{
    // First iterate through the in-edges and collect a list of predecessors
    // that need to be split.  Then process the edges.
    // We do it this way because splitting an edge is likely to change
    // the in-edge array ordering.
    Mem_Manager mm(node->in_edge_size() * sizeof(Cfg_Node *));
    Cfg_Node **array = (Cfg_Node **)mm.alloc(node->in_edge_size() * sizeof(Cfg_Node *));
    Cfg_Int num = 0;
    Cfg_Int i;
    for (i=0; i<node->in_edge_size(); i++)
    {
        Cfg_Node *pred = node->in_edges(i);
        if (pred != node && pred->did_fp_spilling() &&
            !sc->post_fpstacks[pred->label].equals(&sc->fpstk))
            array[num++] = pred;
    }
    for (i=0; i<num; i++)
    {
        Cfg_Node *pred = array[i];
        //
        // there is no direct control flow from athrow block to node (epilog)
        //
        if (pred->is_athrow_block())
            continue;

        Cfg_Node *split = pred->split_edge(sc->mm, node);
        split->set_bytecodes(~0u, 0);
        Fp_Mimic_Stack stk;
        stk.copy_from(&sc->post_fpstacks[pred->label]);
        stk.convert_to_global(&sc->fpstk, split->IR_instruction_list(), sc->exprs, sc->mm);
    }
}

static void fp_split_out_edges(Cfg_Node *node, Spill_Closure *sc)
{
    Mem_Manager mm(node->out_edge_size() * sizeof(Cfg_Node *));
    Cfg_Node **array = (Cfg_Node **)mm.alloc(node->out_edge_size() * sizeof(Cfg_Node *));
    Cfg_Int num = 0;
    Cfg_Int i;
    for (i=0; i<node->out_edge_size(); i++)
    {
        Cfg_Node *succ = node->out_edges(i);
        if ((succ == node || succ->did_fp_spilling()) &&
            !sc->pre_fpstacks[succ->label].equals(&sc->fpstk))
            array[num++] = succ;
    }
    for (i=0; i<num; i++)
    {
        Cfg_Node *succ = array[i];
        Cfg_Node *split = node->split_edge(sc->mm, succ);
        split->set_bytecodes(~0u, 0);
        Fp_Mimic_Stack stk;
        stk.copy_from(&sc->fpstk);
        stk.convert_to_global(&sc->pre_fpstacks[succ->label],
            split->IR_instruction_list(), sc->exprs, sc->mm);
    }
}

//
// return true, if there exists an operand on fpstk that is not 
// live at the entry of the node
// 
static bool contain_non_live_var(Cfg_Node        *node,
                                 Fp_Mimic_Stack& fpstk)
{
    unsigned n = fpstk.num_pushed();
    unsigned i;
    for (i = 0; i < n; i++)
    {
        Reg_Operand *opnd = fpstk.stack_loc(i);
        assert(opnd != NULL);
        if (!node->live->is_live_var(opnd->bv_position()))
            return true;
    }
    return false;
}

static void simple_spill_bb(Cfg_Node *node, Closure *c)
{
    // If this is a cold inlined node, the hot portion
    // should be visited first.
    Cfg_Node *pred = NULL;
    Spill_Closure *sc = (Spill_Closure *)c;
    //
    // initialized spill register manager
    //
    sc->spill_manager.value_in = sc->spill_regs + (NUM_SPILL_ENTRIES*node->label);
    // Now merge the spill_regs of node predecessors.
    if (node->in_edge_size() == 0 || node->eh_in_edge() != NULL)
    {
        sc->spill_manager.reset_all();
    }
    else
    {
        Cfg_Int edge;
        for (edge=0; edge<node->in_edge_size(); edge++) {
            Cfg_Node *p = node->in_edges(edge);
            sc->spill_manager.init((int)edge,sc->spill_regs + (NUM_SPILL_ENTRIES*p->label));
        }
    }

    add_integer_compensation_code(node, sc);

    char *defcount = sc->def_cnt;
    Bit_Vector_Group *live = node->live->clone(sc->mm, 0);
    live->copy_from(node->live);
    Inst *last_inst = node->IR_instruction_list();
    Inst *next_inst;
    Inst *first_inst = last_inst->next();
    Operand *vars_to_kill[2*(MAX_SRCS+1)];
    unsigned i;
    pred = NULL;
    if (sc->has_fp &&
        (is_artificially_split_edge(pred, node) ||
        is_edge_into_hot_inlined(pred, node) ||
        is_edge_into_cold_inlined(pred, node) ||
        is_edge_outof_hot_inlined(pred, node)))
        sc->fpstk.copy_from(sc->post_fpstacks + pred->label);
    else
    {
        // If we're doing global FP register allocation, we need to add
        // compensation code to the end of every predecessor node that has
        // already been processed.
        // We first choose the predecessor whose FP stack configuration we
        // will use.  (I don't know a good heuristic, so I'll just go with
        // the first available.)
        // Then we convert the other already-processed predecessors into
        // this configuration through edge splitting.
        if (!sc->has_fp || node->in_edge_size() == 0)
            sc->fpstk.reset();
        else
        {
            // First, find which predecessor already has the "best" FP stack configuration.
            Cfg_Node *master = find_best_fpstk_config(node);
            sc->fpstk.copy_from(sc->post_fpstacks + master->label);
            // Remove unwanted operands from FP stack, without generating any code.
            remove_dead_vars_from_fp_stack(node, sc->fpstk);
            // Split all already-processed predecessors.
            fp_split_in_edges(node, sc);
        }
#if 0 // what we did before trying global FP regalloc
        sc->fpstk.reset();
#else // 0
#endif // 0
    }
    if (sc->has_fp)
        sc->fpstk.copy_to(sc->pre_fpstacks + node->label);
    pred = NULL;
    //
    // fp stack is not empty and node has insts
    // if node has no inst, then first_inst is the inst head (dummy head).
    // spill_top(first_inst,...) may cause problems under this circumstance.
    //
    if (sc->has_fp && !sc->fpstk.is_empty() && first_inst != last_inst)
    {
        // Clear the FP stack.  We need to do it at the beginning of the
        // basic block because there may be push instructions that depend
        // on the values already being spilled.
        if (is_edge_into_cold_inlined(pred, node) ||
            contain_non_live_var(node,sc->fpstk))
        {
            while (!sc->fpstk.is_empty())
                sc->fpstk.spill_top(first_inst, sc->exprs, sc->mm);
        }
    }
    Inst *inst;
    for (inst=first_inst; inst!=last_inst; inst=next_inst)
    {
        next_inst = inst->next();
        if (inst->is_dead())
            continue;

        if (assign_home_locations(inst, sc->next_home_location, sc->bvpmap,defcount) == false) {
            inst->unlink();
            continue;
        }
        bool should_discard_fpstk0 = false;
        if (inst->is_call())
        {
            Call_Inst *cinst = (Call_Inst *) inst;
            assert(sc->fpstk.is_empty());
#if 0       // KEN 6-24-99
            // A call without a method handle, e.g. helper call, still needs to 
            // mimic fp stack if the return type is double/float
            //
            Method_Handle mh = cinst->get_mhandle();
            if (mh != NULL && IS_FP_DBL_TYPE(method_get_return_type(mh)))
#else
            O3_Jit_Type ty = cinst->type();
            if (IS_FP_DBL_TYPE(ty))
#endif
            {
                assert(sc->has_fp);
                sc->fpstk.push(sc->exprs.fp_stk_opnd(0), NULL, NULL, inst, sc->exprs, sc->mm);
                Inst *ret = cinst->get_ret();
                if (ret == NULL)
                    should_discard_fpstk0 = true;
            }
        }
        if (sc->has_fp)
            inst = O3_do_fp_spilling(inst, sc, should_discard_fpstk0);
        next_inst = inst->next();
        turn_array_opnd_into_field(inst, inst->dst(), -1, sc->mm);
        unsigned srcno;
        for (srcno=0; srcno<inst->n_srcs; srcno++)
            turn_array_opnd_into_field(inst, inst->src(srcno), srcno, sc->mm);
        Operand *dst = inst->dst();
        Operand *src1, *src2;
        unsigned spilled = 0;
        // Compute the current set of available registers.
        unsigned regs = live->available_physical_registers(n_reg) & sc->all_available_registers;
        // Record the operands that need to be killed after processing the instruction.
        // It's important to record them now, because the original instruction can be
        // changed during the expansion.
        if (dst != NULL && dst->kind != Operand::GCTrack)
        {
            remember_operand_pair(live, dst,
                (inst->live_ranges_ended & (1u<<0)),
                (inst->live_ranges_ended & (1u<<1)),
                vars_to_kill[0], vars_to_kill[1]);
        }
        else
            vars_to_kill[0] = vars_to_kill[1] = NULL;
        for (i=0; i<inst->n_srcs; i++)
            remember_operand_pair(live,inst->src(i),
                (inst->live_ranges_ended & (1u << (2*i+2))),
                (inst->live_ranges_ended & (1u << (2*i+3))),
                vars_to_kill[2*i+2], vars_to_kill[2*i+3]);
        if (dst == NULL)
        {
            if (inst->n_srcs != 0)
            {
                assert(inst->n_srcs == 1);
                Operand *b = inst->src(0);
                switch (b->kind)
                {
                case Operand::Field:
                    case11(inst, regs, sc, spilled);
                    break;
                case Operand::Array:
                    case9(inst, regs, sc, spilled);
                    break;
                case Operand::GCTrack:
                    replace_mem_access(sc->spill_manager, inst, 0, b, regs);
                    inst->replace_src(0, b);
                    if (inst->is_deref())
                        case13(inst, regs, sc, spilled);
                    break;
                default:
                    break;
                }
            }
        }
        else
        {
            switch (dst->kind)
            {
            case Operand::GCTrack: // a=...
            case Operand::Status:
            case Operand::Static:
                switch (inst->n_srcs)
                {
                case 0: // NextPC_Inst
                    assert(inst->is_nextpc());
                    break;
                case 3:
                    {
                        assert(ISPREG(inst->src(0)) && // not yet implemented
                            ISPREG(inst->src(1)) &&
                            (inst->src(2)->kind != Operand::Array && inst->src(2)->kind != Operand::Field));
                        Operand *src = inst->src(2);
                        replace_mem_access(sc->spill_manager,inst,2,src,regs);
                        sc->spill_manager.reset_entry(dst);
                    }
                    break;
                case 1:
                    src1 = inst->src(0);
                    switch (src1->kind)
                    {
                    case Operand::GCTrack: // Case 8: a=b
                    case Operand::Immediate:
                    case Operand::Static:
                    case Operand::Const:
                    case Operand::Status:
                        case8(inst, regs, sc, spilled);
                        break;
                    case Operand::Array: // Case 9: a=[b,c]
                        case9(inst, regs, sc, spilled);
                        break;
                    case Operand::Field: // Case 11: a=[b]
                        case11(inst, regs, sc, spilled);
                        break;
                    }
                    break;
                case 2:
                    src1 = inst->src(0);
                    src2 = inst->src(1);
                    switch (src1->kind)
                    {
                    case Operand::GCTrack:
                    case Operand::Immediate:
                    case Operand::Status:
                    case Operand::Static:
                    case Operand::Const:
                        switch (src2->kind)
                        {
                        case Operand::GCTrack: // Case 1: a=b+c
                        case Operand::Immediate:
                        case Operand::Status:
                        case Operand::Static:
                        case Operand::Const:
                            case1(inst, regs, sc, spilled);
                            break;
                        case Operand::Field: // Case 5: a=b+[c]
                            case5(inst, regs, sc, spilled);
                            break;
                        case Operand::Array: // Case 2: a=b+[c,d]
                            case2(inst, regs, sc, spilled);
                            break;
                        }
                        break;
                    case Operand::Array: // Case 3: a=[b,c]+d
                        assert(0); // not yet implemented (shouldn't happen)
                        break;
                    case Operand::Field: // Case 6: a=[b]+c
                        assert(0); // not yet implemented (shouldn't happen)
                        break;
                    }
                    break;
                default:
                    assert(0);
                    break;
                }
                break;
            case Operand::Array: // [a,b]=...
                switch (inst->n_srcs)
                {
                case 1: // Case 10: [a,b]=c
                    case10(inst, regs, sc, spilled);
                    break;
                case 2: // Case 4: [a,b]=c+d
                    assert(0); // not yet implemented (shouldn't happen)
                    break;
                default:
                    assert(0);
                    break;
                }
                break;
            case Operand::Field:// [a]=...
                switch (inst->n_srcs)
                {
                case 1: // Case 12: [a]=b
                    case12(inst, regs, sc, spilled);
                    break;
                case 2: // Case 7: [a]=b+c
                    assert(0); // not yet implemented (shouldn't happen)
                    break;
                default:
                    assert(0);
                    break;
                }
                break;
            case Operand::Immediate:
            case Operand::Const:
                assert(0);
                break;
            }
            if (inst->is_idiv())
            {
                if (dst->assigned_preg() == eax_reg)
                    sc->spill_manager.reset_entry(edx_reg);
                else if (dst->assigned_preg() == edx_reg)
                    sc->spill_manager.reset_entry(eax_reg);
                else
                    assert(0);
            }
        }

        restore_registers(spilled, sc, next_inst);

        if ((dst != NULL && IS_FP_DBL_TYPE(dst->type)) ||
            (inst->n_srcs > 0 && inst->src(0) != NULL && IS_FP_DBL_TYPE(inst->src(0)->type)) ||
            (inst->n_srcs > 1 && inst->src(1) != NULL && IS_FP_DBL_TYPE(inst->src(1)->type)) ||
            (inst->n_srcs > 2 && inst->src(2) != NULL && IS_FP_DBL_TYPE(inst->src(2)->type)))
        {
            continue;
        }
        update_liveness_bv(inst, dst, vars_to_kill, live, sc);
    }
    // Spill anything on the FP stack that's dirty.
    Cfg_Node *merge = NULL;
    if (sc->has_fp && is_edge_outof_cold_inlined(node, merge))
    {
        Cfg_Node *hot = NULL;
        is_edge_outof_hot_inlined(hot, merge);  // fill in "hot" with correct value
        sc->fpstk.convert_to(sc->post_fpstacks + hot->label,
            node->IR_instruction_list(), sc->exprs, sc->mm);
    }
    // If we're doing global FP register allocation, we need to add
    // compensation code to the beginning of every successor node that has
    // already been processed.
    Cfg_Node *succ = NULL;
    if (sc->has_fp &&
        !is_artificially_split_edge(node, succ) &&
        !is_edge_into_hot_inlined(node, succ) &&
        !is_edge_into_cold_inlined(node, succ) &&
        !is_edge_outof_hot_inlined(node, succ))
    {
        fp_split_out_edges(node, sc);
    }
#ifdef _DEBUG
    check_fpstk_consistency(node, sc->fpstk, sc->epilog, sc->has_fp);
#endif // _DEBUG
    if (sc->has_fp)
        sc->fpstk.copy_to(sc->post_fpstacks + node->label);
    node->set_did_fp_spilling();
}

// WARNING: using lea causes interior pointers, which could be a problem for GC.
// In the code sequences proposed, the base pointer is lost.
//
// To stress the spill code, turn off global register allocation, so that only
// the 3 caller-saved registers are available.
//
// Case 3: a=[b,c]+d   SHOULD NOT HAPPEN
// Get register r1.  Preferences:
//   1. b, if ispreg(b) && lr_ends(b)
//   2. a, if ispreg(a) && avail(a)
//   3. any available register
//   4. spill something that's not c or d
// Get register r2.  Preferences:
//   1. c, if ispreg(c)
//   2. any available register
//   3. spill something that's not d or r1
// r1=b; r2=c; r1=[r1,r2]; r1+=d; a=r1;
//
// Case 4: [a,b]=c+d   SHOULD NOT HAPPEN
// Set use_lea=(num_avail-!ispreg(a)-!ispreg(b)-(!ispreg(c)||!lr_ends(c)) < 0)
// Get register r1.  Preferences:
//   1. a, if ispreg(a) && (!use_lea || lr_ends(a))
//   2. any available register
//   3. spill something that's not b or c or d
// Get register r2.  Preferences:
//   1. b, if ispreg(b) && (!use_lea || lr_ends(b))
//   2. any available register
//   3. spill something that's not r1 or c or d
// If !use_lea, get register r3.  Preferences:
//   1. c, if ispreg(c) && lr_ends(c)
//   2. any available register
//   3. spill something that's not r1 or r2 or d
// r1=a; r2=b; r1=lea[r1,r2]; r2=c; r2+=d; [r1]=r2;  (if use_lea)
// r1=a; r2=b; r3=c; r3+=d; [r1,r2]=r3;  (if !use_lea)
//
// Case 6: a=[b]+c   SHOULD NOT HAPPEN
// Get register r1.  Preferences:
//   1. b, if ispreg(b) && lr_ends(b)
//   2. any available register, preferably a
//   3. spill something that's not a or c
// r1=b; r1=[r1]; r1+=c; a=r1;
//
// Case 7: [a]=b+c   SHOULD NOT HAPPEN
// Get register r1.  Preferences:
//   1. a, if ispreg(a)
//   2. any available register
//   3. spill something that's not b or c
// Get register r2.  Preferences:
//   1. b, if ispreg(b) && lr_ends(b)
//   2. any available register
//   3. spill something that's not r1 or c
// r1=a; r2=b; r2+=c; [r1]=r2;
static void simple_spill_code(Flow_Graph *fg, Expressions &exprs,char *defcount)
{
    int max_label = fg->reassign_label();
    Mem_Manager mm(max_label*sizeof(Operand*)*n_reg);
    Spill_Hash_Entry *spill_regs = (Spill_Hash_Entry*)mm.alloc(max_label*sizeof(Spill_Hash_Entry)*NUM_SPILL_ENTRIES);
    int i;
    for (i = 0; i < max_label*NUM_SPILL_ENTRIES; i++) {
        spill_regs[i].opnd = NULL;
        spill_regs[i].regmask = 0;
    }
    Fp_Mimic_Stack *pre_fpstacks = NULL;
    Fp_Mimic_Stack *post_fpstacks = NULL;
    if (fg->has_fp)
    {
        pre_fpstacks = (Fp_Mimic_Stack *) mm.alloc(max_label * sizeof(Fp_Mimic_Stack));
        post_fpstacks = (Fp_Mimic_Stack *) mm.alloc(max_label * sizeof(Fp_Mimic_Stack));
    }

    Fp_Mimic_Stack fpstk;

    unsigned regs = fg->callee_saved_registers_used() | ALL_X86_CALLER_REGS;
    Spill_Reg_Manager srm(exprs);
    Spill_Closure sc(regs, fg->callee_saved_registers_used(),
        exprs, fg->bvpmap, fg->mem_manager, srm, spill_regs, defcount,
        fpstk, pre_fpstacks, post_fpstacks, fg->epilog(), fg->has_fp);
    int num_nodes;
    Mem_Manager tmem(max_label * sizeof(Cfg_Node *));
    Cfg_Node **nodearray;
    fg->create_dataflow_ordering(tmem, nodearray, num_nodes);
    int ii;
    for (ii=num_nodes-1; ii>=0; ii--)
        simple_spill_bb(nodearray[ii], &sc);
}

static bool kills(Operand *dst, Operand *src)
{
    if (dst == NULL)
        return false;
    switch (dst->kind)
    {
    case Operand::Immediate:
    case Operand::Const:
    case Operand::Status:
        assert(0);
        return true;
        break;
    case Operand::Field:
    case Operand::Array:
    case Operand::Static:
        return (src->kind == dst->kind && src->type == dst->type);
        break;
    case Operand::GCTrack:
        return (src->bv_position() == dst->bv_position());
        break;
    }
    return true;
}

// Iterate backwards through the instructions.  If we find a compare instruction,
// then search back for a place to re-insert it.
static void schedule_block(Cfg_Node *node, Closure *c)
{
    Inst *head = node->IR_instruction_list();
    Inst *cinst;
    for (cinst = head->prev(); cinst != head; cinst = cinst->prev())
    {
        if (cinst->is_compare())
        {
            unsigned nsrc = 0;
            Operand *srcs[3];
            assert(cinst->n_srcs <= 2);
            unsigned i;
            for (i=0; i<cinst->n_srcs; i++)
            {
                Operand *src = cinst->src(i);
                if (IS_FP_DBL_TYPE(src->type))
                    return;
                if (src->kind == Operand::Field)
                    srcs[nsrc++] = src->base();
                else if (src->kind == Operand::Array)
                {
                    srcs[nsrc++] = src->base();
                    srcs[nsrc++] = src->index();
                }
                else if (src->kind == Operand::GCTrack)
                    srcs[nsrc++] = src;
            }
            assert(nsrc <= 3);
            bool found = false;
            Inst *inspt;
            Inst *prev_inst = NULL;
            // The "!found" part allows the compare to be moved only 1 instruction away.
            for (inspt=cinst->prev(); !found&&inspt!=head; found=true,inspt=inspt->prev())
            {
                if (inspt->affects_flags())
                    break;
                Operand *dst = inspt->dst();
                for (i=0; i<nsrc; i++)
                {
                    if (kills(dst, srcs[i]))
                        break;
                }
                if (i < nsrc)
                    break;
                prev_inst = inspt;
            }
#ifdef JIT_SAPPHIRE
            // Do not do for sapphire.
            if (found & false)
#else
            if (found)
#endif
            {
                // XXX- this only works if we only hoist up a single instruction.
                if (prev_inst->can_use_lea())
                    prev_inst->set_must_use_lea();
                if (prev_inst->is_assignment())
                    ((Assign_Inst *)prev_inst)->reset_ok_with_carry();
                cinst->unlink();
                cinst->insert_after(inspt);
#ifdef PRINTABLE_O3
                // Without this, the dumpjit will fail.
                cinst->bc_index = ~0u;
#endif // PRINTABLE_O3
            }
        }
    }
}

static void code_scheduling(Flow_Graph *fg)
{
    extern bool O3_do_code_scheduling;
    if (O3_do_code_scheduling)
        fg->apply(schedule_block, NULL);
}

extern bool O3_priority_regalloc;
void global_reg_allocation(Flow_Graph *fg, Expressions &exprs,
                           unsigned scratch_regs_used)
{
    unsigned priority_reg_alloc(Flow_Graph *fg, Expressions &exprs);
    unsigned regs = 0;
    //scratch_regs_used |= (1 << eax_reg);
    char *defcount = (char*)exprs.mem.alloc(exprs.reg_map.curr_tmp_reg_id());
    if (O3_priority_regalloc)
        regs = priority_reg_alloc(fg, exprs);
    else
    {
        regs = simple_reg_alloc(fg, exprs, scratch_regs_used, defcount);
    }
    fg->set_callee_saved_regs(regs);


    // This dead code elimination is done only so that the liveness information
    // can be transfered from the virtual registers to the physical registers
    // after the global register allocation.  As an optimization, we could
    // construct a quicker phase that directly transfers the liveness within
    // each basic block.
    //fg->dead_code_eliminate(exprs, false);

    fg->bvpmap = new(fg->mem_manager) bvp_homeloc_mapping(fg->mem_manager, 
                     exprs.reg_map.curr_tmp_reg_id());
#ifdef TRACE_O3
    fg->print_cfg("yyy");
#endif // TRACE_O3
    simple_spill_code(fg, exprs, defcount);
    code_scheduling(fg);
}

class Home_Loc_Closure : public Closure
{
public:
    Home_Loc_Closure(bvp_homeloc_mapping *b) :
      bvpmap(b), next_home_location(0) {}
    int next_home_location;
    bvp_homeloc_mapping *bvpmap;
};

static void home_loc_assignment(Cfg_Node *node, Closure *c)
{
    Home_Loc_Closure *hlc = (Home_Loc_Closure *) c;
    Inst *last = node->IR_instruction_list();
    Inst *inst = last->next();
    for (; inst != last; inst = inst->next())
    {
        assign_home_location(inst->dst(), hlc->next_home_location, hlc->bvpmap);
        unsigned i;
        for (i=0; i<inst->n_srcs; i++)
            assign_home_location(inst->src(i), hlc->next_home_location, hlc->bvpmap);
    }
}

void Flow_Graph::home_location_assignment()
{
    Home_Loc_Closure hlc(bvpmap);
    apply(home_loc_assignment, &hlc);
    set_home_locations(hlc.next_home_location);
    bvpmap->frame_offsets = (int *) bvpmap->mem.alloc(bvpmap->n_bits * sizeof(int));
    bvpmap->use_var_offset = (bool *) bvpmap->mem.alloc(bvpmap->n_bits * sizeof(bool));
//#ifdef _DEBUG
    unsigned ii;
    for (ii=0; ii<bvpmap->n_bits; ii++)
        bvpmap->frame_offsets[ii] = 0x23456789;
//#endif // _DEBUG
    int i;
    for (i=0; i<bvpmap->o_size; i++)
    {
        GCTrack_Operand *gopnd = bvpmap->opnds[i];
        unsigned loc = gopnd->home_location();
        if (loc != -1)
        {
            unsigned bvp = gopnd->bv_position();
            bvpmap->frame_offsets[bvp] = loc;
            bvpmap->use_var_offset[bvp] = (gopnd->is_arg() || gopnd->use_arg_home_loc());
        }
    }
}
