// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o3_jit/flow_graph_nodes.cpp,v 1.7 2002/01/10 03:12:09 xhshi Exp $
//


#include "defines.h"
#include <assert.h>
#include <string.h>
#include "flow_graph.h"

#ifdef PRINTABLE_O3
// It's OK that this is a global variable and thus not thread-safe.
static int next_unique_label = 0;
#endif // PRINTABLE_O3


Cfg_Node::Cfg_Node(Mem_Manager &mm, int lbl, Flow_Graph *fg,
                   Cfg_Node_List *linear_pred):
latest_traversal(0),
// allocate space for at least one edge (for epilogs and splitting)
_out_edges(NULL), _out_edge_size(0), _out_edge_capacity(0),
_in_edges(NULL), _in_edge_capacity(0), _in_edge_size(0),
_eh_out_edge(NULL),
_eh_in_edge(NULL),
label(lbl),
flowgraph(fg),
#ifdef PRINTABLE_O3
unique_label(next_unique_label++),
#endif // PRINTABLE_O3
#if 0
enclosing_subr(NULL), jsr_succ(NULL), jsr_pred(NULL), already_set_subr(false),
where_is_ret_addr_bb(NULL), where_is_ret_addr_subr(NULL),
#else // 0
_already_set_subr(0), _subr_info(NULL),
#endif // 0
live(NULL), idom(NULL), loop_header(NULL), _loop_depth(0),
_first_bc_idx(NO_BC_IDX), _bc_length(0), _code_offset(-1),
_code_length(-1), _mark(' '), _live_lcse(NULL), _cold_code(false), _cold_non_inlined(false),
_did_fp_spilling(false), _contain_athrow(false),
extra_info(NULL)
{
    _linearization_node = new(mm) Cfg_Node_List(this);
    _linearization_node->insert_after(linear_pred);
}

Eh_Node::Eh_Node(int lbl):
  _in_edges(NULL),
  _in_edge_capacity(0), _in_edge_size(0),
  _out_edges(NULL), _out_edge_capacity(0), _out_edge_size(0),
  label(lbl),
#ifdef PRINTABLE_O3
  unique_label(next_unique_label++),
#endif // PRINTABLE_O3
  live(NULL),
  _bounds_exception(NULL),
  latest_traversal(0)
{
}

// Add an edge from "this" to "dest".
void Cfg_Node::add_edge(Mem_Manager &mm, Cfg_Node *dest)
{
    RESIZE_ARRAY(Cfg_Node *, _out_edges, _out_edge_capacity, _out_edge_size, 2, mm);
  // add the edge
  _out_edges[_out_edge_size] = dest;
  _out_edge_size ++;

  // Next, add the back edge from "dest" to "this".

  RESIZE_ARRAY(Cfg_Node *, dest->_in_edges, dest->_in_edge_capacity,
               dest->_in_edge_size, 4, mm);

  // now add the edge
  dest->_in_edges[dest->_in_edge_size] = this;
  dest->_in_edge_size ++;
}

void Cfg_Node::replace_edge(Mem_Manager &mm, Cfg_Node *old_dest, Cfg_Node *new_dest)
{
    // Delete forward edge.
    Cfg_Int edge, i;
    for (edge=0; edge<_out_edge_size; edge++)
    {
        if (_out_edges[edge] == old_dest)
        {
            _out_edges[edge] = new_dest;
            break;
        }
    }
    // Delete backward edge
    for (edge=0; edge<old_dest->_in_edge_size; edge++)
    {
        if (old_dest->_in_edges[edge] == this)
        {
            for (i=edge+1; i<old_dest->_in_edge_size; i++)
                old_dest->_in_edges[i-1] = old_dest->_in_edges[i];
            break;
        }
    }
    old_dest->_in_edge_size --;
    // Add backward edge to the new edge
    RESIZE_ARRAY(Cfg_Node *, new_dest->_in_edges, new_dest->_in_edge_capacity,
               new_dest->_in_edge_size, 4, mm);

  // now add the edge
  new_dest->_in_edges[new_dest->_in_edge_size] = this;
  new_dest->_in_edge_size ++;
}

// Delete the edge from "this" to "dest".
void Cfg_Node::delete_edge(Cfg_Node *dest)
{
    // Delete forward edge.
    Cfg_Int edge, i;
    for (edge=0; edge<_out_edge_size; edge++)
    {
        if (_out_edges[edge] == dest)
        {
            for (i=edge+1; i<_out_edge_size; i++)
                _out_edges[i-1] = _out_edges[i];
            break;
        }
    }
    assert(edge < _out_edge_size);
    _out_edge_size --;

    // Delete backward edge.
    for (edge=0; edge<dest->_in_edge_size; edge++)
    {
        if (dest->_in_edges[edge] == this)
        {
            for (i=edge+1; i<dest->_in_edge_size; i++)
                dest->_in_edges[i-1] = dest->_in_edges[i];
            break;
        }
    }
    assert(edge < dest->_in_edge_size);
    dest->_in_edge_size --;
}

//
// swap the two outgoing edges
//
void Cfg_Node::swap_edges() {
    assert(_out_edge_size == 2);
    Cfg_Node *tmp = _out_edges[0];
    _out_edges[0] = _out_edges[1];
    _out_edges[1] = tmp;
}

Cfg_Node *Cfg_Node::split_edge(Mem_Manager &mm, Cfg_Node *dest)
{
    // Create the new node.
    Cfg_Node *new_node = new(mm) Cfg_Node(mm, label, flowgraph, dest->linearization_node()->prev());
    // Copy the source's EH node.
    if (_eh_out_edge != NULL)
        new_node->add_eh_edge(mm, _eh_out_edge);
    // Copy other relevant information from the source.
    new_node->set_enclosing_subr(get_enclosing_subr());
    new_node->set_jsr_succ(get_jsr_succ());
    new_node->set_jsr_pred(get_jsr_pred());

    replace_edge(mm, dest, new_node);
    new_node->add_edge(mm, dest);
    return new_node;
}

// Add an EH edge from "this" to "dest".
void Cfg_Node::add_eh_edge(Mem_Manager &mm, Eh_Node *dest)
{
  // There should only be at most one EH edge.
  if (_eh_out_edge != NULL)
    assert(_eh_out_edge == NULL);
  _eh_out_edge = dest;

  RESIZE_ARRAY(Cfg_Node *, dest->_in_edges, dest->_in_edge_capacity, dest->_in_edge_size, 4, mm);

  // now add the edge
  dest->_in_edges[dest->_in_edge_size] = this;
  dest->_in_edge_size ++;
}

void Cfg_Node::delete_eh_edge(Eh_Node *dest)
{
    // Delete forward edge.
    assert(_eh_out_edge == dest);
    _eh_out_edge = NULL;

    // Delete backward edge.
    Cfg_Int edge, i;
    for (edge=0; edge<dest->_in_edge_size; edge++)
    {
        if (dest->_in_edges[edge] == this)
        {
            for (i=edge+1; i<dest->_in_edge_size; i++)
                dest->_in_edges[i-1] = dest->_in_edges[i];
            break;
        }
    }
    assert(edge < dest->_in_edge_size);
    dest->_in_edge_size --;
    if (dest->_in_edge_size == 0)
    {
        dest->unlink();
        while (dest->out_edge_size() > 0)
            dest->delete_edge(dest->out_edges(dest->out_edge_size() - 1)->handler);
    }
}

void Eh_Node::delete_edge(Cfg_Node *dest)
{
    unsigned found = 0;
    Cfg_Int i, cur;
    for (i=0,cur=0; i<out_edge_size(); i++)
    {
        if (_out_edges[i].handler == dest)
        {
            found ++;
        }
        else
        {
            if (i != cur)
                _out_edges[cur] = _out_edges[i];
            cur ++;
        }
    }
    assert(found > 0);
    _out_edge_size -= found;

    unsigned found1 = 0;
    Eh_In_Header *ehin = dest->_eh_in_edge;
    for (i=cur=0; i<ehin->_eh_in_edge_size; i++)
    {
        if (ehin->_eh_in_edges[i] == this)
        {
            found1 ++;
        }
        else
        {
            if (i != cur)
                ehin->_eh_in_edges[cur] = ehin->_eh_in_edges[i];
            cur ++;
        }
    }
    assert(found1 == found);
    ehin->_eh_in_edge_size -= found1;
}

// Add an edge from "this" to "dest".
void Eh_Node::add_edge(Mem_Manager &mm, Cfg_Node *dest, void *type, unsigned cp_idx)
{
  RESIZE_ARRAY(struct Eh_Entry, _out_edges, _out_edge_capacity, _out_edge_size, 2, mm);

  // now add the edge
  _out_edges[_out_edge_size].handler = dest;
  _out_edges[_out_edge_size].class_handle = type;
  _out_edges[_out_edge_size].cp_index = cp_idx;
  _out_edge_size ++;

  // Next, add the back edge from "dest" to "this".

  if (dest->eh_in_edge() == NULL)
    dest->_eh_in_edge = new Eh_In_Header;
  RESIZE_ARRAY(Eh_Node *, dest->_eh_in_edge->_eh_in_edges,
               dest->_eh_in_edge->_eh_in_edge_capacity,
               dest->_eh_in_edge->_eh_in_edge_size, 2, mm);

  // now add the edge
  dest->_eh_in_edge->_eh_in_edges[dest->_eh_in_edge->_eh_in_edge_size] = this;
  dest->_eh_in_edge->_eh_in_edge_size ++;
}

// Returns 1 if the handlers for the two nodes are the same.
// The order of the handlers is significant.
bool Eh_Node::is_identical(Eh_Node *node)
{
    Cfg_Int i;
    
    if (_out_edge_size != node->_out_edge_size)
        return false;
    for (i=0; i<_out_edge_size; i++)
    {
        if (_out_edges[i].handler != node->_out_edges[i].handler ||
            _out_edges[i].class_handle != node->_out_edges[i].class_handle ||
            _out_edges[i].cp_index != node->_out_edges[i].cp_index)
            return false;
    }
    return true;
}

Cfg_Node *Cfg_Node::get_fallthrough()
{
    if (_out_edge_size /*!= 2*/ == 0)
        return NULL;
    return _out_edges[0];
}

Cfg_Node *Cfg_Node::get_branch_target()
{
    if (_out_edge_size == 1)
        return _out_edges[0];  // goto or fall-through
    if (_out_edge_size != 2)
        return NULL;
    return _out_edges[1];
}

//#ifdef INLINE_NATIVE
//New a stand-alone cfg node
Cfg_Node* Flow_Graph::new_cfg_node(Cfg_Node* node)
{
    Cfg_Node *new_node = new (mem_manager)
        Cfg_Node(mem_manager, _next_cfg_label++, this,
        node->linearization_node());

	return new_node ;
}
//#endif

// Splits the input node into 2, with a single edge connecting them.
// The IR_instruction_list of the original node remains unchanged,
// and the new node's IR_instruction_list is empty.  first_bc_idx and
// bc_length are now meaningless.  The new node created and returned
// is the successor.  The old node keeps its eh_in_edge, and the new
// node gets a NULL eh_in_edge.  The new node shares the old node's
// eh_out_edge.
Cfg_Node *Flow_Graph::split_cfg_node(Cfg_Node *node)
{
    Cfg_Node *succ = new (mem_manager)
        Cfg_Node(mem_manager, _next_cfg_label++, this,
        node->linearization_node());
    succ->set_enclosing_subr(node->get_enclosing_subr());
    if (node->get_jsr_succ() != NULL)
    {
        succ->set_jsr_succ(node->get_jsr_succ());
        node->get_jsr_succ()->set_jsr_pred(succ);
        node->set_jsr_succ(NULL);
    }
    //
    // add out edges to succ
    //
    Cfg_Int edge;
    for (edge=0; edge<node->out_edge_size(); edge++) 
        succ->add_edge(mem_manager, node->out_edges(edge));

    //
    // XXX remove all out edges from node
    //
    while (node->out_edge_size() > 0)
        node->delete_edge(node->out_edges(0));
    assert(node->out_edge_size() == 0);

    if (node->eh_out_edge() != NULL)
        succ->add_eh_edge(mem_manager, node->eh_out_edge());

    return succ;
}

// same as split_cfg_node, except now the predecessor is the empty block
// and the process is reversed.
Cfg_Node *Flow_Graph::split_cfg_node_pred(Cfg_Node *node)
{
    Cfg_Node *pred = new (mem_manager)
        Cfg_Node(mem_manager, _next_cfg_label++, this,
        node->linearization_node()->prev());
    pred->set_enclosing_subr(node->get_enclosing_subr());
    //
    // move all in edges from node to pred
    //
    assert(node->in_edge_size() > 0);
    while (node->in_edge_size() > 0)
        node->in_edges(0)->replace_edge(mem_manager,node,pred);
    assert(node->in_edge_size() == 0);

    // don't worry about eh edges

    return pred;
}

// insert an empty block between two nodes
Cfg_Node *Flow_Graph::splice_cfg_nodes(Cfg_Node *pred, Cfg_Node *succ)
{
    Cfg_Node *mid_node = new (mem_manager)
        Cfg_Node(mem_manager, _next_cfg_label++, this,
        pred->linearization_node());
    mid_node->set_enclosing_subr(pred->get_enclosing_subr());
    //
    // replace edges
    //
    Cfg_Int edge;
    for (edge=0; edge<pred->out_edge_size(); edge++) 
    {
        if (pred->out_edges(edge)==succ){
            pred->replace_edge(mem_manager, pred->out_edges(edge), mid_node);
		}
    }

    mid_node->add_edge(mem_manager, succ);
    return mid_node;
}

void Cfg_Node::apply(unsigned short traversal_number,
                     Apply_Func fun, Closure *c,
                     Flow_Graph *fg, 
                     bool traverse_sub_graph) {
    if (latest_traversal >= traversal_number)
        return;
    //
    // only traverse sub graph (fg)
    //
    if (traverse_sub_graph && flowgraph != fg) return;

    latest_traversal = traversal_number;
	fun(this,c);
    // Process this node's successors.  
    Cfg_Int edge;
    for (edge=0; edge<_out_edge_size; edge++) 
        _out_edges[edge]->apply(traversal_number, fun, c, fg, traverse_sub_graph);

    // Process this node's exception handler block.
    Eh_Node *eh = _eh_out_edge;
    if (eh == NULL || eh->latest_traversal >= traversal_number)
        return;

    eh->latest_traversal = traversal_number;
    Cfg_Int i;
    for (i=0; i< eh->_out_edge_size; i++)
        eh->_out_edges[i].handler->apply(traversal_number, fun, c, fg, traverse_sub_graph);
}

// Creates a copy of the Cfg_Node.  Outgoing edges of the clone point to
// the same place as the original, but there are no incoming edges in the
// clone.  Exception handler info is the same for both.
Cfg_Node *Cfg_Node::clone(Mem_Manager &m)
{
    Cfg_Node *result = new(m) Cfg_Node(m, label, flowgraph, linearization_node());
    result->_first_bc_idx = _first_bc_idx;
    result->_bc_length = _bc_length;
    if (_eh_out_edge != NULL)
        result->add_eh_edge(m, _eh_out_edge);
    Cfg_Int i;
    for (i=0; i<_out_edge_size; i++)
    {
        result->add_edge(m, _out_edges[i]);
    }
    result->extra_info = extra_info;
    result->set_enclosing_subr(get_enclosing_subr());
    result->set_jsr_succ(get_jsr_succ());
    result->set_jsr_pred(get_jsr_pred());

    return result;
}

static void linearize_1(Flow_Graph *fg, Cfg_Node *pred, 
                        Cfg_Node *node, unsigned short traversal_number,
                        Cfg_Node_List& cold_code, 
                        bool is_linearizing_cold_code,
                        Cfg_Node_List *linear_ordering) {
    Cfg_Int edge;

    if (node->latest_traversal >= traversal_number)
        return;
    Mem_Manager& mm = fg->mem_manager;
    int depth = node->loop_depth();
    //
    // beginning of a cold code 
    //
    if (node->is_cold() && !is_linearizing_cold_code) {
        node->linearization_node()->unlink();
        node->linearization_node()->insert_before(&cold_code);
        return;
    }

    Cfg_Node *succ;
    if (node->loop_header == node && // loop header
        pred->loop_depth() < depth && // enter the loop
        node->out_edge_size() == 2) {
        //
        // determine if node is an exit block
        //
        succ = NULL;
        if (depth > node->out_edges(0)->loop_depth())  // exit edge
            succ = node->out_edges(1);
        else if (depth > node->out_edges(1)->loop_depth()) // exit edge
            succ = node->out_edges(0);
        if (succ != NULL) {
            if (succ->mark() == 'R') succ->set_mark('N'); // being normalized
            linearize_1(fg, node, succ, traversal_number, 
                        cold_code, is_linearizing_cold_code,linear_ordering);
            return;
        }
    }

    node->latest_traversal = traversal_number;
    //
    // traverse exit edges first
    //
    for (edge=0; edge<node->out_edge_size(); edge++) {
        succ = node->out_edges(edge);
        if (succ->loop_depth() < depth)
            linearize_1(fg, node, node->out_edges(edge), traversal_number, 
                        cold_code, is_linearizing_cold_code,linear_ordering);
    }
    //
    // traverse edges within the loop 
    //
    for (edge=node->out_edge_size(); edge > 0; edge--) {
        succ = node->out_edges(edge-1);
        if (succ->loop_depth() >= depth)
            linearize_1(fg, node, node->out_edges(edge-1), traversal_number, 
                        cold_code, is_linearizing_cold_code,linear_ordering);
    }
    //
    // topological sort ordering
    //
    Cfg_Node_List *list = new(mm)Cfg_Node_List(node);
    list->insert_after(linear_ordering);
    node->set_linearization_node(list);
    
}

static void unlink_linear(Cfg_Node *node, Closure *c) {
    node->linearization_node()->unlink();
}

void Flow_Graph::linearize()
{
    build_dom_tree();
    find_loop_depths_and_headers();

    apply(unlink_linear,NULL);
    traversal_number ++;
    Cfg_Node_List cold_code;
    linearize_1(this, prolog(), prolog(), traversal_number, cold_code, false,&linear_node_ordering);
    //
    // linearize code trace
    //
    Cfg_Node_List *last = &cold_code;
    Cfg_Node_List *linear_ordering = linear_node_ordering.prev();
    Cfg_Node_List *cur;
    for (cur=last->next(); cur != last; cur = cur->next()) {
        Cfg_Node *node = cur->node();
        Cfg_Int i;
        for (i=0; i < node->in_edge_size(); i++)
            if (node->in_edges(i)->loop_header != node) break;
        assert(i != node->in_edge_size());
        linearize_1(this, node->in_edges(i), node, traversal_number, cold_code, true,linear_ordering);
    }
    //
    // linearize unvisited exception handlers
    //
    linear_ordering = linear_node_ordering.prev();
    Eh_Node *enode;
    for (enode = _handlers.next(); enode != &_handlers; enode = enode->next())
    {
        Cfg_Int i;
        for (i=0; i<enode->out_edge_size(); i++)
        {
            Cfg_Node *node = enode->out_edges(i)->handler;
            linearize_1(this, node, node, traversal_number, cold_code, true,linear_ordering);
        }
    }
}

bool is_artificially_split_edge(Cfg_Node *&pred, Cfg_Node *&succ)
{
    assert(pred != NULL || succ != NULL);
    if (pred == NULL)
    {
        if (succ->in_edge_size() == 1)
            pred = succ->in_edges(0);
        else
            return false;
    }
    if (succ == NULL)
    {
        if (pred->out_edge_size() == 1)
            succ = pred->out_edges(0);
        else
            return false;
    }
    return (pred->out_edge_size() == 1 &&
        succ->in_edge_size() == 1 &&
        pred->out_edges(0) == succ &&
        pred->eh_out_edge() == succ->eh_out_edge()
        );
}

extern bool use_compensation;

#ifdef _DEBUG
static void validate(Cfg_Node *node)
{
    return;
    if (node == NULL)
        return;
    Flow_Graph *fg = node->flowgraph;
    while (fg->calling_fg != NULL)
        fg = fg->calling_fg;
    assert(fg->has_fp);
}
#endif // _DEBUG

bool is_edge_into_hot_inlined(Cfg_Node *&pred, Cfg_Node *&succ)
{
#ifdef _DEBUG
    validate(pred);
    validate(succ);
#endif // _DEBUG
#ifdef OLD_FP_INLINING
    return false;
#endif // OLD_FP_INLINING
    if (!use_compensation)
        return false;

    assert(pred != NULL || succ != NULL);
    if (pred == NULL)
    {
        if (succ->in_edge_size() == 1)
            pred = succ->in_edges(0);
        else
            return false;
    }
    if (succ == NULL)
    {
        if (pred->out_edge_size() != 2)
            return false;
        succ = pred->out_edges(0);
        if (succ->is_cold_non_inlined())
            succ = pred->out_edges(1);
    }
    if (succ->is_cold_non_inlined())
        return false;
    if (pred->out_edge_size() != 2 ||
        succ->in_edge_size() != 1 ||
        pred->eh_out_edge() != succ->eh_out_edge())
        return false;
    Cfg_Node *cold = pred->out_edges(0);
    if (cold == succ)
        cold = pred->out_edges(1);
    if (!cold->is_cold_non_inlined())
        return false;
    return (cold->in_edge_size() == 1 &&
        cold->out_edge_size() == 1);
}

bool is_edge_into_cold_inlined(Cfg_Node *&pred, Cfg_Node *&succ)
{
#ifdef _DEBUG
    validate(pred);
    validate(succ);
#endif // _DEBUG
#ifdef OLD_FP_INLINING
    return false;
#endif // OLD_FP_INLINING
    if (!use_compensation)
        return false;

    assert(pred != NULL || succ != NULL);
    if (pred == NULL)
    {
        if (succ->in_edge_size() == 1)
            pred = succ->in_edges(0);
        else
            return false;
    }
    if (succ == NULL)
    {
        if (pred->out_edge_size() != 2)
            return false;
        succ = pred->out_edges(0);
        if (!succ->is_cold_non_inlined())
            succ = pred->out_edges(1);
    }
    if (!succ->is_cold_non_inlined())
        return false;
    if (pred->out_edge_size() != 2 ||
        succ->in_edge_size() != 1 ||
        pred->eh_out_edge() != succ->eh_out_edge())
        return false;
    Cfg_Node *hot = pred->out_edges(0);
    if (hot == succ)
        hot = pred->out_edges(1);
    if (hot->is_cold_non_inlined())
        return false;
    return (hot->in_edge_size() == 1);
}

bool is_edge_outof_hot_inlined(Cfg_Node *&pred, Cfg_Node *&succ)
{
#ifdef _DEBUG
    validate(pred);
    validate(succ);
#endif // _DEBUG
#ifdef OLD_FP_INLINING
    return false;
#endif // OLD_FP_INLINING
    if (!use_compensation)
        return false;

    assert(pred != NULL || succ != NULL);
    if (pred == NULL)
    {
        if (succ->in_edge_size() != 2)
            return false;
        pred = succ->in_edges(0);
        if (pred->is_cold_non_inlined())
            pred = succ->in_edges(1);
    }
    if (succ == NULL)
    {
        if (pred->out_edge_size() == 1)
            succ = pred->out_edges(0);
        else
            return false;
    }
    if (pred->is_cold_non_inlined())
        return false;
    if (pred->out_edge_size() != 1 ||
        succ->in_edge_size() != 2 ||
        pred->eh_out_edge() != succ->eh_out_edge())
        return false;
    Cfg_Node *cold = succ->in_edges(0);
    if (cold == pred)
        cold = succ->in_edges(1);
    if (!cold->is_cold_non_inlined())
        return false;
    return (cold->in_edge_size() == 1 &&
        cold->out_edge_size() == 1);
}

bool is_edge_outof_cold_inlined(Cfg_Node *&pred, Cfg_Node *&succ)
{
#ifdef _DEBUG
    validate(pred);
    validate(succ);
#endif // _DEBUG
#ifdef OLD_FP_INLINING
    return false;
#endif // OLD_FP_INLINING
    if (!use_compensation)
        return false;

    assert(pred != NULL || succ != NULL);
    if (pred == NULL)
    {
        if (succ->in_edge_size() != 2)
            return false;
        pred = succ->in_edges(0);
        if (!pred->is_cold_non_inlined())
            pred = succ->in_edges(1);
    }
    if (succ == NULL)
    {
        if (pred->out_edge_size() == 1)
            succ = pred->out_edges(0);
        else
            return false;
    }
    if (!pred->is_cold_non_inlined())
        return false;
    if (pred->out_edge_size() != 1 ||
        succ->in_edge_size() != 2 ||
        pred->eh_out_edge() != succ->eh_out_edge())
        return false;
    Cfg_Node *hot = succ->in_edges(0);
    if (hot == pred)
        hot = succ->in_edges(1);
    if (hot->is_cold_non_inlined())
        return false;
    return (hot->out_edge_size() == 1);
}
