// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o3_jit/expression.cpp,v 1.6 2002/01/10 03:12:09 xhshi Exp $
//


#include "defines.h"
#include <iostream.h>
#include "ir.h"
#include "expression.h"
#include "build_ir_routines.h"
#include "local_cse.h"

//#define NO_CSE
#define PSEUDO_ASGN

//
// Same var no may have different type in the regions of the program.
// We want to assign unique id for different types.
//
unsigned RegID_Map::lookup_reg_id(unsigned id, O3_Jit_Type ty) {
    unsigned entry = (id + (unsigned)ty)%MAX_JSTACK;
    RegID_Link *rl;
    for (rl = _table[entry]; rl != NULL; rl = rl->next)
        if (rl->id == id && rl->ty == (char)ty) return rl->reg_id;

    unsigned reg_id = get_tmp_reg_id(IS_64BIT(ty));
    assert(reg_id < (1<<MAX_REGID_SIZE));
    _table[entry] = new (_mem) RegID_Link(id,(char)ty,reg_id,_table[entry]);
    return reg_id;
}

//
// return  unique kill id for operand o
//
unsigned Kill_ID_Map::lookup(Mem_Manager& mem, void *o) {
    unsigned entry = (((unsigned)o) >> 2)% MAX_KILL_HASH;
    //
    // search if there is an existing kill_set for exp
    //
    ID_Link *l = _table[entry];
    for (; l != NULL; l = l->next)
        if (l->opnd == o) return l->id;
    //
    // create ID link to hold operand o
    //
    _table[entry] = l = new (mem) ID_Link(o,_next_kill_id++,_table[entry]);
    return _next_kill_id-1;
}

const char* Exp::symbol[Exp::n_exp] = {
        "Opnd",    // operand
        "Array",   // array access
        "Field",   // field access
        "Length",  // array length
        "Widen",   // widen instruction (array load of char)
        "=",     // x = y
        "+",     // integer add
        "+c",    // integer add with carry
        "-",     // integer sub
        "-b",    // integer sub with carry
        "+f",    // floating point add (single and double)
        "-f",    // floating point sub (single and double)
        "*",     // integer mutiplication
        "*f",    // floating point mul (single and double)
        "/",     // integer division
        "/f",    // floating point div (single and double)
        "%",     // remainder (div)
        "%f",    // floating point remainder (div)
        "Neg",     // negate int/long
        "Fneg",    // negate float/double
        "Convt",   // conversion (i2l,i2d,...)
        "&",     // bitwise and
        "|",     // bitwise or
        "^",     // bitwise xor
        ">>",    // shift left
        "<<a",   // arithmetic shift right
        "<<",    // shift right
        "Compare", // compare
        "Cmp_LT",  // compare less than
        "Cmp_GT",  // compare greater than
        "Test",    // test 
        "Beq",    // branch if equal 
        "Bne",    // branch if not equal
        "Blt",    // branch if less than 
        "Bge",    // branch if greater than or equal
        "Bgt",    // branch if greater than
        "Ble",    // branch if less than or equal
		"Call",    // method call (virtual/special)
        "New",     // new 
        "Newarray",// newarray 
        "Anewarray",//anewarray
        "Multinew",// multianewarray call  
		"Vtable",  // vtable of a class
        "Return",  // return
        "Next_PC", // the next pc of a jsr
        "Jump",    // jump 
        "Tableswitch", // table switch 
        "Lookupswitch",// lookup switch
        "String",  // get string from a constant pool  
        "Instanceof", // is instance of 
        "Checkcast",  // check cast
        "Monitorenter", // monitor enter
        "Monitorexit",  // monitor exit
        "Monitorenter_static", // monitor enter for static methods
        "Monitorexit_static",  // monitor exit  for static methods
        "Athrow",  // throw exception
        "InterfaceVtable", // resolve interface - get vtable at runtime
        "Push",    // push value onto stack frame
        "Pop",     // pop value out of stack frame
        "ClassInit", // class initialization
        "WriteBarrier", // write barrier
		"ReadBarrier" //read barrier
#ifdef STAT_INDIRECT_CALL
		,"StatIndirectCall" // For statistics of indirect branch
#endif
#ifdef O3_VTune_Support
		,"VTuneMethodCall"
#endif
};

//
// return 1, if kill_id is in kill_set
//
bool Exp::is_in_kill_set(unsigned kill_id) {
    return _kill != NULL && _kill->is_in_kill_set(kill_id);
}

//
// return true, the expression contains an array of ty
//
bool Exp::contain_of_array(O3_Jit_Type ty) {
    return (_kill != NULL && _kill->is_array_set(ty));
}
//
// return 1, if the value may be killed by calls
//
bool Exp::is_alias_across_call() {
    return (_kill != NULL && _kill->is_alias_across_call());
}

int  Operand_Exp::is_same_const(Value *v,O3_Jit_Type t) {
    if (type != t || opnd->kind != Operand::Const) return 0; // type does not match

    Const_Operand *cnst = (Const_Operand*)opnd;
    
#if 0
    if (type == JAVA_TYPE_FLOAT) 
        return cnst->val.f == v->f;
    else if (type == JAVA_TYPE_DOUBLE)
        return cnst->val.d == v->d;
#endif // 0
    if (/*type == JAVA_TYPE_LONG*/IS_64BIT(type))
        return cnst->val.l.lo == v->l.lo && cnst->val.l.hi == v->l.hi;
    else
        return cnst->val.i == v->i;
}

//
// if a local cse is found, then the inst that produces cse is returned
// 
Inst *Expressions::local_cse(Exp *exp) {
    if (is_local_cse(exp)) { 
        Inst *inst = exp->inst();
        assert(inst->dst() && inst->dst()->is_reg());
        ((Reg_Operand*)inst->dst())->set_global_reg_cand();
        return inst;
    }
    return NULL;
}

//
// compute kill set for the operand expression
//
void Operand_Exp::compute_kill(Mem_Manager& mem, unsigned max_bv_size) {
	//
	// if the operand cannot be killed, e.g. imm or constant, then we 
	// don't create kill set for the operand.  In other words, 
	// _kill == NULL indicates that the exp cannot be killed
	//
	if (!opnd->need_kill_set() || _kill != NULL) return;
	_kill = new (mem) Kill_Set(mem,max_bv_size); // create kill_set
    //
    // determine if the expression is killed across calls
    //
    if (opnd->is_aliased_across_call())
        _kill->set_alias_across_call();

    if (IS_VALID_KILL_ID(_kill_id)) {
        assert(_kill_id < max_bv_size);
        _kill->add_to_kill(_kill_id);
    }
}

//
// there is an assignment "opnd = ...", where opnd can be static, vreg, arg, ...
// the assignment kills any lcse that contains kill_id of the operand
//
void Operand_Exp::asgn_kill_local_cse(Live_LCSE *live_lcse) {
    if (!IS_VALID_KILL_ID(_kill_id) || live_lcse == NULL) return;

    // remove dead lcse from _live_lcse
    live_lcse->kill_lcse_contain(_kill_id); 
}

Inst *Operand_Exp::build_inst(Expressions& exprs, Inst *inst_head) {
    return opnd->lookup_exp(exprs, inst_head);
}


//
// compute kill set for the array expression
//
void Array_Exp::compute_kill(Mem_Manager& mem, unsigned max_bv_size) {
	//
	// if the kill set has been computed already, we don't need to 
	// recompute the kill set
	//
	if (_kill != NULL) return;
	_kill = new (mem) Kill_Set(mem,max_bv_size); // create kill_set

	_kill->set_alias_across_call();
	//
	// set array type
	//
	_kill->set_array(type);
	//
	// find kill sets for both base and indx
	// merge them with ks
	//
	base->compute_kill(mem,max_bv_size);
	indx->compute_kill(mem,max_bv_size);
	_kill->union_with(base->kill_set());
	_kill->union_with(indx->kill_set());
}

//
// there is an assignment " [t0 + t1*8 + 8] = ...".
// the assignment kills any lcse that contains array expr having the same type
//
void Array_Exp::asgn_kill_local_cse(Live_LCSE *live_lcse) {
    if (live_lcse == NULL) return;

    // remove dead lcse that contains array of type from _live_lcse
    live_lcse->kill_lcse_contain_array(type);
}

Inst *Array_Exp::build_inst(Expressions& exprs, Inst *inst_head) {
    Inst *b = base->build_inst(exprs, inst_head);
    Inst *i = indx->build_inst(exprs, inst_head);
    Inst *len = exprs.lookup_inst(Exp::Length,b,NULL,JIT_TYPE_INT,inst_head);
    Inst *bnd = gen_array_cmp_inst(exprs.mem,exprs,inst_head,i,len,JIT_TYPE_INT,true);
    return exprs.lookup_array(b,i,bnd,shift,offset,type,inst_head);;
}

//
// compute kill set for the field expression
//
void Field_Exp::compute_kill(Mem_Manager& mem, unsigned max_bv_size) {
	//
	// if the kill set has been computed already, we don't need to 
	// recompute the kill set
	//
	if (_kill != NULL) return;
	_kill = new (mem) Kill_Set(mem,max_bv_size); // create kill_set

	_kill->set_alias_across_call();

    _kill->add_to_kill(_kill_id);
	//
	// find kill sets for base and merge it with ks
	//
	base->compute_kill(mem,max_bv_size);
    //
    // base's kill set may be null, e.g., base is a const
    // 
	_kill->union_with(base->kill_set());
}

//
// there is an assignment "[t0 + 16] = ..."
// the assignment kills any field access that has the same kill_id
//
void Field_Exp::asgn_kill_local_cse(Live_LCSE *live_lcse) {
    assert(IS_VALID_KILL_ID(_kill_id));
    if (live_lcse == NULL) return;

    // remove dead lcse from _live_lcse
    live_lcse->kill_lcse_contain(_kill_id); 
}
Inst *Field_Exp::build_inst(Expressions& exprs, Inst *inst_head) {
    Inst *b = base->build_inst(exprs, inst_head);
    return exprs.lookup_field(b, offset, type, inst_head, _fid);
}
void Inst_Exp::compute_kill(Mem_Manager& mem, unsigned max_bv_size) {
	if (_kill != NULL) return;
	_kill = new (mem) Kill_Set(mem,max_bv_size); // create kill_set
	//
	// find kill sets for both left and rght
	//
	if (_child[0]) {
		_child[0]->compute_kill(mem,max_bv_size);
		_kill->union_with(_child[0]->kill_set());
	}
	if (_child[1]) {
		_child[1]->compute_kill(mem,max_bv_size);
		_kill->union_with(_child[1]->kill_set());
	}
}
Inst *Inst_Exp::build_inst(Expressions& exprs, Inst *inst_head) {
    Inst *l = NULL, *r = NULL;
    if (_child[0] != NULL)
        l = _child[0]->build_inst(exprs, inst_head);
    if (_child[1] != NULL) 
        r = _child[1]->build_inst(exprs, inst_head);
    return exprs.lookup_inst((Exp::Kind)op, l, r, type, inst_head);
}

//
// hash functions for expressions
//
unsigned array_hash(Exp *base, Exp *indx, unsigned off) {
    return ((unsigned)base + (unsigned)indx + off)>>4;
}
unsigned field_hash(Exp *base, unsigned off) {
    return (unsigned)base + off;
}
unsigned inst_hash(Exp::Kind op, Exp *left, Exp *rght, O3_Jit_Type ty) {
    return (unsigned)op + (unsigned)left + (unsigned)rght + (unsigned)ty;
}

//
// get a virtual reg operand
//
Inst *Expressions::lookup_vreg(unsigned var_no, O3_Jit_Type ty,Inst *inst_head) {
    unsigned reg_id = reg_map.virtual_reg_id(var_no,ty);

    Operand_Exp *exp = lookup_reg_exp(reg_id,ty,1); // lookup virtual reg
    // generate tuple inst and append the inst to the inst list
	return  gen_opnd_tuple(inst_head,exp);
}
//
// get a virtual reg operand
//
Inst *Expressions::lookup_preg(unsigned reg_id, O3_Jit_Type ty,Inst *inst_head) {

    Operand_Exp *exp = lookup_reg_exp(reg_id,ty,0); // lookup virtual reg
    // generate tuple inst and append the inst to the inst list
	return gen_opnd_tuple(inst_head,exp);
}
//
// get a reg operand for a Java stack entry
//
Inst *Expressions::lookup_stack(unsigned stk_entry, O3_Jit_Type ty,Inst *inst_head) {
    unsigned reg_id = reg_map.stack_reg_id(stk_entry,ty);

    Operand_Exp *exp = lookup_reg_exp(reg_id,ty,1); // lookup virtual reg
    // generate tuple inst and append the inst to the inst list
	return gen_opnd_tuple(inst_head,exp);
}

//
// get an imm exp
//
Inst *Expressions::lookup_imm(unsigned imm, O3_Jit_Type ty, Inst *inst_head) {
    // generate tuple inst and append the inst to the inst list
	return gen_opnd_tuple(inst_head,lookup_imm_exp(imm, ty));
}

//
// get an argument
// 
Inst *Expressions::lookup_arg(unsigned i, O3_Jit_Type ty, Inst *inst_head) {
    return gen_opnd_tuple(inst_head, lookup_arg_exp(i,ty));
}

//
// get return value
//
Inst *Expressions::lookup_ret(O3_Jit_Type ty, Inst *inst_head) {
    // generate tuple inst and append the inst to the inst list
	return gen_opnd_tuple(inst_head,lookup_ret_exp(ty));
}

//
// get a static
//
Inst *Expressions::lookup_static(void *addr,O3_Jit_Type ty,Inst *inst_head, Field_Handle fh) {
    // generate tuple inst and append the inst to the inst list
	return gen_opnd_tuple(inst_head,lookup_static_exp(addr,ty,addr, fh));
}

//
// get a static constant (float or double)
//
Inst *Expressions::lookup_const(Value *val,O3_Jit_Type ty,Inst *inst_head) {
    // generate tuple inst and append the inst to the inst list
	return gen_opnd_tuple(inst_head,lookup_const_exp(val,ty));
}

//
// get an array operand
//
Inst *Expressions::lookup_array(Inst *base, Inst *index, Inst *bnd,
    unsigned shift, unsigned off, O3_Jit_Type ty,Inst *inst_head) {

    Array_Exp *exp = lookup_array_exp(base->exp,index->exp,bnd->exp,shift,off,ty);
    // generate tuple inst and append the inst to the inst list
	return gen_array_tuple(inst_head,exp,base,index,bnd);
}

//
// get a field operand
//
Inst *Expressions::lookup_field(Inst *base, unsigned off, O3_Jit_Type ty,
                                Inst *inst_head, FIELD_UID unique_id) {

    Field_Exp *exp = lookup_field_exp(base->exp,off,ty,unique_id);
    // generate tuple inst and append the inst to the inst list
	return gen_field_tuple(inst_head,exp,base,unique_id);
}

//
// get an inst expression
//
Inst *Expressions::lookup_inst(Exp::Kind op, Inst *left, Inst *rght, O3_Jit_Type ty,Inst *inst_head) {
    Exp *left_exp = (left != NULL)? left->exp : NULL;
    Exp *rght_exp = (rght != NULL)? rght->exp : NULL;
    Inst_Exp *exp = lookup_inst_exp(op,left_exp,rght_exp,ty);
    // generate tuple inst and append the inst to the inst list
	return gen_inst_tuple(inst_head,exp,op,left,rght);
}

//
// find register operand with reg id == reg_id
// if not found, then create one and insert into hash table
//
Operand_Exp *Expressions::lookup_reg_exp(unsigned reg_id, O3_Jit_Type ty,int is_vreg) {
    unsigned entry  = reg_id%MAX_HASH_ENTRY;
    // search vreg
    Link<Exp> *link;
    for (link = _table[entry]; link != NULL; link = link->next)
        if (link->elem->is_same_reg(reg_id,ty)) 
            return (Operand_Exp*)link->elem;

    // create register operand
    Reg_Operand *vreg;
    if (is_vreg) 
        vreg = create_vreg_opnd(mem,reg_id,ty);
    else
        vreg = new (mem) Physical_Reg((X86_Reg_No)reg_id, ty);
	// create expression
    Operand_Exp *exp = new (mem) Operand_Exp(vreg, _next_exp_id++);
    _table[entry] = new (mem) Link<Exp>(exp,_table[entry]);

    // assign kill id
    exp->set_kill_id(kill_id_map.get_next_kill_id());
    return exp;

}

Operand_Exp *Expressions::lookup_temp_reg_exp(Reg_Operand *treg) {
    unsigned reg_id = treg->id;
    unsigned entry  = reg_id%MAX_HASH_ENTRY;
    // search vreg
    Link<Exp> *link;
    for (link = _table[entry]; link != NULL; link = link->next)
        if (link->elem->is_same_reg(reg_id,treg->type)) 
            return (Operand_Exp*)link->elem;

	// create expression
    Operand_Exp *exp = new (mem) Operand_Exp(treg, _next_exp_id++);
    _table[entry] = new (mem) Link<Exp>(exp,_table[entry]);

    // assign kill id
    exp->set_kill_id(kill_id_map.get_next_kill_id());
    return exp;
}

//
// find arg operand whose num == i
//
Operand_Exp *Expressions::lookup_arg_exp(unsigned i, O3_Jit_Type ty) {
    unsigned entry = (i + (unsigned)ty)%MAX_HASH_ENTRY;
    Link<Exp> *link;
    for (link = _table[entry]; link != NULL; link = link->next)
        if (link->elem->is_same_arg(i,ty)) 
            return (Operand_Exp*)link->elem;

	// create arg expression
    unsigned reg_id = reg_map.get_tmp_reg_id(IS_64BIT(ty));
    Operand *arg = create_arg_opnd(mem,i,reg_id,ty);
    Operand_Exp *exp = new (mem) Operand_Exp(arg,_next_exp_id++);
    _table[entry] = new (mem) Link<Exp>(exp,_table[entry]);

    // assign kill id
    exp->set_kill_id(kill_id_map.get_next_kill_id());
    return exp;
}

//
// find return value whose type == ty
//
Operand_Exp *Expressions::lookup_ret_exp(O3_Jit_Type ty) {
    unsigned entry = ((unsigned)ty)%MAX_HASH_ENTRY;
    Link<Exp> *link;
    for (link = _table[entry]; link != NULL; link = link->next)
        if (link->elem->is_same_ret(ty)) 
            return (Operand_Exp*)link->elem;

	// create arg expression
    Operand *ret = create_ret_opnd(mem,reg_map.get_tmp_reg_id(ty == JIT_TYPE_LONG),ty);
    Operand_Exp *exp = new (mem) Operand_Exp(ret,_next_exp_id++);
    // expression that contains ret is not a cse candidate
    exp->clr_cse_cand();
    _table[entry] = new (mem) Link<Exp>(exp,_table[entry]);
    // assign kill id
    exp->set_kill_id(kill_id_map.get_next_kill_id());
    return exp;
}

//
// search for imm constant
//
Operand_Exp *Expressions::lookup_imm_exp(unsigned imm, O3_Jit_Type ty) {
    assert(ty == JIT_TYPE_INT || ty == JIT_TYPE_CLASS || ty == JIT_TYPE_ADDR || ty == JIT_TYPE_RETADDR);
    unsigned entry = imm%MAX_HASH_ENTRY;
    Link<Exp> *link;
    for (link = _table[entry]; link != NULL; link = link->next)
        if (link->elem->is_same_imm(imm, ty)) 
            return (Operand_Exp*)link->elem;

	// create imm expression
    Operand_Exp *exp =
    new (mem) Operand_Exp(new (mem) Imm_Operand(imm, ty),_next_exp_id++);
    _table[entry] = new (mem) Link<Exp>(exp,_table[entry]);
    return exp;
}

//
// get a constant expression (float, double, address, ..)
//
Operand_Exp *Expressions::lookup_const_exp(Value *val,O3_Jit_Type ty) {
    unsigned entry = ((unsigned)val->i) % MAX_HASH_ENTRY;
    Link<Exp> *link;
    for (link = _table[entry]; link != NULL; link = link->next)
        if (link->elem->is_same_const(val,ty)) 
            return (Operand_Exp*)link->elem;
            
    // creatre new static const expression
    Const_Operand *con = create_const_opnd(mem,val,ty);
	Operand_Exp *exp = new (mem) Operand_Exp(con,_next_exp_id++);
    _table[entry] = new (mem) Link<Exp>(exp,_table[entry]);
    //
    // keep track of the size of data block that is needed 
    //
    if (con->need_data_space())
        _data_space += con->data_space_needed();
    return exp;
}

Operand_Exp *Expressions::lookup_static_exp(void *addr, O3_Jit_Type ty, void *unique_id,
                                            Field_Handle fh)
{
    unsigned entry = ((unsigned)addr) % MAX_HASH_ENTRY;
    Link<Exp> *link;
    for (link = _table[entry]; link != NULL; link = link->next)
        if (link->elem->is_same_addr(addr)) 
            return (Operand_Exp*)link->elem;

	// create imm expression
    Operand_Exp *exp =
    new (mem) Operand_Exp(create_static_opnd(mem,addr,ty, fh),_next_exp_id++);
    _table[entry] = new (mem) Link<Exp>(exp,_table[entry]);

    // assign kill id for the field
    exp->set_kill_id(kill_id_map.lookup(mem,unique_id));
    return exp;
}

//
// search for an array expression
//
Array_Exp *Expressions::lookup_array_exp(Exp *base, Exp *index, Exp *bnd,
    unsigned shift, unsigned off, O3_Jit_Type ty) {
    int i= 0;
    unsigned entry = array_hash(base,index,off)%MAX_HASH_ENTRY;
    Link<Exp> *link;
    for (link = _table[entry]; link != NULL; link = link->next) {
            i++;
        if (link->elem->is_same_array(base,index,shift,off,ty)) 
			return (Array_Exp*)link->elem;
    }

    // create new array expression
	Array_Exp *exp = new (mem) Array_Exp(base,index,shift,off,ty,_next_exp_id++);
    _table[entry] = new (mem) Link<Exp>(exp,_table[entry]);
    // expression that contains ret is not a cse candidate
    if (!base->is_cse_cand() || !index->is_cse_cand()) exp->clr_cse_cand();
    return exp;
}

//
// search for a field expression
//
Field_Exp *Expressions::lookup_field_exp(Exp *base, unsigned off, O3_Jit_Type ty, 
                                         FIELD_UID unique_id) {
    unsigned entry = field_hash(base,off)%MAX_HASH_ENTRY;
    Link<Exp> *link;
    for (link = _table[entry]; link != NULL; link = link->next)
        if (link->elem->is_same_field(base,off,ty)) 
            return (Field_Exp*)link->elem;

	// create field expression
    Field_Exp *exp = new (mem) Field_Exp(base, off, ty,_next_exp_id++,unique_id);
    _table[entry] = new (mem) Link<Exp>(exp,_table[entry]);
    // expression that contains ret is not a cse candidate
    if (!base->is_cse_cand()) exp->clr_cse_cand();

    // assign kill id for the field
    exp->set_kill_id(kill_id_map.lookup(mem,unique_id));
    return exp;
}

//
// search for an inst expression
//
Inst_Exp *Expressions::lookup_inst_exp(Exp::Kind op, Exp *left, Exp *rght, O3_Jit_Type ty) {
    if (op == Exp::Widen) ty = JIT_TYPE_INT;
    unsigned entry = inst_hash(op,left,rght,ty)%MAX_HASH_ENTRY;
    Link<Exp> *link;
    for (link = _table[entry]; link != NULL; link = link->next)
        if (link->elem->is_same_inst(op,left,rght,ty)) 
            return (Inst_Exp*)link->elem;

	// create inst expression
    Inst_Exp *exp;
    if (left == NULL) 
        exp = new (mem) Inst_Exp(op,ty,_next_exp_id++);
    else if (rght == NULL) {
        exp = new (mem) Inst_Exp(op,left,ty,_next_exp_id++);
        if (!left->is_cse_cand()) exp->clr_cse_cand();
    } else {
        exp = new (mem) Inst_Exp(op,left,rght,ty,_next_exp_id++);
        if (op != Exp::Assign &&
           (!left->is_cse_cand() || !rght->is_cse_cand())) exp->clr_cse_cand();
    }
    _table[entry] = new (mem) Link<Exp>(exp,_table[entry]);
    if (op == Exp::Next_PC)
        exp->clr_cse_cand();
    return exp;
}
//
// search for bound checking expression
//
Inst_Exp *Expressions::lookup_array_cmp_exp(Exp *left, Exp *rght,
                                            O3_Jit_Type ty){
    if (left->is_imm_exp()) {
        Inst_Exp *cmp = _live_lcse->array_cmp(left,rght);
        if (cmp != NULL) return cmp;
    }
    return lookup_inst_exp(Exp::Compare,left,rght,ty);
}

//
// exp can be a cse.  we compute kill set for exp and add exp to _live_lcse
//
void Expressions::gen_local_cse(Exp *exp, Inst *inst, bool for_assgn) {

    exp->compute_kill(mem,kill_id_map.get_curr_kill_id());
    if (!exp->is_cse_cand() || _live_lcse == NULL) return; // no cse is needed

#ifdef NO_CSE
    return;
#endif // NO_CSE

    //
    // x = inst;  we will try to do local copy propagation to propagate inst
    //            if inst's value is not available, then we stop propagating
    //
    if (!for_assgn || is_local_cse(inst->exp)) { 
        exp->set_inst(inst); // inst holds the most recent value of exp
        _live_lcse->add_lcse(exp); 
    }
}

//
// move operand to a temporary (T = opnd)
//
Inst *Expressions::gen_opnd_tuple(Inst *inst_head,Operand_Exp *exp) { 
    Inst *inst = local_cse(exp);
    if (inst != NULL) {
        inst->reset_newly_created();
        return inst;  // local cse found
    }

	inst = new (mem) Assign_Inst(create_new_temp_reg(exp->type),exp->opnd,exp,inst_head);

#ifdef PSEUDO_ASGN
    if(exp->opnd->is_vreg())
        ((Assign_Inst*)inst)->enable_pseudo_asgn();
#endif
    // JMS - removing this optimization because it doesn't allow
    // constants to be removed as LCSE candidates after entering
    // a new basic block. 
#if 0
    if (exp->opnd->kind == Operand::Immediate || // for imm and constant, we don't
        exp->opnd->kind == Operand::Const)       // need to add exp to _live_lcse
        exp->set_inst(inst);
    else
#endif // 0
        gen_local_cse(exp,inst);
	return inst;
}

//
// move array operand to a temporary (T = array)
//
Inst *Expressions::gen_array_tuple(Inst *inst_head,Array_Exp *exp,
								   Inst *base, Inst *index, Inst *bound){
    Inst *inst = local_cse(exp);
    if (inst != NULL) {
        // if base or index is just newly created, there is no way that
        // a cse can happen
        if (!base->is_newly_created() && !index->is_newly_created()) {
            inst->reset_newly_created();
            return inst;  // local cse found
        } else _live_lcse->kill_lcse_contain(exp);
    }

	// create array operand
	Array_Operand *array = 
    create_array_operand(mem,base,index,bound,exp->shift,exp->offset,exp->type,true);
	inst = new (mem) Assign_Inst(create_new_temp_reg(exp->type),array,exp,inst_head);

    gen_local_cse(exp,inst);
	return inst;
}

//
// move field operand to a temporary (T = array)
//
Inst *Expressions::gen_field_tuple(Inst *inst_head, Field_Exp *exp, Inst *base, FIELD_UID fid){
    Inst *inst = local_cse(exp);
    if (inst != NULL) {
        // if base is just newly created, there is no way that
        // a cse can happen
        if (!base->is_newly_created()) {
            inst->reset_newly_created();
            return inst;  // local cse found
        } else _live_lcse->kill_lcse_contain(exp);
    }

	// create field operand
	Field_Operand *field = create_field_operand(mem,base,exp->offset,exp->type,fid);
	inst = new (mem) Assign_Inst(create_new_temp_reg(exp->type),field,exp,inst_head);

    gen_local_cse(exp,inst);
	return inst;
}

Inst *Expressions::gen_inst_tuple(Inst *inst_head, Inst_Exp *exp, 
								  Exp::Kind op,Inst *left, Inst *rght) {
    Inst *inst = local_cse(exp);
    if (inst != NULL) {
        // if left or rght is just newly created, there is no way that
        // a cse can happen
        if ((left == NULL || !left->is_newly_created()) &&
            (rght == NULL || !rght->is_newly_created())) {
            inst->reset_newly_created();
            return inst;  // local cse found
        } else _live_lcse->kill_lcse_contain(exp);
    }

    Operand *l = (left != NULL)? left->dst() : NULL, 
            *r = (rght != NULL)? rght->dst() : NULL;
//	assert((!l || l->is_reg()) && (!r || r->is_reg()));

	Temp_Reg *tmp = create_new_temp_reg(exp->type);
    Inst *i;
    switch(op) {
    case Exp::Add:
        i = new (mem) Add_Inst(Add_Inst::add,l,r,exp,inst_head); break;
    case Exp::Fadd:
        i = new (mem) Add_Inst(Add_Inst::fadd,l,r,exp,inst_head); break;
    case Exp::Sub:
        i = new (mem) Sub_Inst(Sub_Inst::sub,l,r,exp,inst_head); break;
    case Exp::Fsub:
        i = new (mem) Sub_Inst(Sub_Inst::fsub,l,r,exp,inst_head); break;
    case Exp::Length:
        i = new (mem) Obj_Info_Inst(Obj_Info_Inst::length,l,exp,inst_head); break;
	case Exp::Vtable:
		i = new (mem) Obj_Info_Inst(Obj_Info_Inst::vtable,l,exp,inst_head); break;
    case Exp::Compare:
        i = new (mem) Compare_Inst(Compare_Inst::cmp,false,l,r,exp,false,inst_head); break;
    case Exp::Cmp_LT:
        i = new (mem) Compare_Inst(Compare_Inst::cmp_lt,false,l,r,exp,false,inst_head); break;
    case Exp::Cmp_GT:
        i = new (mem) Compare_Inst(Compare_Inst::cmp_gt,false,l,r,exp,false,inst_head); break;
    case Exp::Mul:
        i = new (mem) Mul_Inst(Mul_Inst::mul,l,r,exp,inst_head); break;
    case Exp::Fmul:
        i = new (mem) Mul_Inst(Mul_Inst::fmul,l,r,exp,inst_head); break;
//#ifdef INLINE_NATIVE
    case Exp::Smul:
        i = new (mem) Mul_Inst(Mul_Inst::smul,l,r,exp,inst_head); break;
//#endif
    case Exp::Div:
        i = new (mem) Div_Inst(Div_Inst::div,l,r,exp,inst_head); break;
    case Exp::Fdiv:
        i = new (mem) Div_Inst(Div_Inst::fdiv,l,r,exp,inst_head); break;
    case Exp::Rem:
        i = new (mem) Div_Inst(Div_Inst::rem,l,r,exp,inst_head); break;
    case Exp::Frem:
        i = new (mem) Div_Inst(Div_Inst::frem,l,r,exp,inst_head); break;
    case Exp::Neg:
        i = new (mem) Neg_Inst(Neg_Inst::neg,l,exp,inst_head); break;
    case Exp::Fneg:
        i = new (mem) Neg_Inst(Neg_Inst::fneg,l,exp,inst_head); break;
    case Exp::And:
        i = new (mem) Bitwise_Inst(Bitwise_Inst::k_and,l,r,exp,inst_head); break;
    case Exp::Or:
        i = new (mem) Bitwise_Inst(Bitwise_Inst::k_or,l,r,exp,inst_head);  break;
    case Exp::Xor:
        i = new (mem) Bitwise_Inst(Bitwise_Inst::k_xor,l,r,exp,inst_head); break;
    case Exp::Shl:
        i = new (mem) Bitwise_Inst(Bitwise_Inst::shl,l,r,exp,inst_head); break;
    case Exp::Shr:
        i = new (mem) Bitwise_Inst(Bitwise_Inst::shr,l,r,exp,inst_head); break;
    case Exp::Sar:
        i = new (mem) Bitwise_Inst(Bitwise_Inst::sar,l,r,exp,inst_head); break;
    case Exp::Convt: 
        i = new (mem) Convt_Inst(l,exp,inst_head);  break;
    case Exp::Next_PC:
        i = new (mem) NextPC_Inst(exp,inst_head);   break;
    case Exp::String:
        i = new (mem) String_Inst(l,r,exp,inst_head); break;
    case Exp::Checkcast:
        i = new (mem) Type_Inst(Type_Inst::cast,l,r,exp,inst_head); break;
    case Exp::Instanceof:
        i = new (mem) Type_Inst(Type_Inst::instance,l,r,exp,inst_head); break;
    case Exp::Widen:
        i = new (mem) Widen_Inst(l,exp,inst_head); break; 
    case Exp::InterfaceVtable:
		i = new (mem) Obj_Info_Inst(Obj_Info_Inst::intfcvtable,l,r,exp,inst_head); break;
    case Exp::Intr_Sin:
        i = new (mem) Math_Inst(Math_Inst::sin, l, exp, inst_head); break;
    case Exp::Intr_Cos:
        i = new (mem) Math_Inst(Math_Inst::cos, l, exp, inst_head); break;
    case Exp::Intr_Sqrt:
        i = new (mem) Math_Inst(Math_Inst::sqrt, l, exp, inst_head); break;
    case Exp::Intr_Rndint:
        i = new (mem) Math_Inst(Math_Inst::rndint, l, exp, inst_head); break;
    case Exp::Intr_Abs:
        i = new (mem) Math_Inst(Math_Inst::abs, l, exp, inst_head); break;
    default: assert(0);
    }

    i->set_dst(tmp);
    gen_local_cse(exp,i);
	return i;
}
