/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Process a global xxx=yyy option */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "super.h"
#include "options.h"

Option opts[] = {
	{ "patterns",		GLOBAL,		option_patterns },
	{ "syslog",		GLOBAL,		option_syslog },
	{ "logfile",		GLOBAL,		option_logfile },
	{ "loguid",		GLOBAL,		option_loguid },
	{ "mail",		GLOBAL|LOCAL,	option_mail },
	{ "mailany",		GLOBAL|LOCAL,	option_mailany },
	{ "timestampuid",	GLOBAL,		option_timestampuid },
	{ "timestampbyhost",	GLOBAL,		option_timestampbyhost },
	{ "renewtime",		GLOBAL,		option_renewtime },
	{ "timeout",		GLOBAL|LOCAL,	option_timeout },
	{ "password",		GLOBAL|LOCAL,	option_password },
	{ "owner",		GLOBAL|LOCAL,	option_owner },
	{ "nice",		GLOBAL|LOCAL,	option_nice },
	{ "umask",		GLOBAL|LOCAL,	option_umask },
	{ "info",		LOCAL,		option_info },
	{ "relative_path",	GLOBAL,		option_relative_path },
	{ "group_slash",	GLOBAL,		option_group_slash },
	{ "uid",		LOCAL,		option_uid },
	{ "gid",		LOCAL,		option_gid },
	{ "u+g",		LOCAL,		option_u_g },
	{ "env",		LOCAL,		option_env },
	{ "setenv",		LOCAL,		option_setenv },
	{ "fd",			LOCAL,		option_fd },
	{ "nargs",		LOCAL,		option_nargs },
	{ "arg",		IS_ROOT|LOCAL,	option_arg },
	{ "DIE",		LOCAL,		option_die },	/* OBSOLETE */
	{ "die",		LOCAL,		option_die },
	{ "print",		LOCAL,		option_print },
	{ "",			0,		NULL },
};

/* quick-and-dirty find_option(); should optimize later */

Option *
find_option(word)
char *word;
{
    Option *p;
    int l;
    
    for (p=opts; p->process; p++) {
	l = strlen(p->name);
	if (strncmp(word, p->name, l)==0) {
	    if ((p->flags & IS_ROOT)) {
		return p;	/* Don't care if it's followed by <SEP> */
	    } else if (*(word+l) == OPTION_SEP) {
		return p;	/* Must be followed by <SEP> */
	    }
	}
    }
    return NULL;
}

int
handle_option(word, s, isglobal)
char *word;	/* The input xxx<SEP>yyy string; NULL means clear settings */
char *s;	/* pts to yyy in word; can be NULL if wd is NULL */
int isglobal;	/* bool: are we processing a global or local entry? */
{

    Option *opt;

    if (debug>1)
	fprintf(stderr, "\thandle_option(): word=<%s> s=<%s>; is %s\n",
				word, s, isglobal ? "global" : "local");
    
    if (!word)
	return option_clear_settings(word, NULL, isglobal);

    if (*word == '!')
	return Error(0, 0, "%t\n\tsuper.tab syntax error: \n\
options cannot be negated: <%s>\n", word);

    opt = find_option(word);

    if (opt) {
	if (isglobal && !(opt->flags & GLOBAL))
	    return Error(0, 0,
		    "%t\n\t`%s' may not be used as a global option\n", word);
	if (!isglobal && !(opt->flags & LOCAL))
	    return Error(0, 0,
		    "%t\n\t`%s' may not be used as a local option\n", word);

	return opt->process(word, s, isglobal);

    } else {
	if (s-word == 5 && strncmp(word, "time", 4) == 0) {
	    return Error(0, 0, "%t\n\tUnrecognized %s option `%s'.\n\
\tPerhaps you meant to use the condition `time~%s'?\n",
			isglobal ? "global" : "local", word, s+1);
	} else {
	    return Error(0, 0,
			"%t\n\tUnrecognized %s option `%s'.\n",
			isglobal ? "global" : "local", word);
	}
    }
    return 0;
}

int
option_clear_settings(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    if (debug>1) fprintf(stderr,
			"\toption_clear_settings(\"%s\", \"%s\", %d)\n",
				word, s, isglobal);
    if (isglobal)
	return option_global_reset_settings();
    else
	return option_local_clear_settings();
    /* NOTREACHED */
}

int
option_global_reset_settings()
{
    /* reset global user/group/host patterns pointers, so that more names
     * form a new list, and are not appended to the old one.
     * Similarly for time~pattern pointers.
     * *** Note that we don't free any pattern space!
     * *** We assume that there is plenty of memory in the computer
     * *** to slurp up all _global_ patterns.
     */
    char *wd;

    if (debug>1) fprintf(stderr, "\toption_global_reset_settings()\n");

    gi.user_clear = 1;
    gi.time_clear = 1;
    for (strqS_qm = my_qm, strqS_cc = my_cc,
	    wd=strqtokS(NULL, SEP, NULL, NULL, 1); wd;
				wd = strqtokS(NULL, SEP, NULL, NULL, 1)) {
	if (global_arg(wd) == -1)
	    return -1;
    }
    return 0;
}

int
option_local_clear_settings()
{
    /* Clear local settings */
    int i;
    int maxfd = MAXFD;

    if (debug>1) fprintf(stderr, "\toption_local_clear_settings()\n");

    maxfd = MAXFD;
    if (li.info)
	free(li.info);
    li.info = NULL;
    if (li.die)
	free(li.die);
    li.die = NULL;
    if (li.print)
	free(li.print);
    li.print = NULL;
    *li.user = *li.group = *li.u_g = '\0';
    if (li.env)
	free(li.env);
    li.env = NULL;
    li.setenv[0] = NULL;
    for (i=0; i<=MAXSETENV; i++) {
	if (li.setenv[i])
	    free(li.setenv[i]);
	li.setenv[i]=NULL;
    }
    strcpy(li.owner, gi.owner);
    if (li.fdlist)
	free(li.fdlist);
    li.fdlist = NULL;
    if (!li.fd) {
	li.fd = (int *) malloc(sizeof(int) * (maxfd + 1));
	if (!li.fd)
	    return Error(1, 0,
		"%t\n\tFailed to malloc space for file descriptor list: ");
    }
    for (i=0; i<=maxfd; i++) {
	li.fd[i] = 0;
    }
    li.mailcmd[0] = '\0';
    li.mail_success = -1;
    error_command = gi.mailcmd[0] ? gi.mailcmd : NULL;
    init_umask(0);
    init_nice_incr(0);
    free_TimeList(&li.time);
    free_Simple2List(&li.userpats);
    free_SimpleList(&li.origtext);

    li.usr_args[0] = li.usr_args[1] = -1;
    StrEltsUnused(&li.argpats);
    /* should use struct assignment, but that's not portable to
     * all k&r compilers.
     */
    li.passinfo.required = gi.passinfo.required;
    li.passinfo.timeout = gi.passinfo.timeout;
    li.passinfo.renewtime = gi.passinfo.renewtime;
    li.passinfo.perhost = gi.passinfo.perhost;
    strcpy(li.passinfo.user, gi.passinfo.user);

    return 0;
}

static int
option_patterns(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    if (debug)
	fprintf(stderr, "\toption:patterns=%s (%s)\n",
				s, isglobal ? "global" : "local");

    if (strcmp(s, "shell") == 0) {
	pat_compile = shell_compile;
	pat_compare = shell_compare;
	need_re_anchor = 0;
    } else if (strcmp(s, "regex") == 0) {
	pat_compile = re_comp;
	pat_compare = re_exec;
	need_re_anchor = 1;
    } else {
	return Error(0, 0, "%t\n\tInvalid pattern type `%s'.  \
Valid: \"shell\" and \"regex\"\n", s);
    }
    return 0;
}

static int
option_syslog(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    if (debug)
	fprintf(stderr, "\toption:syslog=%s (%s)\n",
				s, isglobal ? "global" : "local");

    /* Don't do any logging if just checking syntax! */
    if (check_syntax)
	return 0;

    if (strcmp(s, "n") == 0) {
	error_syslog = 0;
    } else {
	error_syslog = 1;
	if (strcmp(s, "y") != 0)
	    return Error(0, 0, "%t\n\tInvalid option value in `%s' \
-- value must be `y' or `n'.\n", word);
#ifndef USE_SYSLOG
	return Error(0, 0,
"%t\n\tCan't enable syslog() calls -- not compiled with USE_SYSLOG defined.\n");
#endif
    }
    return 0;
}

static int
option_logfile(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    if (debug)
	fprintf(stderr, "\toption:logfile=%s (%s)\n",
				s, isglobal ? "global" : "local");

    /* Don't do any logging if just checking syntax! */
    if (check_syntax)
	return 0;

    strcpy(gi.log.filename, s);
    return 0;
}

static int
option_loguid(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    if (debug)
	fprintf(stderr, "\toption:loguid=%s (%s)\n",
				s, isglobal ? "global" : "local");

    /* Don't do any logging if just checking syntax! */
    if (check_syntax)
	return 0;

    strcpy(gi.log.user, s);
    return 0;
}

static int
option_mail(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    if (debug)
	fprintf(stderr, "\toption:mail=%s (%s)\n",
				s, isglobal ? "global" : "local");

    /* Don't do any logging if just checking syntax! */
    if (check_syntax)
	return 0;

    if (isglobal) {
	if (strlen(s) > sizeof(gi.mailcmd)-1)
	    return Error(0, 0, "%t\n\tRidiculous length of string: `%s'.\n\
\tMaximum permitted length is %d; the entry used %d\n",
		word, sizeof(gi.mailcmd)-1, strlen(s));

	strcpy(gi.mailcmd, s);
	error_command = gi.mailcmd;
	gi.mail_success = 0;
    } else {
	if (strlen(s) > sizeof(li.mailcmd)-1)
	    return Error(0, 0, "%t\n\tRidiculous length of string: `%s'.\n\
\tMaximum permitted length is %d; the entry used %d\n",
		word, sizeof(li.mailcmd)-1, strlen(s));

	strcpy(li.mailcmd, s);
	error_command = li.mailcmd;
	li.mail_success = 0;
    }
    return 0;
}

static int
option_mailany(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    if (debug)
	fprintf(stderr, "\toption:mailany=%s (%s)\n",
				s, isglobal ? "global" : "local");

    /* Don't do any logging if just checking syntax! */
    if (check_syntax)
	return 0;

    if (isglobal) {
	if (strlen(s) > sizeof(gi.mailcmd)-1)
	    return Error(0, 0, "%t\n\tRidiculous length of string: `%s'.\n\
\tMaximum permitted length is %d; the entry used %d\n",
		word, sizeof(gi.mailcmd)-1, strlen(s));

	strcpy(gi.mailcmd, s);
	error_command = gi.mailcmd;
	gi.mail_success = 1;
    } else {
	if (strlen(s) > sizeof(li.mailcmd)-1)
	    return Error(0, 0, "%t\n\tRidiculous length of string: `%s'.\n\
\tMaximum permitted length is %d; the entry used %d\n",
		word, sizeof(li.mailcmd)-1, strlen(s));

	strcpy(li.mailcmd, s);
	error_command = li.mailcmd;
	li.mail_success = 1;
    }
    return 0;
}

static int
option_timestampuid(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    if (debug)
	fprintf(stderr, "\toption:timestampuid=%s (%s)\n",
				s, isglobal ? "global" : "local");

    strcpy(gi.passinfo.user, s);
    return 0;
}

static int
option_timestampbyhost(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    if (debug)
	fprintf(stderr, "\toption:timestampbyhost=%s (%s)\n",
				s, isglobal ? "global" : "local");

    if (strcmp(s, "n") == 0) {
	gi.passinfo.perhost = 0;
    } else if (strcmp(s, "y") == 0) {
	gi.passinfo.perhost = 1;
    } else {
	return Error(0, 0,
		"%t\n\tInvalid option value in `%s' -- \
value must be `y' or `n'.\n", word);
    }
    return 0;
}

static int
option_renewtime(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    if (debug)
	fprintf(stderr, "\toption:renewtime=%s (%s)\n",
				s, isglobal ? "global" : "local");

    if (strcmp(s, "y") == 0) {
	gi.passinfo.renewtime = 1;
    } else {
	gi.passinfo.renewtime = 0;
	if (strcmp(s, "n") != 0)
	    return Error(0, 0, "%t\n\tInvalid option value \
in `%s' -- value must be `y' or `n'\n", word);
    }
    return 0; 
}

static int
option_timeout(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    if (debug)
	fprintf(stderr, "\toption:timeout=%s (%s)\n",
				s, isglobal ? "global" : "local");

    if (isglobal)
	gi.passinfo.timeout = atoi(s);
    else
	li.passinfo.timeout = atoi(s);

    return 0;
}

static int
option_password(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    int required;

    if (debug)
	fprintf(stderr, "\toption:password=%s (%s)\n",
				s, isglobal ? "global" : "local");

    if (strcmp(s, "n") == 0) {
	required = 0;
    } else {
	required = 1;
	if (strcmp(s, "y") != 0)
	    return Error(0, 0, "%t\n\tInvalid option value \
in `%s' -- value must be `y' or `n'\n", word);
    }

    if (isglobal)
	gi.passinfo.required = required;
    else
	li.passinfo.required = required;

    return 0;
}

static int
option_owner(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    if (debug)
	fprintf(stderr, "\toption:owner=%s (%s)\n",
				s, isglobal ? "global" : "local");

    if (isglobal)
	stringcopy(gi.owner, s, sizeof(gi.owner));
    else
        stringcopy(li.owner, s, sizeof(li.owner));
    return 0; 
}

static int
option_nice(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    int nice_incr;
    char *p;
    
    if (debug)
	fprintf(stderr, "\toption:nice=%s (%s)\n",
				s, isglobal ? "global" : "local");

    nice_incr = strtol(s, &p, 0);
    if (p == word)
	return Error(0, 0, "%t\n\tillegal value in `nice=%s'\n", word);
    store_nice_incr(nice_incr, isglobal);
    return 0;
}

static int
option_umask(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    char *p;
    int mask = strtol(s, &p, 0);

    if (debug)
	fprintf(stderr, "\toption:umask=%s (%s)\n",
				s, isglobal ? "global" : "local");

    if (mask < 0 || p == word)
	Error(0, 0, "%t\n\tillegal value in `%s'\n", word);
    store_umask(mask, isglobal);
    return 0;
}

static int
option_info(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    if (debug)
	fprintf(stderr, "\toption:info=%s (%s)\n",
				s, isglobal ? "global" : "local");

    li.info = strdup(s);
    if (!li.info)
	return Error(0, 0,
	    "Failed to malloc space for info=<%s>\n", s);
    return 0;
}

static int
option_group_slash(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    if (debug)
	fprintf(stderr, "\toption:group_slash=%s (%s)\n",
				s, isglobal ? "global" : "local");

    if (strcmp(s, "n") == 0) {
	gi.group_slash = 0;
    } else {
	gi.group_slash = 1;
	if (strcmp(s, "y") != 0)
	    return Error(0, 0, "%t\n\tInvalid option value \
in `%s' -- value must be `y' or `n'\n", word);
    }
    return 0;
}

static int
option_relative_path(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    if (debug)
	fprintf(stderr, "\toption:relative_path=%s (%s)\n",
				s, isglobal ? "global" : "local");

    if (strcmp(s, "n") == 0) {
	gi.relative_path = 0;
    } else {
	gi.relative_path = 1;
	if (strcmp(s, "y") != 0)
	    return Error(0, 0, "%t\n\tInvalid option value \
in `%s' -- value must be `y' or `n'\n", word);
    }
    return 0;
}


static int
option_uid(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    if (debug)
	fprintf(stderr, "\toption:uid=%s (%s)\n",
				s, isglobal ? "global" : "local");

    stringcopy(li.user, s, sizeof(li.user));
    return 0;
}

static int
option_gid(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    if (debug)
	fprintf(stderr, "\toption:gid=%s (%s)\n",
				s, isglobal ? "global" : "local");

    stringcopy(li.group, s, sizeof(li.group));
    return 0;
}

static int
option_u_g(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    if (debug)
	fprintf(stderr, "\toption:u+g=%s (%s)\n",
				s, isglobal ? "global" : "local");

    stringcopy(li.u_g, s, sizeof(li.u_g));
    return 0;
}

static int
option_env(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    if (debug)
	fprintf(stderr, "\toption:env=%s (%s)\n",
				s, isglobal ? "global" : "local");

    li.env = strdup(s);
    if (!li.env)
	return Error(0, 0,
	    "Failed to malloc space for copy of env=<%s>\n", s);

    return 0;
}

static int
option_setenv(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
   char **p;
    int i;

    if (debug)
	fprintf(stderr, "\toption:setenv=%s (%s)\n",
				s, isglobal ? "global" : "local");

    /* s must have form v=xxx, where v is any string not including `='. */
    if (*s == '=' || !strchr(s+1, '='))
	return Error(0, 0,
	"%t\n\tbad syntax for setenv%cvar=xxx; you used `setenv%c%s'.\n",
	OPTION_SEP, OPTION_SEP, s);

    /* Append this variable: skip past already-set variables */
    for (i=0, p=li.setenv; p[i] && i < MAXSETENV; i++)
	;
    /* then add this definition */
    if (i == MAXSETENV) {
	return Error(0, 0,
		"%t\n\ttoo many setenv%cvar=xxx entries; max is %d.\n",
		OPTION_SEP, MAXSETENV);
    } else {
	char *t = strdup(s);
	if (!t)
	    return Error(0, 0,
	    "Failed to malloc space for setenv using <%s>\n", s);
	li.setenv[i++] = t;
	li.setenv[i] = NULL;
    }
    return 0;
}

static int
option_fd(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    int n;
    char *w, *p;
    char buf[500];
    int maxfd = MAXFD;

    if (debug)
	fprintf(stderr, "\toption:fd=%s (%s)\n",
				s, isglobal ? "global" : "local");

    if (strlen(s) > sizeof(buf)-1)
	return Error(0, 0, "%t\n\tRidiculous length of string: `%s'.\n\
\tMaximum permitted length is %d; the entry used %d\n",
		word, sizeof(buf)-1, strlen(s));

    li.fdlist = strdup(s);
    if (!li.fdlist)
	return Error(0, 0,
	    "Failed to malloc space for copy of fdlist=<%s>\n", s);

    strcpy(buf, s);
    for (w=strtok(buf, ","); w; w=strtok(NULL, ",")) {
	if (((n=strtol(s, &p, 0)) >= 0) && n <= maxfd && p != s)
	    li.fd[n] = 1;
	else
	    return Error(0, 0,
		"%t\n\tRidiculous value in file descriptor list: `%s'\n",
		word);
    }
    return 0;
}

static int
option_nargs(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    int i, m, n;

    if (debug)
	fprintf(stderr, "\toption:nargs=%s (%s)\n",
				s, isglobal ? "global" : "local");

    i = sscanf(s, "%u-%u", &m, &n);
    switch(i) {
    case 1:     li.usr_args[0] = li.usr_args[1] = m;
		break;
    case 2:     li.usr_args[0] = m;
		li.usr_args[1] = n;
		break;
    default:
	return Error(0, 0,
		"%t\n\tnargs must be nargs=nnn or  nargs=mmm-nnn.\n", word);
    }
    return 0; 
}

static int
option_arg(word, pat, isglobal)
char *word;
char *pat;
int isglobal;
{
    /* Must be argMMM-NNN<SEP>SSS or argNNN<SEP>SSS */
    int i, iarg1, iarg2;
    char eq;

    if (debug)
	fprintf(stderr, "\toption:argMMM-NNN=%s (%s)\n",
				pat, isglobal ? "global" : "local");

    if (*pat == '\0')
	return Error(0, 0,
		"%t\n\tImproper use of argNNN%cXXX option\n", OPTION_SEP);

    /* Check that word matches argNNN=SSS */
    i = sscanf(word, "arg%d-%d%c", &iarg1, &iarg2, &eq);
    if (i != 3 || eq != OPTION_SEP) {
	i = sscanf(word, "arg%d%c", &iarg1, &eq);
	if (i == 2)
	    i = 3;
	iarg2 = iarg1;
    }
    if (i != 3 || eq != OPTION_SEP || iarg1 <= 0 || iarg2 < iarg1)
	return Error(0, 0,
		"%t\n\tImproper use of argNNN%cXXX option\n", OPTION_SEP);

    /* Put pattern into the argpat list. */
    {
	char buf[500];
	int iarg;
	if (strlen(pat) > sizeof(buf)-3)
	    return Error(0, 0,
			    "%t\n\tArgument pattern <%s> too long (max %d)\n",
			    pat, sizeof(buf)-3);
	anchor(pat, buf);
	for (iarg = iarg1; iarg <= iarg2; iarg++) {
	    if (StrEltCpy(&li.argpats, iarg, buf) == -1)
		return Error(0, 0,
		    "Failed to malloc space for argument pattern #%d\n", iarg);
	}
    }
    return 0; 
}

static int
option_die(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    if (debug)
	fprintf(stderr, "\toption:die=%s (%s)\n",
				s, isglobal ? "global" : "local");

    li.die = strdup(s);
    if (!li.die)
	return Error(0, 0, "%t\n\tfailed to malloc space for die=<%s>\n", s);
    return 0;
}

static int
option_print(word, s, isglobal)
char *word;
char *s;
int isglobal;
{
    if (debug)
	fprintf(stderr, "\toption:print=%s (%s)\n",
				s, isglobal ? "global" : "local");

    li.print = strdup(s);
    if (!li.print)
	return Error(0, 0, "%t\n\tfailed to malloc space for print=<%s>\n", s);
    return 0;
}
