/*
 * dproc.c - NEXTSTEP process access functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dproc.c,v 1.14 96/02/21 08:08:12 abe Exp $";
#endif

#include "lsof.h"

#if	defined(HASNCACHE)
#include <sys/dnlc.h>
#endif	/* defined(HASNCACHE) */


/*
 * Local static values
 */

static int Mxp = 0;			/* maximum number of processes */
static int Np;				/* number of entries in P[] */
static int Nv = 0;			/* allocated Vp[] entries */
static struct proc *P = NULL;		/* local proc structure table */
static KA_T *Pa = NULL;			/* kernel address for each P[] entry */
static KA_T Kp;				/* kernel process table pointer */
static KA_T *Vp = NULL;			/* vnode address cache */


_PROTOTYPE(static void get_kernel_access,(void));

#if	defined(HASNCACHE)
_PROTOTYPE(static void ncache_load,(void));
#endif	/* defined(HASNCACHE) */

_PROTOTYPE(static void process_map,(caddr_t map));
_PROTOTYPE(static void read_proc,(void));


/*
 * gather_proc_info() -- gather process information
 */

void
gather_proc_info()
{
	int i, nf, px;
	MALLOC_S nb;
	short pss, sf;
	struct task {			/* (Should come from <kern/task.h>.) */

#if	defined(hppa)
		caddr_t d1[6];
#else	/* !defined(hppa) */
		caddr_t d1[3];
#endif	/* defined(hppa) */

		caddr_t map;

#if	defined(hppa)
		caddr_t d2[13];
#else	/* !defined(hppa) */
		caddr_t d2[10];
#endif	/* defined(hppa) */

		struct utask *u_address;
		struct proc *proc;
	} t;
	struct utask *u;
	static struct file **uf = NULL;
	static MALLOC_S ufb = 0;
	struct utask ut;

/*
 * Read kernel name cache and process table.
 */

#if	defined(HASNCACHE)
	ncache_load();
#endif	/* defined(HASNCACHE) */

	(void) read_proc();
/*
 * Process proc structures pre-loaded in initialize().
 */
	for (px = 0, u = &ut; px < Np; px++) {
		if (is_proc_excl(P[px].p_pid, (int)P[px].p_pgrp,
		    (UID_ARG)P[px].p_uid, &pss, &sf))
			continue;
	/*
	 * Read the task associated with the process, and the user
	 * area assocated with the task.
	 */
		if (kread((KA_T)P[px].task, (char *)&t, sizeof(t)))
			continue;
		if ((KA_T)t.proc != Pa[px])
			continue;
		if (kread((KA_T)t.u_address, (char *)&ut, sizeof(ut)))
			continue;
		if ((KA_T)ut.uu_procp != Pa[px])
			continue;
	/*
	 * NEXTSTEP file pointers come from a structure whose pointer is
	 * in the user task area.
	 */
		nf = ut.uu_ofile_cnt;
		nb = (MALLOC_S) (sizeof(struct file) * nf);
		if (uf == NULL) {
			if ((uf = (struct file **)malloc(nb)) == NULL) {
				(void) fprintf(stderr,
					"%s: no file table space\n", Pn);
				exit(1);
			}
			ufb = nb;
		} else if (nb > ufb) {
			if ((uf = (struct file **)realloc((MALLOC_P *)uf, nb))
			== NULL) {
				(void) fprintf(stderr,
					"%s: no realloc file space\n", Pn);
				exit(1);
			}
			ufb = nb;
		}
		if (kread((KA_T)ut.uu_ofile, (char *)uf, nb))
			continue;
	/*
	 * Allocate a local process structure and start filling it.
	 */
		if (is_cmd_excl(u->u_comm, &pss, &sf))
			continue;
		alloc_lproc(P[px].p_pid, (int)P[px].p_pgrp,
			(UID_ARG)P[px].p_uid, u->u_comm, (int)pss, (int)sf);
		Plf = NULL;
	/*
	 * Save current working directory information.
	 */
		if (u->u_cdir) {
			alloc_lfile(CWD, -1);
			process_node((caddr_t)u->u_cdir);
			if (Lf->sf)
				link_lfile();
		}
	/*
	 * Save root directory information.
	 */
		if (u->u_rdir) {
			alloc_lfile(RTD, -1);
			process_node((caddr_t)u->u_rdir);
			if (Lf->sf)
				link_lfile();
		}
	/*
	 * Print information on the text files of the virtual memory
	 * address map.
	 */
		if (t.map)
			process_map(t.map);
	/*
	 * Save information on file descriptors.
	 */
		for (i = 0; i < nf; i++) {
			if (uf[i]) {
				alloc_lfile(NULL, i);
				process_file(uf[i]);
				if (Lf->sf)
					link_lfile();
			}
		}
	/*
	 * Examine results.
	 */
		if (examine_lproc())
			return;
	}
}


/*
 * get_kernel_access() - access the required information in the kernel
 */

static void
get_kernel_access()
{

#if	defined(HAS_AFS)
	struct nlist *nl = (struct nlist *)NULL;
#define	NLSIZE	((X_LAST + 1) * sizeof(struct nlist))
#endif	/* defined(HAS_AFS) */

#if	defined(WILLDROPGID)
/*
 * If kernel memory isn't coming from KMEM, drop setgid permission
 * before attempting to open the (Memory) file.
 */
	if (Memory)
		(void) dropgid();
#else	/* !defined(WILLDROPGID) */
/*
 * See if the non-KMEM memory file is readable.
 */
	if (Memory && !is_readable(Memory, 1))
		exit(1);
#endif	/* defined(WILLDROPGID) */

/*
 * Access the kernel memory file.
 */
	if ((Kmem = open(Memory ? Memory : KMEM, O_RDONLY, 0)) < 0) {
		(void) fprintf(stderr, "%s: can't open %s: %s\n", Pn,
			Memory ? Memory : KMEM, strerror(errno));
		exit(1);
	}

#if	defined(WILLDROPGID)
/*
 * Drop setgid permission, if necessary.
 */
	if (!Memory)
		(void) dropgid();
#else	/* !defined(WILLDROPGID) */
/*
 * See if the name list file is readable.
 */
	if (Nmlst && !is_readable(Nmlst, 1))
		exit(1);
#endif	/* defined(WILLDROPGID) */

#if	defined(HAS_AFS)
	if (!Nmlst) {

	/*
	 * If AFS is defined and we're getting kernel symbol values from
	 * from N_UNIX, make a copy of Nl[] for possible use with the AFS
	 * module name list file.
	 */
		if ((nl = (struct nlist *)malloc(NLSIZE))
		== (struct nlist *)NULL)
		{
			(void) fprintf(stderr,
				"%s: no space (%d) for Nl[] copy\n",
				Pn, NLSIZE);
			exit(1);
		}
		(void) bcopy((char *)Nl, (char *)nl, (int)NLSIZE);
	}
#endif	/* defined(HAS_AFS) */

/*
 * Access the name list file.
 */
	if (nlist(Nmlst ? Nmlst : VMUNIX, Nl) < 0) {
		(void) fprintf(stderr, "%s: can't read namelist from %s\n",
			Pn, Nmlst ? Nmlst : VMUNIX);
                exit(1);
	}
	if (Nl[X_ALLPROC].n_value == 0l) {
		(void) fprintf(stderr, "%s: no address for %s\n",
			Pn, Nl[X_ALLPROC].n_un.n_name);
		exit(1);
	}

#if	defined(HAS_AFS)
	if (nl) {

	/*
	 * If AFS is defined and we're getting kernel symbol values from
	 * N_UNIX, and if any X_AFS_* symbols isn't there, see if it is in the
	 * the AFS module name list file.  Make sure that other symbols that
	 * appear in both name list files have the same values.
	 */
		if (!Nl[X_AFS_FID].n_value || !Nl[X_AFS_OPS].n_value
		||  !Nl[X_AFS_VOL].n_value)
			(void) ckAFSsym(nl);
		(void) free((MALLOC_P *)nl);
	}
#endif	/* defined(HAS_AFS) */

/*
 * Establish a maximum process count estimate.
 */
	if (Nl[X_MAXPROC].n_value == 0
	||  kread((KA_T)Nl[X_MAXPROC].n_value, (char *)&Mxp, sizeof(Mxp))
	||  Mxp < 1)
		Mxp = PROCDFLT;
}


/*
 * initialize() - perform all initialization
 */

void
initialize()
{
	get_kernel_access();
	(void) iuidcache(UIDCACHEL);
}


/*
 * kread() - read from kernel memory
 */

int
kread(addr, buf, len)
	KA_T addr;			/* kernel memory address */
	char *buf;			/* buffer to receive data */
	READLEN_T len;			/* length to read */
{
	int br;

	if (lseek(Kmem, addr, L_SET) == (off_t)-1L)
		return(-1);
	br = read(Kmem, buf, len);
	return((br == len) ? 0 : 1);
}


/*
 * process_map() - process vm map for vnode references
 */

static void
process_map(map)
	caddr_t map;
{
	int i, j, n;
	struct vm_map_entry {	/* (Should come from <vm/vm_map.h>). */
		struct vm_map_entry *prev;
		struct vm_map_entry *next;
		unsigned int start;
		unsigned int end;
		caddr_t object;
		unsigned int offset;
		unsigned int
			is_a_map:1,
			is_sub_map:1,
			copy_on_write:1,
			needs_copy:1;
		int protection;
		int max_protection;
		int inheritance;
		int wired_count;
	} vme, *vmep;
	struct vm_map {		/* (Should come from <vm/vm_map.h>.) */

#if	defined(hppa)
		caddr_t d1[6];
#else	/* !defined(hppa) */
		caddr_t d1[3];
#endif	/* defined(hppa) */

		struct vm_map_entry header;
		int nentries;
		caddr_t pmap;
		unsigned int size;
		boolean_t is_main_map;
	} vmm;
	struct vm_object {	/* (Should come from <vm/vm_object.h>.) */

#if	defined(hppa)
		caddr_t d1[8];
#else	/* !defined(hppa) */
		caddr_t d1[5];
#endif	/* defined(hppa) */

		unsigned int size;
		short ref_count, resident_page_count;
		caddr_t copy;
		caddr_t pager;
		int pager_request, pager_name;
		unsigned int paging_offset;
		caddr_t shadow;
	} vmo, vmso;
	struct vstruct {	/* (Should come from <vm/vnode_pager.h>.) */
		boolean_t is_device;
		caddr_t vs_pf;
		caddr_t pfMapEntry;
		unsigned int vs_swapfile:1;
		short vs_count;
		int vs_size;
		caddr_t vs_vp;
	} vmp;
/*
 * Read the vm map.
 */
	if (map == NULL
	||  kread((KA_T)map, (char *)&vmm, sizeof(vmm)))
		return;
	if ( ! vmm.is_main_map)
		return;
/*
 * Look for non-map and non-sub-map vm map entries that have an object
 * with a shadow whose pager pointer addresses a non-swap-file istruct
 * that has a vnode pointer.  Process the unique vnodes found.
 */ 
	for (i = n = 0, vme = vmm.header; i < vmm.nentries; i++) {
		if (i) {
			if (vme.next == NULL
			||  kread((KA_T)vme.next, (char *)&vme, sizeof(vme)))
				continue;
		}
		if (vme.is_a_map || vme.is_sub_map)
			continue;
		if (vme.object == NULL
		||  kread((KA_T)vme.object, (char *)&vmo, sizeof(vmo)))
			continue;
		if (vmo.shadow == NULL
		||  kread((KA_T)vmo.shadow, (char *)&vmso, sizeof(vmso)))
			continue;
		if (vmso.pager == NULL
		||  kread((KA_T)vmso.pager, (char *)&vmp, sizeof(vmp)))
			continue;
		if (vmp.is_device || vmp.vs_swapfile || vmp.vs_vp == NULL)
			continue;
	/*
	 * See if the vnode has been processed before.
	 */
		for (j = 0; j < n; j++) {
			if ((KA_T)vmp.vs_vp == Vp[j])
				break;
		}
		if (j < n)
			continue;
	/*
	 * Process a new vnode.
	 */
		alloc_lfile("txt", -1);
		process_node(vmp.vs_vp);
		if (Lf->sf)
			link_lfile();
	/*
	 * Allocate space for remembering the vnode.
	 */
		if (Vp == NULL) {
			if ((Vp = (KA_T *)malloc((MALLOC_S)
			    (sizeof(struct vnode *) * 10)))
			== NULL) {
				(void) fprintf(stderr,
					"%s: no txt ptr space, PID %d\n",
					Pn, Lp->pid);
				exit(1);
			}
			Nv = 10;
		} else {
			Nv += 10;
			if ((Vp = (KA_T *)realloc((MALLOC_P *)Vp,
			     (MALLOC_S)(Nv * sizeof(struct vnode *))))
			== NULL) {
				(void) fprintf(stderr,
					"%s: no more txt ptr space, PID %d\n",
					Pn, Lp->pid);
				exit(1);
			}
		}
		Vp[n++] = (KA_T)vmp.vs_vp;
	}
}


/*
 * read_proc() - read proc structures
 */

static void
read_proc()
{
	static int sz = 0;
	int try;
	KA_T kp;
	struct proc *p;
/*
 * Try PROCTRYLM times to read a valid proc table.
 */
	for (try = 0; try < PROCTRYLM; try++) {

	/*
	 * Read kernel's process list pointer.  This needs to be done each
	 * time lsof rereads the process list.
	 */
	    if (kread((KA_T)Nl[X_ALLPROC].n_value, (char *)&Kp, sizeof(Kp))) {
		if (!Fwarn)
		    (void) fprintf(stderr,
			"%s: WARNING: can't read %s from %#x\n",
			Pn, Nl[X_ALLPROC].n_un.n_name, Nl[X_ALLPROC].n_value);
		continue;
	    }

	/*
	 * Pre-allocate proc structure space.
	 */
		if (sz == 0) {
		    sz = Mxp;
		    if ((P = (struct proc *)malloc((MALLOC_S)
				(sz * sizeof(struct proc))))
		    == (struct proc *)NULL)
		    {
			(void) fprintf(stderr, "%s: no proc table space\n",
			    Pn);
			exit(1);
		    }
		    if ((Pa = (KA_T *)malloc((MALLOC_S)(sz * sizeof(KA_T))))
		    == (KA_T *)NULL)
		    {
			(void) fprintf(stderr, "%s: no proc pointer space\n",
			    Pn);
			exit(1);
		    }
		}
	/*
	 * Accumulate proc structures.
	 */
		for (kp = Kp, Np = 0; kp; ) {
			if (kread(kp, (char *)&P[Np], sizeof(struct proc))) {
				Np = 0;
				break;
			}
			Pa[Np] = kp;
			kp = (KA_T)P[Np].p_nxt;
			if (P[Np].p_stat == 0 || P[Np].p_stat == SZOMB)
				continue;
			Np++;
			if (Np >= sz) {

			/*
			 * Expand the local proc table.
			 */
				sz += PROCDFLT/2;
				if ((P = (struct proc *)realloc((MALLOC_P *)P,
					(MALLOC_S)(sizeof(struct proc) * sz)))
				== NULL) {
					(void) fprintf(stderr,
						"%s: no more (%d) proc space\n",
						Pn, sz);
					exit(1);
				}
				if ((Pa = (KA_T *)realloc((MALLOC_P *)Pa,
					(MALLOC_S)(sizeof(KA_T) * sz)))
				== NULL) {
					(void) fprintf(stderr,
					    "%s: no more (%d) proc ptr space\n",
					    Pn, sz);
					exit(1);
				}
			}
		}
	/*
	 * If not enough processes were saved in the local table, try again.
	 */
		if (Np >= PROCMIN)
			break;
	}
/*
 * Quit if no proc structures were stored in the local table.
 */
	if (try >= PROCTRYLM) {
		(void) fprintf(stderr, "%s: can't read proc table\n", Pn);
		exit(1);
	}
	if (Np < sz && !RptTm) {

	/*
	 * Reduce the local proc structure table size to a minimum if
	 * not in repeat mode.
	 */
		if ((P = (struct proc *)realloc((MALLOC_P *)P,
			 (MALLOC_S)(sizeof(struct proc) * Np)))
		== NULL) {
			(void) fprintf(stderr,
				"%s: can't reduce proc table to %d\n",
				Pn, Np);
			exit(1);
		}
		if ((Pa = (KA_T *)realloc((MALLOC_P *)Pa,
			  (MALLOC_S)(sizeof(KA_T) * Np)))
		== NULL) {
			(void) fprintf(stderr,
				"%s: can't reduce proc ptrs to %d\n",
				Pn, Np);
			exit(1);
		}
	}
}


/*
 * The ncache_addr(), ncache_load(), and ncache_lookup() functions are
 * obtained from ../common/rnch.frag.
 */

#define	n_name	n_un.n_name
