

/*
 * $Id: rnam.frag,v 1.13 96/02/20 06:45:35 abe Exp $
 */


#if	defined(HASNCACHE)
/*
 * rnam.frag - read BSD format (struct namecache or nch) name cache
 *
 * This fragment is effective only when HASNCACHE is defined.
 */

/*
 * The caller must:
 *
 *	#include the relevant header file -- e.g., <sys/namei.h>.
 *
 *	Define X_NCACHE as the Nl[] index to the kernel cache address.
 *
 *	Define X_NCSIZE as the Nl[] index to the size of the kernel
 *	cache.
 *
 *	Define NCACHE_NXT if the kernel's name cache is a linked
 *	list starting at Nl[X_NCACHE].n_value, rather than a table
 *	starting at that address.
 *
 *	Define NCACHE_NO_ROOT if the calling dialect doesn't support
 *	the locating of the root node of a file system.
 *
 *	    _PROTOTYPE(extern int ncache_isroot,(unsigned long na, char *cp));
 *	or
 *	    _PROTOTYPE(static int ncache_isroot,(unsigned long na, char *cp));
 *
 *	Define the name of the name cache structure -- e.g.,
 *
 *		#define NCACHE	<structure name>
 *
 *	Define the names of these elements of struct NCACHE:
 *
 *	must		#define NCACHE_NM	<name>
 *	must		#define NCACHE_NMLEN	<name length
 *	optional	#define NCACHE_NXT	<link to next entry>
 *	must		#define NCACHE_NODEADDR	<node address>
 *	must		#define NCACHE_NODEID	<node capability ID)
 *	optional	#define NCACHE_PARADDR	<parent node address>
 *	optional	#define NCACHE_PARID	<parent node capability ID)
 *
 * The caller may need to:
 *
 *	Define NCHNAMLEN as the length of the name element of NCACHE, if it's
 *	not defined in the header file that defines the NCACHE structure.
 *
 *	Define this prototype for ncache_load() if it is called
 *	before rnam.frag is included.
 *
 *		_PROTOTYPE(static void ncache_load,(void));
 */


/*
 * Local static values
 */

static int Mch;				/* name cache hash mask */
static int Nc = 0;			/* size of name cache */
static int Nch = 0;			/* size of name cache hash pointer
					 * table */
struct l_nch {
	unsigned long na;		/* node address */
	unsigned long id;		/* capability ID */

# if	defined(NCACHE_PARADDR) && defined(NCACHE_PARID)
	unsigned long pa;		/* parent node address */
	struct l_nch *pla;		/* parent local node address */
	unsigned long did;		/* parent capability ID */
# endif	/* defined(NCACHE_PARADDR) && defined(NCACHE_PARID) */

	char nm[NCHNAMLEN+1];		/* name */
	int nl;				/* name length */
};

static struct l_nch *Ncache = NULL;	/* the local name cache */
static struct l_nch **Nchash = NULL;	/* Ncache hash pointers */
static int Ncfirst = 1;			/* first-call status */

#define ncachehash(i,n)		Nchash+(((((int)(n)>>2)+((int)(i)))*31415)&Mch)
#define DEFNCACHESZ	512	/* local size if X_NCACHE kernel value < 1 */
#define	LNCHINCRSZ	64	/* local size increment */

_PROTOTYPE(static struct l_nch *ncache_addr,(unsigned long i, unsigned long n));

#if	!defined(NCACHE_NO_ROOT)
_PROTOTYPE(static int ncache_isroot,(unsigned long na, char *cp));
#endif	/* !defined(NCACHE_NO_ROOT) */


/*
 * ncache_addr() - look up a node's local ncache address
 */

static struct l_nch *
ncache_addr(i, n)
	unsigned long i;		/* node's capability ID */
	unsigned long n;		/* node's address */
{
	struct l_nch **hp;

	for (hp = ncachehash(i, n); *hp; hp++) {
	    if ((*hp)->id == i && (*hp)->na == n)
		return(*hp);
	}
	return(NULL);
}


#if	!defined(NCACHE_NO_ROOT)
/*
 * ncache_isroot() - is head of name cache path a file system root?
 */

static int
ncache_isroot(na, cp)
	unsigned long na;			/* kernel node address */
	char *cp;				/* partial path */
{
	char buf[MAXPATHLEN];
	int i;
	MALLOC_S len;
	struct mounts *mtp;
	static int nca = 0;
	static int ncn = 0;
	static unsigned long *nc = (unsigned long *)NULL;
	struct stat sb;
	struct vnode v;

	if (!na)
	    return(0);
/*
 * Search the root vnode cache.
 */
	for (i = 0; i < ncn; i++) {
	    if (na == nc[i])
		return(1);
	}
/*
 * Read the vnode and see if it's a VDIR node with the VROOT flag set.  If
 * it is, then the path is complete.
 *
 * If it isn't, and if the file has an inode number, search the mount table
 * and see if the file system's inode number is known.  If it is, form the
 * possible full path, safely stat() it, and see if it's inode number matches
 * the one we have for this file.  If it does, then the path is complete.
 */
	if (kread((KA_T)na, (char *)&v, sizeof(v))
	||  v.v_type != VDIR || !(v.v_flag & VROOT)) {

	/*
	 * The vnode tests failed.  Try the inode tests.
	 */
	    if (Lf->inp_ty != 1 || !Lf->inode
	    ||  !Lf->fsdir || (len = strlen(Lf->fsdir)) < 1)
		return(0);
	    if ((len + 1 + strlen(cp) + 1) > sizeof(buf))
		return(0);
	    for (mtp = Mtab; mtp; mtp = mtp->next) {
		if (!mtp->dir || !mtp->inode)
		    continue;
		if (strcmp(Lf->fsdir, mtp->dir) == 0)
		    break;
	    }
	    if (!mtp)
		return(0);
	    (void) strcpy(buf, Lf->fsdir);
	    if (buf[len - 1] != '/')
		buf[len++] = '/';
	    (void) strcpy(&buf[len], cp);
	    if (statsafely(buf, &sb) != 0
	    ||  (unsigned long)sb.st_ino != Lf->inode)
		return(0);
	}
/*
 * Add the node address to the root node cache.
 */
	if (ncn >= nca) {
	    if (nca == 0) {
		len = (MALLOC_S)(10 * sizeof(unsigned long));
		if ((nc = (unsigned long *)malloc(len))
		!=  (unsigned long *)NULL)
		    nca = 10;
	    } else {
		len = (MALLOC_S)((nca + 10) * sizeof(unsigned long));
		if ((nc = (unsigned long *)realloc(nc, len))
		!=  (unsigned long *)NULL)
		    nca += 10;
	    }
	    if (nc == (unsigned long *)NULL) {
		(void) fprintf(stderr, "%s: no space for root node table\n",
		    Pn);
		exit(1);
	    }
	}
	nc[ncn++] = na;
	return(1);
}
#endif	/* !defined(NCACHE_NO_ROOT) */


/*
 * ncache_load() - load the kernel's name cache
 */

static void
ncache_load()
{
	struct l_nch **hp, *lc;
	int i, len, n;
	static int iNc = 0;
	struct NCACHE *kc;
	static KA_T kp = (KA_T)NULL;

# if	defined(NCACHE_NXT)
	static KA_T kf;
	struct NCACHE nc;
# else	/* !defined NCACHE_NXT) */
	static struct NCACHE *kca = (struct NCACHE *)NULL;
# endif	/* defined(NCACHE_NXT) */

	if (!Fncache)
	    return;
	if (Ncfirst) {

	/*
	 * Do startup (first-time) functions.
	 */
	    Ncfirst = 0;
	/*
	 * Establish kernel cache size.
	 */
	    if (!Nl[X_NCSIZE].n_value
	    ||  kread((KA_T)Nl[X_NCSIZE].n_value, (char *)&Nc, sizeof(Nc)))
	    {
		if (!Fwarn)
		(void) fprintf(stderr,
		    "%s: WARNING: can't read name cache size (%s): %#x\n",
		    Pn, Nl[X_NCSIZE].n_name, Nl[X_NCSIZE].n_value);
		iNc = Nc = 0;
		return;
	    }
	    iNc = Nc;
	    if (Nc < 1) {
		if (!Fwarn) {
		    (void) fprintf(stderr,
			"%s: WARNING: kernel name cache size: %d\n", Pn, Nc);
		    (void) fprintf(stderr,
			"      Cache size assumed to be: %d\n", DEFNCACHESZ);
		}
		iNc = Nc = DEFNCACHESZ;
	    }
	/*
	 * Establish kernel cache address.
	 */
	    if (!Nl[X_NCACHE].n_value
	    ||  kread((KA_T)Nl[X_NCACHE].n_value, (char *)&kp, sizeof(kp))) {
		if (!Fwarn)
		    (void) fprintf(stderr,
			"%s: WARNING: can't read name cache ptr, %s: %#x\n",
			Pn, Nl[X_NCACHE].n_name, Nl[X_NCACHE].n_value);
		iNc = Nc = 0;
		return;
	    }

# if	defined(NCACHE_NXT)
	    kf = kp;

# else	/* !defined(NCACHE_NXT) */
	/*
	 * Allocate space for a local copy of the kernel's cache.
	 */
	    len = Nc * sizeof(struct NCACHE);
	    if ((kca = (struct NCACHE *)malloc((MALLOC_S)len)) == NULL) {
		if (!Fwarn)
		    (void) fprintf(stderr,
			"%s: can't allocate name cache space: %d\n", Pn, len);
		exit(1);
	    }
# endif	/* defined(NCACHE_NXT) */

	/*
	 * Allocate space for the local cache.
	 */
	    len = Nc * sizeof(struct l_nch);
	    if ((Ncache = (struct l_nch *)malloc((MALLOC_S)len)) == NULL) {

no_local_space:

		if (!Fwarn)
		    (void) fprintf(stderr,
			"%s: no space for %d byte local name cache\n", Pn, len);
		exit(1);
	    }
	} else {

	/*
	 * Do setup for repeat calls.
	 */
	    if ((Nc = iNc) == 0)
		return;
	    if (Nchash) {
		(void) free((FREE_P *)Nchash);
		Nchash = NULL;
	    }

# if    defined(NCACHE_NXT)
	    kp = kf;
# endif /* defined(NCACHE_NXT) */

	}

# if    !defined(NCACHE_NXT)

/*
 * Read the kernel's name cache.
 */
	if (kread(kp, (char *)kca, (Nc * sizeof(struct NCACHE)))) {
	    if (!Fwarn)
		(void) fprintf(stderr,
		    "%s: WARNING: can't read kernel's name cache: %#x\n",
		    Pn, kp);
	    Nc = 0;
	    return;
        }
# endif /* !defined(NCACHE_NXT) */

/*
 * Build a local copy of the kernel name cache.
 */

# if	defined(NCACHE_NXT)
	for (kc = &nc, lc = Ncache, n = 0; kp; )
# else	/* !defined(NCACHE_NXT) */
	for (i = n = 0, kc = kca, lc = Ncache; i < Nc; i++, kc++)
# endif	/* defined(NCACHE_NXT) */

	{

# if	defined(NCACHE_NXT)
	    if (kread(kp, (char *)kc, sizeof(nc)))
		break;
	    if ((kp = (KA_T)kc->NCACHE_NXT) == kf)
		kp = NULL;
# endif	/* defined(NCACHE_NXT) */

	    if (!kc->NCACHE_NODEADDR)
		continue;
	    if ((len = kc->NCACHE_NMLEN) < 1 || len > NCHNAMLEN)
		continue;
	    if (len < 3 && kc->NCACHE_NM[0] == '.') {
		if (len == 1 || (len == 2 && kc->NCACHE_NM[1] == '.'))
		    continue;
	    }

# if	defined(NCACHE_NXT)
	    if (n >= Nc) {
		Nc += LNCHINCRSZ;
		if ((Ncache = (struct l_nch *)realloc(Ncache,
		    (MALLOC_S)(Nc * sizeof(struct l_nch))))
		== NULL) {
		    (void) fprintf(stderr,
			"%s: no more space for %d entry local name cache\n",
			Pn, Nc);
		    exit(1);
		}
		lc = &Ncache[n];
	    }
# endif	/* defined(NCACHE_NXT) */

	    lc->na = (unsigned long)kc->NCACHE_NODEADDR;
	    lc->id = kc->NCACHE_NODEID;

#  if	defined(NCACHE_PARADDR)
	    lc->pa = (unsigned long)kc->NCACHE_PARADDR;
	    lc->pla = (struct l_nch *)NULL;
#  endif	/* defined(NCACHE_PARADDR) */

#  if	defined(NCACHE_PARID)
	    lc->did = kc->NCACHE_PARID;
#  endif	/* defined(NCACHE_PARID) */

	    (void) strncpy(lc->nm, kc->NCACHE_NM, len);
	    lc->nm[len] = '\0';
	    lc->nl = strlen(lc->nm);
	    n++;
	    lc++;

# if	defined(NCACHE_NXT)
	    if (n >= (20 * DEFNCACHESZ)) {
		if (!Fwarn)
		    (void) fprintf(stderr,
			"%s: WARNING: name cache truncated at %d entries\n",
			Pn, n);
		break;
	    }
# endif	/* defined(NCACHE_NXT) */

	}
/*
 * Reduce memory usage, as required.
 */

# if	!defined(NCACHE_NXT)
	if (!RptTm)
	    (void) free((FREE_P *)kca);
# endif	/* !defined(NCACHE_NXT) */

	if (n < 1) {
	    Nc = 0;
	    if (!RptTm) {
		(void) free((FREE_P *)Ncache);
		Ncache = NULL;
	    }
	    if (!Fwarn)
		(void) fprintf(stderr,
		    "%s: WARNING: unusable name cache size: %d\n", Pn, n);
	    return;
	}
	if (n < Nc) {
	    Nc = n;
	    if (!RptTm) {
		len = Nc * sizeof(struct l_nch);
		if ((Ncache = (struct l_nch *)realloc(Ncache, len)) == NULL)
		    goto no_local_space;
	    }
	}
/*
 * Build a hash table to locate Ncache entries.
 */
	for (Nch = 1; Nch < Nc; Nch <<= 1)
	    ;
	Nch <<= 1;
	Mch = Nch - 1;
	if ((Nchash = (struct l_nch **)calloc(Nch + Nc, sizeof(struct l_nch *)))
	== NULL) {
	    if (!Fwarn)
		(void) fprintf(stderr,
		    "%s: no space for %d name cache hash pointers\n",
		    Pn, Nch + Nc);
	    exit(1);
	}
	for (i = 0, lc = Ncache; i < Nc; i++, lc++) {
	    for (hp = ncachehash(lc->id, lc->na), n = 1; *hp; hp++) {
		if ((*hp)->na == lc->na && (*hp)->id == lc->id
		&&  strcmp((*hp)->nm, lc->nm) == 0

# if	defined(NCACHE_PARADDR) && defined(NCACHE_PARID)
		&&  (*hp)->pa == lc->pa && (*hp)->did == lc->did
# endif	/* defined(NCACHE_PARADDR) && defined(NCACHE_PARID) */

		) {
		    n = 0;
		    break;
		}
	    }
	    if (n)
		*hp = lc;
	}

# if	defined(NCACHE_PARADDR) && defined(NCACHE_PARID)
/*
 * Make a final pass through the local cache and convert parent node
 * addresses to local name cache pointers.
 */
	for (i = 0, lc = Ncache; i < Nc; i++, lc++) {
	    if (!lc->pa)
		continue;
	    lc->pla = ncache_addr(lc->did, lc->pa);
	}
# endif	/* defined(NCACHE_PARADDR) && defined(NCACHE_PARID) */
}


/*
 * ncache_lookup() - look up a node's name in the kernel's name cache
 */

char *
ncache_lookup(buf, blen, fp)
	char *buf;			/* receiving name buffer */
	int blen;			/* receiving buffer length */
	int *fp;			/* full path reply */
{
	char *cp = buf;
	struct l_nch *lc;
	struct mounts *mtp;
	int nl, rlen;

	*cp = '\0';
	*fp = 0;

# if	defined(HASFSINO)
/*
 * If the entry has an inode number that matches the inode number of the
 * file system mount point, return an empty path reply.  That tells the
 * caller to print the file system mount point name only.
 */
	if (Lf->inp_ty == 1 && Lf->fs_ino && Lf->inode == Lf->fs_ino)
	    return(cp);
# endif	/* defined(HASFSINO) */

/*
 * Look up the name cache entry for the node address.
 */
	if (Nc == 0 || (lc = ncache_addr(Lf->id, Lf->na)) == NULL)
	{

	/*
	 * If the node has no cache entry, see if it's the mount
	 * point of a known file system.
	 */
	    if (!Lf->fsdir || !Lf->dev_def || Lf->inp_ty != 1)
		return(NULL);
	    for (mtp = Mtab; mtp; mtp = mtp->next) {
		if (!mtp->dir || !mtp->inode)
		    continue;
		if (Lf->dev == mtp->dev
		&&  (unsigned long)mtp->inode == Lf->inode
		&&  strcmp(mtp->dir, Lf->fsdir) == 0)
		    return(cp);
	    }
	    return(NULL);
	}
/*
 * Start the path assembly.
 */
	if ((nl = lc->nl) > (blen - 1))
	    return(NULL);
	cp = buf + blen - nl - 1;
	rlen = blen - nl - 1;
	(void) strcpy(cp, lc->nm);

# if	defined(NCACHE_PARADDR) && defined(NCACHE_PARID)
/*
 * Look up the name cache entries that are parents of the node address.
 * Quit when:
 *
 *	there's no parent;
 *	the name length is too large to fit in the receiving buffer.
 */
	for (;;) {
	    if (!lc->pla) {

#  if	!defined(NCACHE_NO_ROOT)
		if (ncache_isroot(lc->pa, cp))
		    *fp = 1;
#  endif	/* !defined(NCACHE_NO_ROOT) */

		break;
	    }
	    lc = lc->pla;
	    if (((nl = lc->nl) + 1) > rlen)
		break;
	    *(cp - 1) = '/';
	    cp--;
	    rlen--;
	    (void) strncpy((cp - nl), lc->nm, nl);
	    cp -= nl;
	    rlen -= nl;
	}
# endif	/* defined(NCACHE_PARADDR) && defined(NCACHE_PARID) */
	return(cp);
}
#endif	/* defined(HASNCACHE) */
