#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <gtk/gtk.h>

#include "../include/string.h"

#include "cdialog.h"
#include "obj.h"
#include "win.h"
#include "winlist.h"
#include "winresultsfio.h"
#include "core.h"
#include "config.h"

/* Utilities */
static gchar *CLIST_GET_CELL_TEXT(GtkCList *clist, gint row, gint column);

void WinResultsOpen(win_struct *win, const gchar *path);
void WinResultsSave(win_struct *win, const gchar *path);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


static gchar *CLIST_GET_CELL_TEXT(GtkCList *clist, gint row, gint column)
{
	gchar *text = NULL;
	guint8 spacing;
	GdkPixmap *pixmap;
	GdkBitmap *mask;

	/* Handle by cell type */
	switch(gtk_clist_get_cell_type(clist, row, column))
	{
	  case GTK_CELL_TEXT:
	    gtk_clist_get_text(
		clist, row, column, &text
	    );
	    break;
	  case GTK_CELL_PIXTEXT:
	    gtk_clist_get_pixtext(
		clist, row, column, &text,
		&spacing, &pixmap, &mask
	    );
	    break;
	  case GTK_CELL_PIXMAP:
	  case GTK_CELL_WIDGET:
	  case GTK_CELL_EMPTY:
	    break;
	}

	return(text);
}

/*
 *	Opens the results from the specified file to the Win's
 *	Results List.
 */
void WinResultsOpen(win_struct *win, const gchar *path)
{
	gchar buf[1024];
	FILE *fp;
	GtkWidget *toplevel;
	GtkCList *clist;
	core_struct *core;

	if((win == NULL) || STRISEMPTY(path))
	    return;

	toplevel = win->toplevel;
	clist = GTK_CLIST(win->results_clist);
	core = win->core;

	/* Open the results file for reading */
	fp = fopen(path, "rb");
	if(fp == NULL)
	{
	    const gint error_code = (gint)errno;
	    gchar *s, *error_msg = STRDUP(g_strerror(error_code));
	    if(error_msg != NULL)
	    {
		*error_msg = toupper(*error_msg);
		s = g_strdup_printf(
		    "%s:\n\n    %s",
		    error_msg,
		    path
		);
#ifdef HAVE_EDV2
                EDVPlaySoundWarning(core->edv_ctx);
#endif
                CDialogSetTransientFor(toplevel);
                CDialogGetResponse(
                    "Open Results Failed",
		    s,
                    NULL,   
                    CDIALOG_ICON_WARNING,
                    CDIALOG_BTNFLAG_OK,
                    CDIALOG_BTNFLAG_OK
                );
                CDialogSetTransientFor(NULL);
		g_free(s);
		g_free(error_msg);
	    }
	    return;
	}

	gtk_clist_freeze(clist);

	/* Clear the results list */
	WinResultsListClear(win);

	/* Begin reading the results file */
	while(fgets(buf, sizeof(buf), fp) != NULL)
	{
	    gint len;
	    gchar *s, *s2, *path = NULL, *virus_name = NULL;

	    buf[sizeof(buf) - 1] = '\0';

	    /* Remove newline characters */
	    s = strpbrk(buf, "\n\r");
	    if(s != NULL)
		*s = '\0';

	    /* Set s to the start of line's contents and seek past
	     * initial spaces
	     */
	    s = buf;
	    while(ISBLANK(*s))
		s++;

	    /* Is a comment or empty line? */
	    if((*s == CFG_COMMENT_CHAR) || (*s == '\0'))
		continue;

	    /* Path and virus|problem format? */
	    if(*s == '"')
	    {
		s++;		/* Seek past initial '"' character */

		/* Seek s2 to next '"' character */
		for(s2 = s; *s2 != '\0'; s2++)
		{
		    if(*s2 == '"')
			break;
		}
		/* Calculate length and copy path */
		len = (gint)(s2 - s);
		path = (gchar *)g_malloc((len + 1) * sizeof(gchar));
		if(path != NULL)
		{
		    memcpy(path, s, len * sizeof(gchar));
		    path[len] = '\0';
		}

		/* Seek s to start of virus|problem */
		s = (*s2 == '"') ? (s2 + 1) : s2; 
		while(ISBLANK(*s))
		    s++;
		virus_name = STRDUP(s);
	    }
	    else
	    {
		virus_name = STRDUP(s);
	    }

	    /* Append this row to the results list */
	    WinResultsListAppend(win, path, virus_name);
		 
	    g_free(path);
	    g_free(virus_name);
	}

	gtk_clist_thaw(clist);

	/* Close results file */
	fclose(fp);
}

/*
 *	Saves the results from the Win's Results List to the
 *	specified file.
 */
void WinResultsSave(win_struct *win, const gchar *path)
{
	gint i;
	FILE *fp;
	gchar *virus_name;
	const obj_struct *obj;
	GtkWidget *toplevel;
	GtkCList *clist;
	core_struct *core;

	if((win == NULL) || STRISEMPTY(path))
	    return;

	toplevel = win->toplevel;
	clist = GTK_CLIST(win->results_clist);
	core = win->core;

	/* Open the results file for writing */
	fp = fopen(path, "wb");
	if(fp == NULL)
	{
	    const gint error_code = (gint)errno;
	    gchar *s, *error_msg = STRDUP(g_strerror(error_code));
	    if(error_msg != NULL)
	    {
		*error_msg = toupper(*error_msg);
		s = g_strdup_printf(
		    "%s:\n\n    %s",
		    error_msg,
		    path
		);
#ifdef HAVE_EDV2
                EDVPlaySoundWarning(core->edv_ctx);
#endif
                CDialogSetTransientFor(toplevel);
                CDialogGetResponse(
                    "Save Results Failed",
		    s,
                    NULL,   
                    CDIALOG_ICON_WARNING,
                    CDIALOG_BTNFLAG_OK,
                    CDIALOG_BTNFLAG_OK
                );
                CDialogSetTransientFor(NULL);
		g_free(s);
		g_free(error_msg);
	    }
	    return;
	}

	/* Begin writing the results */
	for(i = 0; i < clist->rows; i++)
	{
	    obj = OBJ(gtk_clist_get_row_data(clist, i));
	    if(obj == NULL)
		continue;

	    /* Get virus|problem from the cell (it is not specified
	     * in the Object
	     */
	    virus_name = CLIST_GET_CELL_TEXT(clist, i, 1);

	    if(STRISEMPTY(obj->path))
	    {
		if(!STRISEMPTY(virus_name))
		    fprintf(fp, "%s\n", virus_name);
	    }
	    else
	    {
		fprintf(fp, "\"%s\"", obj->path);
		if(!STRISEMPTY(virus_name))
		    fprintf(fp, " %s\n", virus_name);
	    }
	}

	/* Close results file */
	fclose(fp);
}
