/* -*- Mode: C++; c-file-style: "stroustrup"; indent-tabs-mode: nil -*- */
/*
 * Dns.cc
 *
 * $Id: DnsBind.cc,v 1.6 2001/01/21 22:37:48 rex Exp $
 *
 * Copyright (c) 2000-2001 Remi Lefebvre <remi@dhis.net>
 * Copyright (c) 2000 Luca Filipozzi <lfilipoz@dhis.net>
 *
 * DDT comes with ABSOLUTELY NO WARRANTY and is licenced under the
 * GNU General Public License (version 2 or later). This license
 * can be retrieved from http://www.gnu.org/copyleft/gnu.html.
 */

#include "Logger.h"
#include "DnsBind.h"

DnsBind::DnsBind(Logger *log)
{
    this->log = log;

    statp = new struct __res_state;

    if (statp == NULL)
    {
        throw DdtException("Can't allocate memory.");
    }

    if (res_ninit(statp) == -1)
    {
        throw DdtException("Resolver failed to initialize.");
    }

    // not initiliased!?? pfft
    listupdrec.head = NULL;
    listupdrec.tail = NULL;
}

DnsBind::~DnsBind()
{
// do we need that ?
    res_freeupdrec(rrecp_in);
    res_nclose(statp);
    delete statp;
}

// FIXME: use PREREQS
void DnsBind::addDnsRecord(int id, const char *dname, DnsRecordType type,
                           const char *data)
{
    ns_type bind_type;

    /* we need to translate the DnsRecordType into a ns_type */
    if (type == A) bind_type = T_A;
    else if (type == CNAME) bind_type = T_CNAME;
    else if (type == MX) bind_type = T_MX;
    else if (type == NS) bind_type = T_NS;
    else if (type == ANY) bind_type = T_ANY;
    else { log->debug("unsupported dns type"); return; }

    if (updateRecord(S_UPDATE, dname, data, ADD, bind_type) == -1)
    {
        log->debug("Failed to add record.");
        return;
    }

    if (flush() == -1)
    {
        log->debug("Failed to flush list.");
        return;
    }
}

void DnsBind::delDnsRecords(char *dname)
{
    if (updateRecord(S_UPDATE, dname, "", DELETE, T_ANY) == -1)
    {
        log->debug("Failed to remove record");
        return;
    }

    if (flush() == -1)
    {
        log->debug("Failed to flush list.");
        return;
    }
}

void DnsBind::delDnsRecord(int id, const char *dname, DnsRecordType type,
                           const char *data)
{
    ns_type bind_type;

    /* we need to translate the DnsRecordType into a ns_type */
    if (type == A) bind_type = T_A;
    else if (type == CNAME) bind_type = T_CNAME;
    else if (type == MX) bind_type = T_MX;
    else if (type == NS) bind_type = T_NS;
    else if (type == ANY) bind_type = T_ANY;
    else { log->debug("unsupported dns type"); return; }

    // remove the record
    if (updateRecord(S_UPDATE, dname, data, DELETE, bind_type) == -1)
    {
        log->debug("Failed to remove record");
        return;
    }

    if (flush() == -1)
    {
        log->debug("Failed to flush list.");
        return;
    }
}

// update a specific record
void DnsBind::updateDnsRecord(int id, const char *dname, DnsRecordType type,
                              const char *data)
{
    ns_type bind_type;

    /* we need to translate the DnsRecordType into a ns_type */
    if (type == A) bind_type = T_A;
    else if (type == CNAME) bind_type = T_CNAME;
    else if (type == MX) bind_type = T_MX;
    else if (type == NS) bind_type = T_NS;
    else if (type == ANY) bind_type = T_ANY;
    else { log->debug("unsupported dns type"); return; }

    if (updateRecord(S_UPDATE, dname, "", DELETE, bind_type) == -1)
    {
        log->debug("Failed to remove record");
        return;
    }

    if (updateRecord(S_UPDATE, dname, data, ADD, bind_type) == -1)
    {
        log->debug("Failed to add record.");
        return;
    }

    if (flush() == -1)
    {
        log->debug("Failed to flush list.");
        return;
    }
}

/// Add calls in a list
int DnsBind::updateRecord(ns_sect r_section, const char *r_dname, 
                          const char *r_data, int r_opcode, 
                          ns_type r_type)
{
    unsigned int r_size;    // size of r_data
    unsigned int r_ttl;     // time to live
    ns_class r_class;

    if (strlen(r_dname) >= MAX_FQDN_LENGTH)
    {
        log->debug("DnsBind::updateRecord(): r_dname too large!");
        return -1;
    }

    r_size = strlen(r_data);
    if (r_size >= MAX_FQDN_LENGTH)
    {
        log->debug("DnsBind::updateRecord(): r_data too large!");
        return -1;
    }

    // ensure r_opcode is one of ADD/DELETE and set r_ttl accordingly
    if (r_opcode == ADD)
    {
        r_ttl = TTL;
        r_class = C_IN;
    }
    else if (r_opcode == DELETE)
    {
        r_ttl = 0;              // must set ttl to 0 to cause deletion
        r_class = C_IN;
    }
    else if (r_opcode == NXRRSET || r_opcode == YXRRSET)
    {
        r_ttl = 0;
        r_class = C_NONE;
    }
    else
    {
        log->debug ("DnsBind::updateRecord(): r_opcode invalid!");
        return -1;
    }

    // fill in ns_updrec structure
    rrecp_in = res_mkupdrec(r_section, strdup(r_dname), 
                            r_class, r_type, r_ttl);

    if (rrecp_in == NULL)
    {
        log->debug ("DnsBind::updateRecord(): res_mkupdrec() failed!");
        return -1;
    }

    if (r_size > 0)
    {
        if ((rrecp_in->r_data = (u_char *)malloc(r_size)) == NULL)
        {
            res_freeupdrec(rrecp_in);
            return -1;
        }
    }

    rrecp_in->r_opcode = r_opcode;
    rrecp_in->r_size = r_size;
    if (r_size > 0)
        strncpy((char *)rrecp_in->r_data, r_data, r_size);
    
    APPEND(listupdrec, rrecp_in, r_link);

    return 0;
}

/// Flush the list of calls
int DnsBind::flush()
{
    if (!EMPTY(listupdrec))
    {
        if (res_nupdate(statp, HEAD(listupdrec), NULL) < 0)
        {
            log->debug("DnsBind::flush(): res_nupdate() failed!");
        }

        while (!EMPTY(listupdrec))
        {
            ns_updrec *tmp = HEAD(listupdrec);

            UNLINK(listupdrec, tmp, r_link);

            if (tmp->r_size != 0)
                free((char *)tmp->r_data);

            res_freeupdrec(tmp);
        }
    }

    return 0;
}
