///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// geo mmg3d inputs / outputs
//
// author:
//      Pierre.Saramito@imag.fr
//
// date: 23 nov 2009
//

// ============================================================================
//  includes
// ============================================================================

#include "rheolef/georep.h"
#include "rheolef/iorheo.h"
#include "rheolef/rheostream.h" // i/o utility
#include "rheolef/field.h"
#include "rheolef/form.h"
#include "rheolef/tiny_element.h"
using namespace std;
namespace rheolef { 


// ============================================================================
// mmg3d input (3d)
// ============================================================================

istream&
get_nodes_mmg3d (
	istream& s, 
    	georep::iterator_node iter_p, 
    	georep::iterator_node last_p, 
    	point& xmin, 
	point& xmax,
        vector<domain>& domains,
        vector<size_t>& vertice_domain_mmg3d_number)
{
  typedef georep::size_type size_type;
  xmin[0] = xmin[1] = xmin[2] =  numeric_limits<Float>::max();
  xmax[0] = xmax[1] = xmax[2] = -numeric_limits<Float>::max();
  map<size_type, vector<tiny_element> > domain_map;
  size_t vertice_number = 0;
  tiny_element vertice;
  vertice.set_name('p');
  for (; iter_p != last_p; iter_p++, vertice_number++) {
        georep::nodelist_type::reference p = (*iter_p);
	p.get(s, 3);
	// get vertice domain number and build 0d-domains
        size_t vertice_domain_number; 
        s >> vertice_domain_number; 
        if (vertice_domain_number != 0) {
          // skip zero vertice domain number, that is reserved for unmarked vertices
          vertice[0] = vertice_number;
          domain_map[vertice_domain_number].push_back (vertice);
	}
        for (size_t j = 0; j < 3; j++) {
          xmin[j] = std::min(p[j], xmin[j]);
	  xmax[j] = std::max(p[j], xmax[j]);
	}
  }
  // copy domain_map into domains (table)
  vertice_domain_mmg3d_number.resize (domain_map.size()); 
  domains.resize (domain_map.size());
  vector<domain>::iterator iter_domain = domains.begin();
  size_type i_dom = 0;
  for (map<size_type, vector<tiny_element> >::const_iterator iter_domain_map = domain_map.begin();
      iter_domain_map != domain_map.end(); iter_domain_map++, iter_domain++, i_dom++) {
      vertice_domain_mmg3d_number[i_dom] = (*iter_domain_map).first;
      const vector<tiny_element>& dom      = (*iter_domain_map).second;
      if (dom.size() == 0) continue;
      (*iter_domain).set(dom.begin(), dom.size(), 0, "unnamed_vertice"+itos(i_dom));
  }
  return s;
}
#ifdef TO_CLEAN
//
// get boundary edges
//
static
istream&
get_boundary_edges_mmg3d (istream& is, vector<domain>& domains,
    vector<size_t>& edge_domain_mmg3d_number)
{
  typedef georep::size_type size_type;
  size_type n_bdr_edges;
  is >> n_bdr_edges;
  map<size_t, vector<pair<size_type,size_type> > > domain_map;
  for (size_type j = 0; is.good() && j < n_bdr_edges; j++) {

    size_type r, p0, p1;
    is >> p0 >> p1 >> r;
    p0--; p1--; // shift indices start from 1 to 0 
    domain_map[r].push_back (make_pair(p0,p1));
  }
  if (!is.good()) {
      error_macro ("a problem occurs while loading a mmg3d mesh");
  }
  // copy domain_map into domains (table)
  edge_domain_mmg3d_number.resize (domain_map.size()); 
  domains.resize (domain_map.size());
  vector<domain>::iterator iter_domain = domains.begin();
  size_t i_dom = 0;
  for (map<size_t, vector<pair<size_type,size_type> > >::const_iterator iter_domain_map = domain_map.begin();
      iter_domain_map != domain_map.end(); iter_domain_map++, iter_domain++, i_dom++) {
      edge_domain_mmg3d_number[i_dom] = (*iter_domain_map).first;
      const vector<pair<size_type,size_type> >& dom = (*iter_domain_map).second;
      if (dom.size() == 0) continue;
      (*iter_domain).set(dom.begin(), dom.size(), "unnamed"+itos(i_dom));
  }
  return is;
}
#endif // TO_CLEAN
//
// get boundary triangles
//
static
istream&
get_boundary_triangles_mmg3d (istream& is, vector<domain>& domains,
    vector<size_t>& triangle_domain_mmg3d_number)
{
  typedef georep::size_type size_type;
  size_type n_bdr_triangles;
  is >> n_bdr_triangles;
warning_macro ("n_tri = " << n_bdr_triangles);
  map<size_t, vector<tiny_element> > domain_map;
  tiny_element K;
  K.set_variant (reference_element::t);
  for (size_type j = 0; is.good() && j < n_bdr_triangles; j++) {
    size_type r;
    is >> K[0] >> K[1] >> K[2] >> r;
    K[0]--; K[1]--; K[2]--; // shift indices start from 1 to 0
    domain_map[r].push_back (K);
  }
  if (!is.good()) {
      error_macro ("a problem occurs while loading a mmg3d mesh");
  }
  // copy domain_map into domains (table)
  triangle_domain_mmg3d_number.resize (domain_map.size()); 
  domains.resize (domain_map.size());
  vector<domain>::iterator iter_domain = domains.begin();
  size_t i_dom = 0;
  for (map<size_t, vector<tiny_element> >::const_iterator iter_domain_map = domain_map.begin();
      iter_domain_map != domain_map.end(); iter_domain_map++, iter_domain++, i_dom++) {
      triangle_domain_mmg3d_number[i_dom] = (*iter_domain_map).first;
      const vector<tiny_element>& dom = (*iter_domain_map).second;
      if (dom.size() == 0) continue;
      (*iter_domain).set(dom.begin(), dom.size(), 2, "unnamed"+itos(i_dom));
  }
  return is;
}
istream&
georep::get_mmg3d (istream& s, bool check_optional_domain_name) 
{
  // mmg3d has only boundary edges that are numbered
  // and have domain numbers 
  // => this is only a version 1 mesh file format type
  //    not consistent for P2 elements
  _version = 1;

  typedef georep::size_type size_type;
  if (!s) error_macro("bad input stream for geo.");
  bool verbose = iorheo::getverbose(s);
  _dim = 3;
  //
  // get coordinates
  //
  string label;
  size_type n_vert = numeric_limits<size_type>::max();
  size_type n_elt  = numeric_limits<size_type>::max();
  _x.resize(0);
  vector<domain> vertice_domain;
  vector<domain> edge_domain;
  vector<domain> triangle_domain;
  vector<size_t> vertice_domain_mmg3d_number;
  vector<size_t> edge_domain_mmg3d_number;
  vector<size_t> triangle_domain_mmg3d_number;
  resize(0);

  while (s.good() && label != "End") {

      s >> label;

      if (label == "MeshVersionFormatted") {
warning_macro("MeshVersionFormatted...");
	size_t v;
	s >> v;
      } else if (label == "Dimension") {
warning_macro("Dimension...");
	size_t d;
	s >> d;
	check_macro (d == 3, "unsupported mmg3d dimension: " << d);
      } else if (label == "Vertices") {
warning_macro("Vertices...");

  	 s >> n_vert;
         _x.resize(n_vert);
         get_nodes_mmg3d (s, begin_node(), end_node(), _xmin, _xmax, vertice_domain, vertice_domain_mmg3d_number);
         _count_geo [0] = _count_element [0] = n_vert;
         if (!s.good()) return s;
#ifdef TODO
      } else if (label == "Edges") {
         //
         // get boundary edges (domains)
         //
  	 get_boundary_edges_mmg3d (s, edge_domain, edge_domain_mmg3d_number);
#endif // TODO
      } else if (label == "Triangles") {
warning_macro("Triangles...");
         //
         // get boundary edges (domains)
         //
  	 get_boundary_triangles_mmg3d (s, triangle_domain, triangle_domain_mmg3d_number);
      } else if (label == "Tetrahedra") {
warning_macro("Tetrahedra...");
  	 //
  	 // get elements
  	 //
  	 s >> n_elt;
  	 size_type old_size = size();
  	 resize(old_size + n_elt); // make physical copies of elements if triangles + quad
	 size_type n_vert_per_elt = 4;

  	 size_type K_idx = 0;
         georep::iterator iter_elt = begin() + old_size;
  	 georep::iterator last_elt = end();
	 for (; s.good() && iter_elt != last_elt; iter_elt++, K_idx++) {

      	    geo_element& K = (*iter_elt);
      	    K.set_variant (n_vert_per_elt, _dim);
      	    K.set_index(K_idx) ;
      	    for (size_type i = 0 ; i < n_vert_per_elt; i++) {
		s >> K[i];
        	K[i]--;
		if (n_vert != numeric_limits<size_type>::max() && K[i] >= n_vert) {
	    	    error_macro("mmg3d input: element " << K_idx+1 << ": "
		        << i+1 << "-th vertice index "
              	        << K[i] << " out of range 0:" << n_vert-1);
		}
      	    }
            // TODO: get region number and build 2d-domains
            size_type dummy; s >> dummy;
           _count_element [K.variant()]++;
      	   _count_geo     [_dim]++;
	 }
      } else if (label == "End") {
	break;
      } else {
warning_macro("label = \"" << label << "\" skipped...");
      }
  }
  // -------------------------------------------------------------------
  // reduce vertice domains:
  // in mmg3d files, boundary vertices have by default the domain
  // number of the 2d triangle domain number.
  // we delete it and conserve just vertice domain that have specific number
  // -------------------------------------------------------------------
  size_t reduced_vertice_domain_size = vertice_domain_mmg3d_number.size();
  for (size_t i = 0; i < vertice_domain_mmg3d_number.size(); i++) {
    bool founded_as_triangle_domain = false;
    // heavy search, not optimal since index are sorted
    // but domain table size are small in general
    for (size_t j = 0; ! founded_as_triangle_domain && j < triangle_domain_mmg3d_number.size(); j++) {
      if (triangle_domain_mmg3d_number[j] == vertice_domain_mmg3d_number[i]) {
        founded_as_triangle_domain = true;
      }
    }
    if (founded_as_triangle_domain) {
      vertice_domain_mmg3d_number[i] = numeric_limits<size_type>::max();
      reduced_vertice_domain_size--;
    }
  }
  vector<domain> reduced_vertice_domain (reduced_vertice_domain_size);
  vector<size_type> reduced_vertice_domain_mmg3d_number (reduced_vertice_domain_size);
  size_t i_reduced = 0;
  for (size_t i = 0; i < vertice_domain_mmg3d_number.size(); i++) {
    if (vertice_domain_mmg3d_number[i] != numeric_limits<size_type>::max()) {
      reduced_vertice_domain [i_reduced] = vertice_domain[i];
      reduced_vertice_domain_mmg3d_number [i_reduced] = vertice_domain_mmg3d_number[i];
      i_reduced++;
    }
  }
  // ---------------------------------------------------------------
  // check extension to optional domain names
  // ---------------------------------------------------------------
  if (!check_optional_domain_name) return s;
warning_macro("check domain names...");
  char c;
  s >> ws >> c; // skip white and grab a char
  // have "FaceDomainNames", "EdgeDomainNames" or "VerticeDomainNames" ?
  // mmg3d mesh may be followed by field data and such, so be carrefull...
  while (c == 'E' || c == 'F' || c == 'V') {
      s.unget(); // put char back
      if (c == 'F') {
        if (!scatch(s,"FaceDomainNames")) return s;
warning_macro("FaceDomainNames...");
        // ---------------------------------------------------------------
        // get triangle domain names
        // ---------------------------------------------------------------
        size_type n_dom_triangle;
        s >> n_dom_triangle;
        if (n_dom_triangle != triangle_domain.size()) {
           string numbers;
           for (size_t j = 0; j < triangle_domain_mmg3d_number.size(); j++) {
              numbers = numbers + itos(triangle_domain_mmg3d_number[j]) + " ";
           }
           warning_macro("geo input: "
	    << n_dom_triangle << " domain names founded while "
            << triangle_domain.size() << " mmg3d triangle domains numbers are defined: "
            << numbers);
           cerr << endl;
           error_macro("HINT: check domain name file (.dmn)");
        }
        size_t start_triangle_domain = _domlist.size();
        _domlist.resize(_domlist.size() + n_dom_triangle);
        for (size_type k = 0; k < n_dom_triangle; k++) {
          string name;
          s >> name;
          triangle_domain[k].set_name(name);
          _domlist.at(start_triangle_domain+k) = triangle_domain[k];
        }
      } else if (c == 'E') {
        if (!scatch(s,"EdgeDomainNames")) return s;
        // ---------------------------------------------------------------
        // get edge domain names
        // ---------------------------------------------------------------
        size_type n_dom_edge;
        s >> n_dom_edge;
        if (n_dom_edge != edge_domain.size()) {
           string numbers;
           for (size_t j = 0; j < edge_domain_mmg3d_number.size(); j++) {
              numbers = numbers + itos(edge_domain_mmg3d_number[j]) + " ";
           }
           warning_macro("geo input: "
	    << n_dom_edge << " domain names founded while "
            << edge_domain.size() << " mmg3d edge domains numbers are defined: "
            << numbers);
           cerr << endl;
           error_macro("HINT: check domain name file (.dmn)");
        }
        size_t start_edge_domain = _domlist.size();
        _domlist.resize(_domlist.size() + n_dom_edge);
        for (size_type k = 0; k < n_dom_edge; k++) {
          string name;
          s >> name;
          edge_domain[k].set_name(name);
          _domlist.at(start_edge_domain+k) = edge_domain[k];
        }
      } else if (c == 'V') {
        if (!scatch(s,"VerticeDomainNames")) return s;
        // ---------------------------------------------------------------
        // get vertice domain names
        // ---------------------------------------------------------------
        size_type n_dom_vertice;
        s >> n_dom_vertice;
        if (n_dom_vertice != reduced_vertice_domain.size()) {
           string numbers;
           for (size_t j = 0; j < reduced_vertice_domain_mmg3d_number.size(); j++) {
              numbers = numbers + itos(reduced_vertice_domain_mmg3d_number[j]) + " ";
           }
           warning_macro("geo input: "
	    << n_dom_vertice << " domain names founded while "
            << reduced_vertice_domain.size() << " mmg3d vertice domains numbers are defined: "
            << numbers);
           cerr << endl;
           error_macro("HINT: check domain name file (.dmn)");
        }
        size_t start_vertice_domain = _domlist.size();
        _domlist.resize(_domlist.size() + n_dom_vertice);
        for (size_type k = 0; k < n_dom_vertice; k++) {
          string name;
          s >> name;
          reduced_vertice_domain[k].set_name(name);
          _domlist.at(start_vertice_domain+k) = reduced_vertice_domain[k];
        }
      }
      s >> ws >> c; // skip white and grab a char
  }
  // ---------------------------------------------------------------
  // check extension to optional domain names
  // ---------------------------------------------------------------
  return s;
}
// =======================================================================
// mmg3d output
// =======================================================================

typedef georep::size_type size_type;
typedef map <size_type,  size_type > vertex_bdr_type;
//          (i_vertex, i_bdr_vertex)

static 
void
insert_vertex_bdr (
    vertex_bdr_type& vertex_bdr, 
    size_type i_vertex)
{
    vertex_bdr_type::iterator p = vertex_bdr.find(i_vertex);
    if (p == vertex_bdr.end()) {
	size_type i_bdr_vertex = vertex_bdr.size();
	vertex_bdr.insert(make_pair(i_vertex,numeric_limits<size_type>::max()));
    }
}
ostream& 
georep::put_mmg3d (ostream& os, bool only_boundary) const
{
  bool verbose = iorheo::getverbose(os);
  if (dimension() != 3) {
      error_macro ("mmg3d output: " << dimension() << "D geometries not supported.");
  }
  //
  // header
  //
  os << "MeshVersionFormatted" << endl << "1" << endl << endl
     << "Dimension" << endl << "3" << endl << endl
     << "Vertices" << endl
     << n_node() << endl;
  //
  // vertices
  //
  for (const_iterator_node iter_p = begin_node(); iter_p != end_node(); iter_p++) {
      const point& p = *iter_p;
      p.put (os, 3);
      os << " 0" << endl ;
  }
  os << endl;
  //
  // boundary triangles
  //
  size_type n_bdr_triangle = 0;
  domlist_type::const_iterator last = _domlist.end();
  for (domlist_type::const_iterator iter = _domlist.begin(); iter != last; iter++) {
    if ((*iter).dimension() == dimension() - 1) n_bdr_triangle += (*iter).size();
  }
  bool is_boundary_mesh = false;
  if (n_bdr_triangle == 0 && n_tetrahedra() == 0) {
    // boundary mesh: no domains, all triangle in the main element list
    n_bdr_triangle = n_triangle();
    is_boundary_mesh = true;
  }
  os << "Triangles" << endl
     << n_bdr_triangle << endl;
  if (!is_boundary_mesh) {
    size_type i_domain = 0;
    vertex_bdr_type vertex_bdr;
    for (domlist_type::const_iterator iter = _domlist.begin(); iter != last; iter++, i_domain++) {
      const domain& d = *iter;
      if (d.dimension() != dimension() -1) continue;
      for (domain::const_iterator iter_s = d.begin(); iter_s != d.end(); iter_s++) {
          const geo_element& S = *iter_s;
	  check_macro (S.size() == 3, "unsupported 2D quadrilateral face");
          os << S[0]+1 << " " << S[1]+1 << " " << S[2]+1 << " " << i_domain+1 << endl;
      }
    }
  } else {
    // output all triangles in a domain number "1"
    for (const_iterator iter_k = begin(); iter_k != end(); iter_k++) {
      const geo_element& K = *iter_k;
      if (K.size() != 3) continue;
      os << K[0]+1 << " " << K[1]+1 << " " << K[2]+1 << " 1" << endl;
    }
  }
  //
  // elements
  //
  if (!only_boundary && n_tetrahedra() != 0) {
    os << endl << "Tetrahedra" << endl << n_tetrahedra() << endl;
    for (const_iterator iter_k = begin(); iter_k != end(); iter_k++) {
      const geo_element& K = *iter_k;
      if (K.size() != 4) continue;
      os << K[0]+1 << " " << K[1]+1 << " " << K[2]+1 << " " << K[3]+1 << " 0" << endl;
    }
  }
  os << endl << "End" << endl;
  return os;
}
}// namespace rheolef
