// Copyright 2018 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ash/system/bluetooth/bluetooth_feature_pod_controller.h"

#include "ash/resources/vector_icons/vector_icons.h"
#include "ash/session/session_controller.h"
#include "ash/shell.h"
#include "ash/strings/grit/ash_strings.h"
#include "ash/system/bluetooth/tray_bluetooth_helper.h"
#include "ash/system/tray/system_tray_notifier.h"
#include "ash/system/unified/feature_pod_button.h"
#include "ash/system/unified/unified_system_tray_controller.h"
#include "base/i18n/number_formatting.h"
#include "ui/base/l10n/l10n_util.h"

namespace ash {

BluetoothFeaturePodController::BluetoothFeaturePodController(
    UnifiedSystemTrayController* tray_controller)
    : tray_controller_(tray_controller) {
  Shell::Get()->system_tray_notifier()->AddBluetoothObserver(this);
}

BluetoothFeaturePodController::~BluetoothFeaturePodController() {
  Shell::Get()->system_tray_notifier()->RemoveBluetoothObserver(this);
}

FeaturePodButton* BluetoothFeaturePodController::CreateButton() {
  DCHECK(!button_);
  button_ = new FeaturePodButton(this);
  button_->ShowDetailedViewArrow();
  UpdateButton();
  return button_;
}

void BluetoothFeaturePodController::OnIconPressed() {
  bool was_enabled = button_->IsToggled();
  Shell::Get()->tray_bluetooth_helper()->SetBluetoothEnabled(!was_enabled);

  // If Bluetooth was disabled, show device list as well as enabling Bluetooth.
  if (!was_enabled)
    tray_controller_->ShowBluetoothDetailedView();
}

void BluetoothFeaturePodController::OnLabelPressed() {
  Shell::Get()->tray_bluetooth_helper()->SetBluetoothEnabled(true);
  tray_controller_->ShowBluetoothDetailedView();
}

SystemTrayItemUmaType BluetoothFeaturePodController::GetUmaType() const {
  return SystemTrayItemUmaType::UMA_BLUETOOTH;
}

void BluetoothFeaturePodController::UpdateButton() {
  bool is_available =
      Shell::Get()->tray_bluetooth_helper()->GetBluetoothAvailable();
  button_->SetVisible(is_available);
  if (!is_available)
    return;

  // Bluetooth power setting is always mutable in login screen before any
  // user logs in. The changes will affect local state preferences.
  //
  // Otherwise, the bluetooth setting should be mutable only if:
  // * the active user is the primary user, and
  // * the session is not in lock screen
  // The changes will affect the primary user's preferences.
  SessionController* session_controller = Shell::Get()->session_controller();
  button_->SetEnabled(!session_controller->IsActiveUserSessionStarted() ||
                      (session_controller->IsUserPrimary() &&
                       !session_controller->IsScreenLocked()));

  bool is_enabled =
      Shell::Get()->tray_bluetooth_helper()->GetBluetoothEnabled();
  button_->SetToggled(is_enabled);

  if (!is_enabled) {
    button_->SetVectorIcon(kUnifiedMenuBluetoothIcon);
    button_->SetLabel(l10n_util::GetStringUTF16(IDS_ASH_STATUS_TRAY_BLUETOOTH));
    button_->SetSubLabel(l10n_util::GetStringUTF16(
        IDS_ASH_STATUS_TRAY_BLUETOOTH_DISABLED_SHORT));
    SetTooltipState(l10n_util::GetStringUTF16(
        IDS_ASH_STATUS_TRAY_BLUETOOTH_DISABLED_TOOLTIP));
    return;
  }

  BluetoothDeviceList connected_devices;
  for (const auto& device :
       Shell::Get()->tray_bluetooth_helper()->GetAvailableBluetoothDevices()) {
    if (device.connected)
      connected_devices.push_back(device);
  }

  if (connected_devices.size() > 1) {
    const size_t device_count = connected_devices.size();
    button_->SetVectorIcon(kUnifiedMenuBluetoothConnectedIcon);
    button_->SetLabel(l10n_util::GetStringUTF16(
        IDS_ASH_STATUS_TRAY_BLUETOOTH_MULTIPLE_DEVICES_CONNECTED_LABEL));
    button_->SetSubLabel(base::FormatNumber(device_count));
    SetTooltipState(l10n_util::GetPluralStringFUTF16(
        IDS_ASH_STATUS_TRAY_BLUETOOTH_MULTIPLE_DEVICES_CONNECTED_TOOLTIP,
        device_count));
  } else if (connected_devices.size() == 1) {
    const base::string16& device_name = connected_devices.back().display_name;
    button_->SetVectorIcon(kUnifiedMenuBluetoothConnectedIcon);
    button_->SetLabel(device_name);
    button_->SetSubLabel(l10n_util::GetStringUTF16(
        IDS_ASH_STATUS_TRAY_BLUETOOTH_DEVICE_CONNECTED_LABEL));
    SetTooltipState(l10n_util::GetStringFUTF16(
        IDS_ASH_STATUS_TRAY_BLUETOOTH_DEVICE_CONNECTED_TOOLTIP, device_name));
  } else {
    button_->SetVectorIcon(kUnifiedMenuBluetoothIcon);
    button_->SetLabel(l10n_util::GetStringUTF16(IDS_ASH_STATUS_TRAY_BLUETOOTH));
    button_->SetSubLabel(
        l10n_util::GetStringUTF16(IDS_ASH_STATUS_TRAY_BLUETOOTH_ENABLED_SHORT));
    SetTooltipState(l10n_util::GetStringUTF16(
        IDS_ASH_STATUS_TRAY_BLUETOOTH_ENABLED_TOOLTIP));
  }
}

void BluetoothFeaturePodController::SetTooltipState(
    const base::string16& tooltip_state) {
  button_->SetIconTooltip(l10n_util::GetStringFUTF16(
      IDS_ASH_STATUS_TRAY_BLUETOOTH_TOGGLE_TOOLTIP, tooltip_state));
  button_->SetLabelTooltip(l10n_util::GetStringFUTF16(
      IDS_ASH_STATUS_TRAY_BLUETOOTH_SETTINGS_TOOLTIP, tooltip_state));
}

void BluetoothFeaturePodController::OnBluetoothRefresh() {
  UpdateButton();
}

void BluetoothFeaturePodController::OnBluetoothDiscoveringChanged() {
  UpdateButton();
}

}  // namespace ash
