
#include <time.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <simclist.h>

#include "../config.h"
#include "../sshguard_log.h"
#include "../sshguard_fw.h"

#define MAXIPFWCMDLEN           60

#ifndef IPFW_RULERANGE_MIN
#define IPFW_RULERANGE_MIN      55000
#endif

#ifndef IPFW_RULERANGE_MAX
#define IPFW_RULERANGE_MAX      55050
#endif

struct addr_ruleno_s {
    char addr[ADDRLEN];
    int addrkind;
    unsigned int ruleno;
};

list_t addrrulenumbers;


void ipfwmod_logsystemretval(char *command, int returnval);

size_t ipfw_rule_meter(const void *el) { return sizeof(struct addr_ruleno_s); }
int ipfw_rule_comparator(const void *a, const void *b) {
    struct addr_ruleno_s *A = (struct addr_ruleno_s *)a;
    struct addr_ruleno_s *B = (struct addr_ruleno_s *)b;
    return !((strcmp(A->addr, B->addr) == 0) && (A->addrkind == B->addrkind));
}

int fw_init() {
    srandom(time(NULL));
    list_init(&addrrulenumbers);
    list_attributes_copy(& addrrulenumbers, ipfw_rule_meter, 1);
    list_attributes_comparator(& addrrulenumbers, ipfw_rule_comparator);
    return 0;
}

int fw_fin() {
    list_destroy(&addrrulenumbers);
    return 0;
}

int fw_block(char *addr, int addrkind, int service) {
    unsigned int ruleno;
    int ret;
    char command[MAXIPFWCMDLEN];
    struct addr_ruleno_s addendum;

    ruleno = (random() % (IPFW_RULERANGE_MAX - IPFW_RULERANGE_MIN)) + IPFW_RULERANGE_MIN;
    switch (addrkind) {
        case ADDRKIND_IPv4:
            sprintf(command, IPFW_PATH "/" "ipfw add %u drop ip from %s to me", ruleno, addr);
            break;
        case ADDRKIND_IPv6:
            sprintf(command, IPFW_PATH "/" "ip6fw add %u drop ip from %s to any", ruleno, addr);
            break;
        default:
            return FWALL_UNSUPP;
    }

    sshguard_log(LOG_DEBUG, "running: '%s'", command);

    ret = system(command);
    if (ret == 0) { /* success, save rule number */
        strcpy(addendum.addr, addr);
        addendum.ruleno = ruleno;
        addendum.addrkind = addrkind;

        list_append(&addrrulenumbers, &addendum);
    }
    ipfwmod_logsystemretval(command, ret);
    
    return ret;
}

int fw_release(char *addr, int addrkind, int service) {
    struct addr_ruleno_s data;
    char command[MAXIPFWCMDLEN];
    int pos, ret = 0;

    strcpy(data.addr, addr);
    data.addrkind = addrkind;
    if ((pos = list_find(& addrrulenumbers, &data)) < 0) {
        sshguard_log(LOG_ERR, "could not get back rule ID for address %s", addr);
        return FWALL_ERR;
    }
    data = *(struct addr_ruleno_s *)list_get_at(& addrrulenumbers, pos);

    switch (data.addrkind) {
        case ADDRKIND_IPv4:
            snprintf(command, MAXIPFWCMDLEN, IPFW_PATH "/" "ipfw delete %u", data.ruleno);
            break;
        case ADDRKIND_IPv6:
            snprintf(command, MAXIPFWCMDLEN, IPFW_PATH "/" "ip6fw delete %u", data.ruleno);
            break;
        default:
            return FWALL_UNSUPP;
    }
    sshguard_log(LOG_DEBUG, "running: '%s'", command);

    ret = system(command);
    if (ret == 0) {
        list_delete_at(&addrrulenumbers, pos);
    }
    ipfwmod_logsystemretval(command, ret);

    return ret;
}

int fw_flush(void) {
    struct addr_ruleno_s *data;
    char command[MAXIPFWCMDLEN];
    int ret;

    list_iterator_start(&addrrulenumbers);
    while (list_iterator_hasnext(&addrrulenumbers)) {
        data = (struct addr_ruleno_s *)list_iterator_next(& addrrulenumbers);
        switch (data->addrkind) {
            case ADDRKIND_IPv4:
                snprintf(command, MAXIPFWCMDLEN, IPFW_PATH "/" "ipfw delete %u", data->ruleno);
                break;
            case ADDRKIND_IPv6:
                snprintf(command, MAXIPFWCMDLEN, IPFW_PATH "/" "ip6fw delete %u", data->ruleno);
                break;
        }
        sshguard_log(LOG_DEBUG, "running: '%s'", command);
        ret = system(command);
        ipfwmod_logsystemretval(command, ret);
    }
    list_iterator_stop(& addrrulenumbers);
    
    list_clear(&addrrulenumbers);

    return 0;
}

void ipfwmod_logsystemretval(char *command, int returnval) {
    if (returnval < 0) {
        sshguard_log(LOG_ERR, "Unable to execute command \"%s\". Exited %d", command, returnval);
    } else if (returnval == 127) {
        sshguard_log(LOG_ERR, "Unable to execute the shell.");
    }
}

