/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 * 
 * ------------------------------------------------------------------
 * XMMS-Stats
 * A simple plugin for XMMS that create a XML file
 * containing the title, the length and the number
 * of time you listened the song
 *
 * Author: Renaud Galante <diodo@oute.org>
 * 
 * ******************** 
 */

/* include required header files */
#include <stdio.h>
#include <pthread.h>
#include <gtk/gtk.h>
#include "xmmsctrl.h"
#include <plugin.h>
#include "configfile.h"
#include <string.h>
#include <unistd.h>

#include "includes.h"
#include "xmlwriter.h"
#include "sqlite_functions.h"


/* Gtk stuff */
static GtkWidget *conf_dialog, *about_win;
static pthread_t tid;
static GtkWidget *xmlfile_entry, *time_entry, *time_save_entry, *delete_timer_entry;


/* Database*/
sqlite *db;

/* returned statement for a request*/
int rc;
char *zErrMsg = 0;

/* Number of time you listen this music*/
int hits =0;

/* Time you spend listening the song*/
int listen_time=0;

/* file descriptor for the xml file*/
int fd;

/* path for the xml file*/
char *xml_file="/tmp/xmms_stats.xml";

/* XML file will be generating every time_generate seconds*/
gint time_generate = 3600;

/* Before min_time_save seconds, the song won't be save in database*/
int min_time_save = 10;

/* Delete song from database when they're too old. here: 5 days*/
int delete_timer = 432000;

struct RecordStat current;

/* the structure for a GeneralPlugin in xmms */
GeneralPlugin xmmsstats =
{
	NULL,
	NULL,
	-1,
	PLUGIN_NAME " " VERSION,
	xmmsstats_init,
	xmmsstats_about,
	xmmsstats_config,
	xmmsstats_cleanup,
};

/* simply stop the running thread if running plugin gets disabled */
void xmmsstats_cleanup(void)
{
	sqlite_close(db);
	pthread_cancel (tid);
}

/* required by xmms to display info line */
GeneralPlugin *get_gplugin_info(void)
{
	return (&xmmsstats);
}

/* read config file ( just one string in it ) */
void read_config (GtkWidget *wid, gpointer data)
{
	gchar *configfile;
	ConfigFile *config;
	
	configfile = g_strconcat (g_get_home_dir(), CONFIGFILE, NULL);
	if ((config = xmms_cfg_open_file (configfile)) != NULL)
	{
		xmms_cfg_read_string (config, "xmmsstats", "xmlfile", &xml_file);
		xmms_cfg_read_int (config, "xmmsstats", "time", &time_generate);
		xmms_cfg_read_int (config, "xmmsstats", "timesave", &min_time_save);
		xmms_cfg_read_int (config, "xmmsstats", "deletetimer", &delete_timer);
	}
	g_free(configfile);
}

/* initialization of the plugin */
void xmmsstats_init(void)
{
	char *buf = g_strconcat(g_get_home_dir(), DATABASE_PATH, NULL);
	
	db = sqlite_open(buf, 600, &zErrMsg);
  if( db==0 ){
    printf("Can't open|create database: %s\n", zErrMsg);
    exit(1);
  }
	pthread_attr_t attr;
	
	read_config(NULL, NULL);
	update_db();
	pthread_attr_init(&attr);
	pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
	pthread_attr_setinheritsched(&attr, PTHREAD_EXPLICIT_SCHED);
	pthread_attr_setschedpolicy(&attr, SCHED_OTHER);
	pthread_attr_setscope(&attr, PTHREAD_SCOPE_SYSTEM);

	pthread_create(&tid, &attr, xmmsstats_thread, NULL);
	
	return;
}

/* Update a RecordStat structure*/
void update_struct(struct RecordStat* rs, int p)
{
	strncpy(rs->title, xmms_remote_get_playlist_title(xmmsstats.xmms_session, p),
					strlen(xmms_remote_get_playlist_title(xmmsstats.xmms_session, p))+1);
	rs->timelength=xmms_remote_get_playlist_time(xmmsstats.xmms_session, p);
	rs->timelistening=xmms_remote_get_output_time(xmmsstats.xmms_session);
}

/* This function add a new line in the database
 * if the song isn't founded. Otherwise it updates
 * the datas
 */

/* the main part */
void *xmmsstats_thread(void *args)
{
	gint pos; // the current position of the song
	gint p_pos; // the previous position
	
	int s_time =0; // time in seconds since loading
	
	if(xmms_remote_is_playing(xmmsstats.xmms_session))
	{
		pos = xmms_remote_get_playlist_pos (xmmsstats.xmms_session);
		p_pos = pos;
		update_struct(&current,pos);
	}
	
	while(1)
	{
		if(xmms_remote_is_playing(xmmsstats.xmms_session))
		{
			pos = xmms_remote_get_playlist_pos (xmmsstats.xmms_session);
				
			if(pos!=p_pos)
			{
					if (current.timelistening !=0)
						insert_data(current);
					p_pos=pos;
			}
			s_time++;
			if (time_generate > 0)
					if(time_generate <= s_time)
					{
						create_xml_file(xml_file);
						s_time=0;
					}
			update_struct(&current,pos);
		}
		sleep(1);
	}
}

/* the about window */
void xmmsstats_about(void)
{
	GtkWidget *button, *label, *bigbox, *buttonbox;

	create_xml_file(xml_file);

	if(about_win)
		return;
	about_win = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_window_set_title(GTK_WINDOW(about_win), ("About"));
	gtk_window_set_policy(GTK_WINDOW(about_win), FALSE, FALSE, FALSE);
	gtk_window_set_position(GTK_WINDOW(about_win), GTK_WIN_POS_MOUSE);

	bigbox = gtk_vbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(about_win), bigbox);

	label = gtk_label_new(g_strconcat((gchar *)"--------------------------\n", (gchar *)"\n--------------------------\n\nXMMS-Stats plugin\n\nMake statistics of what you listen :)\n\n<diodo@oute.org>", NULL));
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_CENTER);
	gtk_container_add(GTK_CONTAINER(bigbox), label);

	buttonbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(buttonbox), GTK_BUTTONBOX_DEFAULT_STYLE);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(buttonbox), 5);
	gtk_box_pack_start(GTK_BOX(bigbox), buttonbox, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(about_win), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &about_win);
	button = gtk_button_new_with_label("Check.");
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), (gpointer) about_win);
	gtk_box_pack_start(GTK_BOX ((buttonbox)), button, FALSE, TRUE, 5);
	gtk_widget_show_all(about_win);
}

/* save config file */
void xmmsstats_config_save(GtkWidget *wid, gpointer data)
{
	gchar *configfile;
	ConfigFile *config;
	configfile = g_strconcat(g_get_home_dir(), CONFIGFILE, NULL);
	
	if ((config = xmms_cfg_open_file(configfile)) == NULL)
		config = xmms_cfg_new();

	xml_file = g_strdup(gtk_entry_get_text(GTK_ENTRY(xmlfile_entry)));
	time_generate = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(time_entry)) * 60;
	min_time_save = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(time_save_entry));
	delete_timer = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(delete_timer_entry)) * 3600 * 24;
	
	xmms_cfg_write_string (config, "xmmsstats", "xmlfile", xml_file);
	xmms_cfg_write_int (config,"xmmsstats", "time",time_generate);
	xmms_cfg_write_int (config,"xmmsstats", "timesave",min_time_save);
	xmms_cfg_write_int (config,"xmmsstats", "deletetimer",delete_timer);
	xmms_cfg_write_file (config, configfile);
	xmms_cfg_free(config);
	g_free(configfile);

	return;
}

/* the config-ok button */
void xmmsstats_config_ok(GtkWidget *wid, gpointer data)
{
	xmmsstats_config_save(NULL, NULL);
	gtk_widget_destroy(conf_dialog);
	conf_dialog = NULL;
	return;
}

/* the config window */
void xmmsstats_config(void)
{
	GtkWidget 	*ok_button, *apply_button, *cancel_button, *buttonbox,
		*bigbox, *textframe, *xmlfile_label, *time_label, *time_save_label, 
		*delete_timer_label, *ctable;
	GtkObject *delete_timer_o, *time_o, *time_save_o;
	
	if (conf_dialog)
		return;
	conf_dialog = gtk_window_new(GTK_WINDOW_TOPLEVEL);

	gtk_window_set_title(GTK_WINDOW(conf_dialog), ("XMMS-Stats setup"));
	gtk_window_set_policy(GTK_WINDOW(conf_dialog), FALSE, FALSE, FALSE);
	gtk_window_set_position(GTK_WINDOW(conf_dialog), GTK_WIN_POS_MOUSE);
	gtk_container_set_border_width(GTK_CONTAINER(conf_dialog), 5);
	gtk_signal_connect(GTK_OBJECT(conf_dialog), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &conf_dialog);
	
	bigbox = gtk_vbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER (GTK_WINDOW(conf_dialog)), bigbox);
	
	textframe = gtk_frame_new ("Config:");
	gtk_container_add(GTK_CONTAINER(bigbox), textframe);
	ctable = gtk_table_new(4, 2, FALSE);
	gtk_container_add(GTK_CONTAINER(textframe), ctable);
	
	xmlfile_label = gtk_label_new("Absolute name for the xml file");
	gtk_table_attach_defaults(GTK_TABLE(ctable), xmlfile_label, 0, 1, 0, 1 );
	xmlfile_entry = gtk_entry_new();
	if(xml_file != NULL)
		gtk_entry_set_text(GTK_ENTRY(xmlfile_entry), xml_file);
	gtk_table_attach_defaults(GTK_TABLE(ctable), xmlfile_entry, 1, 2, 0, 1);

	
	time_label = gtk_label_new("Generating time (min)");
	gtk_table_attach_defaults(GTK_TABLE(ctable), time_label, 0, 1, 1, 2 );
	
	time_o = gtk_adjustment_new( time_generate/60 , 0, 100, 1, 10, 0);
	time_entry = gtk_spin_button_new( time_o,0.3,0);
	

	gtk_table_attach_defaults(GTK_TABLE(ctable), time_entry, 1, 2, 1, 2);
	
	time_save_label = gtk_label_new("Interval (sec)(0 if you want only save a whole song)");
	gtk_table_attach_defaults(GTK_TABLE(ctable), time_save_label, 0, 1, 2, 3 );

	
	time_save_o = gtk_adjustment_new(min_time_save , 0, 100, 1, 10, 0);
	time_save_entry = gtk_spin_button_new( time_save_o,0.3,0);
	
	gtk_table_attach_defaults(GTK_TABLE(ctable), time_save_entry, 1, 2, 2, 3);

	
	delete_timer_o = gtk_adjustment_new( delete_timer/24/3600, 1, 100, 1, 10, 0);
	delete_timer_entry = gtk_spin_button_new( delete_timer_o ,0.3,0);

	delete_timer_label = gtk_label_new("Delete x day-old songs");
	gtk_table_attach_defaults(GTK_TABLE(ctable), delete_timer_label, 0, 1, 3, 4 );
	gtk_table_attach_defaults(GTK_TABLE(ctable), delete_timer_entry, 1, 2, 3, 4);
	
	buttonbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(buttonbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(buttonbox), 5);
	gtk_box_pack_start(GTK_BOX(bigbox), buttonbox, FALSE, FALSE, 0);
	ok_button = gtk_button_new_with_label("Ok");
	apply_button = gtk_button_new_with_label("Apply");
	cancel_button = gtk_button_new_with_label("Cancel");

	gtk_signal_connect_object(GTK_OBJECT(cancel_button), "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy), (gpointer) conf_dialog);
	gtk_signal_connect_object(GTK_OBJECT(apply_button), "clicked", GTK_SIGNAL_FUNC (xmmsstats_config_save), NULL);
	gtk_signal_connect_object(GTK_OBJECT(ok_button), "clicked", GTK_SIGNAL_FUNC (xmmsstats_config_ok), NULL);
	
	GTK_WIDGET_SET_FLAGS(ok_button, GTK_CAN_DEFAULT);
	GTK_WIDGET_SET_FLAGS(cancel_button, GTK_CAN_DEFAULT);
	GTK_WIDGET_SET_FLAGS(apply_button, GTK_CAN_DEFAULT);

	gtk_box_pack_start(GTK_BOX(buttonbox), ok_button, TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(buttonbox), apply_button, TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(buttonbox), cancel_button, TRUE, TRUE, 0);

	gtk_widget_show_all(conf_dialog);
}

/* end of file */
