/*
    $Id: parse.c,v 1.12 2002/05/24 05:51:59 belyi Exp $

    xkeysw - window bound/multi code keyboard switch
    Copyright (C) 1999  Dima Barsky, Igor Belyi

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <ctype.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include "parse.h"

/* Definition of syntax elements */
#define SEP_CHARS " \t\n;:=,<>"
#define COMMENT_CHAR '#'
#define STRING_CHAR '"'
#define NEWNAME_START '('
#define NEWNAME_END ')'
#define MODIFIER_SEP '+'
#define CODING_SPEC "layout"
#define XMMFILE_SPEC "xmmfile"
#define PREFIX_SPEC "prefix"
#define KEYSW_SPEC "hotkey"
#define DEFAULT_CODING "DEFAULT"
#define NULL_CODING "GLOBAL"

/* import function from init.c */
extern unsigned int modNameToMask(char* modifier);

/* external global variables */
parsedCodeRecord_t *ParsedConfFile = NULL;
parsedCodeRecord_t *DefaultRecord = NULL;
int string_index = 0;

/* local functions */
static char* getNextToken(FILE* cfd);
static char* skipSeparators(char* buff);
static char* findSeparator(char* buff);
static char* createName(char* name);
static parsedSwitchKey_t* createOneKey(parsedSwitchKey_t *currKey);
static void freeCodeRecord(parsedCodeRecord_t* currCode);
static void freeKeysRecord(parsedSwitchKey_t *currKey);

/* Reads the file content into ParsedConfFile list */
int ParseConfigFile(FILE* cfd)
{
    parsedCodeRecord_t* currCode = NULL;
    char *ptr, *ptr2;
    parsedSwitchKey_t *newKeySw;
    unsigned int mask;

    string_index = 0;
    currCode = (parsedCodeRecord_t*) calloc(1, sizeof(parsedCodeRecord_t));
    if(currCode == NULL) {
	fprintf(stderr, "Parse: Cannot create the first code record.\n");
	freeCodeRecord(currCode);
	return 1;
    }

    while((ptr = getNextToken(cfd)) != NULL) {

	if(strcmp(ptr, CODING_SPEC) == 0) {

	    /* New layout description starts. Move current one into the list */
	    if(currCode->name==NULL) DefaultRecord = currCode;

	    currCode->next = ParsedConfFile;
	    ParsedConfFile = currCode;

	    currCode=(parsedCodeRecord_t*)calloc(1,sizeof(parsedCodeRecord_t));
	    if(currCode == NULL) {
		fprintf(stderr, "Parse (line %d): Cannot create another code "
			"record.\n", string_index);
		freeCodeRecord(currCode);
		return 1;
	    }

	    if((ptr = getNextToken(cfd)) == NULL) {
		fprintf(stderr, "Parse (line %d): '%s' should be followed by"
			"name of layout", string_index, CODING_SPEC);
		freeCodeRecord(currCode);
		return 1;
	    }

	    currCode->name = createName(ptr);
	    if(currCode->name == NULL) {
		freeCodeRecord(currCode);
		return 1;
	    }

	} else if (strcmp(ptr, XMMFILE_SPEC) == 0) {

	    if(currCode->xmmfile) {
		fprintf(stderr, "Parse (line %d): '%s' can be specified only "
			"once per code record.\n", string_index, XMMFILE_SPEC);
		freeCodeRecord(currCode);
		return 1;
	    }

	    if((ptr = getNextToken(cfd)) == NULL) {
		fprintf(stderr, "Parse (line %d): '%s' should be followed by "
			"name of an xmmfile", string_index, XMMFILE_SPEC);
		freeCodeRecord(currCode);
		return 1;
	    }

	    currCode->xmmfile = createName(ptr);
	    if(currCode->xmmfile == NULL) {
		freeCodeRecord(currCode);
		return 1;
	    }

	} else if (strcmp(ptr, PREFIX_SPEC) == 0) {

	    if(currCode->prefix) {
		fprintf(stderr, "Parse (line %d): '%s' can be specified only "
			"once per code record.\n", string_index, PREFIX_SPEC);
		freeCodeRecord(currCode);
		return 1;
	    }

	    if((ptr = getNextToken(cfd)) == NULL) {
		fprintf(stderr, "Parse (line %d): '%s' should be followed by"
			"a prefix string\n", string_index, PREFIX_SPEC);
		freeCodeRecord(currCode);
		return 1;
	    }

	    currCode->prefix = createName(ptr);
	    if(currCode->prefix == NULL) {
		freeCodeRecord(currCode);
		return 1;
	    }

	} else if (strcmp(ptr, KEYSW_SPEC) == 0) {

	    if((ptr = getNextToken(cfd)) == NULL) {
		fprintf(stderr, "Parse (line %d): '%s' should be followed by"
			"keysym and layout name.\n", string_index, KEYSW_SPEC);
		freeCodeRecord(currCode);
		return 1;
	    }

	    newKeySw = (parsedSwitchKey_t*)calloc(1,sizeof(parsedSwitchKey_t));
	    if(newKeySw == NULL) {
		fprintf(stderr, "Parse (line %d): Cannot create another key "
			"record.\n", string_index);
		freeCodeRecord(currCode);
		return 1;
	    }

	    /* Retrieve optional modifiers */
	    do {
		for(ptr2=ptr; *ptr2 != MODIFIER_SEP; ptr2++)
		    if(!*ptr2) {
			ptr2 = NULL;
			break;
		    }
		if(ptr2) {
		    *ptr2++ = '\0';
		    if((mask = modNameToMask(ptr)) < 0) {
			fprintf(stderr, "Parse (line %d): Unknown modifier "
				"key '%s'.\n", string_index, ptr);
			freeKeysRecord(newKeySw);
			freeCodeRecord(currCode);			
			return 1;
		    }
		    newKeySw->modifiers |= mask;
		    ptr = ptr2;
		}
	    } while(ptr2);

	    /* Check for an optional KeySym adjustment */
	    for(ptr2=ptr; *ptr2 != NEWNAME_START; ptr2++)
		if(!*ptr2) {
		    ptr2 = NULL;
		    break;
		}
	    if(ptr2) *ptr2++ = '\0';

	    newKeySw->keyName = createName(ptr);

	    if(newKeySw->keyName == NULL) {
		freeKeysRecord(newKeySw);
		freeCodeRecord(currCode);
		return 1;
	    }
		
	    /* ptr2 is NULL if there's no optional KeySym adjustment */
	    if(ptr2) {
		for(ptr=ptr2; *ptr2 != NEWNAME_END; ptr2++)
		    if(!*ptr2) {
			fprintf(stderr, "Parse (line %d): There's no "
				"closing '%c' in a keysym substitution "
				"definition.\n",
				string_index, NEWNAME_END);
			freeKeysRecord(newKeySw);
			freeCodeRecord(currCode);
			return 1;
		    }
		*ptr2 = '\0';

		newKeySw->newName = createName(ptr);

		if(newKeySw->newName == NULL) {
		    freeKeysRecord(newKeySw);
		    freeCodeRecord(currCode);
		    return 1;
		}
	    } /* if (ptr2) */

	    if((ptr = getNextToken(cfd)) == NULL) {
		fprintf(stderr, "Parse (line %d): '%s' should be followed by"
			"keysym and layout name.\n", string_index, KEYSW_SPEC);
		freeKeysRecord(newKeySw);
		freeCodeRecord(currCode);
		return 1;
	    }	    

	    newKeySw->codeName = createName(ptr);
	    if(newKeySw->codeName == NULL) {
		freeKeysRecord(newKeySw);
		freeCodeRecord(currCode);
		return 1;
	    }

	    newKeySw->next = currCode->switchKeys;
	    currCode->switchKeys = newKeySw;

	} else {
	    fprintf(stderr, "Parse (line %d): Keyword '%s' is unknown.\n",
		    string_index, ptr);
	    freeCodeRecord(currCode);
	    return 1;
	}

    }

    if(currCode->name == NULL) DefaultRecord = currCode;
    currCode->next = ParsedConfFile;
    ParsedConfFile = currCode;
    
    return 0;
}

int AdjustParsedRecords()
{
    parsedCodeRecord_t *currCode, *currCode2;
    parsedSwitchKey_t *currKey, *currKey2;
    parsedSwitchKey_t *gKeyList = NULL, *pKeyList, *newKeySw;
    int index, result = 0;

    /* Create a name for the default record */
    DefaultRecord->name = createName(DEFAULT_CODING);
    if(DefaultRecord->name == NULL)
	return 1;

    /* Test name dublications and references to unknown names */    
    for(currCode = ParsedConfFile; currCode; currCode = currCode->next) {
	
	/* Coding name should be unique */
	for(currCode2=currCode->next; currCode2; currCode2 = currCode2->next)
	    if(!strcmp(currCode->name, currCode2->name)) {
		fprintf(stderr,"Parse (adjust): Code '%s' is defined twice.\n",
			currCode->name);
		result = 1;
		goto adjust_cleanup;
	    }

	for(currKey=currCode->switchKeys; currKey; currKey = currKey->next) {

	    /* Key should be unique for this layout */
	    for(currKey2 = currKey->next; currKey2; currKey2 = currKey2->next)
		if(!strcmp(currKey->keyName, currKey2->keyName) &&
		   currKey->modifiers == currKey2->modifiers) {
		    fprintf(stderr, "Parse (adjust): Switching key '%s' is "
			    "defined twice in definition for '%s' code.\n",
			    currKey->keyName, currCode->name);
		    result = 1;
		    goto adjust_cleanup;
		}

	    /* General key (defined by NULL_CODING) should be unique in file */
	    if(!strcmp(currKey->codeName, NULL_CODING)) {
		for(pKeyList=gKeyList; pKeyList; pKeyList=pKeyList->next)
		    if(!strcmp(pKeyList->keyName, currKey->keyName) &&
		       pKeyList->modifiers == currKey->modifiers) {
			fprintf(stderr, "General switching key '%s' is "
				"defined twice.\n", currKey->keyName);
			result = 1;
			goto adjust_cleanup;
		    }

		pKeyList=(parsedSwitchKey_t*)malloc(sizeof(parsedSwitchKey_t));
		if (pKeyList == NULL) {
		    fprintf(stderr, "Cannot create an entry for switching key "
			    "'%s' in code '%s'.\n", currKey->keyName,
			    currCode->name);
		    result = 1;
		    goto adjust_cleanup;
		}
		pKeyList->modifiers = currKey->modifiers;
		pKeyList->keyName = currKey->keyName;
		pKeyList->newName = currKey->newName;
		pKeyList->codeName = NULL;
		pKeyList->code = currCode;
		pKeyList->next = gKeyList;
		gKeyList = pKeyList;
		
		/* Now we can upgrade NULL_CODING to DEFAULT_CODING */
		currKey->code = DefaultRecord;

	    } else {
		/* Specific Key should refer to an existing layout */
		currCode->hasSpecific = 1;
		for(currCode2=ParsedConfFile;
		    currCode2;
		    currCode2=currCode2->next)
		    if(!strcmp(currKey->codeName, currCode2->name))
			break;

		if(currCode2 == NULL) {
		    fprintf(stderr, "Parse (adjust): Switching key '%s' "
			    "defined in '%s' refers to an undefined code "
			    "'%s'.\n", currKey->keyName, currCode->name,
			    currKey->codeName);
		    result = 1;
		    goto adjust_cleanup;
		}

		currKey->code = currCode2;
	    }
	    /* We can free names since we switched to pointers */
	    free(currKey->codeName);
	    currKey->codeName = NULL;
	}
    }
    
    for(currCode = ParsedConfFile, index = 0;
	currCode;
	currCode = currCode->next, index++) {

	currCode->index = index;

	/* Add all general Keys to layouts without specific keys */
	if(currCode->hasSpecific == 0)
	    for(pKeyList = gKeyList; pKeyList; pKeyList = pKeyList->next)
		if(currCode != pKeyList->code) {
		    newKeySw = createOneKey(pKeyList);
		    if (newKeySw == NULL) {
			result = 1;
			goto adjust_cleanup;
		    }

		    newKeySw->next = currCode->switchKeys;
		    currCode->switchKeys = newKeySw;
		}
	
	/* If prefix is not specified add one according to our rules */
	if(currCode->prefix == NULL) {
	    if (currCode != DefaultRecord) {
		currCode->prefix = (char*)malloc(strlen(currCode->name) + 3);
		if (currCode->prefix != NULL) {
		    strcpy(currCode->prefix, currCode->name);
		    strcat(currCode->prefix, ": ");
		}
	    } else
		currCode->prefix = createName("");
	}

	if(currCode->prefix == NULL) {
	    fprintf(stderr, "Parse (adjust): Cannot create prefix for '%s' "
		    "code.\n", currCode->name);
	    result = 1;
	    goto adjust_cleanup;
	}
    }

 adjust_cleanup:

    for(; gKeyList; gKeyList = pKeyList) {
	pKeyList = gKeyList->next;
	free(gKeyList);
    }

    return result;
}

static char* getNextToken(FILE* cfd)
{
    char* result;
    static char* ptr = NULL;
    static char buff[256];

    while(ptr == NULL) {
	if(fgets(buff, sizeof(buff), cfd) == NULL)
	    return NULL;

	string_index++;

	ptr = skipSeparators(buff);
    }

    result = ptr;

    if(*ptr == STRING_CHAR) {
	for(result = ++ptr; *ptr; ptr++)
	    if(*ptr == STRING_CHAR)
		break;
	if(!*ptr) {
	    fprintf(stderr, "Parse (line %d): There's no closing '%c'.\n",
		    string_index, STRING_CHAR);
	    return NULL;
	}
    } else
	ptr = findSeparator(ptr);

    if(*ptr == '\0')
	ptr = NULL;
    else {
	*ptr = '\0';
	ptr = skipSeparators(ptr+1);
    }

    return result;
}

static char* skipSeparators(char* buff)
{
    char *ptr, *p;
    for(ptr=buff; *ptr; ptr++) {
	for(p=SEP_CHARS; *p && *ptr != *p; p++);

	if(!*p)
	    return *ptr == COMMENT_CHAR ? NULL : ptr;
    }

    return NULL;
}

static char* findSeparator(char* buff)
{
    char *ptr=buff, *p;

    for(ptr++; *ptr; ptr++)
	for(p=SEP_CHARS; *p; p++)
	    if(*ptr == *p || *ptr == COMMENT_CHAR)
		return ptr;

    return ptr;
}

static char* createName(char* name)
{
    char* result;

    result = (char*)malloc(strlen(name) + 1);
    if (result != NULL)
	strcpy(result, name);
    else
	fprintf(stderr, "Memory problems to dublicate '%s' name.\n", name);
    return result;
}

/* Dublicate one switch key (without following 'next' field) */
static parsedSwitchKey_t* createOneKey(parsedSwitchKey_t *currKey)
{
    parsedSwitchKey_t* result;

    result = (parsedSwitchKey_t*) calloc (1, sizeof(parsedSwitchKey_t));
    if (result == NULL) {
	fprintf(stderr, "Memory problems to dublicate a switch key.\n");
	return NULL;
    }

    result->code = currKey->code;
    result->codeName = NULL;
    result->modifiers = currKey->modifiers;
    result->keyName = createName(currKey->keyName);
    if(result->keyName == NULL) {
	freeKeysRecord(result);
	return NULL;
    }

    if(currKey->newName) {
	result->newName = createName(currKey->newName);
	if(result->newName == NULL) {
	    freeKeysRecord(result);
	    return NULL;
	}
    }

    return result;
}

void freeParsedRecords()
{
    parsedCodeRecord_t *nextCode;

    for(;ParsedConfFile; ParsedConfFile = nextCode) {
	nextCode = ParsedConfFile->next;
	freeCodeRecord(ParsedConfFile);
    }
}


static void freeCodeRecord(parsedCodeRecord_t *currCode)
{
    if(currCode->name) free(currCode->name);
    if(currCode->prefix) free(currCode->prefix);
    if(currCode->xmmfile) free(currCode->xmmfile);
    freeKeysRecord(currCode->switchKeys);
    free(currCode);
}

static void freeKeysRecord(parsedSwitchKey_t *currKey)
{
    parsedSwitchKey_t *nextKey;
    
    for(; currKey; currKey=nextKey) {
	nextKey = currKey->next;
	if(currKey->codeName) free(currKey->codeName);
	if(currKey->keyName) free(currKey->keyName);
	if(currKey->newName) free(currKey->newName);
	free(currKey);
    }
}
