/*
 *   Copyright (C) 2002,2003 by Jonathan Naylor G4KLX
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "JT44Display.h"
#include "JT44App.h"
#include "JT44Defs.h"

const int BORDER_SIZE = 5;

enum {
	List_Messages        = 150,
	List_Average,
	Menu_Message_Include = 250,
	Menu_Message_Exclude,
	Menu_Message_Decode,
	Menu_Messages_Clear,
	Menu_Average_Clear   = 350
};

BEGIN_EVENT_TABLE(CJT44Display, wxPanel)
	EVT_MENU(Menu_Message_Include, CJT44Display::onIncludeMessage)
	EVT_MENU(Menu_Message_Exclude, CJT44Display::onExcludeMessage)
	EVT_MENU(Menu_Message_Decode,  CJT44Display::onDecodeMessage)
	EVT_MENU(Menu_Messages_Clear, CJT44Display::onClearMessages)
	EVT_MENU(Menu_Average_Clear, CJT44Display::onClearAverage)
	EVT_LIST_ITEM_SELECTED(List_Messages, CJT44Display::onMessageSelected)
	EVT_LIST_ITEM_RIGHT_CLICK(List_Messages, CJT44Display::onMessageClick)
	EVT_LIST_ITEM_RIGHT_CLICK(List_Average, CJT44Display::onAverageClick)
END_EVENT_TABLE()

CJT44Display::CJT44Display(wxWindow* parent, int id) :
wxPanel(parent, id),
m_graph(NULL),
m_audio(NULL),
m_messageListCtrl(NULL),
m_averageListCtrl(NULL),
m_messageMenu(NULL),
m_averageMenu(NULL),
m_listCount(0),
m_selected(-1),
m_averageMessage(),
m_messageList(NULL)
{
	wxASSERT(parent != NULL);

	m_messageMenu = createMessageMenu();
	m_averageMenu = createAverageMenu();

	m_messageList = new CJT44MessageList(wxKEY_NONE);
	m_messageList->DeleteContents(true);

	wxBoxSizer* sizer = new wxBoxSizer(wxVERTICAL);

	wxBoxSizer* graphSizer = new wxBoxSizer(wxHORIZONTAL);

	m_graph = new CJT44LevelGraph(this, -1, wxDefaultPosition, wxSize(JT44_GRAPH_WIDTH, JT44_GRAPH_HEIGHT));
	graphSizer->Add(m_graph, 0, wxTOP | wxRIGHT, BORDER_SIZE);

	m_audio = new CLevelGraph(this, -1, wxDefaultPosition, wxSize(JT44_AUDIO_WIDTH, JT44_GRAPH_HEIGHT));
	graphSizer->Add(m_audio, 0, wxTOP, BORDER_SIZE);

	sizer->Add(graphSizer);

	m_messageListCtrl = new wxListCtrl(this, List_Messages, wxDefaultPosition, wxSize(JT44_MESSAGES_WIDTH, JT44_MESSAGES_HEIGHT));
	m_messageListCtrl->SetSingleStyle(wxLC_REPORT);
	m_messageListCtrl->InsertColumn(0, wxT("Id"));
	m_messageListCtrl->SetColumnWidth(0, 75);
	m_messageListCtrl->InsertColumn(1, wxT("Sync"));
	m_messageListCtrl->SetColumnWidth(1, 50);
	m_messageListCtrl->InsertColumn(2, wxT("dB"));
	m_messageListCtrl->SetColumnWidth(2, 50);
	m_messageListCtrl->InsertColumn(3, wxT("Dt"));
	m_messageListCtrl->SetColumnWidth(3, 50);
	m_messageListCtrl->InsertColumn(4, wxT("DF"));
	m_messageListCtrl->SetColumnWidth(4, 50);
	m_messageListCtrl->InsertColumn(5, wxT("S"));
	m_messageListCtrl->SetColumnWidth(5, 20);
	m_messageListCtrl->InsertColumn(6, wxT("Text"));
	m_messageListCtrl->SetColumnWidth(6, 225);
	m_messageListCtrl->InsertColumn(7, wxEmptyString);
	m_messageListCtrl->SetColumnWidth(7, 30);
	m_messageListCtrl->InsertColumn(8, wxEmptyString);
	m_messageListCtrl->SetColumnWidth(8, 30);

	sizer->Add(m_messageListCtrl, 0, wxTOP, BORDER_SIZE);

	m_averageListCtrl = new wxListCtrl(this, List_Average, wxDefaultPosition, wxSize(JT44_MESSAGES_WIDTH, JT44_AVERAGES_HEIGHT));
	m_averageListCtrl->SetSingleStyle(wxLC_REPORT);
	m_averageListCtrl->InsertColumn(0, wxT("Id"));
	m_averageListCtrl->SetColumnWidth(0, 75);
	m_averageListCtrl->InsertColumn(1, wxT("Sync"));
	m_averageListCtrl->SetColumnWidth(1, 50);
	m_averageListCtrl->InsertColumn(2, wxT("dB"));
	m_averageListCtrl->SetColumnWidth(2, 50);
	m_averageListCtrl->InsertColumn(3, wxEmptyString);
	m_averageListCtrl->SetColumnWidth(3, 120);
	m_averageListCtrl->InsertColumn(4, wxT("Text"));
	m_averageListCtrl->SetColumnWidth(4, 225);
	m_averageListCtrl->InsertColumn(5, wxEmptyString);
	m_averageListCtrl->SetColumnWidth(5, 30);
	m_averageListCtrl->InsertColumn(6, wxEmptyString);
	m_averageListCtrl->SetColumnWidth(6, 30);

	sizer->Add(m_averageListCtrl, 0, wxTOP, BORDER_SIZE);

	m_averageListCtrl->InsertItem(0, wxEmptyString, 0);

	SetAutoLayout(true);
	sizer->Fit(this);
	sizer->SetSizeHints(this);
	SetSizer(sizer);
}

CJT44Display::~CJT44Display()
{
	delete m_messageList;
}

wxMenu* CJT44Display::createMessageMenu()
{
	wxMenu* menu = new wxMenu();
	menu->Append(Menu_Message_Include, wxT("Include"));
	menu->Append(Menu_Message_Exclude, wxT("Exclude"));
	menu->Append(Menu_Message_Decode,  wxT("Decode"));
	menu->AppendSeparator();
	menu->Append(Menu_Messages_Clear, wxT("Clear"));

	return menu;
}

wxMenu* CJT44Display::createAverageMenu()
{
	wxMenu* menu = new wxMenu();
	menu->Append(Menu_Average_Clear, wxT("Clear"));

	return menu;
}

void CJT44Display::onMessageSelected(const wxListEvent& event)
{
	m_selected = event.GetIndex();
}

void CJT44Display::onMessageClick(const wxListEvent& event)
{
	wxASSERT(m_messageListCtrl != NULL);

	if (m_selected != -1) {
		wxPoint point = event.GetPoint();

		m_messageListCtrl->PopupMenu(m_messageMenu, point);
	}
}

void CJT44Display::onAverageClick(const wxListEvent& event)
{
	wxASSERT(m_averageListCtrl != NULL);

	wxPoint point = event.GetPoint();

	m_averageListCtrl->PopupMenu(m_averageMenu, point);
}

void CJT44Display::onIncludeMessage(const wxCommandEvent& event)
{
	wxASSERT(m_messageList != NULL);

	if (m_selected != -1) {
		CJT44Message* message = getMessage(m_selected);

		if (!message->getInclude()) {
			m_averageMessage.addMessage(*message);
			message->setInclude();
			updateMessage(m_selected);
			updateAverage();
		}
	}
}

void CJT44Display::onExcludeMessage(const wxCommandEvent& event)
{
	wxASSERT(m_messageList != NULL);

	if (m_selected != -1) {
		CJT44Message* message = getMessage(m_selected);

		if (message->getInclude()) {
			m_averageMessage.subtractMessage(*message);
			message->setExclude();
			updateMessage(m_selected);
			updateAverage();
		}
	}
}

void CJT44Display::onDecodeMessage(const wxCommandEvent& event)
{
	wxASSERT(m_messageListCtrl != NULL);
	wxASSERT(m_messageList != NULL);

	if (m_selected != -1) {
		CJT44Message* message = getMessage(m_selected);

		wxString   text = message->getText();
		wxString oeText = message->getOddEvenText();
		wxString l4Text = message->getLast4Text();

		m_messageListCtrl->SetItem(m_selected, 6, text);
		m_messageListCtrl->SetItem(m_selected, 7, oeText);
		m_messageListCtrl->SetItem(m_selected, 8, l4Text);
	}
}

void CJT44Display::onClearMessages(const wxCommandEvent& event)
{
	wxASSERT(m_messageListCtrl != NULL);
	wxASSERT(m_messageList != NULL);

	m_messageListCtrl->DeleteAllItems();
	m_messageList->Clear();

	m_listCount = 0;
	m_selected  = -1;

	onClearAverage(event);
}

void CJT44Display::onClearAverage(const wxCommandEvent& event)
{
	wxASSERT(m_averageListCtrl != NULL);
	wxASSERT(m_messageListCtrl != NULL);
	wxASSERT(m_messageList != NULL);

	for (int i = 0; i < m_listCount; i++) {
		CJT44Message* message = getMessage(i);
		message->setExclude();
		updateMessage(i);
	}

	m_averageListCtrl->SetItem(0, 0, wxEmptyString);
	m_averageListCtrl->SetItem(0, 1, wxEmptyString);
	m_averageListCtrl->SetItem(0, 2, wxEmptyString);
	m_averageListCtrl->SetItem(0, 3, wxEmptyString);
	m_averageListCtrl->SetItem(0, 4, wxEmptyString);
	m_averageListCtrl->SetItem(0, 5, wxEmptyString);
	m_averageListCtrl->SetItem(0, 6, wxEmptyString);

	m_averageMessage.clear();
}

void CJT44Display::showMessage(CJT44Message* message)
{
	wxASSERT(m_messageListCtrl != NULL);
	wxASSERT(message != NULL);
	wxASSERT(m_listCount >= 0);

	int minSync = ::wxGetApp().getMinSync();

	wxString id = message->getId();

	int pos = id.Find(wxT('_'), true);
	if (pos != -1) {
		wxString tmp = id.Mid(pos + 1);
		id = tmp;
	}

	m_messageListCtrl->InsertItem(0, id, 0);

	wxString buffer;
	buffer.Printf(wxT("%d"), message->getQuality());
	m_messageListCtrl->SetItem(0, 1, buffer);

	buffer.Printf(wxT("%.0f"), message->getStrength() - 0.5);
	m_messageListCtrl->SetItem(0, 2, buffer);

	buffer.Printf(wxT("%.1f"), message->getDt());
	m_messageListCtrl->SetItem(0, 3, buffer);

	buffer.Printf(wxT("%d"), message->getDF());
	m_messageListCtrl->SetItem(0, 4, buffer);

	if (message->getQuality() >= minSync) {
		message->setInclude();

		wxString   text = message->getText();
		wxString oeText = message->getOddEvenText();
		wxString l4Text = message->getLast4Text();

		m_messageListCtrl->SetItem(0, 5, wxT("*"));
		m_messageListCtrl->SetItem(0, 6, text);
		m_messageListCtrl->SetItem(0, 7, oeText);
		m_messageListCtrl->SetItem(0, 8, l4Text);

		m_averageMessage.addMessage(*message);
		m_averageMessage.setId(message->getId());

		::wxGetApp().logMessage(*message);
	} else {
		message->setExclude();

		m_messageListCtrl->SetItem(0, 5, wxEmptyString);
		m_messageListCtrl->SetItem(0, 6, wxEmptyString);
		m_messageListCtrl->SetItem(0, 7, wxEmptyString);
		m_messageListCtrl->SetItem(0, 8, wxEmptyString);
	}

	m_listCount++;

	updateAverage();

	pos = 0;
	m_messageList->Insert(pos, message);

	// The list is moving down and the selected message has moved down
	// by one
	if (m_selected != -1)
		m_selected++;
}

void CJT44Display::updateMessage(int n)
{
	wxASSERT(n >= 0 && n < m_listCount);
	wxASSERT(m_messageListCtrl != NULL);
	wxASSERT(m_messageList != NULL);

	CJT44Message* message = getMessage(n);

	m_messageListCtrl->SetItem(n, 5, message->getInclude() ? wxT("*") : wxEmptyString);
}

void CJT44Display::updateAverage()
{
	wxASSERT(m_averageListCtrl != NULL);

	int minSync = ::wxGetApp().getMinSync();

	if (m_averageMessage.getCount() >= minSync) {
		wxString id = m_averageMessage.getId();

		int pos = id.Find(wxT('_'), true);
		if (pos != -1) {
			wxString tmp = id.Mid(pos + 1);
			id = tmp;
		}
		m_averageListCtrl->SetItem(0, 0, id);

		wxString buffer;
		buffer.Printf(wxT("%d"), m_averageMessage.getQuality());
		m_averageListCtrl->SetItem(0, 1, buffer);

		buffer.Printf(wxT("%.0f"), m_averageMessage.getStrength() - 0.5);
		m_averageListCtrl->SetItem(0, 2, buffer);

		buffer.Printf(wxT("%d/%d"), m_averageMessage.getCount(), m_listCount);
		m_averageListCtrl->SetItem(0, 3, buffer);

		wxString   text = m_averageMessage.getText();
		wxString oeText = m_averageMessage.getOddEvenText();
		wxString l4Text = m_averageMessage.getLast4Text();

		m_averageListCtrl->SetItem(0, 4, text);
		m_averageListCtrl->SetItem(0, 5, oeText);
		m_averageListCtrl->SetItem(0, 6, l4Text);
	} else {
		m_averageListCtrl->SetItem(0, 0, wxEmptyString);
		m_averageListCtrl->SetItem(0, 1, wxEmptyString);
		m_averageListCtrl->SetItem(0, 2, wxEmptyString);
		m_averageListCtrl->SetItem(0, 3, wxEmptyString);
		m_averageListCtrl->SetItem(0, 4, wxEmptyString);
		m_averageListCtrl->SetItem(0, 5, wxEmptyString);
		m_averageListCtrl->SetItem(0, 6, wxEmptyString);
	}
}

void CJT44Display::showLevels(CJT44Levels* levels) const
{
	wxASSERT(levels != NULL);
	wxASSERT(m_graph != NULL);

	m_graph->addData(levels);
}

void CJT44Display::showAudio(double audio) const
{
	wxASSERT(m_audio != NULL);

	m_audio->addData(audio);
}

CJT44Message* CJT44Display::getMessage(int n) const
{
	wxASSERT(n >= 0 && n < m_listCount);
	wxASSERT(m_messageList != NULL);

	CJT44MessageList::Node* node = m_messageList->Item(n);

	return node->GetData();
}
