/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux 

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "sample.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mixer.h"
#include "debug.h"
#include "application.h"

#ifdef HAVE_MIXER
#include <SDL_mixer.h>
#endif

#include <stdlib.h>    //malloc

namespace wftk {

ResourceRegistry<Sample*> Sample::registry;

#ifdef HAVE_MIXER

Sample::Sample(const Sample& sample) :
  Sound(sample),
  channel_(-1)
{
  Mixer::instance(); // insure init audio

  if(this != &sample)
    {
      if(sample.data_ != 0)
	{
	  Debug out(Debug::SOUND);

	  out << "Sample::Sample(const Sample&) ...";

	  Mix_Chunk *new_data, *old_data = (Mix_Chunk*) sample.data_;

	  new_data = (Mix_Chunk*)malloc(sizeof(Mix_Chunk));
	  if(!new_data)
            throw(Fatal("failed malloc in Sample copy contructor"));
	  memcpy(new_data, old_data, sizeof(Mix_Chunk));

	  //allocate mem for sample data
	  new_data->abuf = (Uint8*)malloc(new_data->alen);
	  if(!new_data->abuf)
            throw(Fatal("failed malloc in Sample copy contructor"));
	  
	  //copy sample data
	  //memcpy(<dest>, <src>, <length>)
	  memcpy((void*)new_data->abuf, 
		 (void*)old_data->abuf, 
		 (size_t)new_data->alen);

	  data_ = new_data;

	  out <<"[OK]"<< Debug::endl;
	}
      else
	{
	  data_ = 0;
	}

    }
}

Sample::~Sample()
{
  if(data_)
    {
      Debug out(Debug::SOUND);

      out << "Sample freeing chunk ... ";
      
      if(Mixer::instance()->audioAvailable())
	{
	  Mix_FreeChunk((Mix_Chunk*)data_);
	}
      else
	{
	  //do it by hand...
	  
	  if( ((Mix_Chunk*) data_)->allocated ) 
	    {
	      free(((Mix_Chunk*) data_)->abuf);
	    }
	  free(data_);       
	}

      out << "[OK]" << Debug::endl;
    }
}

int
Sample::setVolume(int vol_)
{
  int tmp = 0;
  
  if(data_ && channel_ > 0)
    {
      Mix_Volume(channel_, vol_);
      tmp  = Mix_VolumeChunk((Mix_Chunk*) data_, vol_);
    }

  return tmp;
}

bool
Sample::load(const std::string& filename )
{
  Mixer::instance(); // insure init audio

  bool success = false;

  if((Mixer::instance() != 0)&&(Mixer::instance()->audioAvailable()))
     {
       Debug out(Debug::SOUND);

       out << "loading wav ...";
       data_ = Mix_LoadWAV(filename.c_str());
       out << "ok." << Debug::endl;
       
       if(data_ == 0)
	 {
	   out << "Sample: unable to load file." << Debug::endl;
	 }
       else
	 {
	   success = true;
	 }
     }
  else
    {
      data_ = 0;
    }
  
  return success;
}


bool
Sample::play(int loops) const
{
  bool success = false;

  if(Mixer::instance() == 0)
    return false;
  if(!Mixer::instance()->audioAvailable())
    return false;

  if(data_ != 0)
    {
      //play on first available channel
      channel_ = Mix_PlayChannel(-1, (Mix_Chunk*) data_, loops);
      if(channel_ >= 0)
	{
	  success = true;
	}
    }

  return success;
}
  
void 
Sample::stop() const
{
  if(!Mixer::instance())
    return;
  if(!Mixer::instance()->audioAvailable())
    return;

    if(channel_ >= 0)
      Mix_HaltChannel(channel_);
}

bool
Sample::playing() const
{
  if(channel_ < 0)
    return false;

  if(!Mixer::instance())
    return false;
  if(!Mixer::instance()->audioAvailable())
    return false;  
  
  return Mix_Playing(channel_);
}

#else // HAVE_MIXER
Sample::Sample(const Sample& sample)
{

}

Sample::~Sample()
{
 
}

int
Sample::setVolume(int)
{
  return 0;
}

bool
Sample::load(const std::string&)
{
  return false;
}


bool
Sample::play(int ) const
{
  return false;
}

void 
Sample::stop() const
{

}

bool
Sample::playing() const
{
  return false;
}

#endif // HAVE_MIXER
} // namespace

