/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux 

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
  
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/


#ifndef _RESOURCES_H
#define _RESOURCES_H

#include <wftk/string_hash_map.h>
#include <cassert>

namespace wftk {

/// default data freeing mechanism for resources (do nothing)
template<class R>
struct ResDestroy
{
  /// do nothing, the destructor handles destroying this resource instance
  void operator()(const R&) {}
};

/// default data freeing mechanism for resources which are pointers (call delete)
template<class R>
struct ResDestroy<R*>
{
  /// delete this resource instance
  void operator()(R* r) {delete r;}
};

/// default fallback for a failed resource find (return a reference to a static member)
template<class R>
struct ResInval
{
  /// the resource type the registry will return on a find() call
  typedef const R& OutType;
#if 0
  // much rather have this, but the compiler gets confused
  /// default value for find() to return when resource lookup fails
  OutType operator()() const {return r_;}
 private:
  R r_;
#else
  ResInval() : r_(new R()) {}
  ~ResInval() {delete r_;}
  /// default value for find() to return when resource lookup fails
  OutType operator()() const {return *r_;}
 private:
  R* r_;
#endif
};

/// default fallback for a failed resource find for pointer data (return null)
template<class R>
struct ResInval<R*>
{
  /// the resource type the registry will return on a find() call
  typedef const R* OutType;
  /// default value for find() to return when resource lookup fails
  OutType operator()(const std::string&) const {return 0;}
};

/// default loader for resource data
template<class R>
struct ResLoad
{
  /// attempt to load the resource
  std::pair<R,bool> operator()(const std::string& filename)
  {
    R r;
    bool result = r.load(filename);
    return std::make_pair(r, result);
  }
};

/// default loader for pointer resource data
template<class R>
struct ResLoad<R*>
{
  /// attempt to load the resource
  std::pair<R*,bool> operator()(const std::string& filename)
  {
    R* r;
    r = new R();
    if(!r->load(filename)) {
      delete r;
      r = 0;
    }
    return std::make_pair(r, r != 0);
  }
};

/// Resource manager with reference counting.
template<class R, class Destroy = ResDestroy<R> >
class Resource
{
 public:
  ///
  ~Resource() {Destroy d; d(res_);}
  /// create a resource with a reference count of one
  Resource(const R& res) : res_(res), references_(1) {} 
  /// increase the reference count
  void bind() {++references_;}
  /// decrease the reference count
  void free() {if(--references_ == 0) delete this;}
  /// access the resource's data
  const R& res() const {return res_;}

 private:
  ///
  R res_;
  ///
  int references_;
};

/// A registry template for different resource types
/**
 * R is the class of the resource type
 * Loader specifies how to load the resource from a file
 * Invalid specifies the fallback mechanism if a resource isn't found
 * Destroy specifies how to free the resource data when the resource is freed
 **/
template<class R, class Loader = ResLoad<R>, class Invalid = ResInval<R>, class Destroy = ResDestroy<R> >
class ResourceRegistry
{
 public:
  /// the type resource this registry holds
  typedef Resource<R,Destroy> Resource;

  ///
  ~ResourceRegistry()
  {
    for(typename ResourceMap::iterator I = resources_.begin(); I != resources_.end(); ++I)
      I->second->free();
  }
  /// template for loading a Resource from disk
  Resource* load(const std::string& resname, const std::string& spec);
  /// For this one, you own a reference to the resource returned
  Resource* loadAnonymous(const std::string& spec);
  /// insert a Resource into the pool 
  /// The resource must be valid
  void insert(const std::string& resname, Resource* res)
  {
    // Make sure the resource name is unique
    assert(res);
    if(resources_.insert(typename ResourceMap::value_type(resname, res)).second)
      res->bind();
  }
  /// Find a resource, with fallback on failure
  typename Invalid::OutType find(const std::string& resname) const
  {
    typename ResourceMap::const_iterator I = resources_.find(resname);
    return (I != resources_.end()) ? I->second->res() : inval_(resname);
  }
  /// Find a resource, return null on failure
  Resource* get(const std::string& resname) const
  {
    typename ResourceMap::const_iterator I = resources_.find(resname);
    return (I != resources_.end()) ? I->second : 0;
  }
  /// Release all resources in this registry
  void unregisterAll()
  {
    for(typename ResourceMap::iterator I = resources_.begin(); I != resources_.end(); ++I)
      I->second->free();
    resources_.clear();
  }

  /// internal registry map type
  typedef typename StringHash<Resource*>::Map ResourceMap;

  /// begin() function for internal map
  typename ResourceMap::const_iterator begin() const {return resources_.begin();}
  /// end() function for internal map
  typename ResourceMap::const_iterator end() const {return resources_.end();}
  /// size() function for internal map
  typename ResourceMap::size_type size() const {return resources_.size();}
  /// empty() function for internal map
  bool empty() const {return resources_.empty();}

 private:
  ///
  ResourceMap resources_;
  ///
  Loader loader_;
  ///
  Invalid inval_;
};

template<class R, class Loader, class Invalid, class Destroy>
Resource<R,Destroy>*
ResourceRegistry<R,Loader,Invalid,Destroy>::load(
	const std::string& resname, const std::string& spec)
{
  // Make sure the resource name is unique

  std::pair<typename ResourceMap::iterator, bool> result =
    resources_.insert(typename ResourceMap::value_type(resname, 0));

  if(!result.second)
    return 0;

  // Load the resource

  std::pair<R,bool> load_val = loader_(spec);
  if(!load_val.second) {
    resources_.erase(result.first);
    return 0;
  }

  // Actually place the resource in the map and return

  Resource* res = new Resource(load_val.first);
  result.first->second = res;
  return res;
}

template<class R, class Loader, class Invalid, class Destroy>
Resource<R,Destroy>*
ResourceRegistry<R,Loader,Invalid,Destroy>::loadAnonymous(const std::string& spec)
{
  std::pair<R,bool> load_val = loader_(spec);
  return load_val.second ? new Resource(load_val.first) : 0;
}

} // namespace

#endif
