/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "multilineedit.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "debug.h"

#include <sigc++/object_slot.h>

#include <ctype.h>
#include <climits>
#include <stack>

namespace wftk {

MultiLineEdit::MultiLineEdit(const std::string& text, const Font &font,
	bool readonly, bool fill):
  LineEdit("", font),
  maxLinesToStore_(100),
  wordWrap_(true),
  readOnly_(readonly),
  fill_(fill),
  displayLines_(6)
{
  enterPressed.connect(slot(*this, &MultiLineEdit::newline));

  getResourceBackground("multilineedit");
    
  fonts_.resize(16);
  images_.resize(16);
  for(size_t n=0; n < 16; n++)
    images_[n] = NULL;

  if(readonly)
    setClickToFocus(false);

  addText(text);

  setPackingInfo();
}

MultiLineEdit::~MultiLineEdit()
{
//   for(unsigned n=0; n < renderedLines_.size(); n++)
//    delete renderedLines_[n];

  for(unsigned i = 0; i < images_.size(); ++i)
    if(images_[i])
      images_[i]->free();
}

void
MultiLineEdit::breakText(std::string& text, std::vector<std::string>& lines, 
			 unsigned maxLines)
{
  unsigned pos = 0;
  unsigned start = 0;
  unsigned len = 0;
  unsigned linestart = 0;
  int x=0;
  std::vector<unsigned> linebreaks;

  Rect dest;
  bool breakLine = false;
  int size_x = 0;
  int size_y = textFont_.getHeight();
  
  lines.clear();
  
  pos=0;
  while(pos < text.size())
    {
      start = pos;
      breakLine = false;
      
      if(wordWrap_)
	{
	  //dont break words
	  while(pos<text.size())
	    {
	      if(isspace(text[pos]))
		break;
	      pos++;
	      size_x+= textFont_.getChar(text[pos]).width();
	    }
	  
	  if(pos<text.size())
	    {
	      if(text[pos] == '\n')
		{
		  breakLine = true;
		  len = pos-start;
		  pos++;
		}
	      else
		{
		  pos++;
		  size_x+= textFont_.getChar(text[pos]).width();
		  len = pos-start;
		}
	    }
	  else
	    {
	      len = pos-start;
	    }
	}
      else
	{
	  pos++;
	  size_x += textFont_.getChar(text[pos]).width();	    
	}
      
      if(!breakLine && !readOnly_ && (x + size_x >= width()))
	{
	  x = 0;
	  size_y += textFont_.getHeight();
	  
	  if(linestart != text.size())
	    {
	      if(start != linestart)
		{
		  //if the word is not longer than the line....
		  linebreaks.push_back(linestart);
		  lines.push_back(text.substr(linestart,start-linestart));
		  linestart = pos = start;
		}
	      else
		{
		  //the word is longer than the line, thus only the front part
		  //will be visible
		  linebreaks.push_back(linestart);
		  lines.push_back(text.substr(linestart,pos-linestart-1));
		  linestart = start = pos;
		}
	    }
	} 
      
      x += size_x;
      
      if(breakLine)
	{
	  x = 0;
	  size_y += textFont_.getHeight();
	  
	  if(linestart != text.size())
	    {
	      linebreaks.push_back(linestart);
	      lines.push_back(text.substr(linestart,pos-linestart-1));
	      linestart = start = pos;
	    }
	}
      
      size_x = 0;
	  
    }  //while pos < text.size()
  
  //add an empty line if last char is a '\n' and we're not read-only
  if(breakLine && !readOnly_)
    lines.push_back("");
  
  //do not forget the last line (if it is not empty)
  if(linestart != pos)
    lines.push_back(text.substr(linestart,pos-linestart));

  /* before removing lines we need to parse for escape sequences, in
   * order to not lose anything when removing lines ..
   */
  preprocessLines(lines);

  /* limit the number of lines to the requested maximum */
  if(lines.size() > maxLines)
    {
      text.erase(0,linebreaks[lines.size()-maxLines]);
    }
}

unsigned 
MultiLineEdit::visibleLines() const
{
  return height()/textFont_.getHeight();
}

void 
MultiLineEdit::preprocessLines(std::vector<std::string>& lines)
{
  std::list<std::string> prefix;
  for(unsigned n = 0; n < lines.size(); n++)
    {
      if(!lines[n].empty())
	{
	  std::string prefixStr;
	  std::list<std::string>::iterator pItr = prefix.begin();
	  while(pItr != prefix.end())
	    {
	      prefixStr+= *pItr;
	      pItr++;
	    }
	  
	  std::string::const_iterator itr = lines[n].begin();	  
	  while(itr != lines[n].end())
	    {
	      std::string tmp;
				  
	      while(itr != lines[n].end() &&
		    *itr != START_TAG && 
		    *itr != END_TAG)
		itr++;
		
	      if(itr == lines[n].end())
		break;
	      
	      if(*itr == START_TAG)
		{
		  tmp += (*itr);

		  itr++;
		  if(itr == lines[n].end())
		    break;
		  		 
		  tmp += (*itr); 

	
		  itr++;
		  if(itr == lines[n].end())
		    break;		  
		  
		  tmp += (*itr++);

		  prefix.push_back(tmp);
		  tmp = "";
		}
	      else if(*itr == END_TAG)
		{
		  if(!prefix.empty())
		    prefix.pop_back();
		  itr++;
		}
	      
	    } /// while (itr != lines[n].end())

	  lines[n] = prefixStr + lines[n];

	} // if(!lines[n].empty) {
    }
}

void
MultiLineEdit::renderTextLines(const std::vector<std::string>& lines)
{
  Debug out(Debug::TEXT_WIDGETS);

  out << "Rendering " << lines.size() << " text lines in MultiLineEdit" << Debug::endl;

  int font_ind = 0;
  int img_ind = 0;
  int link_ind = 0;
  std::stack<int> context;
				  
  unsigned start = (lines.size() <= maxLines()) ? 0 : (lines.size()-maxLines());

  textLines_.clear();

  for(unsigned n = start; n < lines.size(); n++)
    {
      /* to start just push an empty line, we will
       * add text chunks if needed 
       */
      textLines_.push_back(TextLine());
      if(!lines[n].empty())
	{
	  std::string::const_iterator itr = lines[n].begin();

	  while(itr != lines[n].end())
	    {
	      std::string tmp;

	      while(itr != lines[n].end() &&
	            *itr != START_TAG && 
		    *itr != END_TAG)
		{
		  tmp += (*itr);
		  itr++;
		}

	 //      if(!tmp.empty())
// 		{
		  TextChunk chunk;
		  chunk.link = link_ind;
		  chunk.image = img_ind;
		  chunk.font = font_ind;
		  chunk.text = tmp;

		  textLines_.back().push_back(chunk);
		  //	}
	      	      
	      if(itr == lines[n].end())
		break;
	      
	      if(*itr == START_TAG)
		{
		  itr++;
		  if(itr == lines[n].end())
		    break;
		  
		  if(*itr == FONT_TAG ||
		     *itr == IMAGE_TAG ||
		     *itr == LINK_TAG)
		    {
		      char tag = *itr;
		      itr++;
		      if(itr == lines[n].end())
			break;
		      switch(tag) 
			{
			case FONT_TAG:
			  font_ind = *itr++;
			  break;
			case IMAGE_TAG:
			  img_ind = *itr++;
			  break;
			case LINK_TAG:
			  link_ind= *itr++;
			  break;
			default:
			  break;
			}
		      context.push(tag);
		    }		      
		  else
		    {
		      itr++;
		      if(itr == lines[n].end())
			break; 
		      itr++;
		    }
		}
	      else if(*itr == END_TAG)
		{
		  if(!context.empty())
		    {
		      if(context.top() == FONT_TAG)
			{
			  font_ind = 0;
			  context.pop();
			}
		      else if(context.top() == IMAGE_TAG)
			{
			  img_ind = 0;
			  context.pop();
			}
		      else if(context.top() == LINK_TAG)
			{
			  link_ind = 0;
			  context.pop();
			}
		    }
		  itr++;
		}
	      
	    } /// while (itr != lines[n].end())
	  
	} // if(lines[n].empty) {} else {
    }

  out << "Got " << textLines_.size() << " lines" << Debug::endl;
}

Point
MultiLineEdit::drawText(Surface& surf, const Point& offset, const Region& r)
{
  Debug out(Debug::DRAWING);

  out << "Drawing text in MultiLineEdit" << Debug::endl;

  int xoff = 0;
  int yoff = 0;

  int line_height = textFont_.getHeight();

  linkAreas_.clear();

  bool off_the_end = false;

  out << "Got " << textLines_.size() << " text lines" << Debug::endl;

  for(unsigned n=0; n < textLines_.size(); n++)
    {
      out << "Drawing line " << n << Debug::endl;

      xoff = 0;
      TextLine::iterator itr = textLines_[n].begin();
      while(itr != textLines_[n].end())
        {
          int xsave = xoff;
          int v_offset = 0;
          if(itr->image > 0 && itr->image < images_.size() &&
    	    images_[itr->image])
    	      {
    	        const Surface* img = images_[itr->image]->res();
    	        v_offset= line_height-img->height();
    	        Point dest(xoff, yoff+v_offset);
    	        img->blit(surf, dest + offset, r);
    	        xoff += img->width();
    	      }

          if(!itr->text.empty())
    	    {
    	      Font font;

    	      if(itr->font > 0 && itr->font < fonts_.size())
    	        font = fonts_[itr->font];
    	      else 
    	        font = textFont_;
		  
    	      v_offset = line_height + font.metrics().descender / 64;

	      

    	      xoff += font.blitString(itr->text, surf,
    				      Point(xoff, yoff+v_offset) + offset,
    				      r);
    	    }
	      
          if(itr->link > 0)
    	    {
    	      LinkArea link;
    	      link.rect = Rect(xsave, yoff+v_offset, 
    			       xoff-xsave, line_height-v_offset);
    	      link.id = itr->link;
    	      linkAreas_.push_back(link);
    	    }
          if(xoff > width() || yoff > height())
	    break;
          ++itr;
        }
      yoff += textFont_.getHeight();
      if(itr != textLines_[n].end())
        off_the_end = true; // to much text skipping
    }

  if(off_the_end) // push the cursor out of the viewable area
    yoff += height();

  return Point(xoff, yoff);
}

void
MultiLineEdit::setPackingInfo()
{
  LineEdit::setPackingInfo();

  if(readOnly_) {
    // find the width of the text

    unsigned max_width = 0;
    for(std::vector<TextLine>::iterator line = textLines_.begin();
      line != textLines_.end(); ++line) {
      unsigned width = 0;
      for(TextLine::iterator chunk = line->begin(); chunk != line->end(); ++chunk) {
        Font font = (chunk->font > 0 && chunk->font < fonts_.size()) ?
          fonts_[chunk->font] : textFont_;
        width += font.getExtents(chunk->text).w;
        if(chunk->image > 0 && chunk->image < images_.size() && images_[chunk->image])
          width += images_[chunk->image]->res()->width();
      }
      if(width > max_width)
        max_width = width;
    }

    packing_info_.x.pref = max_width;
    if(packing_info_.x.min > packing_info_.x.pref)
      packing_info_.x.min = packing_info_.x.pref;
    if(textLines_.size() > 0)
      packing_info_.y.pref *= textLines_.size();
    if(textLines_.size() == 0)
      packing_info_.y.min = 0;
  }
  else {
    packing_info_.y.expand = true;
    packing_info_.y.pref *= displayLines_;
  }

  if(fill_) {
    packing_info_.y.filler = PackingInfo::Expander::FILL;
    packing_info_.y.expand = true;
  }
}

void
MultiLineEdit::updateText()
{
  Debug::channel(Debug::TEXT_WIDGETS) << "Updating text in MultiLineEdit"
	<< Debug::endl;

  std::vector<std::string> lines;
  breakText(text_, lines, maxLinesToStore_ + maxLines());
  preprocessLines(lines);
  renderTextLines(lines);
  if(readOnly_)
    packingUpdate();
}

unsigned
MultiLineEdit::maxLines()
{
  return readOnly_ ? UINT_MAX : height()/textFont_.getHeight();
}

void
MultiLineEdit::handleResize(Uint16 w, Uint16 h)
{
  LineEdit::handleResize(w, h);

  if(!readOnly_) // resize will change maxLines()
    updateText();
}
 
void
MultiLineEdit::setWrapping(bool flag)
{
  Debug::channel(Debug::TEXT_WIDGETS) << "Setting word wrap " << flag
	<< " in MultiLineEdit" << Debug::endl;

  if(flag != wordWrap_)
    {
      textUpdate();
      wordWrap_ = flag;
    }
}
  
void MultiLineEdit::addText(const std::string& data)
{
  Debug::channel(Debug::TEXT_WIDGETS) << "Adding text in MultiLineEdit:"
	<< Debug::endl << data << Debug::endl;

  if(data.empty())
    return;
  text_ += data;
  //we need to be reblitted !
  textUpdate();
}

bool MultiLineEdit::buttonEvent(Mouse::Button button, bool pressed, const Point& pos)
{
  if(!acceptsFocus())
    return false;

  if(Widget::buttonEvent(button, pressed, pos))
    return true;

  if(button != Mouse::LEFT || !pressed)
    return false;

  std::vector<LinkArea>::iterator itr = linkAreas_.begin();
  while(itr != linkAreas_.end())
    {
      if(itr->rect.contains(pos))
        {
          linkActivated.emit(itr->id);
          /* hmmm break or return true ? ...
           * a design guideline would answer this, but
           * no one has written it yet ... */
          break;
        }
      ++itr;
    }

  return true;
}

void 
MultiLineEdit::setMarkupColor(unsigned index, const Color& fontCol)
{
  if(index < fonts_.size())
    fonts_[index].setColor(fontCol);

  invalidate();
}

void 
MultiLineEdit::setMarkupFont(unsigned index, const Font& font)
{
  if(index < fonts_.size())    
    fonts_[index] = font;
}

void 
MultiLineEdit::setImage(unsigned index, Surface::Resource* image)
{
  if(index >= images_.size() || images_[index] == image)
    return;

  if(images_[index])
    images_[index]->free();
  images_[index] = image;
  if(image)
    image->bind();
}

void MultiLineEdit::setImage(unsigned index, const Surface& surf)
{
  Surface::Resource* r = new Surface::Resource(new Surface(surf));
  setImage(index, r);
  r->free();
}

void MultiLineEdit::setImage(unsigned index, const std::string& name)
{
  Surface::Resource *res = Surface::registry.get(name);
  if(res)
    setImage(index, res);
}

} // namespace wftk
