/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <time.h>
#include <pwd.h>    /* for getpwuid() */
#include <unistd.h> /* for getuid() */

#include "human.h"
#include "list1.h"
#include "defs.h"

#include "conf.h"
#include "preresolv.h"
#include "timediff.h"

#include "wfipaddr.h"
#include "wfport.h"
#include "wfprotocol.h"

#define TIMESIZE 40


wf_outmodule_human::wf_outmodule_human() :
  wf_outmodule(),
  dns(NULL),
  whois(NULL)
{
  conf = new wf_outmodule_human_conf();
}

wf_outmodule_human::~wf_outmodule_human() {
  if (dns != NULL)
    delete dns;
  if (whois != NULL)
    delete whois;
  delete conf;
}


void
wf_outmodule_human::output_header(ostream& os) const {
  char nowstr[TIMESIZE];
  time_t now = time(NULL);
  strftime(nowstr, TIMESIZE, "%a %b %d %H:%M:%S %Z %Y", localtime(&now));
  
  os << conf->config["title"].tostr() << "\n\n";

  struct passwd* gen_user;
  gen_user = getpwuid(getuid());
  if (gen_user != NULL)
    ostream_printf(os, _("Generated on %s by %s."), nowstr,
		   gen_user->pw_gecos[0] != '\0' ?
		   gen_user->pw_gecos : gen_user->pw_name);
  else
    ostream_printf(os, _("Generated on %s by an unknown user."), nowstr);
  os << "\n\n";
}

static void
output_footer(ostream& os) {
  os << '\n';
  ostream_printf(os, _("Generated by wflogs %s"), VERSION);
  os << " (C) Herve Eychenne " WALLFIRE_URL "." << endl;
}


bool
wf_outmodule_human::print(const wf_logentry* entry, ostream& os) {
  char time1[TIMESIZE], time2[TIMESIZE];

  if (conf->config["stimes"].tobool()) {
    strftime(time1, TIMESIZE, "%b %d %H:%M:%S", localtime(&entry->start_time));

  if (conf->config["etimes"].tobool() && entry->end_time != 0) {
      strftime(time2, TIMESIZE, "%b %d %H:%M:%S", localtime(&entry->end_time));
      os << "from " << time1 << " to " << time2;
    }
    else
      os << "at " << time1;

    os << ", ";
  }

  if (conf->config["duration"].tobool() && entry->count > 1)
    os << "during " << timediff_tohumanstr(entry->start_time,
					   entry->end_time) << ", ";

  os << entry->count << ' ';
  if (entry->count == 1)
    os << "packet";
  else
    os << "packets";

  os << " logged";

  if (conf->config["loghost"].tobool())
    os << " on host " << entry->hostname;

  os << ": ";

  if (conf->config["chain"].tobool())
    os << "chain " << entry->chainlabel << ", ";

  if (conf->config["branch"].tobool())
    os << "branch " << entry->branchname << ", ";

  if (conf->config["interfaces"].tobool()) {
    os << "on ";
    if (entry->input_iface.empty() == false)
      os << "input interface " << entry->input_iface << ", ";
  }

  if (conf->config["src_mac"].tobool() && entry->smacaddr.isdefined()) {
    os << "source mac address " << entry->smacaddr;
    if (conf->config["mac_vendor"].tobool())
      os << " (" << entry->smacaddr.vendor() << ')';
    os << ", ";
  }

  if (conf->config["interfaces"].tobool()) {
    if (entry->output_iface.empty() == false)
      os << "output interface " << entry->output_iface << ", ";
  }

  if (conf->config["dst_mac"].tobool() && entry->dmacaddr.isdefined()) {
    os << "destination mac address " << entry->dmacaddr;
    if (conf->config["mac_vendor"].tobool())
      os << " (" << entry->dmacaddr.vendor() << ')';
    os << ", ";
  }

  string proto; /* make it "global" because we may need it later */
  {
    wf_protocol protocol = wf_protocol(entry->protocol);
    proto = protocol.tostr();
  }

  if (conf->config["proto"].tobool())
    os << "protocol " << proto << ", ";

  if (entry->protocol == IPPROTO_ICMP) {
    os << "type " << entry->sport;
    if (wf_protocol_icmp_type_hascode(entry->sport))
      os << " code " << entry->dport;
    
    if (conf->config["sresolve"].tobool())
      os << " (" <<
	wf_protocol_icmp_type_tostr(entry->sport, entry->dport) << ')';
    
    os << ", ";
  }

  if (conf->config["datalen"].tobool())
    os << "length " << entry->datalen << ", ";

  if (conf->config["src_ip"].tobool()) {
    bool src_resolved = false;

    os << "from " << entry->sipaddr;

    if (conf->config["resolve"].toint() != RESOLV_NO) {
      os << " (";
      wf_dns_entry* dnsentry = dns->resolv(entry->sipaddr);
      if (dnsentry != NULL && dnsentry->name.empty() == false) {
	src_resolved = true;
	os << dnsentry->name;
      }
      else
	os << "unknown hostname";
      os << ')';
    }

    if (conf->config["whois_lookup"].toint() != WHOIS_NO &&
	entry->sipaddr.isroutable()) {
      if (conf->config["whois_lookup"].toint() == WHOIS_YES ||
	  (conf->config["whois_lookup"].toint() == WHOIS_IFNODNS &&
	   src_resolved == false)) {
	wf_whois_entry* we = whois->whois(entry->sipaddr);
	os << " (";
	if (we != NULL)
	  os << *we;
	else
	  os << "no whois entry";
	os << ')';
      }
    }
  }

  if (conf->config["src_port"].tobool()) {
    if (entry->protocol == IPPROTO_UDP || entry->protocol == IPPROTO_TCP) {
      os << " port " << entry->sport;

      if (conf->config["sresolve"].tobool()) {
	string service = wf_port_toservice(entry->sport, proto);
	os << " (";
	if (service.empty())
	  os << "unknown service name";
	else
	  os << service;
	os << ')';
      }
    }
  }

  if (conf->config["dst_ip"].tobool()) {
    bool dst_resolved = false;

    os << " to " << entry->dipaddr;

    if (conf->config["resolve"].toint() != RESOLV_NO) {
      os << " (";
      wf_dns_entry* dnsentry = dns->resolv(entry->dipaddr);
      if (dnsentry != NULL && dnsentry->name.empty() == false) {
	dst_resolved = true;
	os << dnsentry->name;
      }
      else
	os << "unknown hostname";
      os << ')';
    }

    if (conf->config["whois_lookup"].toint() != WHOIS_NO &&
	entry->dipaddr.isroutable()) {
      if (conf->config["whois_lookup"].toint() == WHOIS_YES ||
	  (conf->config["whois_lookup"].toint() == WHOIS_IFNODNS &&
	   dst_resolved == false)) {
	wf_whois_entry* we = whois->whois(entry->dipaddr);
	os << " (";
	if (we != NULL)
	  os << *we;
	else
	  os << "no whois entry";
	os << ')';
      }
    }
  }
  
  if (conf->config["dst_port"].tobool()) {
    if (entry->protocol == IPPROTO_UDP || entry->protocol == IPPROTO_TCP) {
      os << " port " << entry->dport;
      
      if (conf->config["sresolve"].tobool()) {
	string service = wf_port_toservice(entry->dport, proto);
	os << " (";
	if (service.empty())
	  os << "unknown service name";
	else
	  os << service;
	os << ')';
      }
    }
  }
  
  if (entry->protocol == IPPROTO_TCP && conf->config["tcpflags"].tobool())
    os << " with TCP flags" << entry->tcpflags_tostr();

  return true;
}

bool
wf_outmodule_human::print(const wf_logentries& logentries, ostream& os) {
  unsigned char verbose = conf->config["verbose"].toint();
  int lines = conf->config["lines"].toint();

  wf_logentries* newlogentries;
  if (lines) { /* copy n first logentries */
    newlogentries = new wf_logentries();
    if (newlogentries == NULL)
      return false;
    list1_firstncopy(newlogentries->elems, logentries.elems, lines);
  }
  else /* do not copy logentries */
    newlogentries = (wf_logentries*)&logentries;

  if (conf->config["whois_lookup"].toint() != WHOIS_NO) {
    if (whois == NULL) {
      whois = new wf_whois(verbose);
      if (whois == NULL) {
	if (lines)
	  delete newlogentries;
	return false;
      }
    }
    if (whois->isconnected() == false && whois->connect() == false) {
      if (lines)
	delete newlogentries;
      return false;
    }
  }
  
  if (conf->config["resolve"].toint() == RESOLV_YES) {
    if (dns == NULL) {
      dns = new wf_dns(verbose);
      if (dns == NULL) {
	if (lines)
	  delete newlogentries;
	return false;
      }
    }
    preresolv(*newlogentries, dns,
	      conf->config["src_ip"].tobool(), conf->config["dst_ip"].tobool(),
	      verbose);
  }

  if (conf->config["header"].tobool())
    output_header(os);
  if (wf_outmodule::print(*newlogentries, os) == false) {
    if (lines)
      delete newlogentries;
    return false;
  }
  if (conf->config["header"].tobool())
    output_footer(os);

  if (conf->config["whois_lookup"].toint() != WHOIS_NO)
    whois->close();
  if (lines)
    delete newlogentries;

  return true;
}

extern "C" wf_outmodule*
wf_outmodule_human_init() {
  return new wf_outmodule_human();
}
