//<copyright>
//
// Copyright (c) 1993,94,95,96,97
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>

//<file>
//
// Name:        camera.C
//
// Purpose:     implementation of class Camera
//
// Created:      4 May 95   Michael Pichler (extracted from SDFCamera)
//
// Changed:     20 Dec 96   Karl Heinz Wagenbrunn (stereo view)
//
// Changed:     21 Feb 97   Alexander Nussbaumer (editing)
//
// Changed:     23 Apr 97   Michael Pichler
//
// $Id: camera.C,v 1.23 1997/05/05 12:18:40 bmarsch Exp $
//
//</file>


#include "camera.h"
#include "scene3d.h"

#include "vecutil.h"
#include <ge3d/ge3d.h>
#include "svbsptree.h"

#include <math.h>
#include <iostream.h>


float Camera::collisionDistance_ = 1.5;
// convergence plane as fraction of far clipping plane
float Camera::convergencePlane_ = 0.33;
// half eye distance
float Camera::screeneye_ = 0.3;

Camera::Camera ()
{
  reset ();  // default view

  useupvec_ = 1;
  projtype_ = 'P';
  focallen_ = 1.0;
  aper_ = 1.0;  // to check (bit more wideangle than VRML default camera)
  aspect_ = 4.0/3.0;  // ignored (window aspect ratio used instead)
  hither_ = 0.1;
  yon_ = 100.0;

  pos_look_set_ = 0;  // only relevant for SDFCamera:
  // use channel values for camera position/lookat by default
}


Camera::~Camera ()
{ // some compilers confused with virtual inline
}


void Camera::reset ()
{
  // default view (choosen to produce an identity view transformation)
  init3D (position_, 0, 0, 0);
  init3D (lookat_, 0, 0, -1);
  init3D (up_, 0, 1, 0);  // untilted
  // perspective parameters unchanged
}


void Camera::setCamera (const Scene3D* scene, int eye)
{
  if (scene)
  { aspect_ = scene->getWinAspect ();
    useupvec_ = scene->arbitraryRotations ();
  }

  // TODO: possibly change up vector
  if (projtype_ == 'O')
    ge3d_ortho_cam (
      &position_, &lookat_, useupvec_ ? &up_ : (const point3D*) 0,
      aper_ * aspect_, aper_, 0.0, yon_
    );
  else
    ge3d_setcamera (
      &position_, &lookat_, useupvec_ ? &up_ : (const point3D*) 0,
      aper_, focallen_, aspect_, hither_, yon_, eye, convergencePlane_, screeneye_
    );
}



void Camera::print ()
{
  // print information about the camera
  cout << "  projtype: " << projtype_ << " focallen: " << focallen_
       << " aper: " << aper_ << " aspect: " << aspect_ << endl;
  cout << "  left: " << left_ << " top: " << top_ << " right: " << right_ << " bottom: " << bottom_
       << " hither: " << hither_ << " yon: " << yon_ << endl;

  cout << endl;
}


float Camera::getfovy () const
{
  // vertical field of view (total; radians)
  return 2 * atan2 (aper_, 2 * focallen_);
}


// translate (const point3D&, ...)
// move whole camera by a vector
// scene must be known for collision detection,
// slide flag tells whether to "slide" along surface when colliding (otherwise stopping)
// respnear flat tells wheter to respect near clipping plane on (primary) collision test

void Camera::translate (const vector3D& tran, Scene3D* scene, int slide, int respnear)
{
  if (!scene || !scene->collisionDetection ())
  { inc3D (position_, tran);
    inc3D (lookat_, tran);
    // upvector unchanged
    // cerr << "-";
    return;
  }

  const float norm = norm3D (tran);  // also used for sliding
  if (!norm)  // no translation
    return;

  vector3D ntrans = tran;
  float temp = 1.0 / norm;
  scl3D (ntrans, temp);  // normalized translation

  // check whether position_ + tran (i.e. position_ + norm * ntrans)
  // is still colltime away from next object (along ray ntrans)

  point3D hitpoint;
  vector3D normal;
  float hittime;

  int hit = (scene->pickObject (
    position_, ntrans, respnear ? hither_ : 0, yon_,
    hitpoint, normal, &hittime) != 0);

  float colltime = 1.1 * hither_;  // collisionDistance_, at least 1.1 times hither
  if (collisionDistance_ > colltime)
    colltime = collisionDistance_;

  if (!hit)  // no ray intersection
  { inc3D (position_, tran);
    inc3D (lookat_, tran);
    // cerr << "_"; // no intersection
    return;
  }

  vector3D delta = position_;  // pointing from to hitpoint to position
  dec3D (delta, hitpoint);

  float ndist = dot3D (delta, normal);  // perpendicular distance of position from object hit
  // moving forward  x * ntrans  the distance becomes  (hittime - x) / hittime * ndist

  if (hittime && ndist && (hittime - norm) / hittime * ndist < colltime)  // collision
  {
    // might move onto collision plane here

    if (slide)
    {
      // slide along a vector perpendicular to normal, in direction "near" ntrans
      vector3D nslidev, slidevec;
      crp3D (ntrans, normal, nslidev);  // perp. help vector
      crp3D (normal, nslidev, slidevec);

      temp = norm3D (slidevec);
      if (!temp)
        return;
      nslidev = slidevec;
      temp = 1.0 / temp;
      scl3D (nslidev, temp);  // normalized

      // float snorm = norm;  // to slide same distance as would have walked forward
      // slide distance: length of projection of tran onto slidevec
      float snorm = dot3D (tran, slidevec) * temp * temp;  // (tran . slidevec) / (slidevec . slidevec)

      // check we are not colliding (hitpoint and normal vector not needed here)
      hit = (scene->pickObject (position_, nslidev, 0, yon_, 0, 0, 0, 0, 0, 0, 0, &hittime) != 0);

      if (!hit || hittime > snorm + colltime)  // no coll. on sliding
      { scl3D (nslidev, snorm);
        inc3D (position_, nslidev);
        inc3D (lookat_, nslidev);
        // cerr << "*";  // sliding
      }
      // else cerr << "X";  // stopping

    } // sliding
  }
  else
  { inc3D (position_, tran);
    inc3D (lookat_, tran);
    // cerr << "+";  // no collision, moving forward
  }

} // translate



// translate (float x, float y, ...)
// move whole camera parallel to the viewplane
// x and y are fractions of window width and height
// corresponds to movement of plane at distance flen

void Camera::translate (float x, float y, float winasp, float flen, Scene3D* scene)
{
  vector3D look_pos, u, v;  // pointing to the right/upwards on the viewplane

  sub3D (lookat_, position_, look_pos);  // lookat - position
  crp3D (look_pos, up_, u);  // points to right on viewplane
  crp3D (u, look_pos, v);   // points upwards on viewplane

  // work with a window at distance flen instead of focallen
  // focallen is most often not related to the size of the scene

  float winheight = aper_ / focallen_;
  if (projtype_ != 'O')
    winheight *= flen;

  x *= winasp * winheight / norm3D (u);  // x *= windowwidth / norm (u)
  y *= winheight / norm3D (v);  // y *= windowheight / norm (v)

  vector3D tran;  // translation
  lcm3D (x, u, y, v, tran);  

  translate (tran, scene, 0, 0);  // collision detection, but no sliding, near clip irrelevant
} // translate



// zoom_in
// move camera along straight line from position towards lookat
// (distance dist in world coordinates)

void Camera::zoom_in (float dist, Scene3D* scene)
{
  vector3D look_pos;
  sub3D (lookat_, position_, look_pos);  // lookat - position

  float factor = dist / norm3D (look_pos);

  scl3D (look_pos, factor);

  translate (look_pos, scene, 1, 1);  // collision detection with sliding, respect near clip
} // zoom_in



/* common to all rotations:
   to avoid accumulation of rounding errors
   distance of lookat from position is normalized to 1.0
*/


// rotate_camera_right
// rotate camera left to right (around position, axis up_)
// (angle phi in radians)
// name grown historically, rotate_horicontal would be better

void Camera::rotate_camera_right (float phi)
{
  double c = cos (phi), s = sin (phi);

  vector3D look_pos, u;

  sub3D (lookat_, position_, look_pos);  // lookat - position
  crp3D (look_pos, up_, u);  // points to right on viewplane
  c /= norm3D (look_pos);  // normalize
  s /= norm3D (u);
  lcm3D (c, look_pos, s, u, look_pos);

  double factor = 1 / norm3D (look_pos);
  // set distance between position and lookat to 1

  pol3D (position_, factor, look_pos, lookat_);  // set new lookat

} // rotate_camera_right



// rotate_camera_up
// rotate camera upwards (around position, axis look_pos X up_)
// (angle phi in radians)
// name grown historically, rotate_vertical would be better

void Camera::rotate_camera_up (float phi)
{
  double c = cos (phi), s = sin (phi);

  vector3D look_pos, u, v;

  // look-pos and v must be normalized!
  sub3D (lookat_, position_, look_pos);  // lookat - position
  crp3D (look_pos, up_, u);  // points to right on viewplane
  crp3D (u, look_pos, v);   // points upwards on viewplane
  c /= norm3D (look_pos);  // normalize
  s /= norm3D (v);
  lcm3D (c, look_pos, s, v, look_pos);
  // new up vector
  crp3D (u, look_pos, up_);

  double factor = 1 / norm3D (look_pos);
  // set distance between position and lookat to 1
  pol3D (position_, factor, look_pos, lookat_);  // set new lookat

  factor = 1 / norm3D (up_);
  scl3D (up_, factor);  // normalize up vector

} // rotate_camera_up



// rotate_camera
// rotate camera around an arbitrary center
// angle l2r left-to-right, b2t bottom-to-top,
// both in radians as seen on the viewplane

void Camera::rotate_camera (float l2r, float b2t, const point3D& center)
{
  // ignore anything on matrix stack
  ge3dPushIdentity ();

  // compute u and v
  vector3D look_pos, u, v;
  sub3D (lookat_, position_, look_pos);  // lookat - position
  crp3D (look_pos, up_, u);  // points to right on viewplane
  crp3D (u, look_pos, v);   // points upwards on viewplane

  point3D uppoint;  // transform up point instead of vector
  add3D (up_, position_, uppoint);

  // move to origin
  dec3D (position_, center);
  dec3D (lookat_, center);
  dec3D (uppoint, center);

  if (l2r)
    ge3dRotate (&v, l2r);  // horicontal rotation
  if (b2t)
    ge3dRotate (&u, -b2t);  // vertical rotation (-u, b2t)

  // apply transformation
  ge3dTransformVectorMcWc (&position_, &position_);
  ge3dTransformVectorMcWc (&lookat_, &lookat_);
  ge3dTransformVectorMcWc (&uppoint, &uppoint);

  // back to original coordinate system
  inc3D (position_, center);
  inc3D (lookat_, center);
  inc3D (uppoint, center);

  // restore upvector
  sub3D (uppoint, position_, up_);

  // set distance between position and lookat to 1
  vector3D pt_center;
  sub3D (lookat_, position_, pt_center);  // lookat - position
  double factor = 1 / norm3D (pt_center);
  pol3D (position_, factor, pt_center, lookat_);  // set new lookat

  ge3d_pop_matrix ();

  factor = 1 / norm3D (up_);
  scl3D (up_, factor);  // normalize up vector

} // rotate_camera (around center)



// rotate_axis
// rotation about arbitrary center and axis
// contributed by drg@spacetec.com
// angle in radians, CCW (one might use norm3D (axis) as rotation angle)

void Camera::rotate_axis (const point3D& center, const vector3D& axis, float angle)
{
  ge3dPushIdentity ();

  // compute u and v
  vector3D look_pos, u, v;
  sub3D (lookat_, position_, look_pos);  // lookat - position
  crp3D (look_pos, up_, u);  // points to right on viewplane
  crp3D (u, look_pos, v);   // points upwards on viewplane

  point3D uppoint;  // transform up point instead of vector
  add3D (up_, position_, uppoint);

  // move to origin
  dec3D (position_, center);
  dec3D (lookat_, center);
  dec3D (uppoint, center);

  ge3dRotate (&axis, angle);

  // apply transformation
  ge3dTransformVectorMcWc (&position_, &position_);
  ge3dTransformVectorMcWc (&lookat_, &lookat_);
  ge3dTransformVectorMcWc (&uppoint, &uppoint);

  // back to original coordinate system
  inc3D (position_, center);
  inc3D (lookat_, center);
  inc3D (uppoint, center);

  // restore upvector
  sub3D (uppoint, position_, up_);

  // set distance between position and lookat to 1
  vector3D pt_center;
  sub3D (lookat_, position_, pt_center);  // lookat - position
  double factor = 1 / norm3D (pt_center);
  pol3D (position_, factor, pt_center, lookat_);  // set new lookat

  ge3d_pop_matrix ();

  factor = 1 / norm3D (up_);
  scl3D (up_, factor);  // normalize up vector

} // rotate_axis



// rotate_
// rotate a vector vec around 'axis'
// by angle c = cos (phi), s = sin (phi), CCW


static void rotate_ (vector3D& vec, double c, double s, char axis)
{
  float* from,
       * to;

  switch (axis)
  { case 'x':  case 'X':
      from = &vec.y;  to = &vec.z;
    break;
    case 'y':  case 'Y':
      from = &vec.z;  to = &vec.x;
    break;
    default:  // case 'z':  case 'Z':
      from = &vec.x;  to = &vec.y;
    break;
  }

  float oldval = *from;

  *from = c * oldval - s * *to;
  *to   = c * *to    + s * oldval;

} // rotate_



// rotate_caxis
// rotate whole camera around point center
// along an axis, CCW

void Camera::rotate_caxis (float phi, const point3D& center, char axis)
{
  vector3D vec;
  double c = cos (phi),
         s = sin (phi);

  // rotate position
  sub3D (position_, center, vec);  // vec = position - center
  rotate_ (vec, c, s, axis);
  add3D (vec, center, position_);

  // rotate lookat
  sub3D (lookat_, center, vec);  // vec = lookat - center
  rotate_ (vec, c, s, axis);
  add3D (vec, center, lookat_);

  point3D uppoint;  // transform up point instead of vector
  add3D (up_, position_, uppoint);
  // rotate uppoint
  sub3D (uppoint, center, vec);  // vec = lookat - center
  rotate_ (vec, c, s, axis);
  add3D (vec, center, uppoint);
  // restore upvector
  sub3D (uppoint, position_, up_);

} // rotate_caxis



// makeHoricontal
// rotate around position such that position and lookat are on same level

void Camera::makeHoricontal ()
{
  vector3D look_pos, new_look_pos;

  sub3D (lookat_, position_, look_pos);  // lookat - position

  new_look_pos = look_pos;
  new_look_pos.y = 0;

  double newlensquare = dot3D (new_look_pos, new_look_pos);

  if (newlensquare > 0.0)
  { double factor = sqrt (dot3D (look_pos, look_pos) / newlensquare);
    // maintain distance between position and lookat
    // set new lookat
    pol3D (position_, factor, new_look_pos, lookat_);
  }
  // else: line of sight was parallel to y-axis - no change

  init3D (up_, 0, 1, 0);  // untilt as well

} // makehoricontal


void Camera::untilt ()
{
  init3D (up_, 0, 1, 0);  // default up vector
}


// viewingRay
// build a ray (eye + t * look) from a point on the viewing window

void Camera::viewingRay (
  float fx, float fy,
  point3D& eye, vector3D& look,
  float& tnear, float& tfar
)
{
  eye = position_;
  look = lookat_;
  vector3D up;
  if (useupvec_)
    up = up_;
  else
    init3D (up, 0, 1, 0);

//cerr << "eye = " << eye << ", lookat = " << look << endl;

  vector3D u, v, n;     // camera coordinate system (orthonormal)
  point3D vppoint,      // point of interest in world coordinates
          ref;          // view reference point (midpoint of viewplane)
  float temp;

  // setup camera coordinate system (normalize later)
  sub3D (look, eye, n);                 // n = look - pos; from eye to lookat, normal to viewplane
  crp3D (n, up, u);                     // u = n X up;     "to the right" on viewplane
  crp3D (u, n, v);                      // v = u X n;      "upwards" on viewplane

//cerr << "u = " << u << ", v = " << v << ", n = " << n << endl;

  if (projtype_ == 'O')  // orthographic camera
  {
//position, lookat, aper*aspect, aper, 0.0, yon
    temp = (fx - 0.5) * aper_ * aspect_ / norm3D (u);
    scl3D (u, temp);
    temp = (fy - 0.5) * aper_ / norm3D (v);
    pol3D (u, temp, v, u);
    // cerr << "translation: " << u << endl;
    inc3D (eye, u);   // eye += u
    inc3D (look, u);  // look += u

    dec3D (look, eye);  // ray direction: look - eye
    tnear = 0;  // see setCamera
    tfar = yon_ / norm3D (n);
    return;
  } // orthographic camera

  // perspective camera

  // compute view reference point (midpoint of viewplane) 
  temp = focallen_ / norm3D (n);
  pol3D (eye, temp, n, ref);

  // compute vppoint (point on viewplane)
  temp = (fx - 0.5) * aper_ * aspect_ / norm3D (u);
  pol3D (ref, temp, u, vppoint);
  temp = (fy - 0.5) * aper_ / norm3D (v);
  pol3D (vppoint, temp, v, vppoint);

  // eye = position_;
  sub3D (vppoint, eye, look);   // ray direction: vppoint - eye
  tnear = hither_ / focallen_;  // ray reaches near clipping plane
  tfar = yon_ / focallen_;      // ray reaches

} // viewingRay


// viewingPoint (anuss)
// calculate point in view coordinates
// (valid when only Projection part of transformation is active,
// i.e. View transformation set to identity)

void Camera::viewingPoint (
  float fx, float fy,
  point3D& A
)
{
  A.x = (fx - 0.5) * aper_ * aspect_;
  A.y = (fy - 0.5) * aper_;

  if (projtype_ == 'O')  // orthographic camera
  {
    A.z = - yon_ / 2.0;  // ???
  }
  else  // perspective camera
  {
    A.z = (hither_ + yon_) / 2.0;
    float f = A.z / focallen_;  // (near + far) / (2 * focallen)
    A.x *= f;
    A.y *= f;
    A.z = - A.z;  // default camera looks towards -z
  }

  // prevent A from being clipped off
//   if (hither_ > 10 || yon_ < 30)
//     A.z = - A.z;  // between hither and yon
//   else
//     A.z = - 20;

//cerr << "hither: " << hither_ << ", yon: " << yon_ << ". point: " << A << endl;

} // viewingPoint



// perspectiveCam
// turn camera into a perspective one

void Camera::perspectiveCam (float yangle)  // height angle (radians)
{
  reset ();
  projtype_ = 'P';
  focallen_ = 1.0;  // not relevant for camera
  aper_ = 2.0 * tan (yangle / 2.0);
  // hither, yon unchanged
}


// orthographicCam
// turn camera into an orthographic one

void Camera::orthographicCam (float height)
{
  reset ();
  projtype_ = 'O';
  focallen_ = 1.0;  // not relevant for camera
  aper_ = height;
  // hither, yon unchanged; although orthographic camera uses hither of 0.0
}


void Camera::getViewVolume (Plane view_volume[6]) const  // gmes, 19960505    
{
  vector3D x, y, z; // z really is -z
  float norm;

  // get the camera coordinate system
  sub3D(lookat_, position_, z);   // normal vector from eye to viewplane, -z axes
  scl3D(z, 1/focallen_);  // normalize the z vector

  crp3D(z, up_, x);
  norm = norm3D(x);
  scl3D(x, 1/norm);

  crp3D(x, z, y);
  norm = norm3D(y);
  scl3D(y, 1/norm);

  // front clipping plane
  point3D front_point;
  pol3D(position_, hither_, z, front_point);
  view_volume[0].normal = z;  
  view_volume[0].d = dot3D(view_volume[0].normal, front_point);

  // back clipping plane
  vector3D normal_back;
  point3D back_point;
  pol3D(position_, yon_, z, back_point);
  normal_back = z;
  neg3D(normal_back);
  view_volume[1].normal = normal_back;
  view_volume[1].d = dot3D(view_volume[1].normal, back_point);

  // vertices of the camera window on the viewplane
  vector3D point_top_left, point_top_right, point_bottom_left, point_bottom_right;
  point3D top_left, top_right, bottom_left, bottom_right;

  float ylen = aper_/2;
  float xlen = aspect_ * ylen;

  lcm3D(ylen, y, -xlen, x, point_top_left);
  lcm3D(-ylen, y, -xlen, x, point_bottom_left);
  point_top_right = point_bottom_left;
  neg3D(point_top_right);
  point_bottom_right = point_top_left;
  neg3D(point_bottom_right);

  add3D(lookat_, point_top_left, top_left);
  add3D(lookat_, point_bottom_left, bottom_left);
  add3D(lookat_, point_top_right, top_right);
  add3D(lookat_, point_bottom_right, bottom_right);

/*
  Clipping::print3D(position_);
  Clipping::print3D(top_left);
  Clipping::print3D(bottom_left);
  Clipping::print3D(top_right);
  Clipping::print3D(bottom_right);
*/  

  // top clipping plane
  SVBSPTree::calcPlaneEqu(position_, top_left, top_right, view_volume[2]);
  // bottom clipping plane
  SVBSPTree::calcPlaneEqu(position_, bottom_right, bottom_left, view_volume[3]);
  // left clipping plane
  SVBSPTree::calcPlaneEqu(position_, bottom_left, top_left, view_volume[4]);
  // right clipping plane
  SVBSPTree::calcPlaneEqu(position_, top_right, bottom_right, view_volume[5]);
   
} // getViewVolume



/**********  object manipulation computations (anuss) *************/


// fabsf is not very common. bm,  5 May 97: Linux, BSDI
#if defined(SUN) || defined(HPUX) || defined(AIX) || defined(LINUX) || defined(BSDI)
# define fabsf fabs
#endif


// normalizes a 2D vector

static void normalizeVector2D (vector2D& v)
{
  float length = norm2D (v);
  if (length)
  {
    v.x /= length;
    v.y /= length;
  }
} // normalizeVector2D


// TODO comment

void Camera::setWindowXY (const matrix4D seltran, float pressfx, float  pressfy)
{
  // (matrix4D passed by reference)

  // calculate dominant object axis on screen

  objectx_ = 0;
  for (int i = 1;  i < 3;  i++)
    if (fabsf (seltran[i][0]) > fabsf (seltran[objectx_][0]))
      objectx_ = i;

  objecty_ = (objectx_ + 1) % 3;
  for (int i = 0;  i < 3;  i++)
    if (i != objectx_ && fabsf (seltran[i][1]) > fabsf (seltran[objecty_][1]))
      objecty_ = i;

  object0_ = 0;
  for (int i = 1;  i < 3;  i++)
    if (i != objectx_ && i != objecty_)
      object0_ = i;

  // use sign of object vectors for oriantation on screen
  signobjectx_ = (seltran[objectx_][0] < 0) ? -1 : 1;
  signobjecty_ = (seltran[objecty_][1] < 0) ? -1 : 1;
  signobject0_ = (seltran[object0_][2] < 0) ? -1 : 1;


  firstdirection_ = 0;  // reset first mouse drag direction
  dragx_ = dragy_ = 0;  // perspective window dragpostition (for absolute drag direction)

  // copy transformation matrix to winvector matrix
  winvectorx_ = *(const vector3D*) seltran[objectx_];  // seltran: pointer to 4x4 matrix
  winvectory_ = *(const vector3D*) seltran[objecty_];
  winvector0_ = *(const vector3D*) seltran[object0_];


  if ((objectx_ == objecty_) || (objecty_ == object0_) || (objectx_ == object0_)) 
  {
    cerr << "Camera::setWindowXY. internal error." << endl;
    cerr << seltran;
//     vector3D* a = (vector3D*)seltran[0];
//     vector3D* b = (vector3D*)seltran[1];
//     vector3D n;
//     crp3D (*a, *b, n); // that was always ok
//     cerr << n << endl << endl;

    cerr << winvectorx_ << endl << winvectory_ << endl << winvector0_ << endl;
    cerr << objectx_ << " " << objecty_ << " " << object0_ << endl << endl;
  }


  point3D A;
  viewingPoint (pressfx, pressfy, A);

  // calculate perspective drag vectors for window (2D projection of 3D winvectors)
  dragwinx_.x = (A.x + winvectorx_.x)/(A.z + winvectorx_.z) - A.x/A.z;
  dragwinx_.y = (A.y + winvectorx_.y)/(A.z + winvectorx_.z) - A.y/A.z;
 
  dragwiny_.x = (A.x + winvectory_.x)/(A.z + winvectory_.z) - A.x/A.z; 
  dragwiny_.y = (A.y + winvectory_.y)/(A.z + winvectory_.z) - A.y/A.z;

  // normalize vectors (maybe not necessary)
  normalizeVector2D (dragwinx_);
  normalizeVector2D (dragwiny_);

  // use positive x/y orientations
  if (dragwinx_.x < 0) 
  { dragwinx_.x = -dragwinx_.x;
    dragwinx_.y = -dragwinx_.y;
  }
  if (dragwiny_.y < 0)
  { dragwiny_.x = -dragwiny_.x;
    dragwiny_.y = -dragwiny_.y;
  }

  // determinant of dragwin matrix
  detwinxy_ = dragwinx_.x*dragwiny_.y - dragwinx_.y*dragwiny_.x;  // TODO may not be zero
  if (detwinxy_ == 0)
    detwinxy_ = 1.0;  // TODO consider that
} // setWindowXY


// returns three 3D Vectors describing the object orientation

void Camera::getWinVectors (vector3D& wx, vector3D& wy, vector3D& w0)
{
//cerr << "winvectors: " << winvectorx_ << ", " << winvectory_ << ", " << winvector0_ << endl;

  wx = winvectorx_;
  wy = winvectory_;
  w0 = winvector0_;
} // getWinVectors

// returns one 3D vecotor describing the object orientation (the front face)

void Camera::getWinVector (vector3D& vector, vector3D& sign)
{
  vector.x = objectx_;
  vector.y = objecty_;
  vector.z = object0_;

  sign.x = signobjectx_;
  sign.y = signobjecty_;
  sign.z = signobject0_;

} // getWinVector


// calculates a 3D translation vector from 2D mouse mouvement  

void Camera::getTranslationVector (float delta_x, float delta_y, float f, vector3D& v3D)
{
  vector2D delta = { delta_x, delta_y };
  float dragx = (dragwiny_.y*delta.x - dragwiny_.x*delta.y) / detwinxy_,
        dragy = (-dragwinx_.y*delta.x + dragwinx_.x*delta.y) / detwinxy_;

  // 3D vector
  float* v = &v3D.x;
  v[objectx_] = signobjectx_*dragx*f; // *normlookat_;
  v[objecty_] = signobjecty_*dragy*f; // *normlookat_;
  v[object0_] = 0;

  // for the case: while draging: control is pressed, released and once more pressed
  // firstdirection_ = 0;
} // getTranslationVector


// same as above, but only one mouse direction is used for calculation

void Camera::getTranslationAxisVector (float delta_x, float delta_y, float f, vector3D& v3D, int thirdaxis)
{
  init3D (v3D, 0, 0, 0);
  float* v = &v3D.x;

  if (thirdaxis)
  { v[object0_] = - delta_y * f * signobject0_;
    return;
  }

  vector2D delta = { delta_x, delta_y };
  float dragx = (dragwiny_.y*delta.x - dragwiny_.x*delta.y) / detwinxy_,
        dragy = (-dragwinx_.y*delta.x + dragwinx_.x*delta.y) / detwinxy_;

  // compute single movement direction
  if (! firstdirection_)
    if (fabsf (dragx) > fabsf (dragy))
      firstdirection_ = 'x';
    else
      firstdirection_ = 'y';

  // 3D vector
  if (firstdirection_ == 'x')
    v[objectx_] = signobjectx_*dragx*f; // *normlookat_;
  else
    v[objecty_] = signobjecty_*dragy*f; // *normlookat_;

} // getTranslationAxisVector 


// calcultes a 3D rotation vector from a 2D mouse movement

void Camera::getRotationVector (float delta_x, float delta_y, vector3D& axis, float& angle)
{
  vector2D delta = { delta_x, delta_y };
  float dragx = (dragwiny_.y*delta.x - dragwiny_.x*delta.y) / detwinxy_,
        dragy = (-dragwinx_.y*delta.x + dragwinx_.x*delta.y) / detwinxy_;

  // calculate rotation vector from absolute mouse movement (origin - current position)
/*
  dragx_ += dragx;
  dragy_ += dragy;
*/
  dragx_ = dragx;
  dragy_ = dragy;

  vector3D v0 = { 0, 0, 0 },
           v1 = { 0, 0, 0 };
  float a0 = 0, a1 = 0;

  // rotation though x-axis
  float* v = &v0.x;
  v[objectx_] = -signobjectx_;  // (1 or -1)
  v[objecty_] = 0;
  v[object0_] = 0;
  a0 = dragy_;

  // rotation though y-axis
  v = &v1.x;
  v[objectx_] = 0;
  v[objecty_] = signobjecty_;  // 1 or -1
  v[object0_] = 0;
  a1 = dragx_;


  axis.x = 0;
  axis.y = 0;
  axis.z = 0;

  // combine both rotations (v0 and v1 are normalized)
  multiplyRotations (v0, a0, v1, a1, axis, angle);  

  // for the case: while draging: control is pressed, release and once more pressed
  //  firstdirection_ = 0;

} // getRotationVector


// same as above, but only one mouse direction is used for calculation

void Camera::getRotationAxisVector (float delta_x, float delta_y, vector3D& axis, float& angle, int thirdaxis)
{
  init3D (axis, 0, 0, 0);
  float* v = &axis.x;

  if (thirdaxis)
  { v[object0_] = -signobject0_; 
    angle = delta_x;
    /*
    dragx_ += delta_x;
    angle = dragx_;
    */
    return;
  }

  vector2D delta = { delta_x, delta_y };
  float dragx = (dragwiny_.y*delta.x - dragwiny_.x*delta.y) / detwinxy_,
        dragy = (-dragwinx_.y*delta.x + dragwinx_.x*delta.y) / detwinxy_;

  dragx_ = dragx;
  dragy_ = dragy;
/*
  dragx_ += dragx;
  dragy_ += dragy;
*/
  if (! firstdirection_)
    if (fabsf (dragx) > fabsf (dragy))
      firstdirection_ = 'x';
    else
      firstdirection_ = 'y';

  if (firstdirection_ == 'x')
  { v[objecty_] = signobjecty_;  // 1 or -1
    angle = dragx_;
  }
  else
  { v[objectx_] = -signobjectx_;  // 1 or -1
    angle = dragy_;
  }
} // getRotationAxisVector


// calculate scaling vector (three scaling factors) from relative mouse movement

void Camera::getScaleVector (float delta_x, float delta_y, vector3D& scale)
{
  float delta;
  if (fabsf (delta_x) > fabsf (delta_y))
    delta = delta_x;
  else 
    delta = delta_y;

  scale.x = delta;
  scale.y = delta;
  scale.z = delta;
} // getScaleVector


// same as above, but only one mouse direction is used for calculation

void Camera::getScaleAxisVector (float delta_x, float delta_y, vector3D& scale, int thirdaxis)
{
  init3D (scale, 0, 0, 0);
  float* v = &scale.x;

  if (thirdaxis)
  { v[object0_] = delta_y;
    return;
  }

  vector2D delta = { delta_x, delta_y };
  float dragx = (dragwiny_.y*delta.x - dragwiny_.x*delta.y) / detwinxy_,
        dragy = (-dragwinx_.y*delta.x + dragwinx_.x*delta.y) / detwinxy_;

  if (! firstdirection_)
    if (fabsf (dragx) > fabsf (dragy))
      firstdirection_ = 'x';
    else
      firstdirection_ = 'y';

  if (firstdirection_ == 'x')
    v[objectx_] = dragx;
  else
    v[objecty_] = dragy;

  // for the case: while draging: control is pressed, release and once more pressed
  // firstdirection_ = 0;
} // getScaleAxisVector
