/* Logical UPS line numbers. These are used as arguments to the
   UPS_Assert() and UPS_Deassert() functions, which map 'logical' UPS
   lines to physical inputs/outputs. */
#define L_UPSPWR  0
#define L_KILL    1
#define L_ONBAT   2
#define L_LOBAT   3
#define L_ERR     4

#define B_UPSPWR  (1<<L_UPSPWR)  /* PC output: power to UPS status pullups */
#define B_KILL    (1<<L_KILL)    /* PC output: kill UPS power */
#define B_ONBAT   (1<<L_ONBAT)   /* PC input: UPS is on battery (AC fail) */
#define B_LOBAT   (1<<L_LOBAT)   /* PC input: UPS has low battery */
#define B_ERR     (1<<L_ERR)     /* PC input: cable detect/UPS error */

/* maximum string storage for UPS line parameters */
#define MAXLEN 12
/* maximum string length for file input */
#define MAXLINELEN 82

typedef struct _upsmap {
  int ioctl_mask;       /* actual bitmask to diddle line (as in termios.h) */
  char serial_pin[MAXLEN];       /* name of serial pin */
  char ups_funcname[MAXLEN];     /* meaning of this pin to UPS */
  char active_state;     /* 'H' or 'L' */
} UPS_Map;

/*-------------------------------------------------------------------*
     Config_Read

     Read UPS pin assignments from config file.

     Parameters: char *fname - filename (incl. path) of config file
                 int test_mode - whether we're in test mode or not

     Returns: FALSE if config file couldn't be read or has improper
              syntax; TRUE otherwise.
 *-------------------------------------------------------------------*/
int Config_Read(char *fname, int test_mode);

/*-------------------------------------------------------------------*
     Serial_Set

     Set specified lines on the serial port.

     Parameters:  int fd   - File descriptor of monitor device.
                  int lstate - bit mask of handshake outputs to set.

     Returns:     Nothing.
 *-------------------------------------------------------------------*/
void Serial_Set (int fd, int lstate);

/*-------------------------------------------------------------------*
     Serial_Clear

     Clear specified lines on the serial port.

     Parameters:  int fd   - File descriptor of monitor device.
                  int lstate - bit mask of handshake outputs to clear.

     Returns:     Nothing.
 *-------------------------------------------------------------------*/
void Serial_Clear (int fd, int lstate);

/*-------------------------------------------------------------------*
     UPS_Read

     Read UPS physical lines and convert into active-high logical
     status.

     Parameters: fd - file handle of UPS device

     Returns: logical status  of each UPS line.
 *-------------------------------------------------------------------*/
int UPS_Read(int fd);

/*-------------------------------------------------------------------*
     UPS_Deassert

     De-asserts a UPS line.

     Parameters: int fd - file handle of UPS device
                 int logical_line - logical UPS line to de-assert

     Returns: <none>

     NOTE: References global table UPSMapping[] to map logical_line to
           a physical UPS line.
 *-------------------------------------------------------------------*/
void UPS_Deassert (int fd, int logical_line);

/*-------------------------------------------------------------------*
     UPS_Assert

     Asserts a UPS line.

     Parameters: int fd - file handle of UPS device
                 int logical_line - logical UPS line to assert

     Returns: <none>

     NOTE: References global table UPSMapping[] to map logical_line to
           a physical UPS line.
 *-------------------------------------------------------------------*/
void UPS_Assert (int fd, int logical_line);
