/* ==================================================== ======== ======= *
 *
 *  uunatdisp.cpp  [Native Layer: platform dependent implementation]
 *  Ubit Project [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uunatdisp.cpp	ubit:03.06.04"
#include "config.h"
#include <iostream>
#include <stdio.h>
#include <math.h>
#include <ubrick.hpp>
#include <ucall.hpp>
#include <uerror.hpp>
#include <ufont.hpp>
#include <ucolor.hpp>
#include <ucursor.hpp>
#include <unatdisp.hpp>
#include <ucolorImpl.hpp>
#include <uappli.hpp>
#include <uflow.hpp>
#include <X11/Xatom.h>
#include <uwin.hpp>
#include <uwinImpl.hpp>
#include <ugraph.hpp>
#include <update.hpp>
#include <utextsel.hpp>
#include <umsclient.hpp>
#include <umsproto.hpp>
using namespace std;

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UNatDisp::UNatDisp(UDisp& d) :
  disp(d),
  id(d.id)
{
  stat = UDisp::NotOpened;

  xdisplay = XOpenDisplay(disp.display_name.chars());
  if (!xdisplay) {
    stat = UDisp::CannotOpen;
    UError::error("UNatDisp::UNatDisp",UError::Cant_open_display,
		  disp.display_name);
    return;
  }

  xscreen  = DefaultScreenOfDisplay(xdisplay);
  if (!xscreen) {
    stat = UDisp::CannotOpen;
    UError::error("UNatDisp::UNatDisp", UError::Invalid_screen,
		  disp.display_name);
    return;
  }

  xconnection = XConnectionNumber(xdisplay);
  xcmap    = DefaultColormapOfScreen(xscreen);
  depth    = DefaultDepthOfScreen(xscreen);
  xvisual  = DefaultVisualOfScreen(xscreen);
  xwin     = None;

  // option -sync => mode synchronized
  if (d.conf.sync) XSynchronize(xdisplay, True); 

  sharedGraph = null;
  clientGraph = null;
  fontFamilyMap = null; fontFamilyCount = 0;
  defaultPixmap = None;
  server_selection_str = null;
  server_selection_pos = 0;
  
#ifdef WITH_GL
  glvisual = null;
  glxc = null;
  glFontFamilyMap = null;
  if (!initGL()) return;   // can change depth, xvisual...
  stat = UDisp::OpenGL;
#else
  if (d.conf.truecolor_depth > 0) {  // can change depth, xvisual...
    if (!setVisual(TrueColor, d.conf.truecolor_depth, d.conf.linear_gamma,
                   /*cmap*/None))
      return;
  }
  stat = UDisp::Opened;
#endif
}

/* ==================================================== ======== ======= */

UNatDisp::~UNatDisp() {
  //cerr << ">> UNATDisp::~UNATDisp " << this << endl;  // NEVER DESTROYED!!!

  if (xdisplay && defaultPixmap != None)
    XFreePixmap(xdisplay, defaultPixmap);

  // avant dans UNatAppli::~UNatAppli
  if (xdisplay) XCloseDisplay(xdisplay);
  xdisplay = null;

  //cerr << "<< UNATDisp::~UNATDisp " << this << endl;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
#ifdef WITH_GL

bool UNatDisp::initGL() {
  int screen_no = XScreenNumberOfScreen(xscreen);
  int dummy, glxmajor = 0, glxminor = 0;

  // Check if GLX is supported 
  if (!glXQueryExtension(xdisplay, &dummy, &dummy)) {
    UError::error("NatDisp::initGL","X server has no OpenGL extension");
    return false;
  }

  glXQueryVersion(xdisplay, &glxmajor, &glxminor);
  cout << "glXQueryVersion: GLX"<<glxmajor<<"."<<glxminor<<"\n";

  int depth_hint = (disp.conf.truecolor_depth > 0 ? 
		    disp.conf.truecolor_depth : depth);
		    
  int gl_options[] = {
    GLX_RGBA, GLX_DEPTH_SIZE, depth_hint,
    (disp.conf.double_buffering ? GLX_DOUBLEBUFFER : None),  // [MODE OGL_DOUBLEBUF]
    None
  };

  // find a Visual that match the requested depth and OpenGL options
  if (! (glvisual = glXChooseVisual(xdisplay, screen_no, gl_options))) {
    UError::error("NatDisp::initGL","can't get appropriate OpenGL visual");
    return false;
  }

  cout <<"GL: ChooseVisual: requested depth="<< depth_hint
       <<" actual depth="<<glvisual->depth
       <<" visual id="<<glvisual->visualid
       <<" visual class="<<glvisual->c_class 
       << (glvisual->c_class == PseudoColor ? " (PseudoColor)" 
	   : glvisual->c_class == TrueColor ? " (TrueColor)" : " (Unknown)")
       << endl;

  //NB: this function also creates the Colormap
  setVisual(*glvisual);

  // Create an OpenGL rendering context
  if (! (glxc = glXCreateContext(xdisplay, glvisual, None, True))) {
    UError::error("NatDisp::initGL","can't create OpenGL rendering context");
    return false;
  }
  
  glShadeModel(GL_FLAT);
  glDisable(GL_COLOR_MATERIAL);
  glClearDepth(1.0f);
  glClear(GL_DEPTH_BUFFER_BIT | GL_COLOR_BUFFER_BIT);
  return true;
}

void UNatDisp::resizeGLViewport(u_dim w, u_dim h) {
  //cerr << "Resize Viewport: " << " w: " << w << " h: " << h <<endl;
  // synchro necessaire entre X et GL sinon pbms de mise a jour
  // des windows dans certains cas aleatoires
  glXWaitX();
  glViewport(0, 0, w, h);
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho(0, w, -h, 0, 0.0, 1.0);
  glMatrixMode(GL_MODELVIEW);
}

#endif
/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

//  UMS [Ubit Mouse Server]
void UNatDisp::showNotify(UX_Window w, bool shows) {
  /*
  if (!disp.umsclient) return;
  if (shows) disp.umsclient->winOpened(w);
  else disp.umsclient->winClosed(w);
  */
}

int UNatDisp::getStatus() const {
  return stat;
}

bool UNatDisp::isRealized() const {
  return (xwin != None);
}

UX_Window UNatDisp::getXRootWindow() const {
  return RootWindowOfScreen(xscreen);
}

int UNatDisp::getScreenWidth() const {
  return WidthOfScreen(xscreen);
}

int UNatDisp::getScreenHeight() const {
  return HeightOfScreen(xscreen);
}

int UNatDisp::getScreenDefaultDepth() const {
  return DefaultDepthOfScreen(xscreen);
}

int UNatDisp::getScreenNumber() const {
  return XScreenNumberOfScreen(xscreen);
}

int UNatDisp::getScreenCount() const {
  return ScreenCount(xdisplay);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

static void count_bits(u_long mask, int& bits, int& shift) {
  bool in_tab = true;
  shift = 0; bits = 0;

  while (mask != 0) {
    if (in_tab) {
      if ((mask & 0x1) == 0) shift++;
      else {in_tab = false; bits++;}
    }
    else {
      if ((mask & 0x1) == 0) break;
      else bits++;
    }
    mask = mask >> 1;
  }
}

/* ==================================================== ======== ======= */

void UNatDisp::realize() {
  // si xwin != None c'est que: 
  // - soit qu'on a deja appele cette fct
  // - soit que xwin a ete fournie comme telte a l'intialisation
  //   (ie. qu'elle a ete creee par une autre appli)

  if (xwin == None) {
    // comme d'habitude, delire X bordelien typique:
    // - d'une part, pour creer une Window de depth != de celle de RootWindow,
    //   il faut obligatoirement fournir un BorderPixmap de meme depth que cette
    //   Window MEME si son BorderDepth = 0
    // - ensuite, pour creer ce BorderPixmap (qui est utilise par initWinAttr)
    //   il faut un drawable (typiquement cette Window) en argument: bref le 
    //   cercle vicieux!
    // - mais comme l'API de X est decidemment tres bien faite, en realite 
    //   ce drawable ne sert qu'a indiquer le Screen (et non la depth!)
    // - DONC, la manip qui consiste a prendre RootWindowOfScreen marche meme
    //   si depth de RootWindowOfScreen != depth du Pixmap.
    // Conclusion: on arrive toujours a trouver une bonne petite bidouille
    // pour contrer l'API merdeuse de X

    if (!xdisplay || !xscreen) {
      UError::error("fatal@UNatDisp::realize", UError::Null_display);
      return;
    }

    // default Pixmap with same depth as UNatDisp
    // !ATT: necessaire pour XCreateWindow ==> a faire AVANT!
    defaultPixmap = XCreatePixmap(xdisplay, 
				  RootWindowOfScreen(this->getXScreen()),
				  10, 10, depth);
    XSetWindowAttributes wattr;
    unsigned long wattr_mask;
    UNatWin::initWinAttributes(this, wattr, wattr_mask);

    // !!ATT: on ne devrait creer de window que pour UNatAppli!
    xwin = XCreateWindow(this->getXDisplay(),
			 RootWindowOfScreen(this->getXScreen()),//fenetre mere
			 0,0,	         // x,y position
			 10, 10,       // width , height (0,0 crashes!)
			 0,	         // border_width
			 depth,
			 InputOutput,  // Class
			 xvisual,
			 wattr_mask,
			 &wattr);

    if (xwin == None) {
      UError::error("fatal@UNatDisp::realize", UError::Cant_create_window);
      return;
    }

#ifdef WITH_GL
    glXMakeCurrent(xdisplay, xwin, glxc);
#endif

    if (xcmap == None) 
      UError::error("UNatDisp::realize", UError::Null_colormap);

    // !attention: les masks varient suivant le type de Visual
    red_mask   = xvisual->red_mask;
    green_mask = xvisual->green_mask;
    blue_mask  = xvisual->blue_mask;

    count_bits(red_mask,   red_bits,   red_shift);
    count_bits(green_mask, green_bits, green_shift);
    count_bits(blue_mask,  blue_bits,  blue_shift);
/*
    cout << "UNatDisp: " << this << " Visual: " << xvisual->visualid
	 << " Depth: " << depth << endl;
    cout << "red_mask "    << red_mask 
	 << "\t red_shift "  << red_shift
	 << "\t red_bits "   << red_bits <<endl;
    cout << "green_mask "  << green_mask 
	 << "\t green_shift "<< green_shift
	 << "\t green_bits " << green_bits << endl;
    cout << "blue_mask "   << blue_mask
	 << "\t blue_shift "  << blue_shift 
	 << "\t blue_bits "  << blue_bits << endl;
*/
  }

  // TOUJOURS FAIRE CE QUI SUIT!! (meme si create_widget == false)
  atoms.PRIMARY_SELECTION   = XA_PRIMARY;
  atoms.SECONDARY_SELECTION = XA_SECONDARY;
  atoms.WM_PROTOCOLS     = XInternAtom(xdisplay, "WM_PROTOCOLS", False);
  atoms.WM_DELETE_WINDOW = XInternAtom(xdisplay, "WM_DELETE_WINDOW", False);
  atoms.WM_TAKE_FOCUS    = XInternAtom(xdisplay, "WM_TAKE_FOCUS", False);
  
  atoms.UBIT_SELECTION   = XInternAtom(xdisplay, 
				       UMSprotocol::UBIT_SELECTION,False);
  atoms.UBIT_MESSAGE     = XInternAtom(xdisplay, 
				       UMSprotocol::UBIT_MESSAGE, False);
  atoms.UBIT_WINDOW      = XInternAtom(xdisplay, 
				       UMSprotocol::UBIT_WINDOW, False);

  // used by UWinGraph objects for toolkit drawing
  // Ce natgraph sera partagee par tous les UWinGraph relatifs a ce UDisp
  sharedGraph = new UNatGraph(this);

  // used by UGraph objects for appli. drawing
  clientGraph = new UNatGraph(this);

  // DEFAULT FONTS
  // !NB: ATTENTION: xdisp DOIT AVOIR ETE INITALISES !
  // ces 4 fontes doivent obligatoirement etre realisees
  realizeFont(UFont::standard);
  realizeFont(UFont::normal);
  realizeFont(UFont::bold);
  realizeFont(UFont::italic);

  // DEFAULT COLORS
  // !NB: ATTENTION: xdisp et xcmap DOIVENT AVOIR ETE INITALISES !

  realizeColor(UColor::white); UBgcolor::white.set(UColor::white);
  realizeColor(UColor::black); UBgcolor::black.set(UColor::black);
  realizeColor(UColor::disabled);

  realizeColor(UColor::grey);      UBgcolor::grey.set(UColor::grey);
  realizeColor(UColor::lightgrey); UBgcolor::lightgrey.set(UColor::lightgrey);
  realizeColor(UColor::darkgrey);  UBgcolor::darkgrey.set(UColor::darkgrey);
  realizeColor(UColor::navy);      UBgcolor::navy.set(UColor::navy);
  realizeColor(UColor::blue);      UBgcolor::blue.set(UColor::blue);
  realizeColor(UColor::red);       UBgcolor::red.set(UColor::red);
  realizeColor(UColor::green);     UBgcolor::green.set(UColor::green);
  realizeColor(UColor::yellow);    UBgcolor::yellow.set(UColor::yellow);
  realizeColor(UColor::orange);    UBgcolor::orange.set(UColor::orange);
  realizeColor(UColor::wheat);     UBgcolor::wheat.set(UColor::wheat);
  realizeColor(UColor::teal);      UBgcolor::teal.set(UColor::teal);
}

/* ==================================================== [OpenGL] ======= */
/* ==================================================== ======== ======= */

int UNatDisp::setVisual(const XVisualInfo& vinfo, UX_Colormap c) {

  if (vinfo.screen >= ScreenCount(xdisplay)) {
    UError::error("UNatDisp::setVisual",UError::Invalid_screen, vinfo.screen);
    return 0;
  }

  depth   = vinfo.depth;
  xvisual = vinfo.visual;
  xscreen = ScreenOfDisplay(xdisplay, vinfo.screen);

  if (xcmap != None) XFreeColormap(xdisplay, xcmap);

  if (c != None)
    xcmap = c;
  else
    xcmap = XCreateColormap(xdisplay, RootWindow(xdisplay, vinfo.screen),
                            xvisual, AllocNone);

  if (xcmap == None) 
    UError::error("UNatDisp::setVisual", UError::Null_colormap);

  return depth;
}

/* ==================================================== ======== ======= */

int UNatDisp::setVisual(int visual_class, int depth_hint,
			bool linear_gamma, UX_Colormap cmap) {
  XVisualInfo vinfo;

  if (chooseVisual(vinfo, visual_class, depth_hint, linear_gamma)) {
    return setVisual(vinfo, cmap);
  }
  else {
    UError::error("UNatDisp::setVisual",
		  "appropriate visual not found; requested depth ", 
		  depth_hint);
    return 0;
  }
}

/* ==================================================== ======== ======= */

#ifdef HAVE_X11_XMU_XMUSOLARIS_H
#  include <X11/Xmu/XmuSolaris.h>
#endif

bool UNatDisp::findExactVisual(XVisualInfo& vinfo,
			       int visual_class, int depth, 
			       bool linear_gamma) {  
  vinfo.depth   = depth;
  vinfo.c_class = visual_class;
  vinfo.screen  = XScreenNumberOfScreen(xscreen);
  long vinfo_mask = VisualScreenMask|VisualDepthMask|VisualClassMask;;

  bool found = false;
  int count = 0;
  XVisualInfo* tab = XGetVisualInfo(xdisplay, vinfo_mask, &vinfo, &count);

  for(int k = 0; k < count; k++) {
    double gamma = 0.;

#ifdef HAVE_XSOLARISGETVISUALGAMMA
    XSolarisGetVisualGamma(xdisplay, vinfo.screen, tab[k].visual, &gamma);
    //cerr << "XSolarisGetVisualGamma "<< gamma << endl;
#else
    linear_gamma = false;
#endif
 
    if (!found) { // on prend le premier
      found = true;
      vinfo = tab[k];
      if (!linear_gamma) break;
    }

    // si linear, ecraser eventuel choix precedent
    if (linear_gamma && gamma > 0.9 && gamma < 1.1) {
      vinfo = tab[k];
      break;
    }
  }

  if (tab) XFree(tab);
  return found;
}

/* ==================================================== ======== ======= */

int UNatDisp::chooseVisual(XVisualInfo& vinfo, 
			   int visual_class, int depth_hint, 
			   bool linear_gamma) {

  int scr_no = XScreenNumberOfScreen(xscreen);
  if (scr_no >= ScreenCount(xdisplay)) {
    UError::error("UNatDisp::chooseVisual", UError::Invalid_screen, scr_no);
    return 0;
  }

  if (findExactVisual(vinfo, visual_class, depth_hint, linear_gamma)) {
    return vinfo.depth;         // exact match
  }


  // find closest available depth
  int depth_count, possible_d = 0;
  int *depths = XListDepths(xdisplay, scr_no, &depth_count);
  XVisualInfo vi;

  if (depths) {
    for (int k = 0; k < depth_count; k++) {
      if (findExactVisual(vi, visual_class, depths[k], linear_gamma)) {

	if (possible_d == 0) {
	  possible_d = depths[k];
	  vinfo = vi;
	}

	else if (abs(depths[k] - depth_hint)
		 < abs(possible_d - depth_hint)) {
	  possible_d = depths[k];
	  vinfo = vi;
	}
      }
    }

    XFree(depths);
  }

  return possible_d;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// Event Management
/* ==================================================== ======== ======= */


UFlow* UNatDisp::retrieveFlow(unsigned int event_state, 
			      unsigned long flow_id) {
  UFlow* f = null;
  if (event_state & UEvent::UbitEventFlow) {
    // create a new flow (or returns an existing flow if ID already used).
    //EX: f = disp.getAppli().openFlow(int(flow_id));
    f = disp.openFlow(int(flow_id));
  }

  //EX: if (!f) f = &disp.getAppli().getDefaultFlow();
  if (!f) f = disp.openFlow(0);  // default
  return f;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// NB: une autre option serait de combiner les refresh events et ne reafficher 
// qu'une seule fois une zone complexe (if (ev->xexpose.count == 0) ...)

void UNatDisp::on_expose(UWin* win, UView* winview, XEvent* xev){
  UFlow& f = *retrieveFlow(0,0);
  UEvent e(UEvent::viewPaint, &f, winview, xev);
  e.setSource(winview);

  switch (xev->type) {
  case Expose:
    //cerr << "Expose : " << xev->xexpose.x <<","<<xev->xexpose.y
    //	 << " " << xev->xexpose.width <<"x" << xev->xexpose.height <<endl;

    // ne reafficher que la zone correspondant au xexpose
    e.sp.redrawClip.set(xev->xexpose.x, xev->xexpose.y, 
			xev->xexpose.width, xev->xexpose.height);
    break;

  case GraphicsExpose:
    //cerr << "GraphicsExpose : " << xev->xexpose.x <<","<<xev->xexpose.y
    //	 << " " << xev->xexpose.width <<"x" << xev->xexpose.height <<endl;

    // these events are generated when XCopyArea or XCopyPlanes is called 
    // and a part of the view is obscured. They are useful for repainting
    // these parts when scrolling data.
    // see fct. copyArea() in ugraph.hpp for details
    e.sp.redrawClip.set(xev->xgraphicsexpose.x, xev->xgraphicsexpose.y, 
			xev->xgraphicsexpose.width, xev->xgraphicsexpose.height);
    break;

  case NoExpose:
    //cerr << "NoExpose : " << xev->xexpose.x <<","<<xev->xexpose.y
    //	 << " " << xev->xexpose.width <<"x" << xev->xexpose.height <<endl;
  default:
    //nop
    return;
  }

  e.sp.redrawStatus = true;
  win->updateView(e, winview, UUpdate::PAINT);
}

/* ==================================================== ======== ======= */
// moves and resizes faits interactivement par l'utilisateur (PAS par
// programme: voir UNatWin::move & resize pour ce cas)

// Remarques:
// a) l'event n'indique pas si c'est un move ou un resize (ou autre chose 
//    par ex. un chgt taille border)
//    ==> on ne traite que les cas ou la taille a change
//
// b) attention: les XMoveWindow et les XResizeWindow generent des events
//    X qui sont ensuite renvoyes a l'appli ==> on ne peut alors savoir
//    le move/resize a ete fait par programme ou par l'utilisateur
//    ==> c'est pourquoi ces 2 cas sont bien separes (NatWin::move & resize
//    dans un cas, UNatDisp::on_configure dans l'autre)

void UNatDisp::on_configure(UWin* win, UView* winview, XEvent* xev){
  //cerr << "on_configure "<< xev->xconfigure.x << " " << xev->xconfigure.y
  //     << " " << xev->xconfigure.width << " " << xev->xconfigure.height
  //     << endl;

  // OLD: on testait WIN_MAPPED pour ne prendre en compte les configure 
  // que lorsque la window est mapped (voir MapNotify/UnmapNotify))
  // PBM: ceci empeche les resize initiaux, par ex par le WM de l'IPAQ
  // if (//win->isDef(0,UMode::WIN_MAPPED)  // inutile et genant (cf note)

  if (winview->getWidth() != xev->xconfigure.width
      || winview->getHeight() != xev->xconfigure.height
    ) {
    // met a jour l'affichage (mais ne change pas la taille de la X Window,
    // sinon generation d'events parasites et boucle infini (voir note ci-dessus)
    UUpdate upmode(UUpdate::ALL);
    win->impl.hard->updateImpl(upmode, win, winview, true, 
			       xev->xconfigure.width, xev->xconfigure.height);
#ifdef WITH_GL
    resizeGLViewport(xev->xconfigure.width, xev->xconfigure.height);
#endif
  }
  // notify event?
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// Text selection (cut and paste)

void UNatDisp::on_selection(UWin *win, UView* winview, UX_Event xev){
  //NB: no time, no state!
  UFlow& f = *retrieveFlow(0,0);  // ???? A REVOIR

  switch (xev->type) {
  case SelectionClear: {
    // effacer la selection
    UTextsel& ts = f.getTextsel();          // EventFlow!!
    ts.reset(true);
  } break;

  case SelectionRequest: {
    // Cas ou une autre application demande a obtenir la valeur de la 
    // selection courante (detenue par 'this' appli)
    
    UTextsel& ts = f.getTextsel();            // EventFlow!!
    changeSelectionContentRequest(xev, ts);
    } break;

  case SelectionNotify:
    // cas ou ce NatDisp a demande le contenu de la selection 
    // en appellant askServerSelection()
    retrieveSelectionContentRequest(xev);
    break;
  }

  if (win->isDef(UMode::NOTIFY_EVENT_CB, 0)) {
    UEvent e(UEvent::notifyEvent, &f, winview, xev);
    e.setSource(winview);
    win->fire(e, UOn::notifyEvent);
  }
}

/* ==================================================== ======== ======= */

void UNatDisp::on_notify(UWin* win, UView* winview, XEvent* xev) {
  UFlow& f = *retrieveFlow(0,0);

  switch (xev->type) {
  case ClientMessage:
    if (xev->xclient.message_type == atoms.UBIT_MESSAGE) {
      /*
      cerr << "MESG: appli='" << f.getAppli().getCommandName()
	   << "' (xwindow: " <<  xev->xclient.window
	   << " uwin:" <<  win
	   << " serial: " <<  xev->xclient.serial
	   << " send_ev: " <<  xev->xclient.send_event
	   << ")" << " MESG:'" << xev->xclient.data.b << "'"
	// << " format " <<  xev->xclient.format
	   << endl;
       */
      if (win->isDef(UMode::UMESSAGE_CB, 0)) {
	UEvent e(UEvent::umessage, &f, winview, xev);
	e.setSource(winview);
	win->fire(e, UOn::umessage);
      }
    } 
    
    //fermeture des windows par le WM:
    // on a clique' sur le bouton Close du WM et celui-ci le fait savoir
    else if (xev->xclient.message_type == atoms.WM_PROTOCOLS) {
      if (((Atom)xev->xclient.data.l[0]) == atoms.WM_DELETE_WINDOW) {
	// this fct. hides or kills the UWin depending on subclass
	// (and possible redefinitions of the fct. by client)
	win->close(0);
      }
    }
    break;

  case PropertyNotify:
    // the atom that was changed on xev->window
    if (xev->xproperty.atom == atoms.UBIT_MESSAGE) {
      //cerr << ">UBIT message propertyNotify" << endl;
      if (win->isDef(UMode::UMESSAGE_CB, 0)) {
	UEvent e(UEvent::umessage, &f, winview, xev);
	e.setSource(winview);
	win->fire(e, UOn::umessage);
      }
    }
    break;

  case MapNotify:		// la fenetre apparait
    win->setCmodes(UMode::CAN_SHOW, true);
    // ceci permet de ne pas prendre en compte les Configure Events
    // tant que la window n'a pas apparu (voir on_configure)
    win->setCmodes(UMode::WIN_MAPPED, true);
    break;

  case UnmapNotify:		// la fenetre disparait
    win->setCmodes(UMode::CAN_SHOW, false);
    win->setCmodes(UMode::WIN_MAPPED, false);
    break;

  case DestroyNotify:		// la fenetre est detruite
    //fprintf(stderr, ">DestroyNotify!!\n");
    break;

  case ColormapNotify:
    //fprintf(stderr, ">ColormapNotify\n");
    break;
  }

  if (win->isDef(UMode::NOTIFY_EVENT_CB, 0)) {
    UEvent e(UEvent::notifyEvent, &f, winview, xev);
    e.setSource(winview);
    win->fire(e, UOn::notifyEvent);
  }
}

/* ==================================================== ======== ======= */

void UNatDisp::on_raw_event(UWin* win, UView* winview, XEvent* xev){
  // any X event
  UEvent e(UEvent::rawEvent, null, winview, xev);  // flow inconnu !!
  e.setSource(winview);
  win->fire(e, UOn::rawEvent);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UNatDisp::on_mpress(UWin *win, UView *winview, XEvent *xev) {
  //cerr << "mpress " << endl;
  UFlow* f = retrieveFlow(xev->xbutton.state, xev->xbutton.subwindow);
  UEvent e(UEvent::mpress, f, winview, xev);
  e.setTime(xev->xbutton.time);
  e.locateSource(xev->xbutton.x, xev->xbutton.y);
  f->winMousePress(win, e);
}

void UNatDisp::on_mrelease(UWin *win, UView *winview, XEvent *xev) {
  //cerr << "mrelease " << endl;
  UFlow* f = retrieveFlow(xev->xbutton.state, xev->xbutton.subwindow);
  UEvent e(UEvent::mrelease, f, winview, xev);
  e.setTime(xev->xbutton.time);
  e.locateSource(xev->xbutton.x, xev->xbutton.y);
  f->winMouseRelease(win, e);
}

void UNatDisp::on_mmove(UWin *win, UView *winview, XEvent *xev) {
  //cerr << "mmove " << endl;
  UFlow* f = retrieveFlow(xev->xmotion.state, xev->xmotion.subwindow);
  UEvent e(UEvent::mmove, f, winview, xev);
  e.setTime(xev->xmotion.time);
  e.locateSource(xev->xmotion.x, xev->xmotion.y);
  f->winMouseMove(win, e);
}

/* ==================================================== ======== ======= */

void UNatDisp::on_kpress(UWin *win, UView *winview, XEvent *xev){
  UFlow* f = retrieveFlow(xev->xkey.state, xev->xkey.subwindow);
  UEvent e(UEvent::kpress, f, winview, xev);
  e.setTime(xev->xkey.time);
  e.locateSource(xev->xkey.x, xev->xkey.y);
  //cerr << "keypress " << xev->xkey.keycode << " flow " << f << endl;
  f->winKeyPress(win, e);
}

void UNatDisp::on_krelease(UWin *win, UView *winview, XEvent *xev){
  UFlow* f = retrieveFlow(xev->xkey.state, xev->xkey.subwindow);
  UEvent e(UEvent::krelease, f, winview, xev);
  e.setTime(xev->xkey.time);
  e.locateSource(xev->xkey.x, xev->xkey.y);
  //cerr << "keyrelease " << xev->xkey.keycode << " flow " << f << endl;
  f->winKeyRelease(win, e);
}

/* ==================================================== ======== ======= */

void UNatDisp::on_enter(UWin *win, UView *winview, XEvent *xev){

  if (UAppli::getDefaults().force_winfocus
      && xev->xcrossing.window != None) {     // ?? PBM MULTIFLOW ??
    UXtry xtry;  // sert a catcher les X Errors safely
    // aussi bizarre que ca paraisse, X est tellement mal fait que le seul
    // moyen d'eviter un plantage c'est de redefinir le XErrorHandler

    // la fenetre dans laquelle on rentre prend le focus : c'est obligatoire 
    // pour les menus sinon ils ne prennent pas les entrees clavier
    XSetInputFocus(xev->xcrossing.display, xev->xcrossing.window,
    		   RevertToPointerRoot, // ex: RevertToParent, 
		   CurrentTime);

    // synchroniser pour que X detecte l'erreur maintenant, en mode
    // protege en non 10 mn plus tard!
    XSync(xev->xcrossing.display, False);
  }

  // !!!! COMPLETER: EVENTS PERDUS !!!!
  // UEvent e(....)
  // e.time = xev->xcrossing.time;

  // CORRECT: ne PAS faire: win->boxMotion(boxlink,ev);
  // incompatible avec gestion des grabbedMenu !
}

void UNatDisp::on_leave(UWin *win, UView *winview, XEvent *xev){
  UFlow* f = retrieveFlow(xev->xcrossing.state, xev->xbutton.subwindow);

  UEvent e(UEvent::leave, f, winview, xev);
  e.setTime(xev->xcrossing.time);
  e.locateSource(xev->xcrossing.x, xev->xcrossing.y);

  // enorme bug corrige ci-dessous qui generait des LeaveEvent (X type 8)
  // qui etaient traites comme des MotionEvent (X type 6) et foutaient
  // un bordel immonde dans la gestion des boxMotion
  // FAUX-EX: winview->getWin()->boxMotion(a, &e);  <- faux!
  f->winLeave(win, e);
}

void UNatDisp::on_focus(UWin *win, UView *winview, XEvent *xev){
  /*                                         PBM avec certains WMs ???
  if (xev->type == FocusIn)
    fprintf(stderr, ">Focus In!!\n");
  else if (xev->type == FocusOut)
    fprintf(stderr, ">Focus Out!!\n");
  else 
    fprintf(stderr, ">Focus ???\n");
  */
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UNatDisp::dispatchEvents(UX_Event xev) {
  UX_Window event_win = xev->xany.window;  // att: xwin = member
  UWin *win = null;

  for (ULink *l = disp.hardwin_list.first(); l!=null; l = l->getNext()) {
    win = static_cast<UWin*>(l->getChild());  // par construction
    UWinGraph *g;
    if (win && win->impl.hard 
	&& ((g = win->impl.hard->getWinGraph()))
	&& (g->getNatWin()->getXWindow() == event_win)
	) break;
  }
  if (!win) return;

  UView* v = win->getWinView(&disp);

  if (win->isDef(UMode::RAW_EVENT_CB, 0)) {
    on_raw_event(win, v, xev);
  }

  switch(xev->xany.type) {
  case MotionNotify:
    on_mmove(win, v, xev);
    break;
  case EnterNotify:
    on_enter(win, v, xev);
    break;
  case LeaveNotify:
    on_leave(win, v, xev);
    break;
  case Expose:
  case GraphicsExpose:
  case NoExpose:
    on_expose(win, v, xev);
    break;
  case ConfigureNotify:
    on_configure(win, v, xev);
    break;
  case ButtonPress:
    on_mpress(win, v, xev);
    break;
  case ButtonRelease:
    on_mrelease(win, v, xev);
    break;
  case KeyPress:
    on_kpress(win, v, xev);
    break;
  case KeyRelease:
    on_krelease(win, v, xev);
    break;
  case FocusIn:
  case FocusOut:
    on_focus(win, v, xev);
    break;
  case SelectionClear:
  case SelectionNotify:
  case SelectionRequest:
    on_selection(win, v, xev);
    break;
  default:
    on_notify(win, v, xev);
    break;
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/* *** Fonts and FontFamilys *** */
/* ==================================================== ======== ======= */

UX_Font UNatDisp::getFont(const UFontDesc& f) {
  if (f.family->index >= fontFamilyCount || !fontFamilyMap[f.family->index])
    realizeFontFamily(*f.family);

  if (!fontFamilyMap[f.family->index][f.font_ix]) realizeFont(f);

  // dans tous les cas (pas de else)
  return fontFamilyMap[f.family->index][f.font_ix];
}

#ifdef WITH_GL
GLuint UNatDisp::getGLFont(const UFontDesc& f) {
  if (f.family->index >= fontFamilyCount || !glFontFamilyMap[f.family->index])
    realizeFontFamily(*f.family);

  if (!glFontFamilyMap[f.family->index][f.font_ix]) realizeFont(f);

  // dans tous les cas (pas de else)
  return glFontFamilyMap[f.family->index][f.font_ix];
}
#endif

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

bool UNatDisp::realizeFontFamily(const UFontFamily& family) {
  // increase the 'fontFamilyMap' vector
  if (family.index >= fontFamilyCount) {
    int newCount = family.index + 1;

    if (!fontFamilyMap) {
      fontFamilyMap = (UX_Font**)malloc(newCount*sizeof(UX_Font*));
#ifdef WITH_GL
      glFontFamilyMap = (GLuint**)malloc(newCount*sizeof(GLuint*));
#endif
    }
    else {
      fontFamilyMap = (UX_Font**)realloc(fontFamilyMap, newCount*sizeof(UX_Font*));
#ifdef WITH_GL
      glFontFamilyMap = (GLuint**)realloc(glFontFamilyMap, newCount*sizeof(GLuint*));
#endif
    }

    if (!fontFamilyMap) {
      UError::error("UNatDisp::realizeFontFamily", UError::No_more_memory);
      return false;
    }

    else {
      for (int k = fontFamilyCount; k < newCount; k++) {
	fontFamilyMap[k] = null;
#ifdef WITH_GL
	glFontFamilyMap[k] = null;
#endif
      }
      fontFamilyCount = newCount;
    }
  }

  // allocate the corresponding fontFamilyMap
  if (!fontFamilyMap[family.index]) {
    // Faudrait aussi verfier que cette Font Family existe !!!! A_REVOIR!!!
    UX_Font *ffmap = (UX_Font*)malloc(UFont::FONTMAP_SIZE * sizeof(UX_Font));
#ifdef WITH_GL
    GLuint *glffmap = (GLuint*)malloc(UFont::FONTMAP_SIZE * sizeof(GLuint));
#endif

    if (!ffmap) {
      UError::error("UNatDisp::realizeFontFamily", UError::No_more_memory);
      //" Not enough memory to create Font Family map\n - (font family = %s)", family->name);
      // impossible car 'const UFontFamily':  *family = UFontFamily::standard;
      return false;
    }
    else {
      for (int kk = 0; kk < UFont::FONTMAP_SIZE; kk++) {
	ffmap[kk] = null;
#ifdef WITH_GL
	glffmap[kk] = 0;
#endif
      }
      fontFamilyMap[family.index] = ffmap;
#ifdef WITH_GL
      glFontFamilyMap[family.index] = glffmap;
#endif
    }
  }

  return true;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

bool UNatDisp::realizeFont(const UFont& font) {
  UFontDesc f;
  f.set(font);
  return realizeFont(f);
}

/* ==================================================== ======== ======= */

bool UNatDisp::realizeFont(const UFontDesc& f) {
  bool stat = true;

  // initializes the font family (if needed)
  // Note: par construction, fontFamily ne peut etre null

  if (f.family->index >= fontFamilyCount || !fontFamilyMap[f.family->index])
    stat = realizeFontFamily(*f.family);

  if (fontFamilyMap[f.family->index][f.font_ix]
#ifdef WITH_GL
      && glFontFamilyMap[f.family->index][f.font_ix] // font already loaded
#endif
      )  return stat;  // font already loaded

  UX_Font fs = loadNatFont(*f.family, f.styles, f.lsize);
  
  if (fs) {
    fontFamilyMap[f.family->index][f.font_ix] = fs;
#ifdef WITH_GL
    glFontFamilyMap[f.family->index][f.font_ix] = glGenLists(256);
    glXUseXFont(fs->fid, 0, 255, glFontFamilyMap[f.family->index][f.font_ix]);
#endif
  }
  else {
    UError::error("warning@UNatDisp::realizeFont",
		  "Can't load font: ", f.family->name);
    stat = false;

    // if normal (= not bold not italic) exists, use it instead of the
    // bold or italic variants if they dont exist

    if (f.font_ix != 0 && fontFamilyMap[f.family->index][0]) {
      fontFamilyMap[f.family->index][f.font_ix]
	= fontFamilyMap[f.family->index][0];
#ifdef WITH_GL
      glFontFamilyMap[f.family->index][f.font_ix]
	= glFontFamilyMap[f.family->index][0];
#endif
    }

    // try to load 'normal' of same FontFamily

    else if (f.font_ix != 0 && (fs = loadNatFont(*f.family, 0, f.lsize))) {
      // both map entries will point to fs
      fontFamilyMap[f.family->index][0] = fs;
      fontFamilyMap[f.family->index][f.font_ix] = fs;
#ifdef WITH_GL
      glFontFamilyMap[f.family->index][0] = glGenLists(256);
      glXUseXFont(fs->fid, 0, 255, glFontFamilyMap[f.family->index][0]);
      glFontFamilyMap[f.family->index][f.font_ix] = glGenLists(256);
      glXUseXFont(fs->fid, 0, 255, glFontFamilyMap[f.family->index][f.font_ix]);
#endif
    }

    // otherwise use the 'fixed' FontFamily (should always exist)

    else if (f.family != &UFontFamily::fixed)  {
      UFontDesc f2 = f;
      f2.family = &UFontFamily::fixed;

      if (!(fs = getFont(f2))) return false;
      else {
	fontFamilyMap[f.family->index][f.font_ix] = fs;
#ifdef WITH_GL
	glFontFamilyMap[f.family->index][f.font_ix] = 
	  glFontFamilyMap[f2.family->index][f2.font_ix];
#endif
      }
    }

    else {    // (f.family == &UFontFamily::fixed) 

      if (f.lsize == 12)  {
	// plus aucun espoir...
	UError::error("fatal@UNatDisp::realizeFont", "Can't load any font");
	return false;
      }
      else {
	// essayer une autre taille
	UFontDesc f2 = f;
	f2.lsize = 12;
	if (!(fs = getFont(f2))) return false;
	else {
	  fontFamilyMap[f.family->index][f.font_ix] = fs;
#ifdef WITH_GL
	glFontFamilyMap[f.family->index][f.font_ix] = 
	  glFontFamilyMap[f2.family->index][f2.font_ix];
#endif
	}
      }
    }
  }
  return stat;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UX_Font UNatDisp::loadNatFont(const UFontFamily& family, 
			      int styles, int lsize) {
  // lsize==0 means "default"  ///!!!!PBM pas de borne min valable !!!

  if (lsize <1) lsize = UFont::MEDIUM_LSIZE; 
  else if (lsize > UFont::MAX_LSIZE) {
    UError::error("warning@UNatDisp::loadNatFont",UError::Invalid_font_size);
    lsize = UFont::MAX_LSIZE;
  }

  int s = lsize-1; // !att: shift -1
  char font_name[500];
  UX_Font fs = null;

  int n = 0;
  while (n < 2) {
    const char* slant = (styles & UFont::ITALIC) ? family.slant : "r";

    //"-*-times-medium-r-normal-*-12-*-*-*-*-*-*-1",
    sprintf(font_name, "-*-%s-%s-%s-%s-*-%s-*-*-*-*-*-*-%s",
	    family.name, 
	    (styles & UFont::BOLD) ? family.bold_weight :family.normal_weight, 
	    slant,
	    family.compression, 
	    family.font_sizes[s].natspec, 
	    family.encoding);

    //cerr << "load font : " << family.name << " " << family.index << endl;
    fs = XLoadQueryFont(getXDisplay(), font_name);
    if (fs) return fs;

    if ((styles & UFont::ITALIC) && n == 0) {
      if (slant[s] == 'o') slant = "i";
      else if (slant[s] == 'i') slant = "o";
    }
    else return null;
    n++;
  };

  return fs;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/* *** Cursors *** */
/* ==================================================== ======== ======= */

class UNatCursor {
  friend class UNatDisp;
  UX_Cursor xcursor;
public:
  UNatCursor(UNatDisp* nd, int shape);
};

UNatCursor::UNatCursor(UNatDisp* nd, int shape) {
  xcursor = XCreateFontCursor(nd->getXDisplay(), shape);
}

UX_Cursor UNatDisp::getCursor(const UCursor& c) {
  if (id >= (int)c.natcurs.size() || c.natcurs[id] == null) realizeCursor(c);
  return c.natcurs[id]->xcursor;
}

bool UNatDisp::realizeCursor(const UCursor& c) {
  if (id >= (int)c.natcurs.size()) {    // allonger la table;
    for (int k = c.natcurs.size(); k <= id; k++) c.natcurs.push_back(null);
  }
  c.natcurs[id] = new UNatCursor(this, c.shape);
  return (c.natcurs[id] !=  None);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/* *** Colors *** */
/* ==================================================== ======== ======= */

UX_Color UNatDisp::getColor(UColorImpl& ci) {
#ifndef MONODISP
  // cerr <<"getColor : id " <<id << " size" << ci.colors.size() << endl;
  if (id >= (int)ci.colors.size()) realizeColor(ci);
#endif

  if (ci.colors[id] < UColorImpl::UNDEF)
    return ci.colors[id];
  else if (ci.colors[id] > UColorImpl::UNDEF) 
    return BlackPixelOfScreen(getXScreen());
  else {      // == UNDEF
    realizeColor(ci);
    return ci.colors[id];
  }
}

/* ==================================================== ======== ======= */

bool UNatDisp::realizeColor(UColorImpl& ci) {
#ifndef MONODISP
  if (id >= (int)ci.colors.size()) {    // allonger la table;
    UX_Color col =     // prend en compte cas NONE et INHERIT
      (ci.colors[0] <= UColorImpl::UNDEF) ? UColorImpl::UNDEF : ci.colors[0];

    for (int k = ci.colors.size(); k <= id; k++) ci.colors.push_back(col);
  }
#endif

  if (ci.colors[id] < UColorImpl::UNDEF) return true;
  if (ci.colors[id] > UColorImpl::UNDEF) return false;
  // else == UNDEF ...

  if (!ci.natspec || !*ci.natspec) {
    ci.colors[id] = BlackPixelOfScreen(getXScreen());
    UError::error("warning@UNatDisp::realizeColor", 
		  UError::Unknown_color, "null");
    return false;
  }

  XColor xcolor;
  if (!XParseColor(getXDisplay(), getXColormap(), ci.natspec, &xcolor)) {
    ci.colors[id] = BlackPixelOfScreen(getXScreen());
    UError::error("warning@UNatDisp::realizeColor", 
		  UError::Unknown_color, ci.natspec);
    return false;
  }

  // !ICI il faudrait chercher une couleur approchee dans la x-cmap
  if (XAllocColor(getXDisplay(), getXColormap(), &xcolor)) {
    ci.colors[id] = xcolor.pixel;
    return true;
  }

  else {
    UError::error("warning@UNatDisp::realizeColor", 
		  UError::Full_colormap, ci.natspec);
    // if > to a certain value use white, otherwise, use black
    // [ 65535 is the max val for each color)

    if (xcolor.red + xcolor.blue + xcolor.green > 65535*2) 
      ci.colors[id] = WhitePixelOfScreen(getXScreen());
    else 
      ci.colors[id] = BlackPixelOfScreen(getXScreen());

    return false;
  }
}

/* ==================================================== ======== ======= */
#ifdef WITH_GL

void UNatDisp::setGLColor(GLubyte red, GLubyte green, GLubyte blue, 
			  GLubyte alpha) {
  glColor4ub(red, green, blue, alpha);
}

void UNatDisp::setGLColor(UX_Color pixel, GLubyte alpha) {
  glColor4ub((GLubyte)((pixel & red_mask)   >> red_shift),
	     (GLubyte)((pixel & green_mask) >> green_shift),
	     (GLubyte)((pixel & blue_mask)  >> blue_shift),
	     alpha);
}

void UNatDisp::setGLColor(UX_Color pixel) {
  glColor3ub((GLubyte)((pixel & red_mask)   >> red_shift),
	     (GLubyte)((pixel & green_mask) >> green_shift),
	     (GLubyte)((pixel & blue_mask)  >> blue_shift));
}

void UNatDisp::toGLColor(UX_Color pixel, 
			 GLubyte& red, GLubyte& green, GLubyte& blue){
  red   = (GLubyte)((pixel & red_mask)   >> red_shift);
  green = (GLubyte)((pixel & green_mask) >> green_shift);
  blue  = (GLubyte)((pixel & blue_mask)  >> blue_shift);
}

#endif
/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
//!ATT ces fcts doivent etre appellee sur des Button events
/* ==================================================== ======== ======= */

static long MaxSelRequestSize(Display *disp) {
  long l = XMaxRequestSize(disp);
  if (l > 65536) l = 65536;  // min
  return (l << 2) - 100;
}

bool UNatDisp::askSelectionOwnership(UEvent& e) {
  XButtonEvent* xev = &(e.getXEvent()->xbutton);
  if (xev->type != ButtonPress && xev->type != ButtonRelease) {
    UError::error("warning@UNatDisp::askSelectionOwnership",
		  UError::Unexpected_type, xev->type);
    return false;
  }

  Atom selection = XA_PRIMARY;
  Window owner = xev->window;

  XSetSelectionOwner(xev->display, selection, owner, xev->time);
  return (XGetSelectionOwner(xev->display, selection) == owner);
}

/* ==================================================== ======== ======= */
// Cas ou une autre application demande a obtenir la valeur de la 
// selection courante (detenue par 'this' appli)
// l'event doit etre un XSelectionRequestEvent

void UNatDisp::changeSelectionContentRequest(UX_Event xev, UTextsel& ts) {
  // on verifie qu'il s'agit bien de la section PRIMARY et le type
  // demande par le requestor (STRING) puis:
  // -- on change la property adequate avec la nouvelle valeur
  //    de la selection
  // -- puis on envoie un XSelectionEvent de confirmation ou d'echec
  //    DANS TOUS LES CAS y compris si on n'a rien change!

  // NOTES:
  // -- il peut y avoir plusieurs echanges: certaines applis peuvent
  //    d'abord demander les donnes sous un type specifique (par exple
  //    propre a Qt, etc.), puis, en cas d'echec (si on a renvoye
  //    un SelectionEvent d'echec), redemander la selection sous une
  //    autre forme
  // -- selection != property
  //    selection = PRIMARY = 1 tandis que property depend de l'appli
  //    qui demande les donnees

  XSelectionRequestEvent *xsel = (XSelectionRequestEvent*)xev;
  bool property_changed = false;

  if (xsel->selection == XA_PRIMARY && xsel->target == XA_STRING) {
    UStr selval;
    ts.copyText(selval);
    //cerr << "selval" <<selval << endl;

    // troncature si la taille de selval est trop grande
    int req_max_length = MaxSelRequestSize(xsel->display);
    int nitems = selval.length();
    if (nitems > req_max_length) nitems = req_max_length;
    
    // NOTE: pour bien faire il faudrait faire une boucle pour
    // pouvoir envoyer les messages de longue taille qui excedent
    // les limites du serveur X (ie. MaxSelRequestSize)
    // D'AUTRE PART XGetWindowProperty rajoute le 0 final des char*
    // (voir retrieveSelectionContentRequest() pour details)
    
    XChangeProperty(xsel->display, xsel->requestor, 
		    xsel->property, xsel->target, 
		    8/*format*/, PropModeReplace/*mode*/,
		    (unsigned char*)selval.chars(), nitems);	
    property_changed = true;
  }
  
  // On envoie un XSelectionEvent de CONFIRMATION ou d'ECHEC
  // DANS TOUS LES CAS y compris si on n'a rien change!
  
  XSelectionEvent event_send;
  event_send.type       = SelectionNotify;
  event_send.serial     = xsel->serial;
  event_send.time       = xsel->time;
  event_send.send_event = xsel->send_event;
  event_send.display    = xsel->display;
  event_send.requestor  = xsel->requestor;
  event_send.selection  = xsel->selection;
  event_send.target     = xsel->target;
  
  // send and event with property == None if nothing changed
  if (property_changed)
    event_send.property = xsel->property;  // cas OK
  else
    event_send.property = None;            // cas d'echec
  
  XSendEvent(xsel->display, xsel->requestor, 
	     0,  //propagate
	     0,  //event_mask
	     (XEvent*)&event_send);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// asks the selection of the X Server
// the UEvent must be a button event. 
// The server will send a SelectionNotify event to the UNatDisp that will
// put the result in this UStr at this position

void UNatDisp::askSelectionContent(UEvent& e, UStr *str, int pos) {
  XButtonEvent* xev = &(e.getXEvent()->xbutton);

  if (xev->type != ButtonPress && xev->type != ButtonRelease) {
    UError::error("warning@UAppli::askSelectionContent", 
		  UError::Unexpected_type, xev->type);
    return;
  }
  Atom selection = XA_PRIMARY;
  Atom target    = XA_STRING;
  Atom property  = atoms.UBIT_SELECTION;

  XConvertSelection(xev->display, selection, target, property,
		    xev->window, xev->time);

  //ATT: pbm si str est detruite entre-temps : il faudrait bloquer l'appli !!
  server_selection_str = str;
  server_selection_pos = pos;

  // ceci entraine la generation d'un SelectionNotify event par le
  // serveur si tout marche bien
  // ce SelectionNotify event est ensuite recu par configureEH qui
  // a alors a charge de realiser l'operation d'insertion
}

/* ==================================================== ======== ======= */
// recupere le contenu de la selection lorsque le NatDisp recoit un
// SelectionNotify event apres avoir demande le contenu de la selection 
// en appellant askSelectionContent()
// the XEvent must be a XSelectionEvent event. 

void UNatDisp::retrieveSelectionContentRequest(UX_Event xev) {
  XSelectionEvent* xsel = (XSelectionEvent*)xev;

  if (xsel->property != None
      && xsel->selection == XA_PRIMARY
      && xsel->target == XA_STRING
      && server_selection_str != null
      ) {
    //fprintf(stderr, "SelNotify: xsel->selection %ld, xsel->property %ld, xsel->target %ld, xsel->requestor %ld \n", xsel->selection, xsel->property, xsel->target, xsel->requestor);
    
    long req_offset = 0;
    long req_length = MaxSelRequestSize(xsel->display);    
    Atom type_return;
    int format_return;
    unsigned long nitems_return = 0, bytes_after_return = 1;
    unsigned char *prop_return = null;
   
    // NOTE: pour bien faire il faudrait faire une boucle pour pouvoir
    // recuperer les messages de longue taille qui excedent les limites
    // du serveur X (ie. MaxSelRequestSize). Dans ce cas il faudrait
    // alors appeler XGetWindowProperty() avec delete_property == FALSE
    // puis appeler XDeleteProperty() qunad tout a ete recupere

    // XGetWindowProperty always allocates one extra byte in prop_return
    // (even if the property is zero length) and sets it to zero so that
    // simple properties consisting of characters do not have to be
    // copied into yet another string before use.
        
    Bool req_stat =
      XGetWindowProperty(xsel->display, xsel->requestor, xsel->property,
			 req_offset, req_length,
			 True,/*delete property*/
			 AnyPropertyType,/*req_type= Any type*/
			 &type_return, &format_return, 
			 &nitems_return, &bytes_after_return,
			 &prop_return);
    
    if (req_stat == Success
	&& nitems_return > 0
	&& prop_return != null
	&& type_return == XA_STRING
	&& format_return == 8
	) {
      server_selection_str->insert(server_selection_pos, (char*)prop_return);
    }
    
    //desallouer la memoire allouee par XGetWindowProperty
    if (prop_return != null) XFree((char*)prop_return);
  }

  // DEVRAIT debloquer appli !!!!
  server_selection_str = null;
  server_selection_pos = 0;
}

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
