namespace eval cat {
    variable cat

    # Define the cat array structure so that all variables are
    # defined for the callbacks in the radiobuttons and checkbuttons.
    array set cat {
	list.reset ""
	list.clear ""
	entry.file ""
	dialog ""
	file ""
	number_nonblank_lines ""
	number_lines ""
	squeeze_blanks ""
	show_non_printing ""
	show_ends ""
	show_tabs ""
	show_help ""
	show_version ""
	show_non_printing_ends ""
	show_non_printing_tabs ""
	show_all ""
	ignored ""
    }
}

# cat::create --
#
#   Method to create the dialog box for the cat command.
#
# Note
#
#   This dialog will not grab focus so the user can keep it open
#   and run other tkWorld dialogs.  Imagine how tedious it would be
#   if you had to close the dialog to run your command, then reopen
#   it to modify it.  By not making this a modal dialog, we do not
#   have to implement any last command saving characteristics since
#   the user can just leave the dialog open.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc cat::create { } {
    global tkWorld
    variable cat

    # Put the focus on the cat dialog if it is already open.
    if [winfo exists $cat(dialog)] {
	switch -- [wm state $cat(dialog)] {
	    normal {
		raise $cat(dialog)
	    }
	    withdrawn -
	    iconic {
		wm deiconify $cat(dialog)
	    }
	}
	focus $cat(dialog)
	return
    } else {
	set cat(dialog) [dialog::create .cat Cat]
    }

    # The first tab has file and regexp information, along with
    # the standard command line options.
    set tab1 [tabnotebook::page [dialog::interior $cat(dialog)] "Options"]

    # Use a frame to encapsulate the File and Regexp options so that
    # the frame can be centered accross the grid columns.
    set f1 [frame $tab1.f1 \
	    -class TabnotebookFrame]
    button $f1.file_button \
	    -text "File" \
	    -width 5 \
	    -command cat::open
    set cat(entry.file) [entry $f1.entry_file \
	    -width 35 \
	    -textvariable cat::cat(file)]
    grid $f1.file_button $f1.entry_file \
	    -sticky w \
	    -padx 2 \
	    -pady 2

    # Now build the individual checkbutton options available to
    # the user.
    set f2 [frame $tab1.f2 \
	    -class TabnotebookFrame]
    checkbutton $f2.number_nonblank_lines \
	    -text "Number Non-blank Lines" \
	    -variable cat::cat(number_nonblank_lines) \
	    -onvalue "b" \
	    -offvalue ""
    checkbutton $f2.number_lines \
	    -text "Number All Lines" \
	    -variable cat::cat(number_lines) \
	    -onvalue "n" \
	    -offvalue ""
    checkbutton $f2.squeeze_blanks \
	    -text "Replace Multiple Blanks With 1" \
	    -variable cat::cat(squeeze_blanks) \
	    -onvalue "s" \
	    -offvalue ""
    checkbutton $f2.show_non_printing \
	    -text "Show Non-printing Characters" \
	    -variable cat::cat(show_non_printing) \
	    -onvalue "v" \
	    -offvalue ""
    checkbutton $f2.show_ends \
	    -text "Show End of Line" \
	    -variable cat::cat(show_ends) \
	    -onvalue "E" \
	    -offvalue ""
    checkbutton $f2.show_tabs \
	    -text "Show Tabs" \
	    -variable cat::cat(show_tabs) \
	    -onvalue "T" \
	    -offvalue ""
    checkbutton $f2.show_help \
	    -text "Display Help" \
	    -variable cat::cat(show_help) \
	    -onvalue "-help" \
	    -offvalue ""
    checkbutton $f2.show_version \
	    -text "Show Version" \
	    -variable cat::cat(show_version) \
	    -onvalue "-version" \
	    -offvalue ""

    grid $f2.number_nonblank_lines $f2.show_non_printing \
	    -padx 2 \
	    -pady 2 \
	    -sticky w 
    grid $f2.number_lines $f2.squeeze_blanks \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f2.show_ends $f2.show_tabs \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f2.show_help $f2.show_version \
	    -padx 2 \
	    -pady 2 \
	    -sticky w

    # Build a tab for the advanced combination options.
    set tab2 [tabnotebook::page [dialog::interior $cat(dialog)] "Combinations"]

    # Now build the frame for the combination type checkbuttons.
    set f3 [frame $tab2.f3 \
	    -class TabnotebookFrame]
    checkbutton $f3.show_non_printing_ends \
	    -text "Show Non-printing Characters and End of Line" \
	    -variable cat::cat(show_non_printing_ends) \
	    -onvalue "e" \
	    -offvalue ""
    checkbutton $f3.show_non_printing_tabs \
	    -text "Show Non-printing Characters and Tabs" \
	    -variable cat::cat(show_non_printing_tabs) \
	    -onvalue "t" \
	    -offvalue ""
    checkbutton $f3.show_all \
	    -text "Show Non-printing Characters, Tabs, and End of Line" \
	    -variable cat::cat(show_all) \
	    -onvalue "A" \
	    -offvalue ""

    grid $f3.show_non_printing_ends \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f3.show_non_printing_tabs \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f3.show_all \
	    -padx 2 \
	    -pady 2 \
	    -sticky w

    # Build a tab for the compatibility option.
    set tab3 [tabnotebook::page [dialog::interior $cat(dialog)] "Advanced"]

    # Now build the frame for the compatibility option.
    set f4 [frame $tab3.f4 \
	    -class TabnotebookFrame]
    checkbutton $f4.ignored \
	    -text "Ignored: -u Flag for Compatibility" \
	    -variable cat::cat(ignored) \
	    -onvalue "u" \
	    -offvalue ""

    grid $f4.ignored \
	    -padx 2 \
	    -pady 2 \
	    -sticky w

    # Build all of the tabs.
    pack $f1 $f2 $f3 $f4 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

    # Duhhhhhh......
    focus $cat(entry.file)

    # Define the lists for the reset and clear methods
    set cat(list.reset) "number_nonblank_lines number_lines \
	    squeeze_blanks show_non_printing show_ends show_tabs \
	    show_help show_version show_non_printing_ends \
	    show_non_printing_tabs show_all ignored"
    set cat(list.clear) "file"
}

# cat::ok --
#
#   Method to insert the command the user has created into the CC
#   as a Tcl list.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc cat::ok { } {
    global tkWorld
    variable cat

    # Insert the Tcl command list in the Command Center with the
    # proper formatting of a space between each argument on the
    # command line.  If there are no options given by the user,
    # then don't display it in the CC.

    $tkWorld(cmd_center) insert insert "cat "

    # Build the command line argument.
    set cmd_arg ""
    foreach x $cat(list.reset) {
	if [string length $cat($x)] {
	    switch -- $x {
		show_help -
		show_version {
		    $tkWorld(cmd_center) insert insert "-$cat($x) "
		}
		default {
		    append cmd_arg $cat($x)
		}
	    }
	}
    }
    if [string length $cmd_arg] {
	$tkWorld(cmd_center) insert insert "-$cmd_arg "
    }
    if [string length $cat(file)] {
	$tkWorld(cmd_center) insert insert "$cat(file)"
    }

    # Activate the buttons in the toolbar for the command center.
    toolbar::group_state cmd_center active
    toolbar::button_state $toolbar::toolbar(stop) disabled

    # Change to the default background/foreground of the file entry.
    $cat(entry.file) configure -bg white -fg black
}

# cat::reset --
#
#   Method to reset the radio and checkbuttons in the dialog.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc cat::reset { } {
    variable cat

    # It is easier to list the elements to exclude rather than list
    # all of the values to reset.
    foreach x $cat(list.reset) {
	set cat($x) ""
    }
}

# cat::clear --
#
#   Method to clear entry items of their text and reset the
#   background and foreground properties.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc cat::clear { } {
    variable cat

    # Reset the data structure elements and bg/fg.
    foreach x $cat(list.clear) {
	set cat($x) ""
	$cat(entry.$x) configure -bg #ffffff -fg #000000
    }

    focus $cat(entry.file)
}

# cat::help --
#
#   Method to invoke the Cat Command Help.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc cat::help { } {
    global tkWorld

    help::create "help/cat.html" "Cat Command Help"
}

# cat::close --
#
#   Close the dialog up.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc cat::close { } {
    variable cat
    
    balloonhelp::cancel
    destroy $cat(dialog)
}

# cat::open --
#
#   Method to add a file to the file entry and move the cursor
#   index of the entry to the end.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc cat::open { } {
    variable cat

    # Insert the file list and move the cursor.
    $cat(entry.file) insert insert [file::select]
    $cat(entry.file) icursor end

    # Set the focus on the entry with the file list in it.
    focus $cat(entry.file)
}
