#        -*- tcl -*-

#
# This file was written by
#         Donal K. Fellows (fellowsd@cs.man.ac.uk)
# in the week leading up to
#         3rd April 1994
# to support dirbrowser.tcl (probably in the same directory as this in your installation).
#
#
# You may distribute and modify this file however you like, provided you acknowledge me as the
# original author and include this notice.  Donations to the Donal K. Fellows Beverage Consumption
# Support Fund are not necessary, even if they are recommended (by me! :)
#
#
# If there are bugs in this, and these haven't been introduced by someone else, mail me and I'll
# probably try to fix them, though I'm not making any promises...  Alternatively, as you have The
# Force, Use It... :) :)
#

### -------------------------------------------------------------------------------------------------------
#
# PROCEDURE
#   parseargs
#
# DESCRIPTION
#   Given a list of acceptable arguments, parses these out of the given list (with args, if requested), and
#   returns any unparseable arguments.
#
# ARGUMENTS
#   allowed_switches      Name of array of numbers, indexed by allowed switches, stating the defaults for
#                         the returned_value array (the next argument)
#                             0  -  set to zero                        -  no argument (1 if switch present)
#                             1  -  set to empty string                -  takes argument
#                             2  -  default preset by calling routine  -  takes argument
#   returned_value        Name of array to fill in with parsed switch values
#   actual_list           List of switches,etc. to parse
#
# RETURN VALUE
#   unparsed_args         Anything not recognised/not a switch value
#
# ERRORS
#   missing_switch_value  If a switch taking an argument appears as the last item without a value
#

proc parseargs {allowed_switches returned_values actual_list} {
    upvar $allowed_switches names
    upvar $returned_values results

    set count [llength $actual_list]
    set rejected {}
    set namelist [array names names]

    foreach i $namelist {
	switch -exact $names($i) {
	    0 {set results($i)  0 }
	    1 {set results($i) {} }
	}
    }

    for {set i 0} {$i < $count} {incr i} {
	set item [lindex $actual_list $i]
	if {[lsearch -exact $namelist $item]+1} {
	    if {$names($item)} {
		incr i
		if {$i<$count} {
		    set results($item) [lindex $actual_list $i]
		} else {
		    error "Argument $item requires a value"
		}
	    } else {
		set results($item) 1
	    }
	} else {
	    lappend rejected $item
	}
    }

    return $rejected
}

### -------------------------------------------------------------------------------------------------------
#
# PROCEDURE
#   recursivebind
#
# DESCRIPTION
#   Binds all widgets beneath a given widget that satisfy a given predicate
#
# ARGUMENTS
#   widget     (0)       Parent window to start at
#   condition  (1)       Guard on binding action
#   binding    (2)       Pretty obvious,  like <Return>
#   action     (3)       Ditto,           like {.foo activate}
#   -notself             Flag to indicate that only descendants of the widget should be bound, not the
#                            widget itself.  Always 1 on recursive calls!  Defaults to 1
#
# RETURN VALUE
#   none
#

proc recursivebind args {
    set as(-bindself) 0
    set arg [parseargs as ss $args]; # return {widget condition binding action}
    if $ss(-bindself) {
	bind [lindex $arg 0] [lindex $arg 2] [lindex $arg 3]
    }
    foreach win [winfo children [lindex $arg 0]] {
	if [lindex $arg 1] {
	    bind $win [lindex $arg 2] [lindex $arg 3]
	    eval [concat [list recursivebind $win] [lrange $arg 1 end]]
	}
    }
}

### -------------------------------------------------------------------------------------------------------
#
# PROCEDURE
#   atomic
#
# DESCRIPTION
#   Ensures that an operation is atomic (assuming that incr is an atomic operation) by using a semaphore.
#
#   You might want this on a system where you have several event handlers, all of which need to delay 
#      during their execution for user interaction, and which must be very careful about how they access
#      some shared data. This also might be useful in some bizarre multiprocessing environment... :)
#
# ARGUMENTS
#   semaphore             Name of semaphore
#   operation             Protected operation (evaluated at level of calling routine)
#
# RETURN VALUE
#   That of operation
#
# PROBLEMS
#   Assumes that incr is atomic. No idea if this is true :)
#
# INITIALLY
#   Before entering any guarded code, the semaphore should be initialised to 0
#

proc atomic {semaphore operation} {
    global errorInfo errorCode
    upvar $semaphore sem

    while {[incr sem] > 1} {
	incr sem -1
	tkwait variable sem
    }
    set code [catch {set value [uplevel $operation]} string]
    set sem 0
    if {$code == 1} {
	return -code error -errorinfo $errorInfo -errorcode $errorCode $string
    } elseif {$code == 2} {
	return -code return $string
    } elseif {$code > 2} {
	return -code $code $string
    }
    set value
}

### -------------------------------------------------------------------------------------------------------
#
# PROCEDURE
#   onexit
#
# DESCRIPTION
#   Performs a given operation upon exit of another piece of code, no matter how the code exits (except
#   for when the OS terminates the process anyway! :)
#
# ARGUMENTS
#   operation             Operation (evaluated at level of calling routine) which is protected
#   exit_script           Operation (evaluated at level of calling routine) executed on exit
#
# RETURN VALUE
#   That of operation (unless operation succeeds, and exit_script fails)
#
# PROBLEMS
#   Bound to be some... :)
#

proc onexit {operation exit_script} {
    global errorInfo errorCode

    set code [catch {
	uplevel 1 $operation
    } string]
    if {$code == 1} {
	return -code error -errorinfo $errorInfo -errorcode $errorCode $string
    }
    set c [catch { uplevel 1 $exit_script } s]
    if {$c == 1} {
	return -code error -errorinfo $errorInfo -errorcode $errorCode $s
    }
    return -code $code $string
}

### -------------------------------------------------------------------------------------------------------
#
# PROCEDURE
#   with_grab_focus
#
# DESCRIPTION
#   Performs a given operation with a grab set on a piece of code and the focus set (initially) to a given
#   window, restoring things properly afterwards.
#
# ARGUMENTS
#   grab_window           Window to grab (actually toplevel within window)
#   focus_window          Window to focus on
#   operation             Operation (evaluated at level of calling routine) which is packaged
#
# RETURN VALUE
#   That of operation
#
# PROBLEMS
#   Bound to be some... :)
#

proc with_grab_focus {grabw focusw args} {
    set rgrab [winfo toplevel $grabw]

    switch [llength $args] {
	1 {
	    set op [lindex $args 0]
	    set global 0
	}
	2 {
	    set op [lindex $args 1]
	    if {[lindex $args 0] != "-global"} {
		return -code error "Usage: with_grab_focus grabwin focuswin ?-global? command"
	    }
	    set global 1
	}
	default {
	    return -code error "Usage: with_grab_focus grabwin focuswin ?-global? command"
	}
    }

    set foc [focus]
    set grb [grab current $rgrab]
    if [llength $grb] {
	set type [grab status $grb]
    }
    if $global {
	grab set -global $rgrab
    } else {
	grab set $rgrab
    }
    focus $focusw

    set code [catch { uplevel 1 $op } str]

    if [llength $foc] {
	catch {focus $foc}
    }
    if [llength $grb] {
	if {$type == "global"} {
	    grab set -global $grb
	} else {
	    grab set $grb
	}
    }
    
    if {$code == 1} {
	global errorInfo errorCode
	return -code error -errorinfo $errorInfo -errorcode $errorCode $string
    }
    return -code $code $str
}
