/*
 * SwamiMidi.c - MIDI driver base class
 *
 * Swami
 * Copyright (C) 1999-2003 Josh Green <jgreen@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Swami homepage: http://swami.sourceforge.net
 */
#include "config.h"

#include <stdio.h>
#include <glib.h>

#include "SwamiMidi.h"
#include "SwamiLog.h"
#include "SwamiObject.h"


/* --- signals and properties --- */

enum {
  LAST_SIGNAL
};

enum {
  PROP_0,
};

/* --- private function prototypes --- */

static void swami_midi_class_init (SwamiMidiClass *klass);
static void swami_midi_init (SwamiMidi *midi);


/* --- private data --- */


// static guint midi_signals[LAST_SIGNAL] = { 0 };


/* --- functions --- */


GType
swami_midi_get_type (void)
{
  static GType item_type = 0;

  if (!item_type) {
    static const GTypeInfo item_info = {
      sizeof (SwamiMidiClass),
      NULL,
      NULL,
      (GClassInitFunc) swami_midi_class_init,
      NULL,
      NULL,
      sizeof (SwamiMidi),
      0,
      (GInstanceInitFunc) swami_midi_init,
    };

    item_type = g_type_register_static (G_TYPE_OBJECT, "SwamiMidi",
					&item_info, G_TYPE_FLAG_ABSTRACT);
  }

  return (item_type);
}

static void
swami_midi_class_init (SwamiMidiClass *klass)
{
}

static void
swami_midi_init (SwamiMidi *midi)
{
  midi->active = FALSE;
}

SwamiMidi *
swami_midi_new (void)
{
  return SWAMI_MIDI (g_object_new (SWAMI_TYPE_MIDI, NULL));
}

/**
 * swami_midi_init_driver:
 * @midi: Swami MIDI object
 *
 * Initialize active driver
 *
 * Returns: SWAMI_OK on success, SWAMI_FAIL otherwise
 */
int
swami_midi_init_driver (SwamiMidi *midi)
{
  SwamiMidiClass *oclass;
  int retval = SWAMI_OK;

  g_return_val_if_fail (midi != NULL, SWAMI_FAIL);
  g_return_val_if_fail (SWAMI_IS_MIDI (midi), SWAMI_FAIL);

  oclass = SWAMI_MIDI_CLASS (G_OBJECT_GET_CLASS (midi));

  if (midi->active) return (SWAMI_OK);

  if (oclass->init_driver)
    retval = (*oclass->init_driver) (midi);

  if (retval == SWAMI_OK)
    midi->active = TRUE;

  return (retval);
}

/**
 * swami_midi_close_driver:
 * @midi: Swami MIDI object
 *
 * Close active driver
 */
void
swami_midi_close_driver (SwamiMidi *midi)
{
  SwamiMidiClass *oclass;

  g_return_if_fail (midi != NULL);
  g_return_if_fail (SWAMI_IS_MIDI (midi));

  if (!midi->active) return;

  oclass = SWAMI_MIDI_CLASS (G_OBJECT_GET_CLASS (midi));

  if (oclass->close_driver)
    (*oclass->close_driver) (midi);

  midi->active = FALSE;
}

/**
 * swami_midi_send_event:
 * @midi: MIDI driver object
 * @event: Event type enumeration
 * @chan: MIDI channel to send event on
 * @param1: First parameter for event type
 * @param2: Second parameter for event type
 *
 * Send MIDI event
 */
void
swami_midi_send_event (SwamiMidi *midi, SwamiMidiEvent event, int chan,
		       int param1, int param2)
{
  SwamiMidiClass *oclass;

  g_return_if_fail (midi != NULL);
  g_return_if_fail (SWAMI_IS_MIDI (midi));
  g_return_if_fail (midi->active);

  oclass = SWAMI_MIDI_CLASS (G_OBJECT_GET_CLASS (midi));

  if (oclass->send_event)
    (*oclass->send_event)(midi, event, chan, param1, param2);
}

/**
 * swami_midi_note_on:
 * @midi: MIDI driver object
 * @chan: MIDI channel to send note on
 * @note: MIDI note number
 * @vel: MIDI note velocity
 *
 * A convenience function to send a note-on event.
 */
void
swami_midi_note_on (SwamiMidi *midi, int chan, int note, int vel)
{
  swami_midi_send_event (midi, SWAMI_MIDI_NOTE_ON, chan, note, vel);
}

/**
 * swami_midi_note_off:
 * @midi: MIDI driver object
 * @chan: MIDI channel to send note off
 * @note: MIDI note number
 * @vel: MIDI note velocity
 *
 * A convenience function to send a note-off event.
 */
void
swami_midi_note_off (SwamiMidi *midi, int chan, int note, int vel)
{
  swami_midi_send_event (midi, SWAMI_MIDI_NOTE_OFF, chan, note, vel);
}

/**
 * swami_midi_set_bank:
 * @midi: MIDI driver object
 * @chan: MIDI channel to send bank change to
 * @bank: Bank number to change to
 *
 * A convenience function to send a bank change event.
 */
void
swami_midi_set_bank (SwamiMidi *midi, int chan, int bank)
{
  swami_midi_send_event (midi, SWAMI_MIDI_BANK_SELECT, chan, bank, 0);
}

/**
 * swami_midi_set_preset:
 * @midi: MIDI driver object
 * @chan: MIDI channel to send preset change to
 * @preset: Preset number to change to
 *
 * A convenience function to send a preset change event.
 */
void
swami_midi_set_preset (SwamiMidi *midi, int chan, int preset)
{
  swami_midi_send_event (midi, SWAMI_MIDI_PRESET_SELECT, chan, preset, 0);
}

/**
 * swami_midi_set_bend_range:
 * @midi: MIDI driver object
 * @chan: MIDI channel to send event to
 * @range: Range value in semitones (MIDI notes, example: 12 = 1 octave,
     default = 2 semitones)
 *
 * A convenience function to set pitch wheel bend range
 */
void
swami_midi_set_bend_range (SwamiMidi *midi, int chan, int range)
{
  swami_midi_send_event (midi, SWAMI_MIDI_BEND_RANGE, chan, range, 0);
}

/**
 * swami_midi_set_pitch_wheel:
 * @midi: MIDI driver object
 * @chan: MIDI channel to send event to
 * @val: MIDI pitch wheel value (14 bit, 8192 = center)
 *
 * A convenience function to set the pitch wheel value.
 */
void
swami_midi_set_pitch_wheel (SwamiMidi *midi, int chan, int val)
{
  swami_midi_send_event (midi, SWAMI_MIDI_PITCH_WHEEL, chan, val, 0);
}
