/**
 * @file scim_x11_frontend.h
 * @brief definition of X11FrontEnd related classes.
 */

/* 
 * Smart Common Input Method
 * 
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_x11_frontend.h,v 1.37 2004/08/24 14:42:20 suzhe Exp $
 */

#if !defined (__SCIM_X11_FRONTEND_H)
#define __SCIM_X11_FRONTEND_H

using namespace scim;

class X11FrontEnd : public FrontEndBase
{
    X11ICManager      m_ic_manager;
    XIMS              m_xims;
    Display          *m_display;
    Window            m_xims_window;
    String            m_server_name;
    String            m_display_name;

    SocketClient      m_panel_socket;
    SocketTransaction m_send_transaction;
    uint32            m_socket_magic_key;

    String            m_panel_socket_address;
    int               m_panel_socket_timeout;

    X11IC            *m_focus_ic;

    KeyEventList      m_trigger_keys;
    KeyEventList      m_next_factory_keys;
    KeyEventList      m_previous_factory_keys;
    KeyEventList      m_show_factory_menu_keys;

    bool              m_xims_dynamic;
    bool              m_wchar_ucs4_equal;
    bool              m_broken_wchar;
    int               m_valid_key_mask;

    bool              m_should_exit;

    IConvert          m_iconv;
    ConfigPointer     m_config;

    std::map<String, String> m_default_factories;

    IMEngineFactoryPointer  m_fallback_factory;
    IMEngineInstancePointer m_fallback_instance;

    int (*m_old_x_error_handler) (Display *, XErrorEvent *);

public:
    X11FrontEnd (const BackEndPointer &backend,
                 const ConfigPointer &config,
                 const String &server_name = String ("SCIM"));

    virtual ~X11FrontEnd ();

protected:
    virtual void show_preedit_string   (int id);
    virtual void show_aux_string       (int id);
    virtual void show_lookup_table     (int id);

    virtual void hide_preedit_string   (int id);
    virtual void hide_aux_string       (int id);
    virtual void hide_lookup_table     (int id);

    virtual void update_preedit_caret  (int id, int caret);
    virtual void update_preedit_string (int id, const WideString & str, const AttributeList & attrs = AttributeList ());
    virtual void update_aux_string     (int id, const WideString & str, const AttributeList & attrs = AttributeList ());
    virtual void commit_string         (int id, const WideString & str);
    virtual void forward_key_event     (int id, const KeyEvent & key);
    virtual void update_lookup_table   (int id, const LookupTable & table);

    virtual void register_properties   (int id, const PropertyList & properties);
    virtual void update_property       (int id, const Property     & property);

public:
    virtual void init (int argc, char **argv);
    virtual void run ();

private:
    String get_supported_locales (void);

    void init_ims (void);

    String get_factory (const String &locale);
    void set_factory (const String &locale, const String &sf_uuid);
    void next_factory (const String &locale);
    void previous_factory (const String &locale);

    XKeyEvent keyevent_scim_to_x11 (const KeyEvent& key);
    KeyEvent keyevent_x11_to_scim (const XKeyEvent& key);

    int ims_open_handler                (XIMS ims, IMOpenStruct *call_data);
    int ims_close_handler               (XIMS ims, IMCloseStruct *call_data);
    int ims_create_ic_handler           (XIMS ims, IMChangeICStruct *call_data);
    int ims_set_ic_values_handler       (XIMS ims, IMChangeICStruct *call_data);
    int ims_get_ic_values_handler       (XIMS ims, IMChangeICStruct *call_data);
    int ims_destroy_ic_handler          (XIMS ims, IMDestroyICStruct *call_data);
    int ims_set_ic_focus_handler        (XIMS ims, IMChangeFocusStruct *call_data);
    int ims_unset_ic_focus_handler      (XIMS ims, IMChangeFocusStruct *call_data);
    int ims_reset_ic_handler            (XIMS ims, IMResetICStruct *call_data);
    int ims_trigger_notify_handler      (XIMS ims, IMTriggerNotifyStruct *call_data);
    int ims_forward_event_handler       (XIMS ims, IMForwardEventStruct *call_data);
    int ims_sync_reply_handler          (XIMS ims, IMSyncXlibStruct *call_data);
    int ims_preedit_start_reply_handler (XIMS ims, IMPreeditCBStruct *call_data);
    int ims_preedit_caret_reply_handler (XIMS ims, IMPreeditCBStruct *call_data);

    void ims_commit_string (const X11IC *ic, const WideString& str);
    void ims_forward_key_event (const X11IC *ic, const KeyEvent& key);
    bool ims_wcstocts (XTextProperty &tp, const X11IC *ic, const WideString& src);

    bool ims_is_preedit_callback_mode (const X11IC *ic);
    void ims_preedit_callback_start (X11IC *ic);
    void ims_preedit_callback_done (X11IC *ic);
    void ims_preedit_callback_draw (X11IC *ic, const WideString& str, const AttributeList & attrs = AttributeList ());
    void ims_preedit_callback_caret (X11IC *ic, int caret);

    void ims_turn_on_ic (X11IC *ic);
    void ims_turn_off_ic (X11IC *ic);

    void ims_sync_ic (X11IC *ic);

    void set_focus_ic (X11IC *ic);
    void unset_focus_ic ();

    bool is_forward_mode () {
        return !validate_ic (m_focus_ic) || !m_focus_ic->xims_on;
    }

    bool socket_connect_panel (int argc, char **argv);
    bool socket_prepare_transaction (const X11IC *ic);
    bool socket_send_request (void);
    bool socket_receive_reply (void);
    bool socket_open_connection ();

    void socket_req_turn_on_panel (const X11IC *ic);
    void socket_req_turn_off_panel (const X11IC *ic);
    void socket_req_update_display (const X11IC *ic);
    void socket_req_update_screen (const X11IC *ic);
    void socket_req_show_help (const X11IC *ic);
    void socket_req_show_factory_menu (const X11IC *ic);

    void socket_req_focus_in (const X11IC *ic);
    void socket_req_focus_out (const X11IC *ic);
    void socket_req_update_factory_info (const X11IC *ic);
    void socket_req_update_spot_location (const X11IC *ic);
    void socket_req_show_preedit_string (const X11IC *ic);
    void socket_req_hide_preedit_string (const X11IC *ic);
    void socket_req_update_preedit_string (const X11IC *ic, const WideString &str, const AttributeList &attrs);
    void socket_req_update_preedit_caret (const X11IC *ic, int caret);
    void socket_req_show_aux_string (const X11IC *ic);
    void socket_req_hide_aux_string (const X11IC *ic);
    void socket_req_update_aux_string (const X11IC *ic, const WideString &str, const AttributeList &attrs);
    void socket_req_show_lookup_table (const X11IC *ic);
    void socket_req_hide_lookup_table (const X11IC *ic);
    void socket_req_update_lookup_table (const X11IC *ic, const LookupTable &table);
    void socket_req_register_properties (const X11IC *ic, const PropertyList &properties);
    void socket_req_update_property (const X11IC *ic, const Property &property);

    String get_help_info (const X11IC *ic);

    bool check_socket_connection (const Socket &socket);

    void launch_panel (int argc, char **argv);

    void reload_config_callback (const ConfigPointer &config);

    void fallback_commit_string_cb (IMEngineInstanceBase * si, const WideString & str);

private:
    static int ims_protocol_handler (XIMS ims, IMProtocol *call_data);

    static bool match_key_event (const std::vector <KeyEvent> &keys, const KeyEvent &key);

    static int  x_error_handler (Display *display, XErrorEvent *error);

    static bool validate_ic (const X11IC * ic) {
        if (ic && ic->si_id >= 0) return true;
        return false;
    }
};

#endif

/*
vi:ts=4:nowrap:ai:expandtab
*/
