
/*
 * Copyright (C) 1999-2001, Ian Main <imain@stemwinder.org>.
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */

#include <roy.h>

#ifdef BUFSIZ
#define FILE_BUFFER_SIZE BUFSIZ
#else
#define FILE_BUFFER_SIZE 4096
#endif

/* Uncomment to test chunk reading */
/* #define FILE_BUFFER_SIZE 1 */


static void
error_handler (RThermlParser *p, int linenum, int colnum, const char *description)
{
    printf ("Error line %d, col %d: %s\n", linenum, colnum, description);
}

static int lvl = 0;

static int
tag_attrib_value_handler (RThermlParser *parser, RBuf *attrib, RBuf *value)
{
    int i;

    for (i = 0; i < lvl; i++) {
        printf ("  ");
    }

    printf ("%s='%s'\n", rbuf_str (attrib), rbuf_str (value));
    rbuf_free (attrib);
    rbuf_free (value);

    return (0);
}

static int
tag_start_handler (RThermlParser *parser, RBuf *type)
{
    int i;

    printf ("\n");
    for (i = 0; i < lvl; i++) {
        printf ("  ");
    }

    lvl++;

    printf ("<%s>\n", rbuf_str (type));

    return (0);
}

static int
tag_end_handler (RThermlParser *p, RBuf *type, RBuf *data)
{
    int i;
    
    if (data) {
        for (i = 0; i < lvl; i++) {
            printf ("  ");
        }
        printf ("data='%s'\n", rbuf_str (data));
        rbuf_free (data);
    }
    
    lvl--;

    return (0);
}



static void
parse_file (RThermlParser *parser, char *name)
{
    FILE *p;
    char buf[FILE_BUFFER_SIZE];
    int nread;

    p = fopen (name,"r");
    if (!p) {
        printf ("Unable to open file '%s', exiting.\n", name);
        exit (-1);
    }

    while (!feof (p)) {
        nread = fread (buf, 1, FILE_BUFFER_SIZE, p);
    
        if (rtherml_parse_chunk (parser, buf, nread, nread != FILE_BUFFER_SIZE) != 0) {
            fclose (p);
            fprintf (stderr, "Aborting file read due to errors.\n");
            return;
        }
    }
    fclose(p);
}

/* When you do the loop test, no handlers etc. are hooked up, so it's the raw
 * speed of the parser, with no output to stdio etc. */

static void
loop_test (char *filename)
{  
    int i;

    printf ("looping 10000 times\n");

    for (i=0; i < 10000; i++) {
        RThermlParser *parser;

        parser = rtherml_parser_new ();
        parse_file (parser, filename);
        rtherml_parser_free (parser);  
    }
}


int
main (int argc, char **argv) 
{
    RThermlParser *parser;
    int do_loop = FALSE;
    int silent = FALSE;
    char *filename = NULL;
    char *arg;

    RARGV_SWITCH {
        case 'l':
            do_loop = TRUE;
            break;
        case 'g':
            arg = RARGV_NEXT;
            if (arg) {
                rdebug_enable (arg);
            } else {
                printf ("-g requires debug domain argument.\n");
                exit (0);
            }
            break;
        case 's':
            silent = TRUE;
            break;
        case '*':
            if (filename)
                printf ("Multiple files specified, ignoring previous\n");

            filename = RARGV_CURRENT;
            break;

        default:
            printf ("ignoring invalid option: %s\n", RARGV_CURRENT);
            break;
    } RARGV_CLOSE;

    rinit ();

    parser = rtherml_parser_new ();

    rtherml_set_error_handler (parser, error_handler);
    if (!silent) {
	    rtherml_set_attrib_value_handler (parser, tag_attrib_value_handler);
	    rtherml_set_node_start_handler (parser, tag_start_handler);
	    rtherml_set_node_end_handler (parser, tag_end_handler);
    }

    if (filename == NULL) {
        printf("Usage: testrtherml [-g debug domains] [-l -s] <filename>\n");
        return 1;
    }

    if (TRUE == do_loop) {
        loop_test (filename);
    } else {
        parse_file (parser, filename);
    }

    rtherml_parser_free (parser);

    rcleanup ();
    
    return (0);
}



