/*
 * robvars.h
 * 
 * Copyright (c) 2000-2004 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

/** @file
  * Declares basic RoboTour constants and enumerations
  * as well as some global variables and typedefs. 
  */

// In this file, you must not include anything!

#ifndef __RT_ROBVARS__
#define __RT_ROBVARS__

namespace rt {

/// RoboTour's version, user-displayed version and copyright text
extern const char* version; ///< version string for use by the program. must be parsable as a floating-point number.
extern const char* userVersion; ///< same as version, but formatted for the user. 
extern const char* aboutText; ///< copyright and release info
extern const int languageVersion;

/// The "RoboCom Integer" (rint) is an all-purpose 16-bit signed integer.
typedef short rint;

const rint MAX_RINT = 32767;
const rint MIN_RINT = -32768;

/////// for The Application:
enum VerboseMode {
	verbRemote = -999,	///< Remote control
	verbNone = 0,		///< Silence
	verbSome = 2,		///< Some output
	verbNormal = 5		///< Quite a lot of output
};


/////// for Simulation: ///////////////////////

/// Simulation variable array indices
enum SimVar {
	simFields = 0,		///< $FIELDS
	simPub,				///< #PUB
	simTime,			///< $TIME
	simMaxGeneration,	///< $MaxGeneration
	simMaxVars,			///< $MaxVars
	simMaxTasks,		///< $MaxTasks
	simMaxMybots,		///< $MaxMybots
	simTimeout,			///< $Timeout
	simElimTrigger,		///< $ElimTrigger
	simMaxLifetime,		///< $MaxLifetime
	simVarLength		///< sim variable count (array length)
};

/////// for Program: //////////////////////////
/// Header field types
enum HeaderType {
	headerInternal,
	headerSecret,
	headerPublished
};

/// Program variable array indices
enum ProgramVar {
	progMybots = 0,		///< $MYBOTS
	progOtherbots,		///< $OTHERBOTS
	progId,				///< $ID
	progVarLength		///< prog variable count (array length)
};

////// for Bot: ///////////////////////////////
/// Bot variable array indices
enum BotVar {
	botActive = 0,		///< #ACTIVE
	botNumBanks,		///< $NUMBANKS
	botGeneration,		///< $GENERATION
	botPosX,			///< x position on the field
	botPosY,			///< y position on the field
	botInstrSet,		///< $INSTRSET
	botMobile,			///< $MOBILE
	botNumTasks,		///< $TASKS
	botVarLength		///< bot variable count
};

////// for Task: //////////////////////////////
/// Instruction timing modes
enum TimingMode {
	timingNumExec = 0,	///< Number of actually executed cycles
	timingNumWaited	= 1	///< Number of cycles waited (only used for SLEEP)
};

/// Task variable array indices
enum TaskVar {
	taskCurBank = 0,	///< current executing bank (RUNBANK)
	taskCurInstr,		///< $INSTRPOS
	taskDir,			///< current heading (direction)
	taskNumCycles,		///< number of cycles the current instruction takes to execute
	taskCyclesDone,		///< number of cycles already done for the current instruction
// value is larger than a rint
//	taskInstrStarted,	///< game time when the instruction was started
	taskTimingMode,		///< current timing mode (see <tt>TimingMode</tt>)
	taskVarLength		///< task variable count
};


//////// for Instr: ///////////////////////////
/// Directions
enum Direction {
	dirRight = 0,
	dirUp = 1,
	dirLeft = 2,
	dirDown = 3
};

/// Turn directions 
enum TurnType {
	turnLeft = 1,
	turnRight = 3
};

/// Operand types (sources)
enum OpType {
	opTypeNumber,     //!< a rint literal (269)
	opTypeTask,		  //!< task variable ($INSTRPOS)
	opTypeActive,     //!< only #%ACTIVE
	opTypeNamedVar,   //!< a named robot variable or constant ($BANKS...)
	opTypeUnnamedVar, //!< the standard integers (#1..#20 (or more?))
	opTypeArray,      //!< array access to the standard integers (##1)
	opTypeProgram,    //!< a program variable ($ID...)
	opTypeGlobal,     //!< a simulation variable (#PUB...)
	opTypeFields,	  //!< only $FIELDS (because it's direction dependent)
	opTypeAge,		  //!< only $AGE (because it has to be calculated)
	opTypeLabel,      //!< a label-jump-call (@begin)
	opTypeAbsLabel,   //!< an absolute label call (@@begin)
	opTypeRandom = 99 //!< a random number ($RANDNUM)
};

/// Instr->exec() return types (error codes)
enum ExecReturnType {

	execSleeps = 1,		//!< returned by Task::exec() : Task sleeps, do another task

	execOK = 0,			//!< Instruction executed normally

	// For all failures: kill the bot
	// Need continuous fail codes! (see robstrings.h: failMsg[])
	failElim = -1,		//!< Elinination Trigger released
	failData = -2,		//!< Data Hunger (Bank 1 empty)
	failDiv0 = -3,		//!< Division by zero
	failBank = -4,		//!< Source bank not present in TRANS
	failInstrSet = -5,	//!< Higher Instruction Set required
	failMobile = -6,	//!< Mobility required
	failDie = -7,		//!< DIE executed
	failParam = -8,		//!< invalid parameter (e.g. Create x,-1,x)
	failUnemploy = -9,	//!< No more tasks left in a robot
	failInstrDur = -10,	//!< Instruction duration too high
	failAge = -11,		//!< Maximum Lifetime exceeded
	failQuit = -12,		//!< Quit executed, still enough tasks (This will not destroy the bot!)
	failLast = failQuit
};

/////// for SimSupervisor: /////////////////////
/// Preferred position in the supervisor list
enum SimSuperPosition {
	supPosFront = -100,		///< In the first part of the list
	supPosDontCare = 0,		///< Irrelevant for this supervisor
	supPosBack = 100,		///< In the last part of the list
	supPosBotCreator = 999,	///< Special value for supervisors which create Bots in initSim()
	supPosBotPresenter = 1000 ///< Special value for sup's which present the loaded bots in initSim()
};


/// Type of current game event
enum GameEventType {
	gameAbort = -1,		///< User abort
	gameNothing = 0,	///< No special event, continue game normally
	gameTie,			///< Game is tied (e.g. by a timeout)
	gameFinished		///< Game is finished, one of the programs won
};

////// for RobLoader: //////////////////////////
/// Constants
enum RobLoadConstants {
	rlMaxRecurse = 10		///< Maximum Define Recursion Depth
};

/// Load fail codes
enum RobLoadReturnType {
	rlOK = 0,					///< everything ok
	rlFailNoBanks = 1,			///< no banks defined in bot
	rlFailUnknownCommand = 2,	///< unknown command (e.g. AOGZ)
	rlFailUnknownParam = 3,		///< unknown parameter (e.g. $g02)
	rlFailWrongParCount = 4,	///< wrong number of parameters for instruction (e.g. MOVE 5)
	rlFailExpectComma = 5,		///< Comma expected
	rlFailExpectBank = 6,		///< Bank expected
	rlFailUndefinedLabel = 7,	///< Call of undefined label
	rlFailRedefinedLabel = 8,	///< Label declared twice
	rlFailNoFile = 9,			///< File not found (or not accessible)
	rlFailNoMemory = 10,		///< Insufficient memory
	rlFailRC3Required = 11,		///< RC3 instruction found, but disabled (e.g. MUL)
	rlFailMuTaRequired = 12,	///< MuTa instruction found, but disabled (e.g. INIT)
	rlFailTooManyBanks = 13,	///< Too many banks in bot
	rlFailTooManyInstrs = 14,	///< Too many instructions in bank
	rlFailDefines = 15,			///< Generic Define error
	rlFailWrongDefinePar = 16,	///< No such define parameter
	rlFailTooManyDefinePar = 17, ///< Too many define parameters
	rlFailExpectAnd = 18,		///< Expect "&"
	rlFailExpectBOpen = 19,		///< Expect "{"
	rlFailExpectBClose = 20,	///< Expect "}"
	rlFailExpectOpen = 21,		///< Expect "("
	rlFailExpectClose = 22,		///< Expect ")"
	rlFailExpectQuestionMark = 23, ///< Expect "?"
	rlFailRecurse = 24,			///< Too many recursion levels
	rlFailUndefinedDefine = 25,	///< Undeclared define
	rlFailRedefinedDefine = 26,	///< Define declared twice
	rlFailInvalidIdentifier = 27 ///< Invalid character in identifier (e.g. '<')
};

////// for RbiLoader: ///////////////////////////
/// Load fail codes
enum RbiLoadReturnType {
	rbOK = 0,					///< everything ok
	rbFailNoBanks = 1,			///< no banks defined in bot
	rbFailUnknownCommand = 2,	///< unknown command (e.g. AOGZ)
	rbFailUnknownParam = 3,		///< unknown parameter (e.g. $g02)
	rbFailWrongParCount = 4,	///< wrong number of parameters for instruction (e.g. MOVE 5)
	rbFailNoFile = 5,			///< File not found (or not accessible)
	rbFailNoMemory = 6,			///< Insufficient memory
	rbFailRC3Required = 7,		///< RC3 instruction found, but disabled (e.g. MUL)
	rbFailMuTaRequired = 8,		///< MuTa instruction found, but disabled (e.g. INIT)
	rbFailTooManyBanks = 9,		///< Too many banks in bot
	rbFailTooManyInstrs = 10,	///< Too many instructions in bank
	rbFailEndOfFile = 11,		///< End of file unexpected
	rbFailUndefined = 12		///< Other error
};

////// for ProgramLoader: ////////////////////////
enum GenLoadFailType {
	genFailLanguageVersion = 0	///< unsupported language version
};


} // namespace

#endif
