/*
 * robstreamif.cpp
 * 
 * Copyright (c) 2000-2004 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

// Implements the Stream Interface (see Streamif.txt)

#include "robstreamif.h"

#include <rtsystem.h>
#include <rtfile.h>

namespace rt {

////////////// Defines all known commands and options for them
////////////// These are used in the "commands" and "options" commands
////////////// options are only given for commands with special string options

/// writes the array, then _done_
void StreamInterface::writeArray(SifSimpleStringArray* arr) 
{
	for(int i = 0; i < arr->length; i++)
		write(arr->strs[i]);
	write("_done_");
}

////////////// setup mode
SifSimpleStringArray StreamInterface::setupCommands = { 
	13,
	{ "version", "langversion", "num", "randomize", "rco",
	  "insert", "assemble", "listbots", "ttype", "mode", 
	  "start", "commands", "options" }
};

SifSimpleStringArray StreamInterface::ttypeOptions = { 
	3,
	{ "Single","Charts", "All_In_One"
	}
};

SifSimpleStringArray StreamInterface::modeOptions = { 
	2,
	{ "Old", "RT3"
	}
};

/////////// run mode
SifSimpleStringArray StreamSimSupervisor::runCommands = { 
	11,
	{ "step", "cycle", "abort", "allbotnums", "allbotnames",
	  "allids", "display", "field", "next", "commands",
	  "options"
	}
};

SifSimpleStringArray StreamSimSupervisor::displayOptions = { 
	5,
	{ "Running_Bank_Owner", "Generation", "Num_Banks", "Running_Bank", "NONE"
	}
};

////////////// End definitions ///////////////////////////////

StreamInterface::StreamInterface(Frontend* parent) : 
	FrontendPlugin(parent), curLine(0), curPos(0), mode(sifModeOld), curBotNums(0)
{}
StreamInterface::~StreamInterface()
{}

String StreamInterface::read()
{
	while(!curLine || curPos >= curLine->length()) {
		delete curLine;
		String cl;
		do{
			cl = System::readLine("");
			cl = cl.trim();
		}while(cl.length() == 0);
		curLine = new Array<String>(cl.split(" \r\t\n", "\""));
		curPos = 0;
	}

	if((*curLine)[curPos] == "_exit_")
		System::exit();

	return (*curLine)[curPos++];
}

void StreamInterface::write(const String& result)
{
	System::println(result);
}

FpInterpretePos StreamInterface::getInterpretePos()
{
	return fpInterPosAfter;
}

String StreamInterface::getName()
{
	return "Stream Interface";
}

String StreamInterface::getHelpText()
{
	// The stream interface is secret and should not be presented to the user
	return "";
}

bool StreamInterface::interpreteParams(const Array<String>& params, Array<bool>& used)
{
	// fetch params from params (command line)
	for(int p = 0; p < params.length(); p++)
	{
		if(used[p]) continue;
		
		if((params[p] == "-v") || (params[p] == "-V"))
		{
			p++; // look at next arg
			if(used[p])
				parent->handleSystemError(1, "Parameter missing for argument -v");
			active = (params[p] == "-999");
		}
	}

	// if unactive, return
	if(!active) return true;

	// the stream interface is really not (user)-interactive
	System::setInteractive(false);

	// fetch params from stream
	String command, val;
	Globals tempGlob(parent);
	tempGlob.getGlobals("assemble.rco"); // that's a resource
	do{
		command = read();

		if(command == "version")
		{
			write(version);
			continue;
		}
		if(command == "langversion")
		{
			write(String("RC") + languageVersion);
			continue;
		}
		if(command == "commands")
		{
			writeArray(&setupCommands);
			continue;
		}
		if(command == "options")
		{
			val = read();
			if(val == "ttype")
			  writeArray(&ttypeOptions);
			else if(val == "mode")
			  writeArray(&modeOptions);
			else
			  write("_done_");
			continue;
		}
		if(command == "num")
		{
			val = read();
			parent->numRepeats = val.intValue(10);
			continue;
		}
		if(command == "randomize")
		{
			val = read();
			if(val == "false")
				parent->noRandom = true;
			else
				parent->noRandom = false;
			continue;
		}
		if(command == "rco")
		{
			val = read();
			parent->optionFile = val;
			continue;
		}
		if(command == "insert")
		{
			val = read();
			write(parent->files.length()); // index to which I'll put the program

			Program* prog = Program::create(val, tempGlob, -1);
			
			if((prog == 0) || !File(val).exists())
			{
				write("_unable_to_open_");
				continue;
			}
			else {
				write(prog->headers["name"].value);
				parent->files += val;
				delete prog;
				continue;
			}
		}
		if(command == "assemble")
		{
			val = read();
			Program* prog = Program::create(val, tempGlob, -1);
			if((prog == 0) || !File(val).exists())
			{
				write("_unable_to_open_");
				write("_done_");
				continue;
			}
			ErrorHandler* sifEH = new StreamErrorHandler(this);
			prog->getLoader()->setErrorHandler(sifEH);
			prog->getBanks();
			
			write("_done_");
			delete prog;
			delete sifEH;
			continue;
		}
		if(command == "listbots")
		{
			write(parent->files.length());
			
			for(int p = 0; p < parent->files.length(); p++)
			{
				write(p);
				Program* prog = Program::create(parent->files[p], tempGlob, -1);
				write(prog->headers["name"].value);
				delete prog;
			}
			continue;
		}
		if(command == "mode")
		{
			val = read();
			if(val == "RT3")
				mode = sifModeRT3;
			else if(val == "Old")
				mode = sifModeOld;
			continue;
		}
		if(command == "ttype")
		{
			val = read();
			if(val == "Single")
				parent->fightType = Frontend::ftSingle;
			else if(val == "Charts")
				parent->fightType = Frontend::ftCharts;
			else if(val == "All_In_One")
				parent->fightType = Frontend::ftAllInOne;
			continue;
		}
	}while(command != "start");

	if(mode == sifModeRT3)
		parent->doAllAgain = true;

	return true;
}

void StreamInterface::fillOptions(SimOptions& options)
{
	if(!active) return;

	options.supervisors += new StreamSimSupervisor(this);
	options.tourDisps += new StreamTourDisplayer(this);
	fightType = parent->fightType;
	curBotNums.clear();

}


/////////////// StreamSimSupervisor

StreamSimSupervisor::StreamSimSupervisor(StreamInterface* parent) : parent(parent),
	step(1), dispType(sifDispRunningBankOwner), retState()
{
	retState.event = gameNothing;
	retState.affectedProgram = 0;
}

StreamSimSupervisor::~StreamSimSupervisor() 
{}

SimSuperPosition StreamSimSupervisor::getPreferredPosition()
{
	return supPosBotPresenter;
}

unsigned long StreamSimSupervisor::getStepCycles()
{
	return step;
}

void StreamSimSupervisor::initSim(Simulation* const sim)
{
	retState.event = gameNothing;
	parent->curBotNums.clear();
	for(int p = 0; p < sim->getNumPrograms(); p++)
		parent->curBotNums += sim->getProgram(p)->globalNum;

	if((parent->fightType == Frontend::ftSingle) || 
		(parent->fightType == Frontend::ftCharts))
	{
		parent->write("_bots_");
		for(int q = 0; q < parent->curBotNums.length(); q++)
			parent->write(parent->curBotNums[q]);
	}
}

SimSupervisor::GameState StreamSimSupervisor::exec(Simulation* const sim)
{
	parent->write("_done_");

	String command, val;
	do{
		command = parent->read();

		if(command == "step")
		{
			val = parent->read();
			step = val.intValue(1);
			continue;
		}
		if(command == "cycle")
		{
			parent->write(sim->curCycle);
			continue;
		}
		if(command == "abort")
		{
			retState.event = gameAbort;
			return retState;
		}
		
		if(command == "allbotnums")
		{
			parent->write(sim->getNumPrograms());
			for(int p = 0; p < sim->getNumPrograms(); p++)
			{
				Program* prog = sim->getProgram(p);
				parent->write(prog->programNum);
				parent->write(prog->vars[progMybots]);
			}
			parent->write("_done_");
			continue;
		}

		if(command == "allnames")
		{
			parent->write(sim->getNumPrograms());
			for(int p = 0; p < sim->getNumPrograms(); p++)
			{
				Program* prog = sim->getProgram(p);
				parent->write(prog->programNum);
				parent->write(prog->headers["name"].value);
			}
			parent->write("_done_");
			continue;
		}

		if(command == "allids")
		{
			parent->write(sim->getNumPrograms());
			for(int p = 0; p < sim->getNumPrograms(); p++)
			{
				Program* prog = sim->getProgram(p);
				parent->write(prog->programNum);
				parent->write(prog->globalNum);
			}
			parent->write("_done_");
			continue;
		}

        if(command == "display")
        {
			val = parent->read();
            if(val == "Running_Bank_Owner") 
              dispType = sifDispRunningBankOwner;
            if(val == "Generation") 
              dispType = sifDispGeneration;
            if(val == "Num_Banks") 
              dispType = sifDispNumBanks;
            if(val == "Running_Bank") 
              dispType = sifDispRunningBank;
            if(val == "NONE") 
              dispType = sifDispNone;
			continue;
        }

        if(command == "field")
        {
			if(parent->mode == sifModeOld) 
				parent->write(sim->glob.fields);	// Fields
			else if(parent->mode == sifModeRT3)
			{
				parent->write(sim->glob.fieldsX);	// FieldsX
				parent->write(sim->glob.fieldsY);	// FieldsY
			}
            for( int y = 0; y < sim->fieldHeight; y++ )
              for( int x = 0; x < sim->fieldWidth; x++ )
              {
				  Bot* bot = sim->field[x + y * sim->fieldWidth];
				  if(bot != 0)
				  {
					  parent->write(bot->owner->programNum);	// program num

					  if(parent->mode == sifModeOld)			// heading
					      parent->write(bot->tasks[0]->vars[taskDir]);
					  else if(parent->mode == sifModeRT3)
					  {
						  parent->write(bot->tasks.length());	// numTasks
						  for(int t = 0; t < bot->tasks.length(); t++)
							  parent->write(bot->tasks[0]->vars[taskDir]);
					  }

					  parent->write(bot->vars[botActive]);		// active

                      switch( dispType )
                      {
                        case sifDispRunningBankOwner:
                        {
							int end = ((parent->mode == sifModeRT3) ?  bot->tasks.length() : 1);
							
							for(int t = 0; t < end; t++) {		// running bank owner
								Program* owner = bot->banks[bot->tasks[t]->vars[taskCurBank]]->owner;
								if(owner) 
									parent->write(owner->programNum);
								else
									parent->write(bot->owner->programNum);
							}
						}
						break;
                        case sifDispGeneration:
                          parent->write(bot->vars[botGeneration]); // generation
                          break;
                        case sifDispNumBanks:
                          parent->write(bot->vars[botNumBanks]); // num banks
                          break;
                        case sifDispRunningBank:
						{
							int end = ((parent->mode == sifModeRT3) ?  bot->tasks.length() : 1);
							
							for(int t = 0; t < end; t++) {		// running bank
                                parent->write(bot->tasks[t]->vars[taskCurBank]);
							}
						}
                        break;
                        case sifDispNone:
                          break;
                        default:
						  parent->write("???");
                      }
                  }
                  else
                  {
                      parent->write("-1");	// no bot
                  }
                  parent->write("_next_");	// next bot
              }
            parent->write("_done_");
			continue;
        }

		if(command == "commands")
		{
			parent->writeArray(&runCommands);
			continue;
		}
		if(command == "options")
		{
			val = parent->read();
			if(val == "display")
			  parent->writeArray(&displayOptions);
			else
			  parent->write("_done_");
			continue;
		}

    }while(command != "next");
	return retState;
}

void StreamSimSupervisor::exitSim(Simulation* const sim, const GameState& simResult)
{
	parent->write("_gameover_");
	parent->write(sim->curCycle);

	if(parent->fightType == Frontend::ftAllInOne) 
	{
		if(simResult.event == gameFinished) // one of the players won
		{
			parent->write("_winner_");
			Program* prog = sim->getProgram(simResult.affectedProgram);
			parent->write(prog->globalNum); // id
			parent->write(prog->headers["name"].value);  // name
		}
		else
			parent->write("_tie_");
		parent->write("_done_");
	}

	// for other fight types, the TourDisplayer does the result display.

}


//////////////// StreamTourDisplayer

StreamTourDisplayer::StreamTourDisplayer(StreamInterface* parent) : parent(parent)
{
}

StreamTourDisplayer::~StreamTourDisplayer()
{}

void StreamTourDisplayer::init(const TourInitInfo& info)
{}

TourStatusType StreamTourDisplayer::update(const Array<TourResult>& status)
{
	// only display scores in single or charts mode
	if(!((parent->fightType == Frontend::ftSingle) ||
		(parent->fightType == Frontend::ftCharts)))
		return tourContinue;

	parent->write("_scores_");
	for(int p = 0; p < parent->curBotNums.length(); p++)
	{
		TourResult res = status[parent->curBotNums[p]];
		parent->write(parent->curBotNums[p]); // id
		parent->write(res.wins);	// wins
		parent->write(res.looses);	// looses
		parent->write(res.ties);	// ties
	}
	parent->write("_done_");
	return tourContinue; 
}

void StreamTourDisplayer::exit(TourStatusType exitType, const Array<TourResult>& result)
{
	parent->write("_ready_");
}

///////////////////// StreamErrorHandler

StreamErrorHandler::StreamErrorHandler(StreamInterface* parent) : parent(parent)
{
}

StreamErrorHandler::~StreamErrorHandler()
{}

void StreamErrorHandler::handleBotError(const Bot* affectedBot, ExecReturnType type) const
{
}

void StreamErrorHandler::handleLoadError(const String& affectedFile, const String& message) const
{
	parent->write(affectedFile + ": " + message);
}

void StreamErrorHandler::handleSystemError(int num, const String& message) const
{
}

} // namespace

