/*
 * rtstring.cpp
 * 
 * Copyright (c) 2000-2004 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

// String.cpp: Implementierung der Klasse String.
//
//////////////////////////////////////////////////////////////////////

#include "rtstring.h"
#include "rtchar.h"
#include "rtmath.h"

#ifdef __SYMBIAN32__
#include <e32std.h>
#else
#include <stdio.h>
#endif


namespace lrt {

//////////////////////////////////////////////////////////////////////
// Konstruktion/Destruktion
//////////////////////////////////////////////////////////////////////

String::String() : Vector<char>(0)
{


}

String::String(const char* cStr) : Vector<char>(cStrLength(cStr))
{
	System::copyMemory(cStr, data, len);
}

String::String(const char chr, int length) : Vector<char>(length)
{
	System::fillMemory(data, len, chr);
}
/*
String::String(const short* ucStr) : Vector<char>(cStrLength(cStr))
{
	for(int i = 0; i < len; i++)
		data[i] = (char)(cStr[i]);
}*/

String::String(int i, int radix, int minLength) : Vector<char>(0)
{
	if((radix < 2) || (radix > Char::MAX_RADIX))
		System::exit(-5,String("Unsupported radix: ") + radix);
	bool neg = false;
	if(i < 0) {
		neg = true;
		//i = -i;
	}
	if(i == 0) {
		(*this)+='0';
	}
	else { // something to do
		int last = -1;
		do { // compute characters
			(*this)+= Char::charValue(Math::abs(i % radix));
			i /= radix;
			last++;
		}while(i != 0);
		// inverse order in string
		for(int j = last / 2; j >= 0; j--)
		{
			char temp = data[j];
			data[j] = data[last - j];
			data[last - j] = temp;
		}
		if(neg) insert('-', 0);
	}
	if(len < minLength)
		for(int s = len; s < minLength; s++)
			(*this)+= ' ';
}

String::String(double d, int length) : Vector<char>(0)
{
#ifdef __SYMBIAN32__
	TBuf<KDefaultRealWidth + 3> buf;
	TRealFormat myFormat;
	myFormat.iPoint = '.';
	buf.Num(d, myFormat);
	(*this) += (char*)(buf.PtrZ());
#else
	char* buf = new char[15];
	sprintf(buf, "%g", d);
	(*this) += buf;
#endif

	// we're too long, try to cut it (if after the dot)
	if(len > length) {
		int pidx = indexOf('.');
		len = Math::max(length, pidx);
	}
	// we're too short, append spaces
	else if(len < length) {
		check(length);
		for(int i = len; i < length; i++)
			data[i] = ' ';
	}
	
#ifndef __SYMBIAN32__ // cleanup buffer for ANSI C implementation
	delete[] buf;
#endif
}

String::String(const String& str) : Vector<char>(str.len)
{
   copy(&str, 0, this, 0, str.len);
}
/*
String String::operator += (const char*  cStr)
{
  Vector<char>::operator+=( String(cStr) );
  return *this;
}
*/
String& String::operator += (const int i)
{
  Vector<char>::operator+=( String(i) );
  return *this;
}

String& String::operator += (const char ch)
{
  Vector<char>::operator+=( ch );
  return *this;
}

String& String::operator += (const String& str)
{
  Vector<char>::operator+= (str );
  return *this;
}

String& String::operator= (const String& str)
{
	Vector<char>::operator =(str);
	return *this;
}

/*
const char& String::operator[](int index) const
{
	return Array<char>::operator[](index);
}
char& String::operator[](int index)
{
	return Vector<char>::operator[](index);
}
*/

//! Returns a new string like this one, but with any whitespace 
//! from the beginning and end of the string removed.
String String::trim() const
{
	if(len == 0) { return String(); } // that could be critical
	int start = 0, end = len - 1;
	while((start < len) && Char::isSpace(data[start]))
		start++;
	while((end > start) && Char::isSpace(data[end]))
		end--;
	return String(*this, start, end + 1);
}

String String::substring(int start, int end) const
{
	// special cases, may lead to problems
	if(len == 0) { return String(); }
	if(end <= start) { return String(); }
	if(start == len) { return String(); }

	// so that the optional parameter works.
	if(end > len) end = len;

	checkConst(start);
	return String(*this, start, end);
}

String String::lowerCase() const
{
	String ret(*this);
	for(int i = 0; i < len; i++)
		ret.data[i] = Char::lowerCase(ret.data[i]);
	return ret;
}

String String::upperCase() const
{
	String ret(*this);
	for(int i = 0; i < len; i++)
		ret.data[i] = Char::upperCase(ret.data[i]);
	return ret;
}
/* // obsolete because Vector<> does now contain this method too
int String::indexOf(char ch, int fromIndex) const
{
	int curIndex = Math::max(fromIndex, 0);
	while(curIndex < len) {
		if(data[curIndex] == ch) return curIndex;
		curIndex++;
	}
	return -1;
}
*/

int String::indexOf(const String& str, int fromIndex) const
{
	if(str.len > len) return -1; // cannot be contained in this string
	int curIndex = Math::max(fromIndex, 0);
	while(curIndex <= len - str.len) {
		if(!compare(curIndex, str, 0, str.len)) return curIndex;
		curIndex++;
	}
	return -1;
}

int String::lastIndexOf(char ch, int fromIndex) const
{
	int curIndex = Math::min(fromIndex, len - 1);
	while(curIndex >= 0) {
		if(data[curIndex] == ch) return curIndex;
		curIndex--;
	}
	return -1;
}

int String::lastIndexOf(const String& str, int fromIndex) const
{
	if(str.len > len) return -1; // cannot be contained in this string
	int curIndex = Math::min(fromIndex, len - str.len);
	while(curIndex >= 0) {
		if(!compare(curIndex, str, 0, str.len)) return curIndex;
		curIndex--;
	}
	return -1;
}

String String::replace(char oldCh, char newCh) const
{
	String ret(*this);
	for(int i = 0; i < len; i++)
		if(ret.data[i] == oldCh)
			ret.data[i] = newCh;
	return ret;
}

String String::replace(const String& oldStr, const String& newStr) const
{
    String ret;
    int pos = 0, end;
    while(pos < len)
    {
      end = indexOf(oldStr, pos);
      if(end < 0) { 
        ret += substring(pos);
        break;
      }
      ret += substring(pos, end);
      ret += newStr;
      pos = end + oldStr.len;
    }
    return ret;
}

Array<String> String::split(const String& separators, const String& combiners, 
							int maxParts, SplitFlags flags) const
{
	if(maxParts <= 0) maxParts = Math::MAX_INT;

    Vector<String> parts(0);
        
    char currComb;
    int end; // hinters Ende
    int begin, nextBegin;
    bool isSep = true; // derzeitiger Teil ist Space
    
    String str = trim();
    
    while(str.length() > 0)
    {
      if(separators.indexOf(str.data[0]) >= 0)
      {
		if(isSep && (flags & CREATE_EMPTY)) // last was a separator too
			parts += String();
        isSep = true;
        end = 1;
        nextBegin = end;
        begin = 0;
      }
      else
      {
        isSep = false;
		// if we are at the last allowed part, just take the rest of string and stuff it 
		// into the parts as the last one
		if(parts.length() >= maxParts - 1) {
			parts += str;
			break;
		}

        int cbin = combiners.indexOf(str.data[0]);
        if(cbin >= 0)
        {
          currComb = combiners.data[cbin];
          end = str.indexOf(currComb, 1);
          if(end < 0)
          {
            end = str.length();
            nextBegin = end;
          }
          else nextBegin = end + 1;
          begin = 1;
        }
        else
        {
          end = str.findNextOf(separators, 0);
          nextBegin = end;
          begin = 0;
        }
      }

	  // add the last part to the string
      String part = str.substring(begin, end);
      str = str.substring(nextBegin);
      if(!isSep) parts += part; // part was a string, do add it to the array

      str = str.trim();
    }
	if(isSep && (flags & CREATE_EMPTY))
		parts += String(); // the last part was a separator, add an empty string after it

    return parts;
}

String String::join(const Array<String>& arr, const String& joiner, const int upTo)
{
	if((arr.length() == 0) || (upTo < 0)) {return String();}
    String ret;
    int end = Math::min(arr.length() - 1, upTo);
    for(int i = 0; i < end; i++)
    {
      ret += arr[i];
      ret += joiner;
    }
    ret += arr[end];
    return ret;
}

int String::intValue(int def, int radix) const
{
	if((radix < 2) || (radix > Char::MAX_RADIX))
		System::exit(-5,String("Unsupported radix: ") + radix);
	bool neg = false, valid = false;
	int ret = 0;
	for(int i = 0; i < len; i++)
	{
		if(Char::isDigit(data[i], radix))
		{
			ret = ret * radix + Char::numericValue(data[i]);
			valid = true;
		}
		else if(!valid)
		{
			if(data[i] == '-')
				neg = !neg;
			else if(data[i] == '+')
				; // do nothing
			else break; // invalid string
		}
		else break; // invalid string
	}
	if(!valid) return def;
	if(neg) ret = -ret;
	return ret;
}

#ifdef __SYMBIAN32__
#include "rtetools.cpp"
double String::doubleValue(double def) const
{
	TLex lex(ESTRING((*this)));
	double ret;
	if(lex.Val(ret, '.') < 0)
		return def;
	else return ret;
}
#else
double String::doubleValue(double def) const
{
	float t;
	int res = sscanf(cStr(), "%g", &t);
	if((res != 0) && (res != EOF))
		return (double)t;
	else
		return def;
}
#endif

const char*  String::cStr() const
{
  String* const localThis = 
    const_cast<String* const>(this);

  localThis->check(len); // increase capacity by 1
  data[len] = char(0);
  return data;
}

////////// destructor
String::~String()
{
}

/////////////// private
//! Copies the part from this[start] up to and excluding this[end] (i.e. including this[end-1])
//! to a new string.
//! Does not check start, end for correctness! That's why it's private.
String::String(const String& str, int start, int end) : Vector<char>(end - start)
{
	copy(&str, start, this, 0, end - start);
}

//! Returns a negative, zero, or a positive number as this is less, equal, or greater than other.
//! Does not check start, end for correctness! That's why it's private.
int String::compare(int thisPos, const String& other, int otherPos, int length) const
{
	char chThis, chOther;
	for(int i = 0; i < length; i++)
	{
		chThis = data[thisPos + i];
		chOther = other.data[otherPos + i];
		if(chThis != chOther)
			return chThis - chOther;
	}
	return 0;
}

//! Returns a negative, zero, or a positive number as this is less, equal, or greater than other.
//! Does not check start, end for correctness! That's why it's private.
int String::compareIgnoreCase(int thisPos, const String& other, int otherPos, int length) const
{
	char chThis, chOther;
	for(int i = 0; i < length; i++)
	{
		chThis = Char::lowerCase(data[thisPos + i]);
		chOther = Char::lowerCase(other.data[otherPos + i]);
		if(chThis != chOther)
			return chThis - chOther;
	}
	return 0;
}

int String::cStrLength(const char* cStr)
{
	int len = 0;
	while(cStr[len] != 0)
		len++;
	return len;
}
/*
int String::cStrLength(const short* cStr)
{
	int len = 0;
	while(cStr[len] != 0)
		len++;
	return len;
}*/

/*! Returns the first position of one of the characters in what, or the
          position of the last character in the string if none of
          them was found. */
int String::findNextOf(const String& what, int start) const
{
	int pos = start;
	while((pos < len) && (what.indexOf(data[pos]) < 0))
	  pos++;

	return pos;
}


/////////////////// not-a-member
String operator + (const String& s1, const String& s2)
{
	String ret(s1);
	ret += s2;
	return ret;
}
} // namespace

