/*
 *  Copyright (C) 2000 heXoNet Support GmbH, D-66424 Homburg.
 *  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

#include "rfbClient.h"

#include <iostream>
#include <fstream>

#include <stdio.h>
#include <stdlib.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <netdb.h>

#include <xclass/utils.h>
#include <xclass/OXClient.h>
#include <xclass/OXWindow.h>
#include <xclass/OXMsgBox.h>
#include <xclass/OXMainFrame.h>
#include <xclass/OXTextButton.h>
#include <xclass/OXComboBox.h>
#include <xclass/OXLabel.h>
#include <xclass/OXMenu.h>
#include <xclass/OString.h>
#include <xclass/OXFont.h>
#include <xclass/OXIcon.h>
#include <xclass/OPicture.h>
#include <xclass/OFileHandler.h>
#include <xclass/OXCanvas.h>
#include <xclass/OTimer.h>
#include <xclass/OXFileDialog.h>

#include <X11/cursorfont.h>
#include <X11/keysym.h>

#include "OXAbout.h"
#include "OXViewerApplet.h"
#include "OXConnectionOptions.h"
#include "OXConnectionDetails.h"
#include "OXPassword.h"

#include "XFramebuffer.h"
#include "ScaleFramebuffer.h"
#include "PixelFormatFramebuffer.h"

#include "d3des.h"


#ifdef USE_ZLIB_WARREN
#include "ZlibConnection.h"
#endif // USE_ZLIB_WARREN

#include "FBStreamRecorder.h"
#include "RFMacroRecorder.h"

#include "../icons/rfb_viewer_large.xpm"
#include "version.h"

#define DEBUG

#ifdef DEBUG
#include <iostream>
#define debug(msg) (cerr << msg << endl)
#else // DEBUG
#define debug(msg)
#endif // DEBUG




OXClient *clientX;

int doConnect( char *hostname, int port );


namespace rfb {


class Client95;

Client95 *client95;


static Cursor
CreateDotCursor( Display *dpy )
{
  Cursor cursor;
  Pixmap src, msk;
  static char srcBits[] = { 0,  4, 14,  4, 0 };
  static char mskBits[] = { 4, 14, 31, 14, 4 };
  XColor fg, bg;

  src = XCreateBitmapFromData(dpy, DefaultRootWindow(dpy), srcBits, 5, 5);
  msk = XCreateBitmapFromData(dpy, DefaultRootWindow(dpy), mskBits, 5, 5);
  XAllocNamedColor(dpy, DefaultColormap(dpy,DefaultScreen(dpy)), "black", &fg, &fg);
  XAllocNamedColor(dpy, DefaultColormap(dpy,DefaultScreen(dpy)), "white", &bg, &bg);
  cursor = XCreatePixmapCursor(dpy, src, msk, &fg, &bg, 2, 2);
  XFreePixmap(dpy, src);
  XFreePixmap(dpy, msk);

  return cursor;
}


class OXFramebufferFrame: public OXFrame
{
 public:
  XFramebuffer *framebuffer;
  GC gc;

  OXFramebufferFrame( OXWindow *parent,
                      unsigned int width,
		      unsigned int height )
   : OXFrame( parent, width, height, OWN_BKGND, 0 )
  {
    framebuffer = new XFramebuffer( _client->GetDisplay(),
                                    GetId(),
	                            width,
                                    height
                                  );
    XSetWindowAttributes attr;
    attr.backing_store = Always;
    attr.cursor = CreateDotCursor(clientX->GetDisplay() );
    XChangeWindowAttributes(clientX->GetDisplay(),
                            GetId(),
			    CWBackingStore | CWCursor,
			    &attr);


    AddInput(PointerMotionMask);
    AddInput(ButtonPressMask);
    AddInput(ButtonReleaseMask);
  }

  virtual int HandleMotion(XMotionEvent * event);
  virtual int HandleButton(XButtonEvent * event);


  virtual ODimension GetDefaultSize() const 
                 { return ODimension( framebuffer->width,
                                      framebuffer->height ); }
  virtual int refreshArea(XExposeEvent *event)
  {
//    framebuffer->update(event->x,event->y,event->width,event->height);
    return True;
  }
};



class Client95MainFrame: public OXMainFrame
{
 public:
  Client95MainFrame( Client95 *_client );
  virtual ~Client95MainFrame();

  virtual int HandleFileEvent(OFileHandler *fh, unsigned int mask);

  virtual int HandleTimer( OTimer *_timer );
  
  virtual int HandleKey(XKeyEvent * event);
    
  XComposeStatus compose;
 
  Client95 *client;
  OXCanvas *canvas;
  OXPopupMenu *popup;
};

class Client95Connection;




class Client95: public Client
{
  public:
    Client95( ConnectionInfo &_connectionInfo, int _fd0, int _fd1 );
    virtual ~Client95();

    virtual void handleProtocolVersion( ProtocolVersion &protocolVersion );
    virtual void handleVNCAuthentication( CARD8 challenge[16] );
    virtual void handleAuthenticated();
    virtual void handleAuthenticationFailed();
    virtual void handleServerInitialisation( ServerInitialisation &serverInitialisation );
    virtual void handleFramebufferUpdate( FramebufferUpdate &framebufferUpdate );
    virtual void handleFramebufferUpdateCompleted( FramebufferUpdate &framebufferUpdate );
    virtual void handleDecodeRectangle( Rectangle &rectangle );
    virtual void handleRectangleDecoded( Rectangle &rectangle );
    virtual void handleBell();
#ifdef USE_ZLIB_WARREN
    virtual void handleZlibEnabled();
#endif // USE_ZLIB_WARREN

    ConnectionInfo *connectionInfo;
    int fd0, fd1;
    OFileHandler *fh0;
    OFileHandler *fh1;
    Client95MainFrame *mainFrame;
    OTimer *timer;
    OXFramebufferFrame *fbf;
    int buttonMask;
    struct timeval tvStart;
    int needUpdate;

    struct {
      Client95Connection *client95;
#ifdef USE_ZLIB_WARREN
      ZlibConnection *buffered;
#else
      BufferedConnection *buffered;
#endif // USE_ZLIB_WARREN
    } connectionLayer;

    bool recordFBS;
    FBStreamRecorder *fbsRecorder;

    RFMacroRecorder *rfmRecorder;

    CARD32 getTimestamp();
};



Client95MainFrame::~Client95MainFrame()
{
  client->mainFrame = NULL;
  delete client;
}




CARD32 Client95::getTimestamp()
{
  struct timeval tv;
  gettimeofday( &tv, NULL );
  
  tv.tv_sec -= tvStart.tv_sec;
  tv.tv_usec -= tvStart.tv_usec;
  if ( tv.tv_usec < 0 ) {
    tv.tv_sec--;
    tv.tv_usec += 1000000;
  }
  return tv.tv_sec * 1000 + tv.tv_usec / 1000;
}



class Client95Connection: public Connection
{
 public:
  Client95Connection( Client95 *_client, Connection *_connection )
    : client( _client )
    , connection( _connection )
    {}

  virtual ~Client95Connection()
    {}

  virtual int send( unsigned char* _data, unsigned int _size )
    {
//      cerr << "send() " << _size << endl;
      if ( !client->fh1 )
        if ( client->fd0 == client->fd1 ) {
	  delete client->fh0;
//          client->fh1 = new OFileHandler( client->mainFrame,
//	                                  client->fd1,
//		                          XCM_READABLE | XCM_WRITABLE | XCM_EXCEPTION );
          client->fh1 = new OFileHandler( client->mainFrame,
	                                  client->fd1,
		                          XCM_READABLE | XCM_WRITABLE );
          client->fh0 = client->fh1;
	}
	else
          client->fh1 = new OFileHandler( client->mainFrame,
	                                  client->fd1,
		                          XCM_WRITABLE | XCM_EXCEPTION );
//      cerr << "return send()" << endl;
      return connection->send( _data, _size );
    }

  virtual int receive( unsigned char* _data, unsigned int _size )
    {
      int retval = connection->receive( _data, _size );
      if ( client->recordFBS ) client->fbsRecorder->record( _data, retval );
      return retval;
    }

  virtual bool isAlive()
    {
      return connection->isAlive();
    }

  Client95 *client;
  Connection *connection;
};



int OXFramebufferFrame::HandleMotion(XMotionEvent * event)
{
  if ( client95->connectionInfo->viewOnly ) return True;
  MessagePointerEvent message;
  message.message_type = 5;
  message.pointerEvent.button_mask = client95->buttonMask;
  message.pointerEvent.x_position = (event->x >= 0)? (event->x * client95->framebuffer->width / GetWidth()) : 0;
  message.pointerEvent.y_position = (event->y >= 0)? (event->y * client95->framebuffer->height / GetHeight()) : 0;
  client95->connection->send( (unsigned char*) &message, 6 );
  if ( client95->rfmRecorder )
    client95->rfmRecorder->recordPointer( message.pointerEvent.x_position,
                                          message.pointerEvent.y_position );
  return True;
}


  int OXFramebufferFrame::HandleButton(XButtonEvent * event) {
    if ( client95->connectionInfo->viewOnly ) return True;
    MessagePointerEvent message;
    message.message_type = 5;

    if ( event->type == ButtonPress )
      client95->buttonMask |= ( 1 << (event->button-1) );
    else
      client95->buttonMask &= ~( 1 << (event->button-1) );

    message.pointerEvent.button_mask = client95->buttonMask;
    message.pointerEvent.x_position = (event->x >= 0)? (event->x * client95->framebuffer->width / GetWidth()) : 0;
    message.pointerEvent.y_position = (event->y >= 0)? (event->y * client95->framebuffer->height / GetHeight()) : 0;
    client95->connection->send( (unsigned char*) &message, 6 );

    if ( client95->rfmRecorder ) {
      if ( event->type == ButtonPress )
        client95->rfmRecorder->recordButtonDown( event->button-1 );
      else
        client95->rfmRecorder->recordButtonUp( event->button-1 );
    }
      
    return True;
  }





  int Client95MainFrame::HandleKey(XKeyEvent * event) {
    MessageKeyEvent message;
    message.message_type = 4;
    message.keyEvent.down_flag = ( event->type == KeyPress );
    char tmp[10];
    int n;
    KeySym keysym;
    n = XLookupString(event, tmp, sizeof(tmp)-1, &keysym, &compose);
    tmp[n] = 0;
    message.keyEvent.key = keysym;

    if ( keysym == XK_F8 ) {
        popup->AddInput(PointerMotionMask);
        popup->AddInput(ButtonPressMask);
        popup->AddInput(ButtonReleaseMask);
        popup->PlaceMenu( event->x_root, event->y_root, false, false );
        popup->RemoveInput(PointerMotionMask);
        popup->RemoveInput(ButtonPressMask);
        popup->RemoveInput(ButtonReleaseMask);
	return True;
    }

    if ( client->connectionInfo->viewOnly ) return True;

    client->connection->send( (unsigned char*) &message, 8 );

    if ( client95->rfmRecorder ) {
      if ( event->type == KeyPress )
        client95->rfmRecorder->recordKeyDown( keysym );
      else
        client95->rfmRecorder->recordKeyUp( keysym );
    }

    return True;
  }






Client95MainFrame::Client95MainFrame( Client95 *_client95 )
  : OXMainFrame( clientX->GetRoot(), 32, 32 )
  , client( _client95 )
{
    ChangeOptions( GetOptions() | OWN_BKGND );
    compose.compose_ptr = NULL;
    compose.chars_matched = 0;
    AddInput(KeyPressMask);
    AddInput(KeyReleaseMask);

    popup = new OXPopupMenu( _client->GetRoot() );
    popup->AddEntry( new OHotString("About"), 1000 );
    popup->AddEntry( new OHotString("Close Viewer"), 1001 );
    popup->Associate( this );
}









Client95::Client95( ConnectionInfo &_connectionInfo, int _fd0, int _fd1 )
  : Client()

  , connectionInfo( &_connectionInfo )
  , fd0( _fd0 )
  , fd1( _fd1 )
  
  , fh0( NULL )
  , fh1( NULL )

  , buttonMask( 0 )
  , recordFBS( false )
  , fbsRecorder( NULL )
  , rfmRecorder( NULL )
{
  client95 = this;
  framebuffer = NULL;
  mainFrame = new Client95MainFrame(this);

#ifdef USE_ZLIB_WARREN
  connectionLayer.buffered = new ZlibConnection();
#else
  connectionLayer.buffered = new BufferedConnection();
#endif // USE_ZLIB_WARREN

  connection = new Client95Connection( this, connectionLayer.buffered );

//  fh0 = new OFileHandler( mainFrame, fd0, XCM_READABLE | XCM_EXCEPTION );
  fh0 = new OFileHandler( mainFrame, fd0, XCM_READABLE );

  clientX->WaitFor( mainFrame );
}



Client95::~Client95()
{
  if ( rfmRecorder ) delete rfmRecorder;
  if ( fbsRecorder ) delete fbsRecorder;
  delete connectionLayer.buffered;
  delete connection;
  if ( fh0 ) delete fh0;
  if ( fh1 ) delete fh1;
  if ( mainFrame ) delete mainFrame;
}



void Client95::handleFramebufferUpdateCompleted( FramebufferUpdate &framebufferUpdate )
{
  Client::handleFramebufferUpdateCompleted( framebufferUpdate );
  if ( fbsRecorder ) recordFBS = false;
  if ( connectionInfo->framerate ) {
    needUpdate = true;
  } else {
    MessageFramebufferUpdateRequest message;
    message.message_type = 3;
    message.framebufferUpdateRequest.incremental = 1;
    message.framebufferUpdateRequest.x_position = 0;
    message.framebufferUpdateRequest.y_position = 0;
    message.framebufferUpdateRequest.width = framebuffer->width;
    message.framebufferUpdateRequest.height = framebuffer->height;
    connection->send( (unsigned char*) &message, 10 );
  }
  XSync( fbf->GetDisplay(), False );
}




int Client95MainFrame::HandleTimer( OTimer *_timer ) {
    delete client->timer;
    client->timer = new OTimer( this, 1000 / client->connectionInfo->framerate );
    if ( client->needUpdate ) {
       MessageFramebufferUpdateRequest message;
       message.message_type = 3;
       message.framebufferUpdateRequest.incremental = 1;
       message.framebufferUpdateRequest.x_position = 0;
       message.framebufferUpdateRequest.y_position = 0;
       message.framebufferUpdateRequest.width = client->framebuffer->width;
       message.framebufferUpdateRequest.height = client->framebuffer->height;
       client->connection->send( (unsigned char *) &message, 10 );
       client->needUpdate = false;
    }
    return True;
}




int Client95MainFrame::HandleFileEvent(OFileHandler *_fh, unsigned int mask) {
  BufferedConnection *connection = client->connectionLayer.buffered;
  int fd = _fh->GetFd();
  
//  cerr << "HandleFileEvent() " << fd << endl;

  if ( (mask & XCM_EXCEPTION) ) {
//    cerr << "XCM_EXCEPTION" << endl;

//    cerr << "return HandleFileEvent() XCM_EXCEPTION " << endl;
    delete this;
    return true;
  }

  if ( (mask & XCM_READABLE) && (fd == client->fd0) ) {
//    cerr << "XCM_READABLE" << endl;

    int readSize = connection->receiverBuffer.size - connection->receiverBuffer.end;
    int bytesRead;
    bytesRead = read( fd,
                      connection->receiverBuffer.data + connection->receiverBuffer.end,
                      readSize );
    
    if (bytesRead <= 0) {
//      cerr << "return HandleFileEvent() XCM_READABLE" << endl;
      delete this;
      return true;
    }
    connection->receiverBuffer.end += bytesRead;
    if ( client->fbsRecorder ) client->fbsRecorder->updateTimestamp();

    while ( connection->hasReceiverBufferData() ) client->update();
    connection->receiverBuffer.end = 0;
    connection->receiverBuffer.pos = 0;
  }


  if ( (mask & XCM_WRITABLE) && (fd == client->fd1) ) {
//    cerr << "XCM_WRITEABLE" << endl;

    int writeSize = connection->senderBuffer.end - connection->senderBuffer.pos;
    if ( writeSize > 0 ) {
      int bytesWritten =
        write( fd,
               connection->senderBuffer.data + connection->senderBuffer.pos,
               writeSize );
      if (bytesWritten < 0) {
//        cerr << "return HandleFileEvent() XCM_WRITABLE" << endl;
        delete this;
        return true;
      }
      connection->senderBuffer.pos += bytesWritten;
    }  else {
      delete client->fh1;
      if ( client->fd0 == client->fd1 )
//        client->fh0 = new OFileHandler( this, client->fd0, XCM_READABLE | XCM_EXCEPTION );
        client->fh0 = new OFileHandler( this, client->fd0, XCM_READABLE );
      client->fh1 = NULL;
    }
  }

//  cerr << "return HandleFileEvent() " << endl;
  return True;
}



void Client95::handleProtocolVersion( ProtocolVersion &protocolVersion )
{
  Client::handleProtocolVersion( protocolVersion );
  protocolVersion[12] = 0;
//  cerr << "RFB protocol version: " << protocolVersion;
}


void Client95::handleBell()
{
  XBell( fbf->GetDisplay(), 100 );
}



#ifdef USE_ZLIB_WARREN
void Client95::handleZlibEnabled()
{
//  cerr << "Zlib (WARREN) enabled" << endl;
  connectionLayer.buffered->enableReceiverInflation();
}
#endif


void Client95::handleVNCAuthentication( CARD8 challenge[16] )
{
  char key[16] = "\0\0\0\0\0\0\0\0";

  if ( !getPassword( key, sizeof (key), clientX->GetRoot(), mainFrame ) )
    delete this;
  
  unsigned char response[16];

  deskey( (unsigned char*) key, EN0 );
  des( challenge, response );
  des( challenge+8, response+8 );

  connection->send( response, 16 );
}


void Client95::handleAuthenticationFailed()
{
  mainFrame->Move((clientX->GetDisplayWidth() - mainFrame->GetWidth()) >> 1,
                  (clientX->GetDisplayHeight() - mainFrame->GetHeight()) >> 1);
  new OXMsgBox( clientX->GetRoot(), mainFrame,
                new OString( "RFB info" ),
                new OString( "Failed to connect to server:\nAuthentication failed!" ),
                MB_ICONEXCLAMATION,
                ID_OK );
  delete this;
}



void Client95::handleAuthenticated()
{
  if ( connectionInfo->recordFBS ) {
    OFileInfo finfo;
    char *filetypes[] = { "FrameBuffer Stream [*.fbs]", "*.fbs",
                          "All Files",          "*" ,
                          NULL,                 NULL };
    finfo.file_types = filetypes;
    mainFrame->Move((clientX->GetDisplayWidth() - mainFrame->GetWidth()) >> 1,
                    (clientX->GetDisplayHeight() - mainFrame->GetHeight()) >> 1);
    new OXFileDialog ( clientX->GetRoot(),
                       mainFrame,
                       FDLG_SAVE, &finfo );
    connectionInfo->recordFBS = (finfo.filename)? 1 : 0;
    if ( connectionInfo->recordFBS ) {
      sprintf( connectionInfo->filenameFBS, "%s/%s", finfo.ini_dir, finfo.filename );
//      cerr << "FBS - filename: " << connectionInfo->filenameFBS << endl;
      
      fbsRecorder = new FBStreamRecorder( connectionInfo->filenameFBS );
      fbsRecorder->record( (unsigned char*) "RFB 003.003\n", 12 );
      CARD32 auth = 1;
      fbsRecorder->record( (unsigned char*) &auth, 4 );
    }
  }
  if ( connectionInfo->recordRFM ) {
    OFileInfo finfo;
    char *filetypes[] = { "Remote Framebuffer Macro [*.rfm]", "*.rfm",
                          "All Files",          "*" ,
                          NULL,                 NULL };
    finfo.file_types = filetypes;
    mainFrame->Move((clientX->GetDisplayWidth() - mainFrame->GetWidth()) >> 1,
                    (clientX->GetDisplayHeight() - mainFrame->GetHeight()) >> 1);
    new OXFileDialog ( clientX->GetRoot(),
                       mainFrame,
                       FDLG_SAVE, &finfo );
    connectionInfo->recordRFM = (finfo.filename)? 1 : 0;
    if ( connectionInfo->recordRFM ) {
      sprintf( connectionInfo->filenameRFM, "%s/%s", finfo.ini_dir, finfo.filename );
//      cerr << "RFM - filename: " << connectionInfo->filenameRFM << endl;
      
      rfmRecorder = new RFMacroRecorder( connectionInfo->filenameRFM );
    }
  }


  ClientInitialisation message;
  message.shared_flag = connectionInfo->requestSharedSession;
  connection->send( (unsigned char*) &message, 1 );
  currentState = &stateInitialisation;

  gettimeofday( &tvStart, NULL );
}


void Client95::handleServerInitialisation( ServerInitialisation &serverInitialisation )
{
  Client::handleServerInitialisation( serverInitialisation );

//  cerr << "Desktop: " << serverInitialisation.name_string << endl;
//  cerr << "FB-Width: " << serverInitialisation.framebuffer_width << endl;
//  cerr << "FB-Height: " << serverInitialisation.framebuffer_height << endl;

  
  mainFrame->canvas = new OXCanvas( mainFrame,
                                    serverInitialisation.framebuffer_width,
				    serverInitialisation.framebuffer_height,
				    0 );


  if ( !connectionInfo->scale ) {
    fbf = new OXFramebufferFrame( mainFrame->canvas->GetViewPort(),
                                  serverInitialisation.framebuffer_width,
                                  serverInitialisation.framebuffer_height
                                );
    framebuffer = fbf->framebuffer;
    mainFrame->canvas->SetContainer( fbf );
     
    OLayoutHints *_lh = new OLayoutHints( LHINTS_EXPAND_X | LHINTS_EXPAND_Y, 0, 0, 0, 0);
    mainFrame->AddFrame( mainFrame->canvas, _lh );


    mainFrame->SetWindowName( (char*) serverInitialisation.name_string );
    mainFrame->SetIconName( (char*) serverInitialisation.name_string );
    mainFrame->SetWMSizeHints( 0, 0, serverInitialisation.framebuffer_width, serverInitialisation.framebuffer_height, 1, 1 );

    mainFrame->MapSubwindows();
    mainFrame->Resize( serverInitialisation.framebuffer_width,
                       serverInitialisation.framebuffer_height );
                       
  } else {
    fbf = new OXFramebufferFrame( mainFrame->canvas->GetViewPort(),
                                  connectionInfo->scaleFactor 
                                    * serverInitialisation.framebuffer_width
                                    / connectionInfo->scaleDivisor,
                                  connectionInfo->scaleFactor 
                                    * serverInitialisation.framebuffer_height
                                    / connectionInfo->scaleDivisor
                                );
    if ( connectionInfo->scaleSmooth )
        framebuffer = new SmoothScaleFramebuffer( *fbf->framebuffer,
                                            serverInitialisation.framebuffer_width,
                                            serverInitialisation.framebuffer_height
                                          );
    else
        framebuffer = new ScaleFramebuffer( *fbf->framebuffer,
                                            serverInitialisation.framebuffer_width,
                                            serverInitialisation.framebuffer_height
                                          );
    mainFrame->canvas->SetContainer( fbf );
    OLayoutHints *_lh = new OLayoutHints( LHINTS_EXPAND_X | LHINTS_EXPAND_Y, 0, 0, 0, 0);
    mainFrame->AddFrame( mainFrame->canvas, _lh );


    mainFrame->SetWindowName( (char*) serverInitialisation.name_string );
    mainFrame->SetIconName( (char*) serverInitialisation.name_string );
    mainFrame->SetWMSizeHints( 0, 0, fbf->GetWidth(), fbf->GetHeight(), 1, 1 );

    mainFrame->MapSubwindows();
    mainFrame->Resize( fbf->GetWidth(), fbf->GetHeight() );
  }
  
  
  mainFrame->MapWindow();

  XFlush( mainFrame->GetDisplay() );
  XSync( mainFrame->GetDisplay(), False );



  #ifdef USE_ZLIB_WARREN
  if ( connectionInfo->useZlib ) {
    CARD8 zlib = 10;
    CARD8 level = connectionInfo->useZlib;
    connection->send( (unsigned char*) &zlib, 1 );
    connection->send( (unsigned char*) &level, 1 );
  }
  #endif USE_ZLIB_WARREN




 if ( connectionInfo->restrictPixels ) {
   PixelFormat pf8bit;
   pf8bit.bits_per_pixel = 8;
   pf8bit.depth = 8;
   pf8bit.big_endian_flag = false;
   pf8bit.true_colour_flag = true;
   pf8bit.red_max = 7;
   pf8bit.green_max = 7;
   pf8bit.blue_max = 3;
   pf8bit.red_shift = 0;
   pf8bit.green_shift = 3;
   pf8bit.blue_shift = 6;
   framebuffer = new PixelFormatFramebuffer( *framebuffer, pf8bit );
 }

 if ( connectionInfo->useServerPixelFormat ) {
   framebuffer = new PixelFormatFramebuffer( *framebuffer, serverInitialisation.server_pixel_format );
 } else {
  MessagePixelFormat message;
  message.message_type = 0;
  message.pixelFormat = framebuffer->pixelFormat;
  connection->send( (unsigned char*) &message, 20 );
 }

  if ( connectionInfo->recordFBS ) {
    serverInitialisation.server_pixel_format = framebuffer->pixelFormat;
    fbsRecorder->record( (unsigned char*) &serverInitialisation, 24 );
    fbsRecorder->record( (unsigned char*) serverInitialisation.name_string,
                         serverInitialisation.name_length );
    fbsRecorder->startTimer();
  }
  
 if ( rfmRecorder ) rfmRecorder->startTimer();

 { 
   struct {
     CARD8 message_type;
     CARD8 padding;
     CARD16 number_of_encodings;
     CARD32 preferredEncoding;
     CARD32 copyRect;
   } Encodings;
   Encodings.message_type = 2;
   Encodings.number_of_encodings = (connectionInfo->allowCopyRect)? 2 : 1;
   Encodings.preferredEncoding = connectionInfo->preferredEncoding;
   Encodings.copyRect = 1;
   connection->send( (unsigned char*) &Encodings, (connectionInfo->allowCopyRect)? 12 : 8 );
 }
 
  if ( connectionInfo->framerate ) {
    needUpdate = false;
    timer = new OTimer( mainFrame, 1000 / connectionInfo->framerate );    
  }
  {
    MessageFramebufferUpdateRequest message;
    message.message_type = 3;
    message.framebufferUpdateRequest.incremental = 0;
    message.framebufferUpdateRequest.x_position = 0;
    message.framebufferUpdateRequest.y_position = 0;
    message.framebufferUpdateRequest.width = serverInitialisation.framebuffer_width;
    message.framebufferUpdateRequest.height = serverInitialisation.framebuffer_height;
    connection->send( (unsigned char*) &message, 10 );
  }

}


void Client95::handleFramebufferUpdate( FramebufferUpdate &framebufferUpdate )
{
  if ( fbsRecorder ) {
    CARD8 messageType = 0;
    fbsRecorder->record( (unsigned char*) &messageType, 1 );
    fbsRecorder->record( (unsigned char*) &framebufferUpdate, 3 );
    recordFBS = true;
  }
  Client::handleFramebufferUpdate( framebufferUpdate );
}



void Client95::handleDecodeRectangle( Rectangle &rectangle )
{
  Client::handleDecodeRectangle( rectangle );
}


void Client95::handleRectangleDecoded( Rectangle &rectangle )
{
 if ( rectangle.encoding_type != 1 )
    framebuffer->update( rectangle.x_position,
                         rectangle.y_position,
                         rectangle.width,
                         rectangle.height );
  Client::handleRectangleDecoded( rectangle );
}




} // namespace _003_003


char xrfbviewerCommand[1024];





void createViewerProcess( rfb::ConnectionInfo &connectionInfo, int fd0, int fd1 )
{
  if ( fork() ) {
    return;
  } else {
    clientX = new OXClient;

    OXMainFrame *mainWindow = new OXMainFrame(clientX->GetRoot(), 32, 32);
                                              mainWindow->Move((clientX->GetDisplayWidth() - mainWindow->GetWidth()) >> 1,
                                              (clientX->GetDisplayHeight() - mainWindow->GetHeight()) >> 1);
	    new rfb::Client95( connectionInfo, fd0, fd1 );
    exit( 0 );
  }
};



void printVersion_xrfbviewer()
{
  cerr << endl << "heXoNet RFB viewer for the X Window System"
       << endl << "Version " << VERSION_xrfbviewer
       << endl;
}

void printHelp_xrfbviewer()
{
  printVersion_xrfbviewer();
  cerr 
    << endl 
    << "usage: xrfbviewer [<options>] [<host>][:<display>]" << endl
    << "       xrfbviewer [<options>] -listen [<display>]" << endl
    << "       xrfbviewer [<options>] -stdio" << endl
    << "       xrfbviewer -about" << endl
    << endl
    << "<options>" << endl
    << "       -shared" << endl
    << "       -viewonly" << endl
    << "       -bgr233 | -spf" << endl
    << "       -scale <num>/<denum>" << endl;
  ;
  exit( 1 );
}


void parseCommandLine_xrfbviewer( int argc, char **argv )
{
  enum { UNDEF, ABOUT, INET, LISTEN, STDIO } mode;
  mode = UNDEF;

  rfb::ConnectionInfo connectionInfo;
  strcpy( connectionInfo.host, "" );

  int i = 1;
  while ( i < argc ) {

    if ( !strcmp( argv[i], "-help" ) ) {
      printHelp_xrfbviewer();
    } else

    if ( !strcmp( argv[i], "-listen" ) ) {
      if ( mode == UNDEF ) mode = LISTEN;
      else printHelp_xrfbviewer();
      i++;
    } else
    
    if ( !strcmp( argv[i], "-about" ) ) {
      if ( mode == UNDEF ) mode = ABOUT;
      else printHelp_xrfbviewer();
      i++;
    } else
    
    if ( !strcmp( argv[i], "-bgr233" ) ) {
      if ( !connectionInfo.useServerPixelFormat )
        connectionInfo.restrictPixels = true;
      else printHelp_xrfbviewer();
      i++;
    } else

    if ( !strcmp( argv[i], "-spf" ) ) {
      if ( !connectionInfo.restrictPixels )
        connectionInfo.useServerPixelFormat = true;
      else printHelp_xrfbviewer();
      i++;
    } else
    
    if ( !strcmp( argv[i], "-stdio" ) ) {
      if ( mode == UNDEF ) mode = STDIO;
      else printHelp_xrfbviewer();
      i++;
    } else
    
    if ( !strcmp( argv[i], "-scale" ) ) {
      if ( argc < i + 2 ) printHelp_xrfbviewer();
      connectionInfo.scale = true;
      char tmp[32];
      strncpy( tmp, argv[i+1], 31 );
      char *pos = strstr( tmp, "/" );
      (*pos) = 0; pos++;
      connectionInfo.scaleFactor  = atoi( tmp );
      connectionInfo.scaleDivisor = atoi( pos );
      i += 2;
    } else


// [<host>][:<display>] ?    
    if ( argv[i][0] != '-' ) {
      char *pos = strstr( argv[i], ":" );
      if ( pos ) {
        strncpy( connectionInfo.host, argv[i], pos - argv[i] );
        connectionInfo.host[pos - argv[i]] = 0;
	connectionInfo.display = atoi(pos+1);
      } else {
        strcpy( connectionInfo.host, argv[i] );
      }
      mode = INET;
      i++;
    } else printHelp_xrfbviewer();
  }


  
  switch( mode ) {
    case UNDEF: {
      OXMainFrame *mainWindow = new OXMainFrame(clientX->GetRoot(), 32, 32);
                                                mainWindow->Move((clientX->GetDisplayWidth() - mainWindow->GetWidth()) >> 1,
                                                (clientX->GetDisplayHeight() - mainWindow->GetHeight()) >> 1);

      char inifile[1024];
      sprintf( inifile, "%s/.xrfbviewer", getenv( "HOME" ) );
      readConnectionInfo( inifile, connectionInfo );
      if ( getConnectionDetails( connectionInfo, clientX->GetRoot(), mainWindow ) ) {
        if ( !connectionInfo.host[0] ) strcpy( connectionInfo.host, "localhost" );
        
        int fd = doConnect( connectionInfo.host, connectionInfo.display + 5900 );
        new rfb::Client95( connectionInfo, fd, fd );
      }
      exit( 0 );
    }
    break;

    case ABOUT: {
      OXAbout *about = new OXAbout( clientX->GetRoot(), new OXMainFrame( clientX->GetRoot(), 16, 16 ),
				    "About: heXoNet xrfbviewer V" VERSION_xrfbviewer,
				    "xrfbviewer version " VERSION_xrfbviewer,
				    "The RFB viewer from heXoNet",
				    clientX->GetPicture( "rfb_viewer_large", rfb_viewer_large_xpm )
				  );
      about->MapWindow();
      clientX->WaitFor( about );
      exit( 0 );
    }
    break;

    
    case INET: {
      if ( !connectionInfo.host[0] ) strcpy( connectionInfo.host, "localhost" );
      int fd = doConnect( connectionInfo.host, connectionInfo.display + 5900 );
      new rfb::Client95( connectionInfo, fd, fd );
      exit( 0 );
    }
    break;
    
    case LISTEN: {
      new OXViewerApplet( clientX, argv[0] );
      exit( 0 );
    }
    break;
      
    case STDIO: {
      new rfb::Client95( connectionInfo, 0, 1 );
      exit( 0 );
    }
    break;
      
  }
}




int main( int argc, char **argv ) {

  clientX = new OXClient;

  parseCommandLine_xrfbviewer( argc, argv );

  return 0;
}





void init_sockaddr (struct sockaddr_in *name, const char *hostname, uint16_t port)
{
  struct hostent *hostinfo;
  
  name->sin_family = AF_INET;
  name->sin_port = htons (port);
  hostinfo = gethostbyname (hostname);
  if (hostinfo == NULL)
    {
      printf( "Error\n" );
      exit (EXIT_FAILURE);
    }
  name->sin_addr = *(struct in_addr *) hostinfo->h_addr;
}


int doConnect( char *hostname, int port )
{
    struct sockaddr_in name;
    
    int s = socket( PF_INET, SOCK_STREAM, 0 );

    init_sockaddr( &name, hostname, port );

    if ( connect( s, (struct sockaddr*) &name, sizeof( name ) ) ) {
        printf( "connect Error\n" );
    }

    int one = 1;
    setsockopt(s, IPPROTO_TCP, TCP_NODELAY, (char *)&one, sizeof(one));
    fcntl(s, F_SETFL, O_NONBLOCK);

    return s;
}




