#ifndef PROSPECT_H_INCLUDED
#define PROSPECT_H_INCLUDED /* allow multiple inclusions */

/* 
 * Prospect: a developer's system profiler.
 *
 * COPYRIGHT (C) 2001-2004 Hewlett-Packard Company
 *
 * Author: Alex Tsariounov, HP
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: prospect.h,v 1.21 2004/01/09 20:29:28 type2 Exp $ */

/*
********************************************************************************
**
**                          PROSPECT PROJECT
**                    PROSPECT GLOBAL DEFINITIONS
**
********************************************************************************
*/


/*
 * --------------------- System include files -------------------------
 */
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <sys/param.h>
#include <sys/utsname.h>
#include <sys/time.h>
#include <unistd.h>
#include <errno.h>
#include <stdarg.h>
#include <stdio.h>

/*
 * ------------------ Prospect include files ------------------------
 */
#include "malloc_debug.h"
#include "linux_model.h"
#include "linux_module.h"

/*
 * ---------------- Things includes left out  ----------------------
 */
int setresuid(uid_t ruid, uid_t euid, uid_t suid);
int setresgid(gid_t rgid, gid_t egid, gid_t sgid);

/*
 * ---------------------- Prospect macros -------------------------
 */
#define mOUTPUT_IS_FILE  (gConf.strm_scr && gConf.strm_scr!=stdout)
#define mASC_TRACEOUT    (gConf.strm_rec)
#define mTRACEOUT        (gConf.fh_trace_out != -1)
#define mTRACEIN         (gConf.fh_trace_in != -1)

/*
 * Fast time macros - access the hardware counters.
 */
#ifdef __ia64__

/* For ia64, use ar44 register */
#define mTICKNOW(T) __asm__ __volatile__("mov %0=ar.itc" : "=r"(T) :: "memory");

#else

/* For ia32, use  the TSC register */

/* Note: RH (perhaps others too) keep not including the correct 
 *       /usr/include/asm/msr.h, so, just define it here and be done
 *       with it, and don't try to include the file.
 */
#ifndef rdtscl
#define rdtscl(low) \
     __asm__ __volatile__("rdtsc" : "=a" (low) : : "edx")
#endif

#define mTICKNOW(T)    rdtscl(T)

#endif


/*
 * ---------------------- Prospect defines ------------------------
 */

#define TRUE 1
#define FALSE 0

/* maximum buf size for saving command line */
#define MAXCOMMANDLEN 512

/* define following for dtree cache performance tests */
/* #define PERFTEST */

/*
 * --------------------- Prospect data types -----------------------
 */

/*
 * Which is the os we're running on?
 */
typedef enum eCURRENTOS
{
    cCO_UNKNOWN     = 0,
    cCO_LINUX_IA32  = 1,
    cCO_LINUX_IA64  = 2
} tCURRENTOS;

/*
 * Immediate child of prospect struct
 */
typedef struct _mychild {
    pid_t         pid;            /* the child pid */
    process_t    *p;              /* the child process struct ptr */
    
    int           signalled;      /* did we get signalled (gMyChild only) */
    int           terminated;     /* true = term, flase = stop */
    int           ts_signal;      /* terminated/stopped signal number */
} mychild_t;

/*
 * ----------------------- Prospect global data bits  --------------------------
 */

/*
 * Prospect program configuration structure
 */
typedef struct {
    struct {
        int follow_forks;             /* do all children */
        int system_wide;              /* do everything */
        int kernel_only;              /* kernel only profiles/not */
        int do_kernel;                /* output kernel profiles/not */
        int disassemble;              /* instruction level profile */
        int force_disassemble;        /* disassemble all functions */
        int merge_profiles;           /* merge profiles of same bin obj */
        int sort_by_system;           /* sort by system time, not user */
        int no_sort;                  /* don't sort output */
        int xml_output;               /* do xml output not ascii */
        int no_gettimeofday;          /* don't use gettimeofday() calls */
        int dis_debug;                /* debug disassembly system */
        int kmod_debug;               /* debug kernel modules system */
    } flags;
    
    char **argv;                      /* argument array */
    char **command;                   /* command line to execute */     
    char command_line[MAXCOMMANDLEN]; /* buffer for command line */

    char *my_name;                    /* this program's name */
    pid_t my_pid;                     /* this program's process id */
    pid_t my_sid;                     /* this program's session id */
    pid_t my_pgrp;                    /* this program's process group id */
    mychild_t my_child;               /* prospect's child */ 

    process_t **sort_base_pids;       /* ptr to bas of pid array that ran */
    unsigned int num_procs;           /* how many pid ran */
    tCURRENTOS current_os;            /* os we're built for */
    char *prospect_rev;               /* our revision */
    struct utsname my_utsname;        /* current running uname struct */
    char run_time[32];                /* current time of this run */
    double cpumhz;                    /* speed of cpu */
    op_cpu_t cputype;                 /* type of cpu from oprofile */
    unsigned int numcpus;             /* number of cpus */

    FILE *strm_scr;                   /* screen output filedes */
    FILE *strm_rec;                   /* ascii trace output filedes */
    FILE *strm_info;                  /* debug/inform trace output filedes */
    int fh_trace_out;                 /* binary trace output file handle */
    int fh_trace_in;                  /* binary trace input file handle */

    int bug_level;                    /* debug output level */
    double min_value;                 /* output cut off point in seconds */
    int rt_pri;                       /* real time priorty of this proc */
    int inherit_pri;                  /* exec'ed child inherits priority */
    char *system_map;                 /* system.map file for current kernel */
    char *kernel_image;               /* uncompressed kernel image loc */
    int dis_list_size;                /* num of gdb pipes to hold open */
    size_t page_size;                 /* well, the current page size */

} prospect_config_t;

extern prospect_config_t gConf;

/*
 * ------------------ Inform Tracing and ASCII Record Output -------------------
 */

/* Flag to turn on gathering traces of internal workings */
#define INFORM_LOGS 1

#ifdef INFORM_LOGS
#define mINFORM(args...) \
    do \
    { \
        if (gConf.strm_info) inform_trace(__LINE__, __FILE__, args); \
    } while(0)

#define mBUG(args...) \
    do \
    { \
        inform_trace(-__LINE__, __FILE__, args); \
    } while(0)

#else /* INFORM_LOGS */

#define mINFORM(args...)
#define mBUG(args...)

#endif /* INFORM_LOGS */


#define mPTREC(args...) \
    do \
    { \
        if (gConf.strm_rec) prec(args); \
    } while(0)

#define mTIME_MS(T) \
    do \
    { \
        if (gDoGettimeofday) T = get_time_ms(); \
    } while(0)

/*
 * -------------------------- Prospect prototypes ------------------------------
 */
#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#ifdef INFORM_LOGS
void inform_trace(const int, const char*, const char *, ...);
#endif

void prec(const char *fmt, ...);
void pscr(const char *Fmt, ...);
void pscr_fixlen(int len, const char *Fmt, ...);
void hint(const char *fmt, ...);
void ferr(const char *fmt, ...);
void ferr_nt(const char *fmt, ...);
void perr(const char *fmt, ...);
FILE *create_stream(char *s);
int create_fd(char *pStr);
tCURRENTOS get_built_for_os(void);
int correct_os(void);
int make_realtime(int pri);
void printversion(void);
void extract_basename(char *dest, const char *src);
int main(int argc, char *argv[]);
double get_time_ms(void);
void prospect_exit(int Code);
int amroot(void);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif  /* !PROSPECT_H_INCLUDED */

