##############################################################################
#
# Print billing management system - support modules, version 4.1.2
#
# Copyright (C) 2000, 2001, 2002, 2003 Daniel Franklin
#
# This module is free software; you can redistribute it and/or modify it
# under the same terms as Perl itself.
#
# This module provides an implementation of a basic ASCII database format.
# It stores information in the form key:value, with each pair on a separate
# line. The arguments are filename and whether or not it is to be read-only
# (you may still modify a read-only structure, but it won't be written to
# disk).
#
# Note that if the filename starts with http:// or ftp:// we will try to
# suck the file down using wget.
#
##############################################################################

package Printbill::PTDB_File;

use Carp;
use POSIX;
use strict;

sub TIEHASH
{
	my ($class, $filename, $rdonly) = @_;
	my ($err, $line, $key, $value, $tmpfile, $wipe);
	
	my $WGET = "/usr/bin/wget";

	$wipe = 0;
	
# If the filename starts with "http://" or "ftp://" we want to download the
# file first. If this fails, print a warning but don't change the filename.

	if ($filename =~ /^(ht|f)tp:\/\//) {
		if (-x $WGET) {
			$tmpfile = POSIX::tmpnam;
			$err = `$WGET $filename -Y off -O $tmpfile &> /dev/null`;
			
			if ($? >> 8) {
				print "$0 (PTDB_File::TIEHASH): call to $WGET failed: $err\n";
				return undef;
			}
			
			$filename = $tmpfile;
# We're not going to be able to write to it. However, we need to wipe it when done.
			$rdonly = 1;
			$wipe = 1;
		} else {
			print "
Warning - couldn't find $WGET. Normally I'd put this sort of thing in a
config file, but this code is in a perl module which shouldn't be dependent
on being able to read such a file. Hence you just get this warning. To fix
it so that we can find wget, please edit the file PTDB_File.pm and change
the line at the top which says my \$WGET = \"$WGET\" - or make a
symlink.\n\n";

			return undef;
		}
	}

	my $self = {
		FILENAME => $filename,
		LIST => {},
		RDONLY => $rdonly,
		WIPE => $wipe
	};

	if (-r $filename) {
		open TEXTFILE, "<$filename"
			or return undef;

		while ($line = <TEXTFILE>) {
			chomp $line;

# We allow comments - but they will be destroyed when written back

			if ($line !~ /^#.*$/) {
				($key, $value) = split /\s*:\s*/, $line;
				
				if (defined $key && defined $value) {

# This brutally un-taints the text. I'm sure this opens up all sorts of security holes...

					$key =~ /(.*)/;
					$key = $1;
					$value =~ /(.*)/;
					$value = $1;
				
					$self -> {LIST} -> {$key} = $value;
				}
			}
		}

		close TEXTFILE
			or return undef;

# We asked for it read-write... is it?
		stat ($filename);
		
		if (($self -> {RDONLY} eq "FALSE") && ! (-w $filename)) {
			return undef;
		}
	} elsif ($self -> {RDONLY} eq "TRUE") {
		return undef;
	} else { # Test for writability of a file which does not yet exist
		open TMP, ">$filename"
			or return undef;

		close TMP;
	}
	
	bless $self, $class;
}

sub FETCH
{
	my ($self, $key) = @_;
	
	if (defined ($self -> {LIST} -> {$key})) {
		return $self -> {LIST} -> {$key};
	} else {
		return undef;
	}
}

sub STORE
{
	my ($self, $key, $value) = @_;
	
	$self -> {LIST} -> {$key} = $value;
}

sub DELETE
{
	my ($self, $key) = @_;

	delete $self -> {LIST} -> {$key};
}

sub CLEAR
{
	my ($self) = @_;
	
	$self -> {LIST} = ();
}

sub EXISTS
{
	my ($self, $key) = @_;
	
	return exists $self -> {LIST} -> {$key};
}

sub FIRSTKEY
{
	my ($self) = @_;
	my $a = keys %{$self -> {LIST}};
	
	each %{$self -> {LIST}};
}

sub NEXTKEY
{
	my ($self) = @_;

	return each %{$self -> {LIST}};
}

sub DESTROY
{
	my ($self) = @_;

	$self -> sync ();
	unlink ($self -> {FILENAME}) if ($self -> {WIPE});
}

sub sync
{
	my ($self) = @_;
	my $key;
	
	return if ($self -> {RDONLY} eq "TRUE");

	open TEXTFILE, ">" . $self -> {FILENAME}
		or die "$0 (PTDB_File::sync()): unable to open \"" . $self -> {FILENAME} . "\" for writing: $!\n";
	
	foreach $key (keys %{$self -> {LIST}}) {
		print TEXTFILE "$key:" . $self -> {LIST} -> {$key} . "\n"
			or die "$0 (PTDB_File::sync()): unable to write \"" . $self -> {FILENAME} . "\": $!\n";
	}
	
	close TEXTFILE
		or die "$0 (PTDB_File::sync()): unable to close \"" . $self -> {FILENAME} . "\": $!\n";
}

1;
