
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <sys/types.h>

#include <debug/log.h>
#include <debug/memory.h>

#include <abz/typedefs.h>
#include <abz/atou32.h>
#include <abz/bprintf.h>

#include "config.h"

static struct config config;

static __attribute__ ((noreturn)) void usage (const char *progname)
{
   log_printf (LOG_ERROR,
			   "usage: %s [options] <interface> [expression]\n"
			   "\n"
			   "   -s | --snaplen=<bytes>     capture snaplen bytes of data from each packet\n"
			   "   -f | --flows=<n>           maximum number of flows to create\n"
			   "   -a | --active=<minutes>    active flow timeout in minutes\n"
			   "   -i | --inactive=<seconds>  inactive flow timeout in seconds\n"
			   "   -S | --syslog=<facility>   syslog facility to log to\n"
			   "   -h | --help                show this help message\n"
			   "\n",
			   progname);

   config_destroy ();

   exit (EXIT_FAILURE);
}

static __attribute__ ((noreturn)) void nomem (void)
{
   log_printf (LOG_ERROR,"failed to allocate memory: %m\n");
   config_destroy ();
   exit (EXIT_FAILURE);
}

static __attribute__ ((noreturn)) void fmterr (const char *fmt, ...)
{
   va_list ap;

   va_start (ap,fmt);
   log_vprintf (LOG_ERROR,fmt,ap);
   va_end (ap);

   config_destroy ();

   exit (EXIT_FAILURE);
}

static char *copy_argv (int argc,char *argv[])
{
   int i,len = 0;
   char *str;

   for (i = 0; i < argc; i++)
	 len += strlen (argv[i]) + 1;

   if ((str = mem_alloc (len + 1)) == NULL)
	 return (NULL);

   *str = '\0';
   len = 0;

   for (i = 0; i < argc; i++)
	 {
		if (i)
		  strcpy (str + len++," ");

		strcpy (str + len,argv[i]);
		len += strlen (argv[i]);
	 }

   return (str);
}

static int isfacility (const char *str)
{
   static const char *name[] =
	 {
		"authpriv",
		"cron",
		"daemon",
		"ftp",
		"kern",
		"lpr",
		"mail",
		"news",
		"syslog",
		"user",
		"uucp",
		"local0",
		"local1",
		"local2",
		"local3",
		"local4",
		"local5",
		"local6",
		"local7"
	 };
   size_t i;

   for (i = 0; i < ARRAYSIZE (name); i++)
     if (!strcmp (str,name[i]))
	   return (1);

   return (0);
}

const struct config *config_parse (int argc,char *argv[])
{
   static const struct option option[] =
	 {
		{ "snaplen", 1, NULL, 's' },
		{ "flows", 1, NULL, 'f' },
		{ "active", 1, NULL, 'a' },
		{ "inactive", 1, NULL, 'i' },
		{ "syslog", 1, NULL, 'S' },
		{ "help", 0, NULL, 'h' },
		{ NULL, 0, NULL, 0 }
	 };
   const char *facility;
   int finished = 0;
   uint32_t value;

   opterr = 0;
   optind = 1;

   memset (&config,0L,sizeof (struct config));

   (config.progname = strrchr (argv[0],'/')) != NULL ?
	 config.progname++ : (config.progname = argv[0]);

   config.snaplen = 128;
   config.flows = 64;
   config.active = 1800;
   config.inactive = 30;
   facility = "user";

   while (!finished)
	 {
		switch (getopt_long (argc,argv,"s:f:a:i:S:h",option,NULL))
		  {
		   case -1:
			 finished = 1;
			 break;
		   case 's':
			 if (atou32 (optarg,&value))
			   fmterr ("%s: invalid snaplen: %s\n",config.progname,optarg);
			 config.snaplen = value;
			 break;
		   case 'f':
			 if (atou32 (optarg,&value) || value < 1024 || value > 524288)
			   fmterr ("%s: maximum number of flows must be 1024-524288 entries\n",config.progname);
			 config.flows = value;
			 break;
		   case 'a':
			 if (atou32 (optarg,&value) || value < 1 || value > 60)
			   fmterr ("%s: active timeout must be 1-60 minutes\n",config.progname);
			 config.active = value * 60;
			 break;
		   case 'i':
			 if (atou32 (optarg,&value) || value < 10 || value > 600)
			   fmterr ("%s: inactive timeout must be 10-600 seconds\n",config.progname);
			 config.inactive = value;
			 break;
		   case 'S':
			 if (!isfacility (optarg))
			   fmterr ("%s: invalid syslog facility: %s\n",config.progname);
			 facility = optarg;
			 break;
		   case '?':
			 fmterr ("%s: invalid option or missing argument\n",config.progname);
		   default:
			 usage (config.progname);
		  }
	 }

   if (optind + 1 > argc)
	 fmterr ("usage: %s [options] <interface> [expression]\n"
			 "try `%s --help' for more information\n",
			 config.progname,config.progname);

   config.iface = argv[optind++];

   if (optind < argc && (config.expr = copy_argv (argc - optind,argv + optind)) == NULL)
	 nomem ();

   if ((config.syslog = bprintf ("%s.%s",config.progname,facility)) == NULL)
	 nomem ();

   return (&config);
}

void config_destroy (void)
{
   if (config.syslog != NULL)
	 mem_free (config.syslog);

   if (config.expr != NULL)
	 mem_free (config.expr);
}

