// ------------------------------------------------------------------------- //
// $Id: mainwindow.cpp,v 1.55 2003/12/16 22:33:21 pandr Exp $
// ------------------------------------------------------------------------- //
 
/*
 * Copyright (c) 2002 
 *				see AUTHORS list
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

// ------------------------------------------------------------------------- //

#if HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef WIN32			// let's just pretend that we care
# ifndef _WIN32_WCE
#  ifndef HAVE_OPENGL
#   define HAVE_OPENGL	// OpenGL is available on most Windows system
#  endif
# endif
# ifndef __GNUC__
#  include <windows.h>
# endif
#endif

#if HAVE_GL_GL_H
# include <GL/gl.h>
#endif

#if HAVE_OPENGL_GL_H	// MAC OS X
# include <OpenGL/gl.h>	
#endif

#if HAVE_GL_GLU_H
# include <GL/glu.h>
#endif

#if HAVE_OPENGL_GLU_H	// MAC OS X
# include <OpenGL/glu.h>
#endif

#if STDC_HEADERS
# include <stdio.h>
#endif

#if HAVE_STDARG_H
# include <stdarg.h>
#endif

#if HAVE_STRING_H
# include <string.h>
#elif HAVE_STRINGS_H
# include <strings.h>
#endif

#if HAVE_ALGORITHM
# include <algorithm>
#endif 

#include "common.h"
#include "image.h"
#include "node.h"
#include "nodefactory.h"
#include "mainwindow.h"
//#include "particle.h"

// ------------------------------------------------------------------------- //

MainWindow* MainWindow::_instance = 0;

MainWindow::MainWindow()
{
	if (_instance) {
		log_abort("Only one MainWindow supported");
	}

	_keybuf = new EventBuffer(100);

	_frames            = 0;
	_frames_since_time = 0;
	_frames_time       = 0.0f;
	_fps               = 0.0f;

	_last_idle = g_clock.time();

	_root = new Group("ROOT");

	_wireframe = false;
	_debug_osd = false;

	_fullscreen = false;
	_instance = this;
//	_particle_system = 0;
}

MainWindow::~MainWindow() 
{
	delete _keybuf;
	delete _root;
}

void MainWindow::new_root() 
{
	delete _root;
	_root = new Group("ROOT");
}

const Event* MainWindow::getevent(bool block)
{
	event(block);
	while (const Event* k =  _keybuf->peek()) {
		_keybuf->next();
		Event::_type_ et = k->get_type();
		if (et == Event::Key)
		{
			if (k->key().type == KeyEvent::DOWN) return k;
		} else if (et == Event::MouseButton) {
			return k;
		} else {
			return k;
		}
	}
	return 0;
}

void MainWindow::printString(int x, int y, const char *format, ... )
{
	char buf[1024];
	va_list ap;
	va_start (ap, format);
	vsnprintf(buf, sizeof(buf), format, ap);
	va_end(ap);
	glRasterPos2i(x*10, get_height() - (y+1)*13);
	glPushAttrib (GL_LIST_BIT);
	glListBase(m_font_offset);
	glCallLists((GLsizei) strlen(buf), GL_UNSIGNED_BYTE, (GLubyte *) buf);
	glPopAttrib ();
}

void MainWindow::init_gfx()
{
	make_rasterfont();
	init_gl();
//	_particle_system = new ParticleSystem();
}
	
void MainWindow::init_gl()
{
	glViewport(0, 0, get_width(), get_height());

	// set clearing color to black (RGBA is assumed!)
	glClearColor(0.0, 0.0, 0.0, 0.0);	

	// set up alpha-blend
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	// set up camera 
	glMatrixMode(GL_PROJECTION);	// prepare for the projection below
	glLoadIdentity();

	float fov = atan(float(get_height())/2.0f/1000.0f)*2.0f;
	gluPerspective(rad2deg(fov), float(get_width())/float(get_height()), 
			100.0f, 2000.0f);
	glTranslatef(-(float)get_width()/2.0f, -float(get_height())/2.0f, 
			-1000.0f);
}

void MainWindow::reset_time()
{
	g_clock.reset();
	_last_idle = g_clock.time() - 1.0f/60.0f; // Insert artificial frame
}

void MainWindow::frame() 
{
	float now = g_clock.time();
	float time_passed = now - _last_idle;

	_frame_time     = now;
	_frame_duration = time_passed;
	assert(_frame_duration > 0.0f);
	assert(_frame_time > 0.0f);

//	if (_particle_system) _particle_system->update(_frame_duration);

#if 0
    // For debugging of TileBank / textures
	static bool once = true;
	if (once) {
		once = false;
		TileBank::instance()->load_textures();
		Leaf* l = new Leaf("foo");
		l->add_primitive(new Primitive(TileBank::instance()->debug_texture(7),0, 0));
		l->set_pos(v3(500, 50, 0));
		_root->set_alpha(0.3f);
		_root->add(l);

		l = new Leaf("foo");
		l->add_primitive(new Primitive(TileBank::instance()->debug_texture(8),0, 0));
		l->set_pos(v3(500, 350, 0));
		_root->add(l);
	}
#endif


	_frames++;
	_frames_since_time++;
	if (_frames_time + 5.0f < g_clock.time()) {
		_frames_time       = g_clock.time();
		_frames_since_time = 1;
	}

	Node::_controllers_run = 0;
	_root->frame_update();
	display();
	_last_idle = now; 
}

void MainWindow::display()
{
	// clear the window to cleaning color
	glClear(GL_COLOR_BUFFER_BIT);	

	// let the user toggle the polygon mode
	if (_wireframe) {
		glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
	}
	else {
		glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
	}

	// Draw the slide
	glShadeModel(GL_SMOOTH);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	Primitive::_num_prims_drawn = 0;
	_root->draw_prims();
	if (_debug_osd) {
		print_at(18, 0, "Prims: %i", Primitive::_num_prims_drawn);
	}

//	if (_particle_system) _particle_system->draw();

	if (_overlay_items.size()) draw_osd();
	assert(_overlay_items.size() == 0);

	// Print any errors
	PrintGLError;

	// Finally, swap buffers
	swap_buffers();
}

void MainWindow::draw_osd()
{
	glLoadIdentity();
	gluOrtho2D(0.0, (float)get_width(), 0.0, (float)get_height());
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	glColor4f(1.0, 1.0, 1.0, 1.0);
	_fps = 1.0f / _frame_duration;
	if (_debug_osd) {
		print_at(2, 0, "Time: %5.2f  FPS: %7.3f", g_clock.time(), _fps);
	}

	for (_overlay_item_it i = _overlay_items.begin();
	     i != _overlay_items.end(); ++i)
	{
		printString((*i).x, (*i).y, (*i).str);
	}
	_overlay_items.clear();
}

void MainWindow::load_textures()
{
	TileBank::instance()->load_textures();
}

void MainWindow::print_at(uint x, uint y, const char *fmt, ... )
{
	_overlay_item i;
	va_list ap;
	va_start(ap, fmt);
	vsnprintf(i.str, sizeof(i.str), fmt, ap);
	va_end(ap);
	i.x = x;
	i.y = y;
	_overlay_items.push_back(i);
}

bool MainWindow::read_framebuffer(byte *buf, unsigned int buf_size)
{
	if (buf_size < get_width()*get_height()*4) {
		// Not room enough for a RGBA dump
		return false;
	}
	glReadBuffer(GL_BACK);
	glReadPixels(0, 0, get_width(), 
			get_height(), GL_RGBA, GL_UNSIGNED_BYTE, (GLvoid*)buf);
	if (glGetError() != GL_NO_ERROR) {
		return false;
	}
	return true;
}

bool MainWindow::make_screenshot(const char * filename, float scale)
{
	Image *image = new Image(get_width(), get_height(), 4);
	if (!image) { return false; }

	if (!read_framebuffer(image->get_buffer(), image->get_buffer_size())) {
		return false;
	}

	ImageSaver s;

	image->scale(
			(int)(image->get_width()*scale), (int)(image->get_height()*scale));

	bool ok = s.save(image, filename);

	delete image;

	return ok;
}

// ------------------------------------------------------------------------- //


void EventBuffer::insert(const Event &event)
{
	if ((_head+1)%_size == _tail) {
		// buffer full
		log_warning("EventBuffer full. Skipping events");
		return;
	}
	_eventbuf[_head] = event;
	_head = (_head + 1) % _size;
}

// ------------------------------------------------------------------------- //

