// ------------------------------------------------------------------------- //
// $Id: image.cpp,v 1.35 2003/03/15 10:15:01 weismann Exp $
// ------------------------------------------------------------------------- //

/*
 * Copyright (c) 2002 
 *				see AUTHORS list
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

// ------------------------------------------------------------------------- //

#if HAVE_CONFIG_H
# include <config.h>
#endif

#if STDC_HEADERS
# include <stdio.h>
# include <stdlib.h>
#endif

#if HAVE_STRING_H
# include <string.h>
#elif HAVE_STRINGS_H
# include <strings.h>
#endif

#if HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif

#if HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif

#if HAVE_FCNTL_H
# include <fcntl.h>
#endif

#if HAVE_UNISTD_H
# include <unistd.h>
#endif

#if HAVE_ASSERT_H
# include <assert.h>
#endif

extern "C"
{
	#include <png.h>
}

#ifndef png_infopp_NULL
# ifndef PNG_NO_TYPECAST_NULL
#  define png_infopp_NULL           (png_infopp)NULL
# else
#  define png_infopp_NULL           NULL
# endif
#endif


#include "common.h"
#include "image.h"

// ------------------------------------------------------------------------- //

Image::Image(int w, int h, int channels) : _width(w), _height(h),
	_channels(channels)
{
	_buf = new_buf();
	if (!_buf) {
		log_abort("Malloc failed in Image::Image");
	}
	memset(_buf, 0, get_buffer_size());
	assert(_channels == 4);
}

Image::~Image() { del_buf(_buf); }

void Image::default_texture()
{
	GLubyte *bp = _buf;
	for (uint h=0; h<_height; h++) {
		for (uint w=0; w<_width; w++) {
			*bp++ = 255 * h / _height;
			*bp++ = 0;
			*bp++ = 255 * w / _width;
			*bp++ = 255;
		}
	}
}

void Image::copy_from(Image* img, iv2 dst, Rect src)
{
	if (dst.x() + src.width()  > get_width() ||
	    dst.y() + src.height() > get_height())
	{
		log_warning("Image::copy_from(): Cannot fit src in dst");
		return;
	}
	if (src.top() > img->get_height() || src.right() > img->get_width())
	{
		log_warning("Image::copy_from(): Invalid args");
		return;
	}

	GLubyte *dst_bp = _buf + (get_width() * dst.y() + dst.x()) * _channels;
	GLubyte *src_bp = img->get_buffer() + (img->get_width() * src.bottom() 
			+ src.left()) * _channels;

	for (unsigned int row = 0; row < src.height(); row++) {
		memcpy(dst_bp, src_bp, _channels*src.width());
		dst_bp += _channels*get_width();
		src_bp += _channels*img->get_width();
	}
}

double Sinc(double x)
{
	if (x == 0.0) return 1.0;
	return sin(x*M_PI)/(x*M_PI);
}

double Lanczos(double x)
{
	if (x < 0.0) x = -x;
	if (x < 3.0) return Sinc(x)*Sinc(x/3.0);
	return 0.0;
}

void Image::scale(int new_width, int new_height)
{
	if ((uint)new_width != get_width()) {
		scale_horizontal(new_width);
	}
	if ((uint)new_height != get_height()) {
		swap_xy();
		scale_horizontal(new_height);
		swap_xy();
	}
}

void Image::swap_xy()
{
	GLubyte* newbuf = new_buf();
	for (uint y = 0; y < _height; y++) {
		for (uint x = 0; x < _width; x++) {
			for (uint c = 0; c < _channels; c++) {
				pixel_in_buf(newbuf,_height,_width,_channels,y,x)[c] = 
					pixel(x,y)[c];
			}
		}
	}
	replace_buf(newbuf, _height, _width, _channels);
}

void Image::scale_horizontal(int new_width)
{
	struct {
		int   pixelnum;
		float weight;
	} contrib[256];

	GLubyte* x_scaled_buf = new_buf();
	float xscale = (float)new_width / get_width();
	float blur = 1.0;
	float filterscale = blur * fmax(1.0/xscale, 1.0);
	for (int x = 0; x < new_width; x++) {
		//printf("Col: %i\n", x);
		float support = fmax(filterscale*3.0, 0.5);
		float center = (x+0.5)/xscale-0.5;
		int sample_begin = (int)fmax(center-support+0.5, 0);
		int sample_end   = (int)fmin(center+support+0.5, get_width()-1);
		assert(sample_end - sample_begin < 256);
		int samples = sample_end - sample_begin;
		assert(samples > 0);
		//std::cout << "samples: " << samples << std::endl;
		double density = 0;
		for (int n = 0; n < samples; n++) {
			int pixelnum = sample_begin + n;
			contrib[n].pixelnum = pixelnum;
			contrib[n].weight   = Lanczos((pixelnum-center)/filterscale);
			density += contrib[n].weight;
		}
		if (density != 0.0 && density != 1.0) {
			// Normalize
			density = 1.0 / density;
			for (int n = 0; n < samples; n++) {
				contrib[n].weight *= density;
			}
		}
		for (uint y = 0; y < get_height(); y++) {
			//printf("Creating (%i,%i)\n", x, y);
			GLubyte *dst = pixel_in_buf(x_scaled_buf,new_width,_height,_channels,x,y);
			for (uint c = 0; c < _channels; c++) {
				//printf("Mixing chan %i\n", c);
				float a = 0.0;
				for (int s = 0; s < samples; s++) {
					//printf("Adding %i w %f\n", pixel(contrib[s].pixelnum, y)[c], contrib[s].weight);
					a += pixel(contrib[s].pixelnum, y)[c]*contrib[s].weight;
				}
				//printf("Result: %f\n", a);
				if (a < 0.0)        { dst[c] = 0; }
				else if (a > 255.0) { dst[c] = 255; }
				else                { dst[c] = (byte)a; }
			}
		}	
	}
	replace_buf(x_scaled_buf, new_width, _height, _channels);
}

void Image::gray_alpha()
{
	GLubyte *bp = _buf;
	for (uint h = 0; h < _height; h++) {
		for (uint w = 0; w < _width; w++) {
			if (bp[0] == bp[1] && bp[1] == bp[2]) {
				// r == g == b
				bp[3] = 0xff - bp[0];
				bp[0] = bp[1] = bp[2] = 0xff;
				bp += _channels;
			}
		}
	}
}

// ------------------------------------------------------------------------- //

ImageLoader::ImageLoader()
{
	assert(sizeof(byte)==1);
	assert(sizeof(uint16)==2);
	assert(sizeof(uint32)==4);
}

ImageLoader::~ImageLoader() {}

Image* ImageLoader::load(const char *filename)
{
	png_structp png;
	png_infop   info;
	png_infop   endinfo;
	unsigned char header[8];
	png_uint_32 width, height;
	png_uint_32 i;
	int depth, color;
	png_bytep  *row_p;
	png_bytep   data;
	/* Create and initialize the png_struct with the desired error handler
	 * functions.  If you want to use the default stderr and longjump method,
	 * you can supply NULL for the last three parameters.  We also supply the
	 * the compiler header file version, so that we know if the application
	 * was compiled with a compatible version of the library.  REQUIRED
	 */
	png = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
	if (png == NULL) {
		log_warning("Coudn't create a read png struct");
		return 0;
	}
	/* Allocate/initialize the memory for image information.  REQUIRED. */
	info = png_create_info_struct(png);
	if (info == NULL) {
		log_warning("Coudn't create an info png struct");
		png_destroy_read_struct(&png, png_infopp_NULL, png_infopp_NULL);
		return 0;
	}
	endinfo = png_create_info_struct(png);
	if (endinfo == NULL) {
		log_warning("Coudn't create an info png struct");
		png_destroy_read_struct(&png, png_infopp_NULL, png_infopp_NULL);
		return 0;
	}
	FILE *fp = NULL;

	fp = fopen(filename, "rb");
	if (fp && fread(header, 1, 8, fp) && png_check_sig(header, 8)) {
    	/* Set up the input control if you are using standard C streams */
		png_init_io(png, fp);
	}
	else {
		log_warning("Failed in ImageLoader trying to read %s. File is not a proper png file", filename);
		png_destroy_read_struct(&png, &info, &endinfo);
		return 0;
	}
	png_set_sig_bytes(png, 8);

	/* The call to png_read_info() gives us all of the information from the
	 * PNG file before the first IDAT (image data chunk).  REQUIRED
	 */
	png_read_info(png, info);
	png_get_IHDR(png, info, &width, &height, &depth, &color, NULL, NULL, NULL);

	log_info("image %s",filename);
	log_info("height %d",height);
	log_info("width %d",width);
	log_info("depth %d",depth);
	log_info("color %d",color);

	/* Set up the data transformations you want.  Note that these are all
	   optional.  Only call them if you want/need them.  Many of the
	   transformations only work on specific types of images, and many
	   are mutually exclusive.  */

	/* tell libpng to strip 16 bit/color files down to 8 bits/color */
	png_set_strip_16(png);

	/* strip alpha bytes from the input data without combining with th
	 * background (not recommended) */
	/* png_set_strip_alpha(png); */

	/* extract multiple pixels with bit depths of 1, 2, and 4 from a single
	 * byte into separate bytes (useful for paletted and grayscale images).
	 */
	/* png_set_packing(png); */

	/* change the order of packed pixels to least significant bit first
	 * (not useful if you are using png_set_packing). */
	/* png_set_packswap(png); */

	/* expand paletted colors into true RGB triplets */
	if (color == PNG_COLOR_TYPE_PALETTE)
		png_set_expand(png);

	/* expand grayscale images to the full 8 bits from 1, 2, or 4 bits/pixel */
	if (color == PNG_COLOR_TYPE_GRAY && depth < 8)
		png_set_expand(png);

	/* expand paletted or RGB images with transparency to full alpha channels
	 * so the data will be available as RGBA quartets */
	if (png_get_valid(png, info, PNG_INFO_tRNS))
		png_set_expand(png);

	/* convert grayscale to RGB */
	if (color == PNG_COLOR_TYPE_GRAY)
		png_set_gray_to_rgb(png);

	/* Add filler (or alpha) byte (before/after each RGB triplet) */
	 png_set_filler(png, 0xff, PNG_FILLER_AFTER); 

	/* Turn on interlace handling.  REQUIRED if you are not using
	 * png_read_image().  To see how to handle interlacing passes,
	 * see the png_read_row() method below.
	 */

	png_read_update_info(png, info);

	unsigned int im_size = png_get_rowbytes(png, info)*height;
	data = (png_bytep) malloc(im_size);
	row_p = (png_bytep *) malloc(sizeof(png_bytep)*height);

	bool StandardOrientation = true;
	for (i = 0; i < height; i++)
	{
		if (StandardOrientation)
			row_p[height - 1 - i] = &data[png_get_rowbytes(png, info)*i];
		else
			row_p[i] = &data[png_get_rowbytes(png, info)*i];
	}

	//PNG_TRANSFORM_SWAP_ENDIAN instead of PNG_TRANSFORM_IDENTITY FIXME
	//png_read_png(png, info, PNG_TRANSFORM_IDENTITY, png_voidp_NULL);
	png_read_image(png, row_p);
	free(row_p);
	row_p=NULL; /* core will be dumped if row_p is used */
	png_read_end(png, endinfo);
	png_destroy_read_struct(&png, &info, &endinfo);
	if (fp)
    	fclose(fp); /* close the file */

	Image* im = new Image(width, height, 4);

	/* now we are friends of Image */
	GLubyte *im_buf = im->_buf;
	if (memcpy(im_buf,data,im_size) == 0)
		log_warning("memcpy failed in handling of a png image");
	free(data);
	data=NULL; /* core will be dumped if data is used */

	/* that's it */
	return im;

}

// ------------------------------------------------------------------------- //

ImageSaver::ImageSaver()
{
	assert(sizeof(byte)==1);
	assert(sizeof(uint16)==2);
	assert(sizeof(uint32)==4);
}

ImageSaver::~ImageSaver() {}

bool ImageSaver::save(Image *image, const char *filename)
{
	int height = image->get_height();
	int width  = image->get_width();

	byte* rgb_buffer = image->get_buffer();

	FILE *fp = fopen(filename, "wb");
	if (!fp) {
		log_warning("Failed in ImageSaver trying to open %s for writing.", 
				filename);
		return false;
	}

	png_structp png;
	png_infop   info;

	png = png_create_write_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
	if (png == NULL) {
		log_warning("Couldn't create a write png struct");
		fclose(fp);
		return 0;
	}

	info = png_create_info_struct(png);
	if (info == NULL) {
		log_warning("Couldn't create an info png struct");
		fclose(fp);
		return 0;
	}

	png_init_io(png, fp);

	uint bit_depth = 8;
	png_set_IHDR(png, info, width, height,
			bit_depth, PNG_COLOR_TYPE_RGBA, PNG_INTERLACE_NONE,
			PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_DEFAULT);

	// Create array of row-pointers. Each is a byte pointer
	// pointing to the start of a row in the rgba buffer.
	png_bytep *row_p;
	row_p = new png_bytep[height * sizeof(png_bytep)];
	for (int i = 0; i < height; i++) {
		row_p[height - 1 - i] = rgb_buffer + width*4*i;
	}
	png_set_rows(png, info, row_p);

	png_write_png(png, info, PNG_TRANSFORM_IDENTITY, NULL);

	delete [] row_p;
	row_p = NULL;
	png_destroy_write_struct(&png, &info);

	fclose(fp);
	return true;
}

// ------------------------------------------------------------------------- //

