// --------------------------------------------------------------------
// Pdftoipe: convert PDF file to editable Ipe XML file
// --------------------------------------------------------------------

#include <stdio.h>
#include "ocfile.h"
#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include "parseargs.h"
#include "gstring.h"
#include "gmem.h"
#include "globalparams.h"
#include "object.h"
#include "stream.h"
#include "array.h"
#include "dict.h"
#include "xref.h"
#include "catalog.h"
#include "page.h"
#include "pdfdoc.h"
#include "xmloutputdev.h"
#include "error.h"

static int firstPage = 1;
static int lastPage = 0;
static char ownerPassword[33] = "";
static char userPassword[33] = "";
static GBool quiet = gFalse;
static GBool printHelp = gFalse;
static GBool math = gFalse;
static GBool notext = gFalse;

static ArgDesc argDesc[] = {
  {"-f",      argInt,      &firstPage,      0,
   "first page to convert"},
  {"-l",      argInt,      &lastPage,       0,
   "last page to convert"},
  {"-opw",    argString,   ownerPassword,   sizeof(ownerPassword),
   "owner password (for encrypted files)"},
  {"-upw",    argString,   userPassword,    sizeof(userPassword),
   "user password (for encrypted files)"},
  {"-q",      argFlag,     &quiet,          0,
   "don't print any messages or errors"},
  {"-math",   argFlag,     &math,           0,
   "turn all text objects into math formulas"},
  {"-notext", argFlag,     &notext,         0,
   "discard all text objects"},
  {"-h",      argFlag,     &printHelp,      0,
   "print usage information"},
  {"-help",   argFlag,     &printHelp,      0,
   "print usage information"},
  {"--help",  argFlag,     &printHelp,      0,
   "print usage information"},
  {"-?",      argFlag,     &printHelp,      0,
   "print usage information"},
  {NULL, argFlag, 0, 0, 0}
};

int main(int argc, char *argv[])
{
  // parse args
  GBool ok = parseArgs(argDesc, &argc, argv);
  if (!ok || argc < 2 || argc > 3 || printHelp) {
    fprintf(stderr, "pdftoipe version %s\n", PDFTOIPE_VERSION);
    printUsage("pdftoipe", "<PDF-file> [<XML-file>]", argDesc);
    return 1;
  }

  GString *fileName = new GString(argv[1]);

  // read config file
  globalParams = new GlobalParams(0);
  if (quiet) {
    globalParams->setErrQuiet(quiet);
  }

  GString *ownerPW, *userPW;
  if (ownerPassword[0]) {
    ownerPW = new GString(ownerPassword);
  } else {
    ownerPW = 0;
  }
  if (userPassword[0]) {
    userPW = new GString(userPassword);
  } else {
    userPW = 0;
  }

  // open PDF file
  PDFDoc *doc = new PDFDoc(fileName, ownerPW, userPW);
  delete userPW;
  delete ownerPW;

  if (!doc->isOk())
    return 1;
  
  // check for print permission
  if (!doc->okToPrint()) {
    error(-1, "Printing this document is not allowed.");
    return 3;
  }

  // construct XML file name
  GString *xmlFileName;
  if (argc == 3) {
    xmlFileName = new GString(argv[2]);
  } else {
    char *p = fileName->getCString() + fileName->getLength() - 4;
    if (!strcmp(p, ".pdf") || !strcmp(p, ".PDF")) {
      xmlFileName = new GString(fileName->getCString(),
				fileName->getLength() - 4);
    } else {
      xmlFileName = fileName->copy();
    }
    xmlFileName->append(".xml");
  }

  // get page range
  if (firstPage < 1) {
    firstPage = 1;
  }
  if (lastPage < 1 || lastPage > doc->getNumPages()) {
    lastPage = doc->getNumPages();
  }

  // write XML file
  XmlOutputDev *xmlOut = 
    new XmlOutputDev(xmlFileName->getCString(), doc->getXRef(),
		     doc->getCatalog(), firstPage, lastPage, 
		     math, notext);
  int exitCode = 2;
  if (xmlOut->isOk()) {
    doc->displayPages(xmlOut, firstPage, lastPage, 72, 0, gFalse);
    exitCode = 0;
  }

  if (xmlOut->hasUnicode())
    fprintf(stderr, 
	    "The generated XML file contains Unicode (non-ASCII) text.\n"
	    "See Chapter 12 in the Ipe manual on how to handle this.\n"
	    "If your document is in a Western language, it may be "
	    "suffient to add this line:\n\n"
	    "\\usepackage{ucs}\n\n"
	    "to the documents Latex preamble (Edit->Document properties).\n");

  // clean up
  delete xmlOut;
  delete xmlFileName;
  delete doc;
  delete globalParams;

  // check for memory leaks
  Object::memCheck(stderr);
  gMemReport(stderr);

  return exitCode;
}

// --------------------------------------------------------------------
