;;;-*- Mode: Lisp; Package: CCL -*-
;;;
;;;   Copyright (C) 1994-2001 Digitool, Inc
;;;   This file is part of OpenMCL.  
;;;
;;;   OpenMCL is licensed under the terms of the Lisp Lesser GNU Public
;;;   License , known as the LLGPL and distributed with OpenMCL as the
;;;   file "LICENSE".  The LLGPL consists of a preamble and the LGPL,
;;;   which is distributed with OpenMCL as the file "LGPL".  Where these
;;;   conflict, the preamble takes precedence.  
;;;
;;;   OpenMCL is referenced in the preamble as the "LIBRARY."
;;;
;;;   The LLGPL is also available online at
;;;   http://opensource.franz.com/preamble.html


;; L1-processes.lisp

(cl:in-package "CCL")


(let* ((all-processes ())
       (shutdown-processes ())
       (all-processes-lock (make-lock)))
  (defun add-to-all-processes (p)
    (with-lock-grabbed (all-processes-lock)
      (pushnew p all-processes :test #'eq)
      p))
  (defun remove-from-all-processes (p)
    (with-lock-grabbed (all-processes-lock)
      (setq all-processes (delete p all-processes))
      t))
  (defun all-processes ()
    (with-lock-grabbed (all-processes-lock)
      (copy-list all-processes)))
  (defun shutdown-processes ()
    (with-lock-grabbed (all-processes-lock)
      (copy-list shutdown-processes)))
  (defun %clear-shutdown-proceses ()
    (setq shutdown-processes nil))
  (defun add-to-shutdown-processes (p)
    (with-lock-grabbed (all-processes-lock)
      (pushnew p shutdown-processes :test #'eq))
    t)
  (defun pop-shutdown-processes ()
    (with-lock-grabbed (all-processes-lock)
      (pop shutdown-processes)))
  (defun find-process (id)
    (etypecase id
      (process id)
      (integer (with-lock-grabbed (all-processes-lock)
		 (find id all-processes
		       :key #'(lambda (p)
				(process-serial-number p)))))
      (string (with-lock-grabbed (all-processes-lock)
		(find id all-processes
		      :key #'(lambda (p)
			       (process-name p))
		      :test #'equal))))))



(defun not-in-current-process (p operation)
  (if (eq p *current-process*)
    (error "The current process (~s) can't perform the ~a operation on itself."
	   p operation)))

(defun startup-shutdown-processes ()
  (let* ((p))
    (loop
      (unless (setq p (pop-shutdown-processes)) (return))
      (new-tcr-for-thread (process-thread p))
      (%process-preset-internal p)
      (process-enable p)
      )))

; Done with a queue-fixup so that it will be the last thing
; that happens on startup.
(queue-fixup
 (pushnew 'startup-shutdown-processes *lisp-system-pointer-functions*))







(defun wrap-initial-bindings (alist)
  (mapcar #'(lambda (pair)
              (destructuring-bind (symbol . valform) pair
                (cons (require-type symbol 'symbol)
                      (cond ((symbolp valform)
                             (constantly (symbol-value valform)))
                            ((typep valform 'function) valform)
                            ((consp valform)
                             (if (eq (car valform) 'quote)
                               (constantly (cadr valform))
                               #'(lambda () (apply (car valform) (cdr valform)))))
                            (t
                             (constantly valform))))))
          alist))

  
(let* ((psn -1))
  (defun %new-psn () (incf psn)))

(defclass process ()
    ((name :initform nil :initarg :name :accessor process-name)
     (thread :initarg :thread :accessor process-thread)
     (initial-form :initform (cons nil nil) :reader process-initial-form)
     (priority :initform 0 :initarg :priority :accessor process-priority)
     (persistent :initform nil :initarg :persistent :reader process-persistent)
     (whostate :initform "Reset" :accessor %process-whostate)
     (splice :initform (cons nil nil) :accessor process-splice)
     (initial-bindings :initform nil :initarg :initial-bindings
		       :accessor process-initial-bindings)
     (serial-number :initform (%new-psn) :accessor process-serial-number)
     (creation-time :initform (get-tick-count) :reader process-creation-time)
     (total-run-time :initform nil :accessor %process-total-run-time)
     (ui-object :initform (application-ui-object *application*)
                :accessor process-ui-object))
  
  (:primary-p t))

(defmethod print-object ((p process) s)
  (print-unreadable-object (p s :type t :identity t)
    (format s "~a(~d) [~a]" (process-name p)
	    (process-serial-number p)(process-whostate p))))

(defvar *process-class* (find-class 'process))

(defun processp (p)
  (memq *process-class* (class-precedence-list (class-of p))))

(set-type-predicate 'process 'processp)

(defun make-process (name &key 
			  thread
			  persistent
                          (priority 0)
                          (stack-size *default-control-stack-size*)
                          (vstack-size *default-value-stack-size*)
                          (tstack-size *default-temp-stack-size*)
                          (initial-bindings ())
			  (use-standard-initial-bindings t)
                          (class (find-class 'process)))
  (declare (ignore flavor))
  (let* ((p (make-instance
	     class
	     :name name
	     :thread (or thread
			 (new-thread name stack-size  vstack-size  tstack-size))
	     :priority priority
	     :persistent persistent
	     :initial-bindings (append (if use-standard-initial-bindings
					 (standard-initial-bindings))
				       (wrap-initial-bindings
					initial-bindings)))))
    (add-to-all-processes p)
    (setf (car (process-splice p)) p)
    p))


(defglobal *initial-process*
    (let* ((p (make-process
	       "Initial"
	       :thread *initial-lisp-thread*
	       :priority 0)))
      (setf (%process-whostate p) "Active")
      p))


(defvar *current-process* *initial-process*)

(defglobal *interactive-abort-process* *initial-process*)




(defun process-tcr (p)
  (%svref (process-thread p) ppc32::lisp-thread.tcr-cell))





(defun process-exhausted-p (p)
  (let* ((thread (process-thread p)))
    (or (null thread)
	(thread-exhausted-p thread))))
  

(defun process-whostate (p)
  (if (process-exhausted-p p)
    "Exhausted"
    (%process-whostate p)))





(defun process-total-run-time (p)
  (or (%process-total-run-time p)
      (thread-total-run-time (process-thread p))))




(defun initial-bindings (alist)
  (let* ((symbols ())
	 (values ()))
    (dolist (a alist (values (nreverse symbols) (nreverse values)))
      (push (car a) symbols)
      (push (funcall (cdr a)) values))))


                            
(defun symbol-value-in-process (sym process)
  (symbol-value-in-tcr sym (process-tcr process)))

(defun (setf symbol-value-in-process) (value sym process)
  (setf (symbol-value-in-tcr sym (process-tcr process)) value))


(defun process-enable (p &optional (wait 1))
  (setq p (require-type p 'process))
  (not-in-current-process p 'process-enable)
  (unless (car (process-initial-form p))
    (error "Process ~s has not been preset.  Use PROCESS-PRESET to preset the process." p))
  (let* ((thread (process-thread p)))
    (do* ((total-wait wait (+ total-wait wait)))
	 ((thread-enable thread wait)
	  (setf (%process-whostate p) "Active")
	  p)
      (cerror "Keep trying."
	      "Unable to enable process ~s; have been trying for ~s seconds."
	      p total-wait))))


(defun process-resume (p)
  (setq p (require-type p 'process))
  (%resume-tcr (process-tcr p)))

(defun process-suspend (p)
  (setq p (require-type p 'process))
  (%suspend-tcr (process-tcr p)))

(defun process-suspend-count (p)
  (setq p (require-type p 'process))
  (let* ((thread (process-thread p)))
    (if thread
      (lisp-thread-suspend-count thread))))



(defun process-active-p (p)
  (setq p (require-type p 'process))
  (and (eql 0 (process-suspend-count p))
       (not (process-exhausted-p p))))

  
; Used by process-run-function
(defun process-preset (process function &rest args)
  (let* ((p (require-type process 'process))
         (f (require-type function 'function))
         (initial-form (process-initial-form p)))
    (declare (type cons initial-form))
    (not-in-current-process p 'process-preset)
    ; Not quite right ...
    (rplaca initial-form f)
    (rplacd initial-form args)
    (%process-preset-internal process)))

(defun %process-preset-internal (process)
   (let ((initial-form (process-initial-form process))
         (thread (process-thread process)))
     (declare (type cons initial-form))
     (thread-preset
      thread
      #'(lambda (process initial-form)
	  (let* ((*current-process* process))
	    (add-to-all-processes process)
	    (multiple-value-bind (syms values)
		(initial-bindings (process-initial-bindings process))
	      (progv syms values
		(run-process-initial-form process initial-form)))))
      process
      initial-form)
     process))


(defun run-process-initial-form (process initial-form)
  (let* ((exited nil)
	 (kill (handler-case
		   (restart-case
		    (progn
		      (apply (car initial-form) (cdr (the list initial-form)))
		      (setq exited t)
		      nil)
                    (abort-break ())
		    (abort () (setq exited t)))
		 (process-reset (condition)
		   (process-reset-kill condition)))))
    ;; We either exited from the initial form normally, were told to
    ;; exit prematurely, or are being reset and should enter the
    ;; "awaiting preset" state.
    (if (or kill exited) 
      (unless (eq kill :toplevel)
	(process-initial-form-exited process (or kill t)))
      (progn
	(thread-change-state (process-thread process) :run :reset)
	(tcr-set-preset-state (process-tcr process))
	(setf (%process-whostate process) "Reset")))
    nil))

; Separated from run-process-initial-form just so I can change it easily.
(defun process-initial-form-exited (process kill)
  ;; Enter the *initial-process* and have it finish us up
  (without-interrupts
   (if (eq kill :shutdown)
     (progn
       (setf (%process-whostate process) "Shutdown")
       (add-to-shutdown-processes process)))
   (maybe-finish-process-kill process kill)))

(defun maybe-finish-process-kill (process kill)
  (when (and kill (neq kill :shutdown))
    (setf (%process-whostate process) "Dead")
    (remove-from-all-processes process)
    (let ((thread (process-thread process)))
      (unless (or (eq thread *current-lisp-thread*)
                  (thread-exhausted-p thread))
        (kill-lisp-thread thread))))
  nil)


 

(defun require-global-symbol (s &optional env)
  (let* ((s (require-type s 'symbol))
	 (bits (%symbol-bits s)))
    (unless (or (logbitp $sym_vbit_global bits)
		(let* ((defenv (if env (definition-environment env))))
		  (if defenv
		    (eq :global (%cdr (assq s (defenv.specials defenv)))))))
      (error "~s not defined with ~s" s 'defglobal))
    s))


(defmethod print-object ((s lock) stream)
  (print-unreadable-object (s stream :type t :identity t)
    (let* ((val (uvref s ppc32::lock._value-cell))
	   (name (uvref s ppc32::lock.name-cell)))
      (when name
	(format t "~s " name))
      (if (typep val 'macptr)
        (format stream "[ptr @ #x~x]"
                (%ptr-to-int val))))))

(defun lockp (l)
  (eq ppc32::subtag-lock (typecode l)))

(set-type-predicate 'lock 'lockp)

(defun recursive-lock-p (l)
  (and (eq ppc32::subtag-lock (typecode l))
       (eq 'recursive-lock (%svref l ppc32::lock.kind-cell))))

(defun read-write-lock-p (l)
  (and (eq ppc32::subtag-lock (typecode l))
       (eq 'read-write-lock (%svref l ppc32::lock.kind-cell))))

(setf (type-predicate 'recursive-lock) 'recursive-lock-p
      (type-predicate 'read-write-lock) 'read-write-lock-p)


(defun grab-lock (lock)
  (%lock-recursive-lock (recursive-lock-ptr lock)))

(defun release-lock (lock)
  (%unlock-recursive-lock (recursive-lock-ptr lock)))

(defun try-lock (lock)
  (%try-recursive-lock (recursive-lock-ptr lock)))

(defun process-wait (whostate function &rest args)
  (declare (dynamic-extent args))
  (or (apply function args)
      (let* ((p *current-process*)
             (old-whostate (process-whostate p)))
        (unwind-protect
             (progn
               (setf (%process-whostate p) whostate)
               (loop
                 (when (apply function args)
                   (return))
		   ;; Sleep for a tick
		   (%nanosleep 0 *ns-per-tick*)))
          (setf (%process-whostate p) old-whostate)))))


(defun process-wait-with-timeout (whostate time function &rest args)
  (declare (dynamic-extent args))
  (cond ((null time)  (apply #'process-wait whostate function args) t)
        (t (let* ((win nil)
                  (when (+ (get-tick-count) time))
                  (f #'(lambda () (let ((val (apply function args)))
                                    (if val
                                      (setq win val)
                                      (> (get-tick-count) when))))))
             (declare (dynamic-extent f))
             (process-wait whostate f)
             win))))


(defmethod process-interrupt ((process process) function &rest args)
  (let* ((p (require-type process 'process)))
    (if (eq p *current-process*)
      (apply function args)
       (let* ((thread (process-thread p)))
         (when (thread-exhausted-p thread)
           (error "process-interrupt run on exhausted ~s" p))
         (progn
	   (thread-interrupt
	    thread
	    process
            #'apply
	    function args))))))



; This one is in the Symbolics documentation
(defun process-allow-schedule ()
  (yield))


; something unique that users won't get their hands on
(defun process-reset-tag (process)
  (process-splice process))

(defun process-run-function (name-or-keywords function &rest args)
  (if (listp name-or-keywords)
    (%process-run-function name-or-keywords function args)
    (let ((keywords (list :name name-or-keywords)))
      (declare (dynamic-extent keywords))
      (%process-run-function keywords function args))))

(defun %process-run-function (keywords function args)
  (destructuring-bind (&key (name "Anonymous")
                            (priority  0)
			    (stack-size *default-control-stack-size*)
			    (vstack-size *default-value-stack-size*)
			    (tstack-size *default-temp-stack-size*)
			    (initial-bindings ())
                            (persistent nil)
			    (use-standard-initial-bindings t))
                      keywords
    (setq priority (require-type priority 'fixnum))
    (let* ((process (make-process name
                                  :priority priority
                                  :stack-size stack-size
				  :vstack-size vstack-size
				  :tstack-size tstack-size
                                  :persistent persistent
				  :use-standard-initial-bindings use-standard-initial-bindings
				  :initial-bindings initial-bindings)))
      (process-preset process #'(lambda () (apply function args)))
      (process-enable process)
      process)))

(defmethod process-reset ((process process) &optional kill)
  (setq process (require-type process 'process))
  (unless (memq kill '(nil :kill :shutdown))
    (setq kill (require-type kill '(member nil :kill :shutdown))))
  (if (eq process *current-process*)
    (%process-reset kill)
    (if (process-exhausted-p process)
      (maybe-finish-process-kill process kill)
      (progn
	(process-interrupt process '%process-reset kill)))))


(defun %process-reset (kill)
  (signal 'process-reset :kill kill)
  (maybe-finish-process-kill *current-process* kill))

;;; By default, it's just fine with the current process
;;; if the application/user wants to quit.
(defmethod process-verify-quit ((process process))
  t)

(defmethod process-exit-application ((process process) thunk)
  (when (eq process *initial-process*)
    (prepare-to-quit)
    (%set-toplevel thunk)
    (toplevel)))


(defmethod process-kill ((process process))
  (process-reset process  :kill))

(defun process-abort (process &optional condition)
  (process-interrupt process
                     #'(lambda ()
                         (abort condition))))

(defmethod process-reset-and-enable ((process process))
  (not-in-current-process process 'process-reset-and-enable)
  (process-reset process)
  (process-enable process))


(defun tcr->process (tcr)
  (dolist (p (all-processes))
    (when (eq tcr (process-tcr p))
      (return p))))


(def-standard-initial-binding *backtrace-contexts* nil)
