# vcc3client.tcl --
#
#       Provides a detailed interface to the vcc3d control of the Canon
#       Vcc3 cameras.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

Import enable

import DpClient

Class Vcc3Client

namespace eval Vcc3Client {
    variable complexFrame
    variable simpleFrame
}

global g_show g_focusMode

Vcc3Client instproc init {base hostname port} {
    $self instvar serv_ maxZoom_

    if {[catch {new DpClient $hostname $port} serv_]} {
        return -code error "Fatal error:  could not connext to camera server"
    }
    $self makeUI $base

    set maxZoom_ [$serv_ do vcc3_getMaxZoom]
    set numTries 1
    while {($maxZoom_ == -9999) && ($numTries < 10)} {
	set maxZoom_ [$serv_ do vcc3_getMaxZoom]
	set numTries [expr $numTries + 1]
    }
    if {$maxZoom_ == -9999} {
	return -code error "Vcc3Client::init: unable to get maxZoom"
    }
}

Vcc3Client instproc makeUI {base} {
    global g_show
    $self instvar serv_ frames_


    set Vcc3Client::complexFrame [frame $base.complexFrame]
#    pack $base.complexFrame -side top
    set origBase $base
    set base $base.complexFrame

    set curFrame $base.choiceFrame
    frame $curFrame
    pack $curFrame -side left

    set labelStr [$serv_ do vcc3_getLabel]
    label $curFrame.label -text $labelStr -foreground "blue"
    pack $curFrame.label -side top

    set controlList [list]
    lappend controlList [list "panTilt" "Pan/Tilt"]
    lappend controlList [list "zoom" "Zoom"]
    lappend controlList [list "presets" "Presets"]
    lappend controlList [list "focus" "Focus"]
    lappend controlList [list "exposure" "Exposure"]
    lappend controlList [list "fade" "Fade"]
    lappend controlList [list "stepMove" "Step Move"]

    foreach info $controlList {
	set control [lindex $info 0]
	set controlText [lindex $info 1]
	set curFrame $base.choiceFrame
	frame $curFrame.$control
	pack $curFrame.$control -side top -fill x
	set checks($control) [checkbutton $curFrame.$control.button -text $controlText -variable g_show($control) -command "$self show $control"]
	pack $curFrame.$control.button -side left
	# actual interface frames
	set curFrame $base.$control
	set frames_($control) [frame $curFrame -borderwidth 1 -relief solid]
	label $curFrame.label -text $controlText
	pack $curFrame.label
	set str "$self make_$control $curFrame"
	eval $str
    }
    button $base.choiceFrame.refresh -text "Refresh" -command "$self refresh"
    pack $base.choiceFrame.refresh -side top

    # allow a switch to simple mode which just has the arrows to have a small
    #    screen footprint
    button $base.choiceFrame.simple -text "Simple" -command "$self simpleMode"
    pack $base.choiceFrame.simple -side top
    $self makeSimpleUI $origBase

    $self refresh

    $checks(panTilt) invoke
    $checks(zoom) invoke

    $self simpleMode

    # to make debugging easier
#    foreach info $controlList {
#	set control [lindex $info 0]
#	$checks($control) invoke
#    }

}

Vcc3Client instproc make_panTilt {base} {
    $self instvar serv_ widgets_

#    label $base.label -text "Pan/Tilt"
#    pack $base.label

    button $base.home -text "Home" -command "$serv_ do vcc3_home"
    pack $base.home -side top

    # move arrows
    set cur_frame "$base.position_frame"
    frame $cur_frame
    pack $cur_frame -side top -pady 10 -padx 10
    label $cur_frame.label -text "Position"
    pack $cur_frame.label -side top
    set canv "$cur_frame.arrow_canvas"
    canvas $canv -width 100 -height 100 -borderwidth 0
    pack $canv -side top
    set uparrow " -6 -9	 -6 -23	 -14 -23  0 -35	 14 -23	 6 -23	 6 -9  -6 -9"
    set dnarrow " -6  9	 -6  23	 -14  23  0  35	 14  23	 6  23	 6  9  -6  9"
    set ltarrow " -9 -6	 -23 -6	 -23 -14  -35 0	 -23 14	 -23 6	 -9 6  -9 -6"
    set rtarrow "  9 -6	  23 -6	  23 -14   35 0	  23 14	  23 6	  9 6	9 -6"
    set p_up [eval $canv create polygon $uparrow -tags \{up arrow\}]
    set p_dn [eval $canv create polygon $dnarrow -tags \{dn arrow\}]
    set p_lt [eval $canv create polygon $ltarrow -tags \{lt arrow\}]
    set p_rt [eval $canv create polygon $rtarrow -tags \{rt arrow\}]
    $canv bind up <ButtonPress-1> "$serv_ do vcc3_startMove up"
    $canv bind up <ButtonRelease-1> "$serv_ do vcc3_stop"
    $canv bind dn <ButtonPress-1> "$serv_ do vcc3_startMove down"
    $canv bind dn <ButtonRelease-1> "$serv_ do vcc3_stop"
    $canv bind lt <ButtonPress-1> "$serv_ do vcc3_startMove left"
    $canv bind lt <ButtonRelease-1> "$serv_ do vcc3_stop"
    $canv bind rt <ButtonPress-1> "$serv_ do vcc3_startMove right"
    $canv bind rt <ButtonRelease-1> "$serv_ do vcc3_stop"
    $canv scale all 0 0 1.2 1.2
    # 0,0 is the top left, so move 0, 0 to the center of the canvas
    $canv move all 50 50

    # speed bars
    $self makeBar $base "panSpeed" "Pan Speed" 1 76
    $self makeBar $base "tiltSpeed" "Tilt Speed" 1 70
    $self makeBar $base "panPosition" "Pan Position" -800 800
    $self makeBar $base "tiltPosition" "Tilt Position" -266 222
}

Vcc3Client instproc make_zoom {base} {
    $self instvar serv_ widgets_

    set curFrame $base.buttons
    frame $curFrame
    pack $curFrame -side top

    button $curFrame.in -text "In"
    bind $curFrame.in <Button-1> "$serv_ do vcc3_startMove in"
    bind $curFrame.in <ButtonRelease-1> "$serv_ do vcc3_stop"
    pack $curFrame.in
    button $curFrame.out -text "Out"
    bind $curFrame.out <Button-1> "$serv_ do vcc3_startMove out"
    bind $curFrame.out <ButtonRelease-1> "$serv_ do vcc3_stop"
    pack $curFrame.out

    $self makeBar $base "zoomSpeed" "Zoom Speed" 0 7
    $self makeBar $base "zoomPosition" "Zoom Position" 0 [$serv_ do vcc3_getMaxZoom]
}

Vcc3Client instproc make_presets {base} {
    $self instvar serv_ widgets_

    frame $base.go_frame
    pack $base.go_frame -fill x -side top

    for {set x 1} {$x < 7} {incr x 1} {
	button $base.go_frame.$x -text "$x" -command "$serv_ do vcc3_goPreset $x"
	pack $base.go_frame.$x -side left
    }
    label $base.go_frame.label -text "Go"
    pack $base.go_frame.label -side left

    frame $base.set_frame
    pack $base.set_frame -fill x -side top

    for {set x 1} {$x < 7} {incr x 1} {
	button $base.set_frame.$x -text "$x" -command "$serv_ do vcc3_setPreset $x"
	pack $base.set_frame.$x -side left
    }
    label $base.set_frame.label -text "Set"
    pack $base.set_frame.label -side left
}

Vcc3Client instproc make_focus {base} {
    $self instvar widgets_ serv_

    set curFrame $base.radio
    frame $curFrame
    pack $curFrame -side top
    set widgets_(autoFocus) [radiobutton $curFrame.auto -text "Auto" -variable g_focusMode -value "auto" -command "$self handleFocus"]
    pack $curFrame.auto -side top
    set widgets_(manualFocus) [radiobutton $curFrame.manual -text "Manual" -variable g_focusMode -value "manual" -command "$self handleFocus"]
    pack $curFrame.manual -side top

    set curFrame $base.buttons
    frame $curFrame
    pack $curFrame -side top
    set widgets_(focusNear) [button $curFrame.near -text "Near"]
    pack $curFrame.near
    bind $curFrame.near <Button-1> "$serv_ do vcc3_startFocus near"
    bind $curFrame.near <ButtonRelease-1> "$serv_ do vcc3_stopFocus"
    set widgets_(focusFar) [button $curFrame.far -text "Far"]
    pack $curFrame.far
    bind $curFrame.far <Button-1> "$serv_ do vcc3_startFocus far"
    bind $curFrame.far <ButtonRelease-1> "$serv_ do vcc3_stopFocus"

    $self makeBar $base "focusSpeed" "Focus Speed" 0 7
    # the range for focus varies depending on the zoom value
    $self makeBar $base "focusPosition" "Focus Position" 0 2000
}

Vcc3Client instproc make_exposure {base} {
    $self instvar widgets_ serv_

    $self makeBar $base "aeRef" "AE Reference" 0 255
    $self makeBar $base "exposureSpeed" "Exposure Speed" 0 255
    $self makeBar $base "iris" "Iris" 0 255
    $self makeBar $base "shutterSpeed" "Shutter Speed" 0 255
    $self makeBar $base "gain" "Gain" 0 255
}

Vcc3Client instproc make_fade {base} {
    $self instvar widgets_ serv_

    set curFrame $base.buttons
    frame $curFrame
    pack $curFrame -side top

    button $curFrame.in -text "In" -command "$serv_ do vcc3_fadeIn"
    pack $curFrame.in
    button $curFrame.out -text "Out" -command "$serv_ do vcc3_fadeOut"
    pack $curFrame.out

    $self makeBar $base "fadeSpeed" "Fade Speed (secs)" 1 20
    $self makeBar $base "fadeGain" "Fade Gain" 0 255
    $self makeBar $base "fadeLevel" "Fade Level" 0 255
}

Vcc3Client instproc make_stepMove {base} {
    $self instvar widgets_ serv_

    button $base.home -text "Home" -command "$serv_ do vcc3_home"
    pack $base.home -side top

    # move arrows
    set cur_frame "$base.position_frame"
    frame $cur_frame
    pack $cur_frame -side top -pady 10 -padx 10
    label $cur_frame.label -text "Position"
    pack $cur_frame.label -side top
    set canv "$cur_frame.arrow_canvas"
    canvas $canv -width 100 -height 100 -borderwidth 0
    pack $canv -side top
    set uparrow " -6 -9	 -6 -23	 -14 -23  0 -35	 14 -23	 6 -23	 6 -9  -6 -9"
    set dnarrow " -6  9	 -6  23	 -14  23  0  35	 14  23	 6  23	 6  9  -6  9"
    set ltarrow " -9 -6	 -23 -6	 -23 -14  -35 0	 -23 14	 -23 6	 -9 6  -9 -6"
    set rtarrow "  9 -6	  23 -6	  23 -14   35 0	  23 14	  23 6	  9 6	9 -6"
    set p_up [eval $canv create polygon $uparrow -tags \{upStep arrow\} -fill blue]
    set p_dn [eval $canv create polygon $dnarrow -tags \{dnStep arrow\} -fill blue]]
    set p_lt [eval $canv create polygon $ltarrow -tags \{ltStep arrow\} -fill blue]]
    set p_rt [eval $canv create polygon $rtarrow -tags \{rtStep arrow\} -fill blue]]
    $canv bind upStep <ButtonPress-1> "$self handleStep up"
    $canv bind dnStep <ButtonPress-1> "$self handleStep down"
    $canv bind ltStep <ButtonPress-1> "$self handleStep left"
    $canv bind rtStep <ButtonPress-1> "$self handleStep right"
    $canv scale all 0 0 1.2 1.2
    # 0,0 is the top left, so move 0, 0 to the center of the canvas
    $canv move all 50 50

    set curFrame $base.buttons
    frame $curFrame
    pack $curFrame -side top

    button $curFrame.in -text "In" -command "$self handleStep in"
    pack $curFrame.in
    button $curFrame.out -text "Out" -command "$self handleStep out"
    pack $curFrame.out

    $self makeBar $base "panStep" "Pan Step" 0 400 false
    $self makeBar $base "tiltStep" "Tilt Step" 0 100 false
    $self makeBar $base "zoomStep" "Zoom Step" 0 400 false
    $widgets_(panStep) set 40
    $widgets_(tiltStep) set 20
    $widgets_(zoomStep) set 100
}

Vcc3Client instproc handleStep {dir} {
    $self instvar serv_ widgets_ maxZoom_

    switch -exact -- $dir {
	left -
	right {
	    set val [$widgets_(panStep) get]
	    $serv_ do vcc3_moveRelative $dir $val
	}
	up -
	down {
	    set val [$widgets_(tiltStep) get]
	    $serv_ do vcc3_moveRelative $dir $val
	}
	in -
	out {
	    set val [$widgets_(zoomStep) get]
	    set temp [$serv_ do vcc3_getZoomPosition]
	    if {$dir == "out"} {
		set temp [expr $temp - $val]
	    } else {
		set temp [expr $temp + $val]
	    }
	    if {$temp < 0} {
		set temp 0
	    }
	    if {$temp > $maxZoom_} {
		set temp $maxZoom_
	    }
	    $serv_ do vcc3_setZoomPosition $temp
	}
	default {
	    puts stderr "Vcc3Client::handleStep: error, unrecognized dir $dir"
	}
    }
}

Vcc3Client instproc handleFocus {} {
    global g_focusMode
    $self instvar serv_ widgets_

    switch -exact -- $g_focusMode {
	auto {
	    $serv_ do vcc3_setFocusMode $g_focusMode
	    $widgets_(focusNear) configure -state disabled
	    $widgets_(focusFar) configure -state disabled
	}
	manual {
	    $serv_ do vcc3_setFocusMode $g_focusMode
	    $widgets_(focusNear) configure -state normal
	    $widgets_(focusFar) configure -state normal
	}
	default {
	    puts stderr "Vcc3Client::handleFocus: bad focus value"
	}
    }
}

Vcc3Client instproc makeBar {base name textLabel min max {makeButton true}} {
    $self instvar widgets_

    set curFrame "$base.$name"
    frame $curFrame
    pack $curFrame -side top
    set size 100

    set widgets_($name) [scale $curFrame.bar -from $min -to $max -length $size -variable x.$name -orient horizontal -label $textLabel -showvalue true]
    pack $curFrame.bar -side left
    if {$makeButton == "true"} {
	button $curFrame.button -text "Set" -command "$self setBarValue $name"
	pack $curFrame.button -side left
    }
}

Vcc3Client instproc show {control} {
    global g_show
    $self instvar frames_

    if {$g_show($control)} {
	pack $frames_($control) -side left -padx 2 -pady 2
    } else {
	pack unpack $frames_($control)
    }
}

Vcc3Client instproc setBarValue {control} {
    $self instvar widgets_ serv_

    set val [$widgets_($control) get]
#    puts stdout "widget $control has value $val"
    switch -exact -- $control {
	panSpeed {
	    $serv_ do vcc3_setPanSpeed $val
	}
	tiltSpeed {
	    $serv_ do vcc3_setTiltSpeed $val
	}
	panPosition {
	    $serv_ do vcc3_setPanPosition $val
	}
	tiltPosition {
	    $serv_ do vcc3_setTiltPosition $val
	}
	zoomSpeed {
	    $serv_ do vcc3_setZoomSpeed $val
	}
	zoomPosition {
	    $serv_ do vcc3_setZoomPosition $val
	}
	focusSpeed {
	    $serv_ do vcc3_setFocusSpeed $val
	}
	fadeSpeed {
	    set speed [expr $val * 60]
    	    $serv_ do vcc3_setFadeSpeed $speed
	}
	fadeGain {
    	    $serv_ do vcc3_setFadeGain $val
	}
	fadeLevel {
    	    $serv_ do vcc3_setFadeLevel $val
	}
	aeRef {
	    $serv_ do vcc3_setAERef $val
	}
	# FIXME - add the rest of the exposure stuff to the vcc3d server
	exposureSpeed {
	    puts stderr "this feature is not implemented yet"
	    #$serv_ do vcc3_setExposureSpeed $val
	}
	gain {
	    puts stderr "this feature is not implemented yet"
	    #    $serv_ do vcc3_setGain $val
	}
	shutterSpeed {
	    puts stderr "this feature is not implemented yet"
	    #	    $serv_ do vcc3_setShutterSpeed $val
	}
	iris {
	    puts stderr "this feature is not implemented yet"
	    #	    $serv_ do vcc3_setIris $val
	}
	default {
	    puts stderr "Error: unrecognized control: $control"
	}
    }
}


Vcc3Client instproc refresh {} {
    $self instvar widgets_ serv_

    set info [$serv_ do vcc3_getPanTiltInfo]
    if {[llength $info] > 0} {
	set panSpeed [lindex $info 0]
	set tiltSpeed [lindex $info 1]
	set panPosition [lindex $info 2]
	set tiltPosition [lindex $info 3]
	$widgets_(panSpeed) set $panSpeed
	$widgets_(tiltSpeed) set $tiltSpeed
	$widgets_(panPosition) set $panPosition
	$widgets_(tiltPosition) set $tiltPosition
    } else {
	puts stderr "Vcc3Client::refresh: error calling getPanTiltInfo"
    }

    set info [$serv_ do vcc3_getZoomInfo]
    if {[llength $info] > 0} {
	set zoomSpeed [lindex $info 0]
	set zoomPosition [lindex $info 1]
	$widgets_(zoomSpeed) set $zoomSpeed
	$widgets_(zoomPosition) set $zoomPosition
    } else {
	puts stderr "Vcc3Client::refresh: error calling getZoomInfo"
    }

    set info [$serv_ do vcc3_getFocusInfo]
    if {[llength $info] > 0} {
	set focusSpeed [lindex $info 0]
	set focusPosition [lindex $info 1]
	$widgets_(focusSpeed) set $focusSpeed
	$widgets_(focusPosition) set $focusPosition
    } else {
	puts stderr "Vcc3Client::refresh: error calling getFocusInfo"
    }

    set info [$serv_ do vcc3_getExposureInfo]
    if {[llength $info] > 0} {
	set exposureSpeed [lindex $info 0]
	set aeRef [lindex $info 1]
	set iris [lindex $info 2]
	set shutterSpeed [lindex $info 3]
	set gain [lindex $info 4]
	$widgets_(aeRef) set $aeRef
	$widgets_(exposureSpeed) set $exposureSpeed
	$widgets_(iris) set $iris
	$widgets_(shutterSpeed) set $shutterSpeed
	$widgets_(gain) set $gain
    } else {
	puts stderr "Vcc3Client::refresh: error calling getExposureInfo"
    }

    set info [$serv_ do vcc3_getFadeInfo]
    if {[llength $info] > 0} {
	set fadeSpeed [lindex $info 0]
	set fadeGain [lindex $info 1]
	set fadeLevel [lindex $info 2]
	set speed [expr $fadeSpeed / 60]
	$widgets_(fadeSpeed) set $speed
	$widgets_(fadeGain) set $fadeGain
	$widgets_(fadeLevel) set $fadeLevel
    } else {
	puts stderr "Vcc3Client::refresh: error calling getExposureInfo"
    }
}

Vcc3Client instproc simpleMode {} {
    pack unpack $Vcc3Client::complexFrame
    pack $Vcc3Client::simpleFrame -side top
}

Vcc3Client instproc complexMode {} {
    pack unpack $Vcc3Client::simpleFrame
    pack $Vcc3Client::complexFrame -side top
}

Vcc3Client instproc makeSimpleUI {base} {
    $self instvar serv_

    set Vcc3Client::simpleFrame [frame $base.simpleFrame]
    # don't pack it here
    set base $base.simpleFrame

    set labelStr [$serv_ do vcc3_getLabel]
    label $base.label -text $labelStr -foreground "blue"
    pack $base.label -side top

    button $base.complex -text "Complex" -command "$self complexMode"
    pack $base.complex -side bottom

    frame $base.panTilt
    pack $base.panTilt -side top
    $self make_simplePanTilt $base.panTilt

    frame $base.zoom
    pack $base.zoom -side top
    $self make_simpleZoom $base.zoom
}


Vcc3Client instproc make_simplePanTilt {base} {
    $self instvar serv_ widgets_

    # move arrows
    set cur_frame "$base.position_frame"
    frame $cur_frame
    pack $cur_frame -side top -pady 10 -padx 10
    set canv "$cur_frame.arrow_canvas"
    canvas $canv -width 100 -height 100 -borderwidth 0
    pack $canv -side top
    set uparrow " -6 -9	 -6 -23	 -14 -23  0 -35	 14 -23	 6 -23	 6 -9  -6 -9"
    set dnarrow " -6  9	 -6  23	 -14  23  0  35	 14  23	 6  23	 6  9  -6  9"
    set ltarrow " -9 -6	 -23 -6	 -23 -14  -35 0	 -23 14	 -23 6	 -9 6  -9 -6"
    set rtarrow "  9 -6	  23 -6	  23 -14   35 0	  23 14	  23 6	  9 6	9 -6"
    set p_up [eval $canv create polygon $uparrow -tags \{up arrow\}]
    set p_dn [eval $canv create polygon $dnarrow -tags \{dn arrow\}]
    set p_lt [eval $canv create polygon $ltarrow -tags \{lt arrow\}]
    set p_rt [eval $canv create polygon $rtarrow -tags \{rt arrow\}]

    set homeSize 6
    set home "0 $homeSize -$homeSize 0 0 -$homeSize $homeSize 0"
    set p_home [eval $canv create polygon $home -tags \{home\}]
    $canv bind home <ButtonPress-1> "$serv_ do vcc3_home"

    $canv bind up <ButtonPress-1> "$serv_ do vcc3_startMove up"
    $canv bind up <ButtonRelease-1> "$serv_ do vcc3_stop"
    $canv bind dn <ButtonPress-1> "$serv_ do vcc3_startMove down"
    $canv bind dn <ButtonRelease-1> "$serv_ do vcc3_stop"
    $canv bind lt <ButtonPress-1> "$serv_ do vcc3_startMove left"
    $canv bind lt <ButtonRelease-1> "$serv_ do vcc3_stop"
    $canv bind rt <ButtonPress-1> "$serv_ do vcc3_startMove right"
    $canv bind rt <ButtonRelease-1> "$serv_ do vcc3_stop"
    $canv scale all 0 0 1.2 1.2
    # 0,0 is the top left, so move 0, 0 to the center of the canvas
    $canv move all 50 50
}


Vcc3Client instproc make_simpleZoom {base} {
    $self instvar serv_ widgets_

    set curFrame $base.buttons
    frame $curFrame
    pack $curFrame -side top

    button $curFrame.in -text " In " -justify center
    bind $curFrame.in <Button-1> "$serv_ do vcc3_startMove in"
    bind $curFrame.in <ButtonRelease-1> "$serv_ do vcc3_stop"
    pack $curFrame.in -side left
    button $curFrame.out -text "Out"
    bind $curFrame.out <Button-1> "$serv_ do vcc3_startMove out"
    bind $curFrame.out <ButtonRelease-1> "$serv_ do vcc3_stop"
    pack $curFrame.out -side right
}
