# ui-tools.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#  @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/applications/mbv2/ui-tools.tcl,v 1.17 2002/02/03 04:21:59 lim Exp $


import ToolBar Observer MbIcons Dialog/MBv2Import TipManager

Class MBv2Tool


Class MBv2DrawTextTool -superclass MBv2Tool
Class MBv2ShapeTool    -superclass MBv2Tool
Class MBv2ArrowTool    -superclass MBv2ShapeTool
Class MBv2RectTool     -superclass MBv2ShapeTool
Class MBv2OvalTool     -superclass MBv2ShapeTool
Class MBv2ImportTool    -superclass MBv2Tool
Class MBv2SelectTool   -superclass MBv2Tool
#Class MBv2MoveTool     -superclass MBv2SelectTool
#Class MBv2CopyTool     -superclass MBv2SelectTool
Class MBv2UndoTool     -superclass MBv2Tool
Class MBv2RedoTool     -superclass MBv2Tool


MBv2Tool public init { path toolbar image tip {button 0} } {
	$self next
	$self set path_ $path
	$self set toolbar_ $toolbar
	if $button {
		button $path -image $image -command \
				"$toolbar activate [$self info class] 1"
	} else {
		radiobutton $path -image $image -indicatoron false \
				-value [$self info class] \
				-variable [$toolbar tkvarname curtool_] \
				-command "$toolbar activate [$self info class]"
	}
	TipManager tip $path $tip
}


MBv2Tool public destroy { } {
	$self instvar path_
	destroy $path_
	$self next
}


MBv2Tool public sender { s } {
	$self set sender_ $s
}


MBv2Tool public properties { s } {
	$self set props_ $s
}


MBv2Tool public invoke { } {
	$self instvar path_
	$path_ invoke
}


MBv2Tool public activate { canv } {
}


MBv2Tool public deactivate { } {
}


MBv2DrawTextTool public init { path toolbar } {
	$self next $path $toolbar MbIcons(draw) "Freehand\n+ Text"

	bind DrawTextTags <ButtonPress-1>   "$self button_down %x %y"
	bind DrawTextTags <B1-Motion>       "$self button_move %x %y"
	bind DrawTextTags <ButtonRelease-1> "$self button_release %x %y"
	bind DrawTextTags <Escape>          "$self escape"

	bind DrawTextTags <KeyPress-Shift_L> "$self shift_press"
	bind DrawTextTags <KeyPress-Shift_R> "$self shift_press"
}


MBv2DrawTextTool public activate { canv } {
	$self instvar canv_ shift_
	if [info exists shift_] {
		# we're being reactivated
		unset shift_
	}
	set canv_ $canv
	$canv_ add_bindtag DrawTextTags

	set path [$canv_ path]
	$path bind current_text <Any-Key> "$self keypress %A; break"
	$path bind current_text <Return>  "$self keypress \\n"

	# explicitly catch <Tab> so it doesn't go to the default Tk <Tab>
	# handler
	$path bind current_text <Tab>     "$self keypress \\t; break"
	$path bind current_text <Left>    "$self keyleft;      break"
	$path bind current_text <Right>   "$self keyright;     break"
	$path bind current_text <Up>      "$self keyup;        break"
	$path bind current_text <Down>    "$self keydown;      break"

	$path bind local <Enter> "$self enter_local"
	$path bind local <Leave> "$self leave_local"

	$path configure -cursor pencil
}


MBv2DrawTextTool public deactivate { } {
	$self instvar canv_ shift_
	if [info exists shift_] {
		$self instvar toolbar_
		[$toolbar_ tool MBv2SelectTool] deactivate
	}

	$self finish
	$canv_ remove_bindtag DrawTextTags

	$self leave_local
	set path [$canv_ path]
	$path bind local <Enter> ""
	$path bind local <Leave> ""

	$path configure -cursor {}
}


MBv2DrawTextTool private shift_press { } {
	$self instvar canv_ toolbar_ mode_ shift_
	if { [info exists mode_] } return
	$self deactivate
	[$toolbar_ tool MBv2SelectTool] activate $canv_ $self
	set shift_ 1
}


MBv2DrawTextTool private enter_local { } {
	$self instvar canv_ xterm_cursor_

	set path [$canv_ path]
	set t [$path find withtag current]
	if { [$path type $t] != "text" } return

	set xterm_cursor_ 1
	$path configure -cursor xterm
}


MBv2DrawTextTool private leave_local { } {
	$self instvar xterm_cursor_ canv_

	if [info exists xterm_cursor_] {
		[$canv_ path] configure -cursor pencil
		unset xterm_cursor_
	}
}


MBv2DrawTextTool private button_down { x y } {
	$self instvar x_ y_ mode_ canv_ toolbar_
	if { [info exists mode_] } {
		# we are in the middle of a previous command
		# finish it
		$self finish
	}

	set path [$canv_ path]
	set x_ [$path canvasx $x]
	set y_ [$path canvasy $y]
	set mode_ unknown
}


MBv2DrawTextTool private button_move { x y } {
	$self instvar x_ y_ mode_ canv_ start_ end_ sender_ props_
	if { ![info exists mode_] || $mode_ == "text" } return

	set path [$canv_ path]
	set x [$path canvasx $x]
	set y [$path canvasy $y]

	if { $mode_ != "freehand" } {
		# give some leeway
		set diff [expr abs($x - $x_) + abs($y - $y_)]
		if { $diff < 2.0 } return
	}

	set mode_ freehand

	set end_ [$sender_ create_item line $x_ $y_ $x $y \
			-width [$props_ width] \
			-fill  [$sender_ sanitize_color [$props_ outline]] \
			-arrow none]
	if ![info exists start_] { set start_ $end_ }
	#$path create line $x_ $y_ $x $y -width 2
	set x_ $x
	set y_ $y
}


MBv2DrawTextTool private flush_coords { } {
	$self instvar afterId_ start_ end_ coords_ canv_
	if [info exists afterId_] { unset afterId_ }
	if ![info exists coords_] return

	eval [$canv_ path] create line $coords_ -width 2
	set end_ 0
	if ![info exists start_] { set start_ $end_ }
}


MBv2DrawTextTool private button_release { x y } {
	$self instvar mode_ canv_ current_text_ sender_ props_
	if { ![info exists mode_] || $mode_ == "text" } return

	set path [$canv_ path]
	set x1 [$path canvasx $x]
	set y1 [$path canvasy $y]
	if { $mode_ == "unknown" } {
		# check if we are inside a local text box
		set t [$path find withtag current]
		if { [$path type $t] == "text" && \
				[lsearch [$path gettags $t] local] != -1 } {
			# we should try to re-edit this text box
			set cmdid [$sender_ get_cmdid $t]
			if { $cmdid != {} } {
				set current_text_(canvitem) $t
				set current_text_(edit_text) $cmdid
				$path addtag current_text withtag $t
				set current_text_(index) [$self text_index $t \
						[$path canvasx $x] \
						[$path canvasy $y]]
				set icursor $current_text_(index)
			}
		}
		if ![info exists current_text_(edit_text)] {
			set font [$canv_ interpret_font [$props_ font] \
					[$canv_ get_scale]]
			set current_text_(canvitem) \
					[$path create text $x1 $y1 -font $font\
					-tags {current_text local} \
					-anchor nw -justify [$props_ justify] \
					-fill [$sender_ sanitize_color \
					[$props_ outline]]]
			set icursor end
		}

		set mode_ text
		$path focus current_text
		$path icursor current_text $icursor
		$path configure -cursor xterm
	} else {
		# we are done with the freehand
		$self instvar x_ y_
		if { $x_ != $x1 || $y_ != $y1 } { $self button_move $x $y }
		$self finish
	}
}


MBv2DrawTextTool private text_index { item x y } {
	$self instvar canv_
	set path [$canv_ path]
	set sr [$path cget -scrollregion]
	if { $sr != {} } {
		foreach {sx1 sy1 sx2 sy2} $sr { }
		set x [expr $x - $sx1]
		set y [expr $y - $sy1]
	}
	set i [$path index $item @${x},${y}]
	return $i
}


MBv2DrawTextTool private keyleft { } {
	$self instvar current_text_ canv_
	if { ![info exists current_text_(index)] || $current_text_(index)==0} {
		return
	}

	incr current_text_(index) -1
	[$canv_ path] icursor current_text $current_text_(index)
}


MBv2DrawTextTool private keyright { } {
	$self instvar current_text_ canv_
	if { ![info exists current_text_(index)] } return

	set path [$canv_ path]
	set len [string length [$path itemcget $current_text_(canvitem) -text]]
	if [info exists current_text_(redundant)] { incr len -1 }

	if { $current_text_(index) >= $len } return

	incr current_text_(index)
	[$canv_ path] icursor current_text $current_text_(index)
}


MBv2DrawTextTool private keyup { } {
	$self instvar current_text_ canv_
	if { ![info exists current_text_(index)] } return

	set path [$canv_ path]
	set text [$path itemcget $current_text_(canvitem) -text]

	for { set i [expr $current_text_(index)-1] } { $i >= 0 } {incr i -1} {
		if { [string index $text $i] == "\n" } break
	}

	# there was no \n found
	if { $i < 0 } return

	# find the \n before this one as well
	set newline $i
	for { incr i -1 } { $i >= 0 } {incr i -1} {
		if { [string index $text $i] == "\n" } break
	}

	set prev_newline $i
	set prev_numchars [expr $newline - $i - 1]
	set numchars [expr $current_text_(index) - $newline]

	if { $numchars > $prev_numchars } { set newindex $newline } \
			else { set newindex [expr $prev_newline + $numchars] }

	set current_text_(index) $newindex
	[$canv_ path] icursor current_text $current_text_(index)
}


MBv2DrawTextTool private keydown { } {
	$self instvar current_text_ canv_
	if { ![info exists current_text_(index)] } return

	set path [$canv_ path]
	set text [$path itemcget $current_text_(canvitem) -text]
	set len [string length $text]
	if [info exists current_text_(redundant)] { incr len -1 }

	# find the starting newline on this line
	for { set i [expr $current_text_(index)-1] } { $i >= 0 } {incr i -1} {
		if { [string index $text $i] == "\n" } break
	}
	set newline $i

	# find the ending newline on this line
	for { set i $current_text_(index) } { $i < $len } {incr i} {
		if { [string index $text $i] == "\n" } break
	}

	# there was no \n found
	if { $i >= $len } return
	set next_newline $i

	set numchars [expr $current_text_(index) - $newline]
	set newindex [expr $next_newline + $numchars]
	if { $newindex > $len } { set newindex $len }

	set current_text_(index) $newindex
	[$canv_ path] icursor current_text $current_text_(index)
}


# key may consist of multiple characters
# specifically if keypress is called from paste_text
MBv2DrawTextTool private keypress { key } {
	if { $key == {} } return
	set ascii 0
	# this only checks the first character in $key
	binary scan $key c ascii
	if { $ascii < 32 && $key != "\b" && $key != "\t" && $key != "\n" } \
			return

	$self instvar current_text_ canv_ sender_ props_ x_ y_
	set path [$canv_ path]
	if ![info exists current_text_(cmd)] {
		if [info exists current_text_(edit_text)] {
			set current_text_(cmd) [$sender_ edit_text_item \
					$current_text_(edit_text)]
		} else {
			# this is the first key stroke
			if { $key == "\b" } return

			# create a text item
			$path delete $current_text_(canvitem)

			set current_text_(cmd) [$sender_ create_item text \
					$x_ $y_ -font [$props_ font] \
					-justify [$props_ justify] \
					-fill [$sender_ sanitize_color \
					[$props_ outline]]]
			set current_text_(index) 0
		}
		set current_text_(canvitem) [$sender_ get_canvas_item \
				$current_text_(cmd)]

		$path addtag current_text withtag $current_text_(canvitem)
		$path focus current_text
		$path icursor current_text $current_text_(index)
	}

	if { $key == "\b" && $current_text_(index)==0 } return

	# check if we have a redundant character (if so, remove it)
	if [info exists current_text_(redundant)] {
		unset current_text_(redundant)
		set text [$path itemcget $current_text_(canvitem) \
				-text]
		$path itemconfigure $current_text_(canvitem) \
				-text [string range $text 0 \
				[expr [string length $text] - 2]]
	}

	# send out the MBv2CmdChar
	set current_text_(end) [$sender_ insert_chars $current_text_(cmd) \
			$current_text_(index) $key]
	if { $key == "\b" } {
		incr current_text_(index) -1
		set text [$path itemcget $current_text_(canvitem) \
				-text]
		set last [string index $text [expr [string length $text] - 1]]
	} else {
		set len [string length $key]
		incr current_text_(index) $len
		set last [string index $key [expr $len - 1]]
	}

	# check if we must add a redundant space character
	if { $last == "\n" || $last == "\t" } {
		set current_text_(redundant) 1
		set text [$path itemcget $current_text_(canvitem) \
				-text]
		# use <Tab> as the redundant character coz if you use
		# <Space>, it shows up in underline or overstrike mode
		# whereas <Tab> doesn't
		append text "\t"
		$path itemconfigure $current_text_(canvitem) -text $text
	}

	$path icursor current_text $current_text_(index)
}


MBv2DrawTextTool public paste_text { x y text } {
	$self instvar toolbar_ current_text_ canv_ sender_
	set curtool [$toolbar_ current_tool]

	if { $curtool == $self && [info exists current_text_(canvitem)] } {
		set bbox [[$canv_ path] bbox $current_text_(canvitem)]
		if { $bbox != {} } {
			foreach {bx1 by1 bx2 by2} $bbox { }
			if { $x >= $bx1 && $x <= $bx2 && $y >= $by1 && \
					$y <= $by2 } {
				set paste_in_current_text 1
			}
		}
	}

	if ![info exists paste_in_current_text] {
		# check if we are inside some other local text box
		set path [[$toolbar_ canvas] path]
		set t [$path find withtag current]
		if { [$path type $t] == "text" && \
				[lsearch [$path gettags $t] local] != -1 } {
			# we should try to re-edit this text box
			set cmdid [$sender_ get_cmdid $t]
			if { $cmdid != {} } {
				set idx [$self text_index $t $x $y]
				set paste_in_other_text [list $cmdid $idx]
			}
		}
	}


	if ![info exists paste_in_current_text] {
		# stop what we were doing earlier
		set canv [$toolbar_ canvas]
		if { $curtool!={} && $canv!={} } {
			$curtool deactivate
			$curtool activate $canv
		}

		if [info exists paste_in_other_text] {
			set cmd [eval [list $toolbar_] insert_text \
					$paste_in_other_text [list $text] 1]
		} else {
			set cmd [$toolbar_ insert_text $x $y $text]
		}

		$toolbar_ reset_undo $cmd
	} else {
		set current_text_(index) [$self text_index \
				$current_text_(canvitem) $x $y]
		$self keypress $text
	}
}


MBv2DrawTextTool private escape { } {
	$self instvar mode_ sender_
	if ![info exists mode_] return
	if { $mode_ == "text" } {
		$self finish
	} elseif { $mode_ == "freehand" } {
		set finished [$self finish]

		# escape means we don't really want this object; so delete it
		if { $finished != {} } {
			$sender_ delete_item $finished
		}
	}
}


MBv2DrawTextTool public finish { } {
	$self instvar mode_ start_ end_ x_ y_ sender_ current_text_ canv_ \
			toolbar_
	if ![info exists mode_] return ""

	set finished_cmd ""
	if { $mode_ == "freehand" } {
		if { $start_ != $end_ } {
			# we must group
			set finished_cmd [$sender_ create_group line \
					$start_ $end_]
			$toolbar_ reset_undo $finished_cmd
		}
	} elseif { $mode_ == "text" } {
		if [info exists current_text_(cmd)] {

			# check if we have a redundant character
			# (if so, remove it)
			set path [$canv_ path]
			if [info exists current_text_(redundant)] {
				unset current_text_(redundant)
				set text [$path itemcget \
						$current_text_(canvitem) -text]
				$path itemconfigure $current_text_(canvitem) \
						-text [string range $text 0 \
						[expr [string length $text]-1]]
			}

			# we must group
			set finished_cmd [$sender_ create_group text \
					$current_text_(cmd) \
					$current_text_(end)]
			$toolbar_ reset_undo $finished_cmd

			# remove the current_text tag from the item
			$path dtag $current_text_(canvitem) current_text

			# forget the focus
			$path focus {}
		} elseif [info exists current_text_(canvitem)] {
			if [info exists current_text_(edit_text)] {
				# remove the current_text tag from the item
				[$canv_ path] dtag $current_text_(canvitem) \
						current_text
				# forget the focus
				[$canv_ path] focus {}
			} else {
				[$canv_ path] delete $current_text_(canvitem)
			}
		}
	}

	foreach var { mode_ start_ end_ x_ y_ } {
		if [info exists $var] { unset $var }
	}
	foreach e [array names current_text_] {
		unset current_text_($e)
	}

	$self instvar xterm_cursor_
	if ![info exists xterm_cursor_] {
		[$canv_ path] configure -cursor pencil
	}
	return $finished_cmd
}


MBv2ShapeTool public init { path toolbar icon tip } {
	$self next $path $toolbar $icon $tip

	set cls [$self info class]
	bind ${cls}Tags <ButtonPress-1>   "$self button_down %x %y"
	bind ${cls}Tags <B1-Motion>       "$self button_move %x %y"
	bind ${cls}Tags <ButtonRelease-1> "$self button_release %x %y"
	bind ${cls}Tags <Escape>          "$self escape"

	bind ${cls}Tags <KeyPress-Shift_L> "$self shift_press"
	bind ${cls}Tags <KeyPress-Shift_R> "$self shift_press"
}


MBv2ShapeTool public activate { canv } {
	$self instvar canv_ shift_
	if [info exists shift_] {
		# we're being reactivated
		unset shift_
	}

	set canv_ $canv
	$canv_ add_bindtag [$self info class]Tags
	[$canv_ path] configure -cursor tcross
}


MBv2ShapeTool public deactivate { } {
	$self instvar canv_ shift_
	if [info exists shift_] {
		$self instvar toolbar_
		[$toolbar_ tool MBv2SelectTool] deactivate
	}

	$self finish
	$canv_ remove_bindtag [$self info class]Tags
	[$canv_ path] configure -cursor {}
}


MBv2ShapeTool private shift_press { } {
	$self instvar canv_ toolbar_ shift_ canvitem_
	if { [info exists canvitem_] } return
	$self deactivate
	[$toolbar_ tool MBv2SelectTool] activate $canv_ $self
	set shift_ 1
}


MBv2ShapeTool public button_down { x y } {
	$self instvar x1_ y1_ x2_ y2_ canv_ canvitem_ toolbar_
	set path [$canv_ path]
	set x1_ [$path canvasx $x]
	set y1_ [$path canvasy $y]
	set x2_ $x1_
	set y2_ $y1_

	set canvitem_ [eval [list $path] create [$self item_type] \
			$x1_ $y1_ $x2_ $y2_ [$self get_properties]]
}


MBv2ShapeTool public button_move { x y } {
	$self instvar x1_ y1_ x2_ y2_ canv_ canvitem_
	if ![info exists canvitem_] return
	set path [$canv_ path]
	set x2_ [$path canvasx $x]
	set y2_ [$path canvasy $y]
	$path coords $canvitem_ $x1_ $y1_ $x2_ $y2_
}


MBv2ShapeTool public button_release { x y } {
	$self instvar canvitem_
	if ![info exists canvitem_] return
	$self button_move $x $y
	$self finish
}


MBv2ShapeTool public escape { } {
	$self finish 0
}


MBv2ShapeTool public finish { {create 1} } {
	$self instvar x1_ y1_ x2_ y2_ canv_ canvitem_ sender_ toolbar_
	if ![info exists canvitem_] return
	[$canv_ path] delete $canvitem_
	if $create {
		set cmdid [eval [list $sender_] create_item [$self item_type] \
				$x1_ $y1_ $x2_ $y2_ [$self get_properties]]
		$toolbar_ reset_undo $cmdid
	}
	foreach var { x1_ y1_ x2_ y2_ canvitem_ } {
		if [info exists $var] { unset $var }
	}
}


MBv2ArrowTool public init { path toolbar } {
	$self next $path $toolbar MbIcons(arrow) "Arrow"
}


MBv2ArrowTool public item_type { } { return line }


MBv2ArrowTool public get_properties { } {
	$self instvar props_ sender_
	return [list -width [$props_ width] \
			-fill [$sender_ sanitize_color [$props_ outline]] \
			-arrow [$props_ arrow]]
}


MBv2RectTool public init { path toolbar } {
	$self next $path $toolbar MbIcons(rect) "Rectangle"
}


MBv2RectTool public item_type { } { return rectangle }


MBv2RectTool public get_properties { } {
	$self instvar props_ sender_
	set fill [$sender_ sanitize_color [$props_ fill]]
	if { $fill == "none" } {
		return [list -width [$props_ width] -outline \
				[$sender_ sanitize_color [$props_ outline]]]
	} else {
		return [list -width [$props_ width] -outline \
				[$sender_ sanitize_color [$props_ outline]] \
				-fill [$sender_ sanitize_color [$props_ fill]]]
	}
}


MBv2OvalTool public init { path toolbar } {
	$self next $path $toolbar MbIcons(oval) "Oval"
}


MBv2OvalTool public item_type { } { return oval }


MBv2OvalTool public get_properties { } {
	$self instvar props_ sender_
	set fill [$sender_ sanitize_color [$props_ fill]]
	if { $fill == "none" } {
		return [list -width [$props_ width] -outline \
				[$sender_ sanitize_color [$props_ outline]]]
	} else {
		return [list -width [$props_ width] -outline \
				[$sender_ sanitize_color [$props_ outline]] \
				-fill [$sender_ sanitize_color [$props_ fill]]]
	}
}


MBv2ImportTool public init { path toolbar } {
	$self next $path $toolbar MbIcons(image) "Import" 1
}


MBv2ImportTool public activate { canv } {
	$self instvar dlg_ sender_ toolbar_ next_import_as_

	if { ![info exists dlg_] || ![winfo exists $dlg_] } {
		# create the dialog box
		set dlg_ [Dialog/MBv2Import .importdlg$self \
                                -title {Select file to import} \
				-modal local]
	}

	if [info exists next_import_as_] {
		$dlg_ import_as $next_import_as_
		unset next_import_as_
	} else {
		$dlg_ import_as image
	}

	set result [$dlg_ invoke]
	set import_as [lindex $result 0]
	set files [lindex $result 1]
	if { [llength $files] > 0 } {
		set path [$canv path]
		set cmdid [$self create_item $import_as [lindex $files 0] \
				[$path canvasx 5] [$path canvasy 5]]
		if { $cmdid != {} } { $toolbar_ reset_undo $cmdid }

		foreach file [lrange $files 1 end] {
			# create a new page per image and put the image on
			# that page
			set pageid [$sender_ new_page]
			set cmdid [$self create_item $import_as $file 5 5]
		}
	}

	## switch to the select tool
	#$toolbar_ do_activate MBv2SelectTool
}


MBv2ImportTool public create_item { what file x y } {
	$self instvar sender_ toolbar_
	switch -exact -- $what {
		image {
			return [$sender_ create_item image $x $y -file $file]
		}
		text {
			if [catch {set f [open $file]}] {
				set text "Error opening file: $file"
			} else {
				set text [read $f]
				close $f
			}
			return [$toolbar_ insert_text $x $y $text]
		}
	}
	return ""
}


MBv2ImportTool public next_import_as { what } {
	$self set next_import_as_ $what
}


MBv2SelectTool public init { path toolbar } {
	$self instvar frame_ menu_
	set frame_ $path
	frame $path -relief raised -class Selectbutton
	$self next $path.radiobutton $toolbar MbIcons(pointer) "Select"
	menubutton $path.menubutton -indicatoron 1 -font {Helvetica 6} \
			-text "" -padx 0 -pady 0 -menu $path.menubutton.menu
	pack $path.radiobutton $path.menubutton -fill both -expand 1 -side top
	TipManager tip $path.radiobutton ""
	TipManager tip $path "Select"

	# create the menu
	set menu_ [menu $path.menubutton.menu -tearoff 0 \
			-font {Helvetica 10 bold}]
	$menu_ add command -label "Cut"    -accelerator <Ctl-X> \
			-command "$self cut"    -state disabled
	$menu_ add command -label "Copy"   -accelerator <Ctl-C> \
			-command "$self copy"   -state disabled
	$menu_ add command -label "Paste"  -accelerator <Ctl-V> \
			-command "$self paste"  -state disabled
	$menu_ add command -label "Delete" -accelerator <Del>   \
			-command "$self delete" -state disabled
	$menu_ add separator
	$menu_ add cascade -label "Ordering" -menu $menu_.order -state disabled

	menu $menu_.order -tearoff 0 -font {Helvetica 10 bold}
	$menu_.order add command -label "Bring to front"
	$menu_.order add command -label "Send to back"
	$menu_.order add command -label "Bring forward"
	$menu_.order add command -label "Send backward"

	$self selection_available 0
	$self clipboard_available 0

	bind SelectTags <ButtonPress-1> "$self button_down %x %y"
	bind SelectTags <B1-Motion> "$self button_move %x %y"
	bind SelectTags <ButtonRelease-1> "$self button_release %x %y"
	bind SelectTags <Escape> "$self escape"
	bind SelectTags <Delete> "$self delete"
	bind SelectTags <BackSpace> "$self delete"
	bind SelectTags <Control-x> "$self cut"
	bind SelectTags <Control-X> "$self cut"
	bind SelectTags <Control-c> "$self copy"
	bind SelectTags <Control-C> "$self copy"
	# <Control-v> is bound to the canvas all the time when the
	# canvas is created in MBv2CanvasMgr

	bind SelectTags <KeyPress-Shift_L> "$self shift_press"
	bind SelectTags <KeyPress-Shift_R> "$self shift_press"
	bind SelectTags <KeyRelease-Shift_L> "$self shift_release"
	bind SelectTags <KeyRelease-Shift_R> "$self shift_release"
}


MBv2SelectTool public destroy { } {
	$self instvar frame_
	destroy $frame_
	catch {$self next}
}


MBv2SelectTool public clipboard_available { {state {}} } {
	$self instvar menu_
	if { $state == {} } {
		set state [$menu_ entrycget "Paste*" -state]
		if { $state == "disabled" } { return 0 } else { return 1 }
	}
	if $state { set state normal } else { set state disabled }
	$menu_ entryconfigure "Paste*" -state $state
}


MBv2SelectTool public selection_available { {state {}} } {
	$self instvar menu_
	if { $state == {} } {
		set state [$menu_ entrycget "Cut*" -state]
		if { $state == "disabled" } { return 0 } else { return 1 }
	}

	if $state { set state normal } else { set state disabled }
	$menu_ entryconfigure "Cut*"      -state $state
	$menu_ entryconfigure "Copy*"     -state $state
	$menu_ entryconfigure "Delete*"   -state $state
	$menu_ entryconfigure "Ordering*" -state $state
}


# temp_from is set to some other tool in case we are entering
# the select tool mode only temporarily (via pressing the Shift key)
MBv2SelectTool public activate { canv {temp_from {}} } {
	$self instvar frame_ canv_ toolbar_ temp_from_ shift_
	$frame_ configure -relief sunken

	set canv_ $canv
	set path [$canv_ path]
	$path bind local <Enter> "$self enter"
	$path bind all   <Enter> "$self do_show_owner_under_cursor"
	$canv_ add_bindtag SelectTags

	set temp_from_ $temp_from
	if { $temp_from!={} } {
		set shift_ 1
	} elseif [info exists shift_] { unset shift_ }

	# do the <Enter> stuff if we are already inside a local item
	set t [$path find withtag current]
	if { $t != {} } {
		# make sure it's a local item
		if { [lsearch [$path gettags $t] local] != -1 } {
			$self enter
		}
	}
}


MBv2SelectTool public deactivate { } {
	$self instvar frame_ canv_ temp_from_ shift_
	$frame_ configure -relief raised

	$self finish
	[$canv_ path] bind local <Enter> ""
	[$canv_ path] bind all   <Enter> ""
	$canv_ remove_bindtag SelectTags

	if [info exists shift_] { unset shift_ }

	if { [info exists temp_from_] && $temp_from_ != {} } {
		$temp_from_ activate $canv_
		set temp_from_ {}
	}
}


MBv2SelectTool public shift_press { } {
	$self instvar temp_from_ shift_ canv_
	if { $temp_from_ != {} } {
		set shift_ 1

		# do the <Enter> stuff if we are already inside a local item
		set path [$canv_ path]
		set t [$path find withtag current]
		if { $t != {} } {
			# make sure it's a local item
			if { [lsearch [$path gettags $t] local] != -1 } {
				$self enter
			}
		}
	}
}


MBv2SelectTool public shift_release { } {
	$self instvar shift_ select_
	if { [info exists shift_] } {
		unset shift_
		if { ![info exists select_(target)] } { $self deactivate } \
				else { $self leave }
	}
}


MBv2SelectTool private in_bbox { x y id } {
	$self instvar canv_
	set path [$canv_ path]
	set x [$path canvasx $x]
	set y [$path canvasy $y]
	set bbox [$path bbox $id]
	set x1 [lindex $bbox 0]
	set y1 [lindex $bbox 1]
	set x2 [lindex $bbox 2]
	set y2 [lindex $bbox 3]
	if { $x1 <= $x && $x <= $x2 && $y1 <= $y && $y <= $y2 } { return 1 } \
			else { return 0 }
}


MBv2SelectTool public show_owner_under_cursor { { flag {} } } {
	$self instvar show_owner_
	if { $flag == {} } {return $show_owner_} else {set show_owner_ $flag}
}


MBv2SelectTool public do_show_owner_under_cursor { } {
	$self instvar canv_ sender_ toolbar_ select_ show_owner_
	if { [info exists select_(motion)] || ![info exists show_owner_] || \
			!$show_owner_ } return
	set path [$canv_ path]
	set canvid [$path find withtag current]
	if {$canvid == {}} return

	set srcid [[$sender_ session] find_owner \
			[$toolbar_ current_pageid] $canvid]
	if { $srcid == {} } return
	set name [[$sender_ session] source_name $srcid]
	$canv_ show_owner $srcid $name $canvid 1
}


MBv2SelectTool public enter { } {
	$self instvar canv_ hilit_ select_ shift_ temp_from_

	if { [info exists select_(motion)] || \
			($temp_from_ != {} && ![info exists shift_]) } return

	set path [$canv_ path]
	set t [$path find withtag current]
	if { [info exists hilit_(target)] && \
			$hilit_(target) == $t } return

	$self leave

	if [info exists hilit_(border)] {
		$path delete $hilit_(border)
	}

	set bbox [$path bbox current]
	set x1 [expr [lindex $bbox 0] - 2]
	set y1 [expr [lindex $bbox 1] - 2]
	set x2 [expr [lindex $bbox 2] + 2]
	set y2 [expr [lindex $bbox 3] + 2]

	set hilit_(target) [$path find withtag current]
	set hilit_(border) [$path create rectangle $x1 $y1 $x2 $y2 -width 1 \
			-outline gray50]
	$path bind $hilit_(border) <Leave> "$self leave"
}


MBv2SelectTool public leave { {x {}} {y {}} } {
	$self instvar canv_ hilit_
	if ![info exists hilit_(border)] return

	# do the leave only if we are exiting the bbox
	set path [$canv_ path]
	if { $x!={} && $y!={} } {
		if [$self in_bbox $x $y $hilit_item(border)] return
	}

	$path bind $hilit_(border) <Leave> {}
	$path delete $hilit_(border)
	unset hilit_(border)
	unset hilit_(target)
}


MBv2SelectTool public select { target {x {}} {y {}} } {
	$self instvar select_ hilit_ canv_
	$self unselect
	set select_(target) $target
	if { $x!={} && $y!={} } {
		set select_(startx) $x
		set select_(starty) $y
		set select_(x) $x
		set select_(y) $y
	}
	$self leave

	$self selection_available 1
}


MBv2SelectTool public unselect { } {
	$self instvar select_ hilit_ canv_
	if ![info exists select_(target)] return
	if [info exists select_(border)] {
		set path [$canv_ path]
		$path delete $select_(border)
		$path delete $select_(border,tl)
		$path delete $select_(border,tr)
		$path delete $select_(border,bl)
		$path delete $select_(border,br)
	}

	foreach elem [array names select_] {
		unset select_($elem)
	}

	$self selection_available 0
}


MBv2SelectTool public button_down { x y } {
	$self instvar select_ hilit_
	if [info exists select_(target)] {
		if { [info exists hilit_(target)] && \
				[$self in_bbox $x $y $hilit_(border)] } {
			$self select $hilit_(target) $x $y
			set select_(motion) 1
			return
		}

		if { [info exists select_(border)] && \
				[$self in_bbox $x $y $select_(border)] } {
			# reselect the select region
			# this causes select rectangle to disappear
			# until ButtonRelease happens
			$self select $select_(target) $x $y
			set select_(motion) 1
			return
		}

		if { [info exists select_(border)] && \
				![$self in_bbox $x $y $select_(border)] } {
			# we are outside the select region
			# unselect now
			$self unselect
		}
	} else {
		if { [info exists hilit_(target)] && \
				[$self in_bbox $x $y $hilit_(border)] } {
			$self select $hilit_(target) $x $y
			set select_(motion) 1
			return
		}
	}
}


MBv2SelectTool public button_move { x y } {
	$self instvar select_ hilit_ canv_
	if ![info exists select_(target)] return

	set dx [expr $x - $select_(x)]
	set dy [expr $y - $select_(y)]
	[$canv_ path] move $select_(target) $dx $dy
	set select_(x) $x
	set select_(y) $y
}


MBv2SelectTool public button_release { x y } {
	$self instvar select_ hilit_ canv_ sender_ toolbar_
	if ![info exists select_(motion)] {
		# if we are in temp mode then deactivate ourselves
		$self instvar temp_from_ shift_
		if { $temp_from_ != {} && ![info exists shift_] } \
				{ $self deactivate }
		return
	}

	unset select_(motion)
	if ![info exists select_(target)] {
		# if we are in temp mode then deactivate ourselves
		$self instvar temp_from_ shift_
		if { $temp_from_ != {} && ![info exists shift_] } \
				{ $self deactivate }
		return
	}

	# create the selection border
	if [info exists select_(border)] return

	set path [$canv_ path]
	# reset the moved canvas item
	set dx [expr $select_(startx) - $select_(x)]
	set dy [expr $select_(starty) - $select_(y)]
	if { $dx != 0 || $dy != 0 } {
		$path move $select_(target) $dx $dy
	}
	set dx [expr $x - $select_(startx)]
	set dy [expr $y - $select_(starty)]
	if { $dx !=0 || $dy != 0 } {
		# must have the sender do the actual move
		set cmdid [$sender_ get_cmdid $select_(target)]
		if { $cmdid != {} } {
			set cmdid [$sender_ move_item $cmdid $dx $dy]
			$toolbar_ reset_undo $cmdid
		}
	}

	$self make_selection_border
}


MBv2SelectTool public make_selection_border { } {
	$self instvar select_ canv_
	if ![info exists select_(target)] return

	set path [$canv_ path]
	set bbox [$path bbox $select_(target)]
	set x1 [expr [lindex $bbox 0] - 2]
	set y1 [expr [lindex $bbox 1] - 2]
	set x2 [expr [lindex $bbox 2] + 2]
	set y2 [expr [lindex $bbox 3] + 2]

	set select_(border) [$path create rectangle $x1 $y1 $x2 $y2 -width 1 \
			-outline gray50]
	set select_(border,tl) [$path create rectangle $x1 $y1 [expr $x1+2] \
			[expr $y1+2] -outline black -fill black]
	set select_(border,tr) [$path create rectangle $x2 $y1 [expr $x2-2] \
			[expr $y1+2] -outline black -fill black]
	set select_(border,bl) [$path create rectangle $x1 $y2 [expr $x1+2] \
			[expr $y2-2] -outline black -fill black]
	set select_(border,br) [$path create rectangle $x2 $y2 [expr $x2-2] \
			[expr $y2-2] -outline black -fill black]
}


MBv2SelectTool public escape { } {
	$self instvar select_ temp_from_
	$self finish 1
	if { $temp_from_ != {} } {
		$self deactivate
	}
}


MBv2SelectTool public finish { { escape 0 } } {
	$self instvar select_
	if [info exists select_(motion)] {
		if $escape {
			$self button_release $select_(startx) $select_(starty)
		} else {
			$self button_release $select_(x) $select_(y)
		}
	}

	$self leave
	$self unselect
}


MBv2SelectTool public cut { } {
	$self copy
	$self delete
}


MBv2SelectTool public copy { } {
	$self instvar clipboard_ select_ sender_ toolbar_ canv_
	if ![info exists select_(target)] return

	set pageid [$toolbar_ current_pageid]
	set cmdid  [$sender_ get_cmdid $select_(target)]
	if { $pageid != {} && $cmdid != {} } {
		set clipboard_(pageid) $pageid
		set clipboard_(cmdid)  $cmdid
		$self clipboard_available 1

		# if the target is a text object, put it into the OS clipboard
		set path [$canv_ path]
		if { [$path type $select_(target)] == "text" } {
			selection clear -displayof $path
			selection handle $path "$self os_selection \
					[list [$path itemcget $select_(target)\
					-text]]"
			selection own $path
		}
	}
}


MBv2SelectTool private os_selection { text offset max } {
	return [string range $text $offset [expr $offset + $max - 1]]
}


MBv2SelectTool public paste { } {
	$self instvar clipboard_ sender_ toolbar_ canv_
	if ![info exists clipboard_(cmdid)] return

	# make sure our canv_ instvar is set to the right value
	set canv_ [$toolbar_ canvas]

	# now create the copy
	set cmdid [$sender_ copy_item $clipboard_(cmdid) $clipboard_(pageid)]
	$toolbar_ reset_undo $cmdid
	set item [$sender_ get_canvas_item $cmdid]
	$self select $item
	$self make_selection_border

	set curtool [$toolbar_ current_tool]
	if { $curtool != {} && $curtool != $self } {
		# temporarily activate the selection tool
		catch {
			$curtool shift_press
			$self instvar shift_
			if [info exists shift_] { unset shift_ }
		}
	}
}


MBv2SelectTool public delete { } {
	$self instvar clipboard_ select_ sender_ toolbar_ temp_from_
	if ![info exists select_(target)] return

	set id [$sender_ get_cmdid $select_(target)]
	if { $id != {} } {
		set cmdid [$sender_ delete_item $id]
		$self unselect
		$toolbar_ reset_undo $cmdid
	}

	if { $temp_from_ != {} } {
		$self deactivate
	}
}



#MBv2CopyTool public init { path toolbar } {
#	$self next $path $toolbar MbIcons(copy) "Copy"
#}


MBv2UndoTool public init { path toolbar } {
	$self next $path $toolbar MbIcons(undo) "Undo" 1
}


MBv2UndoTool public activate { canv } {
	$self instvar toolbar_
	$toolbar_ undo
}


MBv2RedoTool public init { path toolbar } {
	$self next $path $toolbar MbIcons(redo) "Redo" 1
}


MBv2RedoTool public activate { canv } {
	$self instvar toolbar_
	$toolbar_ redo
}


WidgetClass MBv2Tools_ToolBar -superclass { ToolBar Observer } -default {
	{ *Button*borderWidth 1 }
	{ *Radiobutton*borderWidth 1 }
	{ *Selectbutton.borderWidth 1 }
	{ *Selectbutton*Radiobutton.borderWidth 0 }
	{ *Selectbutton*Menubutton.borderWidth 0 }
}



MBv2Tools_ToolBar public build_widget { path } {
	$self next $path
	$self instvar tools_ canv_

	set canv_ {}
	foreach cls {
		MBv2DrawTextTool
		MBv2ArrowTool
		MBv2RectTool
		MBv2OvalTool
		MBv2ImportTool
		MBv2SelectTool
		MBv2UndoTool
		MBv2RedoTool
	} {
		set tools_($cls) [new $cls $path.[string tolower $cls] $self]
	}

	$self insert end mbv2drawtexttool mbv2arrowtool mbv2recttool \
			mbv2ovaltool mbv2importtool sep \
			mbv2selecttool mbv2undotool mbv2redotool
	$tools_(MBv2DrawTextTool) invoke
	$self set recv_only_ 0
	$self reset_undo {}
}


MBv2Tools_ToolBar public destroy { } {
	$self instvar tools_ curtool_
	if [info exists curtool_] { $curtool_ deactivate }
	foreach t [array names tools_] {
		delete $tools_($t)
	}
}


MBv2Tools_ToolBar public current_pageid { } {
	$self instvar current_pageid_
	if [info exists current_pageid_] { return $current_pageid_ } \
			else { return "" }
}


MBv2Tools_ToolBar public switch_canvas { canv pageid } {
	$self instvar canv_ curtool_ current_pageid_ tools_
	set current_pageid_ $pageid
	if { $canv_ == $canv } return

	if [info exists curtool_] {
		if { $canv_!={} } {
			$curtool_ deactivate
		}
		set canv_ $canv
		$curtool_ activate $canv_
	} else {
		set canv_ $canv
	}

	$self reset_undo {}
}


# invoked as a notify_observers callback from MBv2CanvasMgr
MBv2Tools_ToolBar public control_v  { } {
        $self instvar tools_
        $tools_(MBv2SelectTool) paste
}


# invoked as a notify_observers callback from MBv2CanvasMgr
MBv2Tools_ToolBar public buttonpress_2  { path x y } {
	$self instvar curtool_ canv_ tools_
	if { ![catch {set text [string trimright [selection get]]}] && \
			$text!={} } {
		$tools_(MBv2DrawTextTool) paste_text [$path canvasx $x] \
				[$path canvasy $y] $text
		return
		# stop what we were doing earlier
		if { [info exists curtool_] && [info exists canv_] && \
				$canv_!={} } {
			$curtool_ deactivate
			$curtool_ activate $canv_
		}

		set cmd [$self insert_text [$path canvasx $x] \
				[$path canvasy $y] $text]
		$self reset_undo $cmd
	}
}


# invoked as a notify_observers callback from MBv2CanvasMgr
MBv2Tools_ToolBar public switch_scale  { scale } {
	$self instvar curtool_ tools_ canv_
	if { [info exists curtool_] && $curtool_ == $tools_(MBv2SelectTool) } {
		$curtool_ deactivate
		$curtool_ activate $canv_
	}
}


MBv2Tools_ToolBar public insert_text { x_or_cmd y_or_index text {edit 0} } {
	$self instvar sender_ props_
	if $edit {
		set start [$sender_ edit_text_item $x_or_cmd]
		set index $y_or_index
	} else {
		set start [$sender_ create_item text $x_or_cmd $y_or_index \
				-font [$props_ font] -fill [$sender_ \
				sanitize_color [$props_ outline]] \
				-justify [$props_ justify]]
		set index 0
	}
	set end [$sender_ insert_chars $start $index $text]
	set cmd [$sender_ create_group text $start $end]
	return $cmd
}


# invoked as a notify_observers callback from MBv2CanvasMgr
MBv2Tools_ToolBar public buttonpress_3  { } {
	$self activate MBv2UndoTool 1
}


MBv2Tools_ToolBar public do_activate { tool } {
	$self tkvar curtool_
	set curtool_ $tool
	$self activate $tool
}


MBv2Tools_ToolBar public activate { tool {dont_switch 0} } {
	$self instvar tools_ curtool_ canv_
	if { [info exists curtool_] && $curtool_ == $tools_($tool) } return
	if $dont_switch {
		if { [info exists curtool_] && $canv_ != {} } \
				{ $curtool_ deactivate }
		if { $canv_ != {} } { $tools_($tool) activate $canv_ }
		if { $canv_ != {} } { $curtool_ activate $canv_ }
	} else {
		if { [info exists curtool_] && $canv_ != {} } \
				{ $curtool_ deactivate }
		set curtool_ $tools_($tool)
		if { $canv_ != {} } { $tools_($tool) activate $canv_ }
	}
}


MBv2Tools_ToolBar public reset_undo { cmdid } {
	$self instvar undo_ canv_
	if { $cmdid=={} && [info exists undo_(next_undo,$canv_)] } {
		# we're just restoring the original undo state
		$self set_undo_state {} {}
	} else {
		set undo_(next_undo,$canv_)  $cmdid
		set undo_(redo_list,$canv_)  {}
		set undo_(redid_list,$canv_) {}

		$self set_undo_state {} 0
	}
}


MBv2Tools_ToolBar public undo { } {
	$self instvar undo_ sender_ canv_
	set undo [lindex $undo_(redid_list,$canv_) 0]
	if { $undo == {} } { set undo $undo_(next_undo,$canv_) }
	if { $undo != {} } {
		set list [$sender_ undo_item $undo]
		if { $undo==$undo_(next_undo,$canv_) } {
			set undo_(next_undo,$canv_) [lindex $list 0]
		} else {
			set undo_(redid_list,$canv_) [lrange \
					$undo_(redid_list,$canv_) 1 end]
		}
		set undo_(redo_list,$canv_) [concat [list [lindex $list 1]] \
				$undo_(redo_list,$canv_)]
		$self set_undo_state {} 1
	}
}


MBv2Tools_ToolBar public redo { } {
	$self instvar undo_ sender_ canv_
	set redo [lindex $undo_(redo_list,$canv_) 0]
	if { $redo != {} } {
		set list [$sender_ undo_item $redo]
		set undo_(redid_list,$canv_) [concat [list [lindex $list 1]] \
				$undo_(redid_list,$canv_)]
		set undo_(redo_list,$canv_) [lrange $undo_(redo_list,$canv_) \
				1 end]
		$self set_undo_state 1 {}
	}
}


MBv2Tools_ToolBar public set_undo_state { u r } {
	$self instvar undo_ canv_
	if { ($u == "1") || ($u == {} && \
			($undo_(next_undo,$canv_) != {} || \
			$undo_(redid_list,$canv_)!={}))} {
		$self subwidget mbv2undotool configure -state normal
	} else {
		$self subwidget mbv2undotool configure -state disabled
	}

	if { ($r == "1") || ($r == {} && $undo_(redo_list,$canv_) != {}) } {
		$self subwidget mbv2redotool configure -state normal
	} else {
		$self subwidget mbv2redotool configure -state disabled
	}
}


MBv2Tools_ToolBar public get_undo_state { } {
	set state [$self subwidget mbv2undotool cget -state]
	if { $state == "disabled" } { set u 0 } else { set u 1 }

	set state [$self subwidget mbv2redotool cget -state]
	if { $state == "disabled" } { set r 0 } else { set r 1 }

	return [list $u $r]
}


MBv2Tools_ToolBar public recv_only { f } {
	$self instvar recv_only_
	set recv_only_ $f
	$self enable_session $f
}


MBv2Tools_ToolBar public enable_session { f } {
	$self instvar tools_
	if $f { set state normal } else { set state disabled }

	set path [$self info path]
	foreach t { mbv2drawtexttool mbv2arrowtool mbv2recttool \
			mbv2ovaltool mbv2importtool \
			mbv2selecttool.radiobutton mbv2selecttool.menubutton \
			mbv2undotool mbv2redotool } {
		$path.$t configure -state $state
	}

	if $f {
		$self set_undo_state {} {}
		$tools_(MBv2DrawTextTool) invoke
	} else {
		# set the variable associated with the radiobuttons to ""
		$self tkvar {curtool_ tkcurtool_}
		set tkcurtool_ ""

		$self instvar curtool_ canv_
		if [info exists curtool_] {
			if { $canv_ != {} } {
				$curtool_ deactivate
			}
			unset curtool_
		}
	}
}


MBv2Tools_ToolBar public sender { s } {
	$self instvar tools_ sender_
	set sender_ $s
	set path [$self info path]
	foreach cls { MBv2DrawTextTool MBv2ArrowTool MBv2RectTool \
			MBv2OvalTool MBv2ImportTool MBv2SelectTool \
			MBv2UndoTool MBv2RedoTool } {
		$tools_($cls) sender $s
	}
}


MBv2Tools_ToolBar public properties { props } {
	$self instvar tools_ props_
	set props_ $props
	set path [$self info path]
	foreach cls { MBv2DrawTextTool MBv2ArrowTool MBv2RectTool \
			MBv2OvalTool MBv2ImportTool MBv2SelectTool \
			MBv2UndoTool MBv2RedoTool } {
		$tools_($cls) properties $props
	}
}


MBv2Tools_ToolBar public tool { cls } {
	$self instvar tools_
	return $tools_($cls)
}


MBv2Tools_ToolBar public invoke { cls } {
	$self instvar tools_
	$tools_($cls) invoke
}


MBv2Tools_ToolBar public current_tool { } {
	$self instvar curtool_
	if [info exists curtool_] { return $curtool_ } else { return "" }
}


MBv2Tools_ToolBar public canvas { } {
	$self instvar canv_
	if [info exists canv_] { return $canv_ } else { return "" }
}
