/*
 * atobj-rcvr.cc --
 *
 *      member functions of
 *          - AtobjRcvr
 *
 * Copyright (c) 1997-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef ATOBJ_RCVR_CC
#define ATOBJ_RCVR_CC

#include <mb/mb.h>
#include <mb/mb-nethost.h>
#include <inet.h>
#include "atobj-rcvr.h"
#include "atobj-sm.h"
#include <misc/str.h>

class AtobjSM;

#ifndef lint
static const char rcsid[] = "$Header: /usr/mash/src/repository/mash/mash-1/atobj/atobj-rcvr.cc,v 1.11 2002/02/03 03:10:21 lim Exp $";
#endif

static class AtObjRcvrClass : public TclClass {
public:
        AtObjRcvrClass() : TclClass("Atobj_rcvr") {}
        TclObject* create(int argc, const char*const* argv);
} atobjRcvr_class;

TclObject* AtObjRcvrClass::create(int argc, const char*const* argv)
{
        // format:
        //    new AtobjRcvr <sess-mgr> [<uid-in-hex> <addr-in-hex>]
        //        (use local addr/getuid() if not specified)

        // REVIEW: Tcl should start arg with 1!
        const int cTclArgStart = 4; // first argument is at argv[4]
        if (argc < cTclArgStart+1) {
                SignalError(("wrong args, usage: new AtobjRcvr"
                             " <sess-mgr> [<uid-in-hex> <addr-in-hex>]"));
        }
        AtobjSM* pMgr = (AtobjSM*)TclObject::lookup(argv[cTclArgStart]);
        SrcId sid;
        if (argc < cTclArgStart+3) {
                sid.ss_uid = getuid();
                sid.ss_addr = LookupLocalAddr();
        } else {
                sid.ss_uid  = strtoul(argv[cTclArgStart+1], (char **)NULL, 16);
                sid.ss_addr = strtoul(argv[cTclArgStart+2], (char **)NULL, 16);
        }
        return (new AtobjRcvr(pMgr, sid));
}

AtobjRcvr::AtobjRcvr(AtobjSM *pSM, const SrcId& srcId)
        // REVIEW: what is the use of 'hdrlen' in SRM?
        : SRM_PacketHandler(0), nextAnmId_(1)
{
        srcId_ = srcId;
        pMgr_ = pSM;
        phtAnimations_ = new Tcl_HashTable;

        // the following must be true
        // otherwise we would use garbage as part of the keys
        assert( sizeof(AnimationId) == sizeof(char*) );
        Tcl_InitHashTable(phtAnimations_, TCL_ONE_WORD_KEYS);

        phtRequests_ = new Tcl_HashTable;
        // keys are pointers
        Tcl_InitHashTable(phtRequests_, TCL_ONE_WORD_KEYS);

        phtReplies_ = new Tcl_HashTable;
        // keys are pointers
        Tcl_InitHashTable(phtReplies_, TCL_ONE_WORD_KEYS);
}

AtobjRcvr::~AtobjRcvr()
{
        Tcl_HashSearch search;
        Tcl_HashEntry *pEntry = Tcl_FirstHashEntry(phtAnimations_, &search);
        while (pEntry) {
                char *szAnmName = (char *)Tcl_GetHashValue(pEntry);
                Tcl::instance().evalf("delete %s", szAnmName);
                delete[] szAnmName;
                pEntry = Tcl_NextHashEntry(&search);
        }
        Tcl_DeleteHashTable(phtAnimations_);
        delete phtAnimations_;

        pEntry = Tcl_FirstHashEntry(phtRequests_, &search);
        while (pEntry) {
                Atobj_request* pRequest =
                        (Atobj_request*) Tcl_GetHashValue(pEntry);
                pRequest->cancel(); // cancel will delete the request
                pEntry = Tcl_NextHashEntry(&search);
        }
        Tcl_DeleteHashTable(phtRequests_);
        delete phtRequests_;

        pEntry = Tcl_FirstHashEntry(phtReplies_, &search);
        while (pEntry) {
                Atobj_reply* pReply =
                        (Atobj_reply*) Tcl_GetHashValue(pEntry);
                pReply->cancel(); // cancel will delete the reply
                pEntry = Tcl_NextHashEntry(&search);
        }
        Tcl_DeleteHashTable(phtReplies_);
        delete phtReplies_;
}

/*
 *----------------------------------------------------------------------
 *
 * AtobjRcvr::create_animator --
 *
 *  Creates a new animation
 *
 *  Note: called by other member functions, return string will be stored
 *        as part of hash table and deleted at destruction time.
 *
 *----------------------------------------------------------------------
 */
char* AtobjRcvr::create_animation(const AnimationId& anmId)
{
        MB_DefTcl(tcl);
        // REVIEW: figure out dynamic creation of objects with diff. types
        tcl.evalf("%s new_animation %lu", name(), anmId);
        char *sz = NULL;
        ::AllocNCopy(&sz, tcl.result());
        Trace(VERBOSE, ("new animation returns %s", sz));
        return sz;
}

/*
 *----------------------------------------------------------------------
 *
 * AtobjRcvr::define_animation --
 *
 *  Creates an animation if we did not know about it yet, otherwise
 *  create a new one.
 *
 *----------------------------------------------------------------------
 */

int AtobjRcvr::define_animation(const AnimationId& anmId)
{
        int created;
        char* szAnim;
        Tcl_HashEntry *pEntry =
                Tcl_CreateHashEntry(phtAnimations_, (char*)anmId, &created);
        if (created) {
                szAnim = create_animation(anmId);
                assert(szAnim);
                Tcl_SetHashValue(pEntry, (ClientData) szAnim);
        }
        return created;
}

/*
 *----------------------------------------------------------------------
 *
 * AtobjRcvr::get_animation --
 *
 * returns the (tcl) name of an animation given anmId
 *    if animation not found, return NULL
 *
 *----------------------------------------------------------------------
 */
const char* AtobjRcvr::get_animation(const AnimationId& anmId)
{
        Tcl_HashEntry *pEntry = Tcl_FindHashEntry(phtAnimations_,
                                                  (char*)anmId);
        if (!pEntry) return NULL;
        else return (char*)Tcl_GetHashValue(pEntry);
}

/*
 *----------------------------------------------------------------------
 *
 * AtobjRcvr::add_animation --
 *
 *  Adds a new animation given its name, this is appropriate for local
 *     receivers
 *
 *  szAnim : animation (tcl) name
 *  anmId : Id of the animation that we want to create
 *
 *----------------------------------------------------------------------
 */

void AtobjRcvr::add_animation(const char* szAnim, const AnimationId& anmId)
{
        int created;
        Tcl_HashEntry *pEntry =
                Tcl_CreateHashEntry(phtAnimations_, (char*)anmId, &created);

        assert(created && "animation was added twice?");
        assert(szAnim && "creating with null animation?");
        char *szName;
        ::AllocNCopy(&szName, szAnim);
        Tcl_SetHashValue(pEntry, (ClientData) szName);

        Tcl& tcl = Tcl::instance();
        tcl.evalf("%s setAnmId %d", szAnim, anmId);
}

/*
 *----------------------------------------------------------------------
 *
 * AtobjRcvr::next_ADU --
 *
 *  return the next ADU from a animator
 *
 *----------------------------------------------------------------------
 */
/*virtual*/
int AtobjRcvr::next_ADU(Byte *pb, int maxPktLen)
{
        char* szADU=NULL;
        AtoPkt_event* pPkt = (AtoPkt_event*) pb;

        // For now we get data from animations in arbitrary order
        // REVIEW: obviously we should poll them in a better sequence
        Tcl& tcl = Tcl::instance();
        Tcl_HashSearch search;
        Tcl_HashEntry *pEntry=NULL;
        for ( pEntry = Tcl_FirstHashEntry(phtAnimations_, &search);
              pEntry != NULL;
              pEntry = Tcl_NextHashEntry(&search) ) {

                char *szAnmName = (char *)Tcl_GetHashValue(pEntry);
                tcl.evalf("%s next_ADU %ld", szAnmName, maxPktLen);
                szADU = tcl.result();
                if (szADU[0] != cchNull) {
                        break;
                }
                pEntry = Tcl_NextHashEntry(&search);
        }
        if (!szADU) return 0;

        AnimationId anmId =
                (AnimationId) Tcl_GetHashKey(phtAnimations_, pEntry);

        pPkt->pe_anmId = host2net(anmId);
        // REVIEW: decide where eventids go
        pPkt->pe_eventId = 0;
        int strLen = PKT_ROUNDUP(strlen(szADU)+1);
        pPkt->pe_strLen = host2net(strLen);
        strcpy(pPkt->pe_achContent, szADU);
                                // zero out the rest of the string
        for (int i=strlen(szADU); i<strLen; i++) {
                pPkt->pe_achContent[i] = cchNull;
        }
        assert((int)sizeof(AtoPkt_event)+strLen <= maxPktLen);
        return (sizeof(AtoPkt_event)+strLen);
}


/*
 *----------------------------------------------------------------------
 *
 * AtobjRcvr::recv --
 *
 *  The callback from SRM_PacketHandler
 *
 *----------------------------------------------------------------------
 */

/*virtual*/
#ifdef MB_DEBUG
void AtobjRcvr::recv(Byte *pb, u_int len)
#else
void AtobjRcvr::recv(Byte *pb, u_int /* len */)
#endif
{
        AtoPkt_event* pEvent=(AtoPkt_event*) pb;
        AnimationId anmId = net2host(pEvent->pe_anmId);
        //    EventId evId = net2host(pEvent->pe_anmId);
        //    int len = net2host(pEvent->len);
        // REVIEW: do something more efficient here
        MB_DefTcl(tcl);
        // REVIEW: check the length .... might not be null terminated...

        // create it if not found
        define_animation(anmId);
        Trace(VERYVERBOSE, ("received %d bytes", len));
        const char* szAnim = get_animation(anmId);
        if (szAnim) {
                char* szCmd = Concat3(szAnim, " recv_ADU ",
                                      pEvent->pe_achContent);
                tcl.eval(szCmd);
                delete[] szCmd;
        } else {
                Trace(VERBOSE, ("ignoring data"));
        }
}

/*
 *----------------------------------------------------------------------
 *
 * AtobjRcvr::handleRequest --
 *
 *      upcall to the animation to handle it
 *
 *----------------------------------------------------------------------
 */
void AtobjRcvr::handleRequest(AtoPkt_request *pPkt)
{
        AnimationId anmId = net2host(pPkt->pr_anmId);
        int isNew = define_animation(anmId);
        // if this is the first time we see this animation, no point
        // handing it the request.
        Tcl& tcl=Tcl::instance();
        if (!isNew) {
                const char* szAnim = get_animation(anmId);
                tcl.evalf("%s handle_request %s", szAnim,
                          pPkt->pr_achContent);
        }
}

/*
 *----------------------------------------------------------------------
 *
 * AtobjRcvr::schedRequest --
 *
 *  Creates a request object and schedules it
 *
 *----------------------------------------------------------------------
 */
Atobj_request* AtobjRcvr::schedRequest(const AnimationId& anmId)
{
        Atobj_request* pRequest = new Atobj_request(this, anmId);
        int created;
        Tcl_HashEntry *pEntry =
                Tcl_CreateHashEntry(phtRequests_, (char*)pRequest, &created);
        assert(created && "2 request at the same memory?");

        assert(sizeof(anmId)==sizeof(ClientData));
        Tcl_SetHashValue(pEntry, (ClientData)anmId);

        pMgr_->schedRequest(pRequest, srcId_);
        return pRequest;
}

/*
 *----------------------------------------------------------------------
 *
 * AtobjRcvr::updateRequest --
 *
 *  returns true if request is still needed, false otherwise
 *
 *----------------------------------------------------------------------
 */
int AtobjRcvr::updateRequest(Atobj_request* pRequest,
                               const AnimationId& anmId)
{
        const char* szAnmName = get_animation(anmId);
        Tcl& tcl = Tcl::instance();
        tcl.evalf("%s update_request %lu", szAnmName, pRequest);
        int ret;
        int ok = str2int(tcl.result(), ret);
        assert(ok && "conversion error!");
        return (int)ok;
}

/*
 *----------------------------------------------------------------------
 *
 * AtobjRcvr::fillRequest --
 *
 *    use an upcall to get request
 *
 *----------------------------------------------------------------------
 */
u_int AtobjRcvr::fillRequest(Atobj_request* pRequest,
                             const AnimationId& anmId, Byte* pb, int len)
{
        const char* szAnmName = get_animation(anmId);
        Tcl& tcl = Tcl::instance();
        tcl.evalf("%s fill_request %lu %d", szAnmName, pRequest,
                  len - sizeof(AtoPkt_request));
        const char* szResult = tcl.result();
        if (szResult[0]==cchNull)
                return 0;

        AtoPkt_request* pPkt = (AtoPkt_request*) pb;
        host2net(srcId_, pPkt->pr_srcId);
        Trace(VERYVERBOSE, ("(%x) filling rqt src=%d@%s", this, srcId_.ss_uid,
                        intoa(srcId_.ss_addr)));
        pPkt->pr_anmId = host2net(anmId);

        int strLen = PKT_ROUNDUP(strlen(szResult)+1);
        strcpy(pPkt->pr_achContent, szResult);
        // zero out the rest of the string
        for (int i=strlen(szResult); i<strLen; i++) {
                pPkt->pr_achContent[i] = cchNull;
        }
        assert((int)sizeof(AtoPkt_request)+strLen < len);
        return (sizeof(AtoPkt_request)+strLen);
}

/*
 *----------------------------------------------------------------------
 *
 * AtobjRcvr::cancelRequest --
 *
 *    cancel an request, we notify the animation also in case the
 *    cancellation is due to # requests exceeding cMaxNumRequests
 *
 *    return false on invalid id's or requests.
 *
 *----------------------------------------------------------------------
 */
int AtobjRcvr::cancelRequest(Atobj_request* pRequest,
                             const AnimationId& anmId, int wantCallback)
{
        Tcl_HashEntry *pEntry =
                Tcl_FindHashEntry(phtRequests_, (char*)pRequest);
        if (!pEntry) return FALSE;

        if (wantCallback) {
                const char* szAnmName = get_animation(anmId);
                if (!szAnmName) return FALSE;

                Tcl& tcl = Tcl::instance();
                tcl.evalf("%s cancel_request %lu", szAnmName, (u_int)pRequest);
        }

        assert(pEntry && "cannot find request!");
        Tcl_DeleteHashEntry(pEntry);
        pRequest->cancel();
        return TRUE;
}


/*
 *----------------------------------------------------------------------
 *
 * AtobjRcvr::cancelReply --
 *
 *    use an upcall to get an reply
 *
 *    return false on invalid id's or requests.
 *----------------------------------------------------------------------
 */
int AtobjRcvr::cancelReply(Atobj_reply* pReply, const AnimationId& anmId)
{
        Tcl_HashEntry *pEntry =
                Tcl_FindHashEntry(phtReplies_, (char*)pReply);
        if (!pEntry) return FALSE;

        const char* szAnmName = get_animation(anmId);
        if (!szAnmName) return FALSE;

        Tcl& tcl = Tcl::instance();
        tcl.evalf("%s cancel_reply %lu", szAnmName, (u_int)pReply);

        assert(pEntry && "cannot find reply!");
        Tcl_DeleteHashEntry(pEntry);
        pReply->cancel();
        return TRUE;
}


/*
 *----------------------------------------------------------------------
 *
 * AtobjRcvr::schedReply --
 *
 *  Creates a reply object and schedules it
 *
 *----------------------------------------------------------------------
 */
Atobj_reply* AtobjRcvr::schedReply(const AnimationId& anmId)
{
        Atobj_reply* pReply = new Atobj_reply(this, anmId);
        int created;
        Tcl_HashEntry *pEntry =
                Tcl_CreateHashEntry(phtReplies_, (char*)pReply, &created);
        assert(created && "2 reply at the same memory?");

        assert(sizeof(anmId)==sizeof(ClientData));
        Tcl_SetHashValue(pEntry, (ClientData)anmId);
        pMgr_->schedReply(pReply, srcId_);

        return pReply;
}


/*
 *----------------------------------------------------------------------
 *
 * AtobjRcvr::handleReply --
 *
 *      upcall to the animation to handle it
 *
 *----------------------------------------------------------------------
 */
void AtobjRcvr::handleReply(AtoPkt_reply *pPkt)
{
        AnimationId anmId = net2host(pPkt->pr_anmId);
        define_animation(anmId);

        const char* szAnim = get_animation(anmId);
        char* szCmd=Concat3(szAnim, " handle_reply ", pPkt->pr_achContent);
        Tcl::instance().eval(szCmd);
        delete[] szCmd;
}

/*
 *----------------------------------------------------------------------
 *
 * AtobjRcvr::fillReply --
 *
 *    use an upcall to get an reply
 *
 *----------------------------------------------------------------------
 */
u_int AtobjRcvr::fillReply(Atobj_reply* pReply,
                             const AnimationId& anmId, Byte* pb, int len)
{
        const char* szAnmName = get_animation(anmId);
        Tcl& tcl = Tcl::instance();
        // -8 so that the rounding up is being taken care of
        // REVIEW: less than -8 ?
        tcl.evalf("%s fill_reply %lu %d", szAnmName, pReply,
                  len - sizeof(AtoPkt_reply)- 6);
        const char* szResult = tcl.result();

        if (szResult[0]==cchNull)
                return 0;

        AtoPkt_reply* pPkt = (AtoPkt_reply*) pb;
        host2net(srcId_, pPkt->pr_srcId);
        pPkt->pr_anmId = host2net(anmId);

        int strLen = PKT_ROUNDUP(strlen(szResult)+1);
        strcpy(pPkt->pr_achContent, szResult);
        // zero out the rest of the string
        for (int i=strlen(szResult); i<strLen; i++) {
                pPkt->pr_achContent[i] = cchNull;
        }
        assert((int)sizeof(AtoPkt_reply) + strLen < len);
        return (sizeof(AtoPkt_reply) + strLen);
}

/*
 *----------------------------------------------------------------------
 *
 * AtobjRcvr::fillSA --
 *
 *----------------------------------------------------------------------
 */
u_int AtobjRcvr::fillSA(Byte* pb)
{
        /* REVIEW: modify this to take multiple animations */
        Tcl_HashSearch hs;
        Tcl_HashEntry *pEntry = Tcl_FirstHashEntry(phtAnimations_, &hs);
        if (!pEntry) {
                bzero(pb, sizeof(AtoPkt_SA));
                return sizeof(AtoPkt_SA);       // nothing
        }

        const char* szAnmName = (char*) Tcl_GetHashValue(pEntry);
        AnimationId anmId =
                (AnimationId) Tcl_GetHashKey(phtAnimations_, pEntry);
        /* REVIEW: get the size right! */
        Tcl& tcl = Tcl::instance();
        tcl.evalf("%s fill_sa", szAnmName);
        const char* szResult = tcl.result();
        if (szResult[0]==cchNull) return 0;

        AtoPkt_SA *pSA = (AtoPkt_SA *)pb;
        host2net(srcId_, pSA->sa_srcId);
        pSA->sa_anmId = host2net(anmId);
        int strLen = PKT_ROUNDUP(strlen(szResult)+1);
        strcpy(pSA->sa_achSA, szResult);
        // zero out the rest of the string
        for (int i=strlen(szResult); i<strLen; i++) {
                pSA->sa_achSA[i] = cchNull;
        }
        assert(sizeof(AtoPkt_SA) + strLen < SRM_MTU);
        return (sizeof(AtoPkt_SA) + strLen);
}


/*
 *----------------------------------------------------------------------
 *
 * AtobjRcvr::handleSA --
 *
 *----------------------------------------------------------------------
 */
void AtobjRcvr::handleSA(AtoPkt_SA* pSA)
{
        AnimationId anmId = net2host(pSA->sa_anmId);
        MB_DefTcl(tcl);
        define_animation(anmId);
        const char* szAnim = get_animation(anmId);
        if (szAnim) {
                char* szCmd = Concat3(szAnim, " handle_sa ",
                                      pSA->sa_achSA);
                tcl.eval(szCmd);
                delete[] szCmd;
        } else {
                Trace(VERBOSE, ("ignoring data"));
        }
}


/*virtual*/
int AtobjRcvr::command(int argc, const char*const* argv)
{
        const char* cszUsage =
                "wrong usage, should be: \n"
                "$atobjRcvr begin_xmit len\n"
                "   add_animation anm_name [id]\n"
                "   sched_request anmId\n"
                "   sched_reply anmId\n"
                "   backoff_request reqId\n"
                "   cancel_request reqId anmId wantCallback\n"
                "   cancel_reply replyId anmId\n";
        int len;

        Tcl& tcl=Tcl::instance();
        if (!strcmp(argv[1], "begin_xmit") && argc==3) {
                if (str2int(argv[2], len)) {
                        len = PKT_ROUNDUP(len+1);
                        len += sizeof(AtoPkt_event);
                        pMgr_->begin_xmit(len);
                } else {
                        goto err;
                }
        } else if (!strcmp(argv[1], "add_animation") && argc>=3) {
                int id;
                if (argc==4) {
                        if (!str2int(argv[3], id))
                                goto err;
                }
                id = nextAnmId_++;
                add_animation(argv[2], id);
        } else if (!strcmp(argv[1], "sched_request") && argc==3) {
                int id;
                if (!str2int(argv[2], id)) goto err;
                Atobj_request* pRequest = schedRequest(id);
                tcl.resultf("%lu", (u_int)pRequest);
        } else if (!strcmp(argv[1], "backoff_request") && argc==3) {
                int p;
                if (!str2int(argv[2], p)) goto err;
                Atobj_request* pRequest=(Atobj_request*)p;
                pRequest->backoff();
        } else if (!strcmp(argv[1], "cancel_request") && argc==5) {
                int p, anmId, wantCallback;
                if (!str2int(argv[2], p)) goto err;
                Atobj_request* pRequest=(Atobj_request*)p;
                if (!str2int(argv[3], anmId)) goto err;
                if (!str2int(argv[4], wantCallback)) goto err;
                if (!cancelRequest(pRequest, anmId, wantCallback)) {
                        Tcl_AddErrorInfo(tcl.interp(),
                                         "cancel request failed");
                        return TCL_ERROR;
                }
        }  else if (!strcmp(argv[1], "sched_reply") && argc==3) {
                int id;
                if (!str2int(argv[2], id)) goto err;
                Atobj_reply* pReply = schedReply(id);
                tcl.resultf("%lu", (u_int)pReply);
        } else if (!strcmp(argv[1], "cancel_reply") && argc==4) {
                int p, anmId;
                if (!str2int(argv[2], p)) goto err;
                Atobj_reply* pReply=(Atobj_reply*)p;
                if (!str2int(argv[3], anmId)) goto err;
                if (!cancelReply(pReply, anmId)) {
                        Tcl_AddErrorInfo(tcl.interp(),
                                         "cancel reply failed");
                        return TCL_ERROR;
                }
        } else {
                goto err;
        }
        return TCL_OK;

 err:
                                // invalid parameter
        Tcl_AddErrorInfo(tcl.interp(), (char *)cszUsage);
        return TCL_ERROR;
}


#endif /* #ifdef ATOBJ_RCVR_CC */
