/* Run builtin functions ... sin/error etc.
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#include "ip.h"

/* Trace builtin calls.
#define DEBUG
 */

/* Spot something that might be an arg to sin/cos/tan etc.
 */
static gboolean
ismatharg( Reduce *rc, PElement *base )
{
	return( PEISIMAGE( base ) || PEISREAL( base ) || PEISCOMPLEX( base ) );
}

/* Spot something that might be an arg to re/im etc.
 */
static gboolean
iscomplexarg( Reduce *rc, PElement *base )
{
	return( PEISIMAGE( base ) || PEISCOMPLEX( base ) );
}

/* Spot anything.
 */
static gboolean isany( Reduce *rc, PElement *base ) { return( TRUE ); }

/* Other PEIS as functions.
 */
static gboolean pe_isimage( Reduce *rc, PElement *base ) 
	{ return( PEISIMAGE( base ) ); }
static gboolean pe_isreal( Reduce *rc, PElement *base ) 
	{ return( PEISREAL( base ) ); }
static gboolean pe_iscomplex( Reduce *rc, PElement *base ) 
	{ return( PEISCOMPLEX( base ) ); }
static gboolean pe_isbool( Reduce *rc, PElement *base ) 
	{ return( PEISBOOL( base ) ); }
static gboolean pe_ischar( Reduce *rc, PElement *base ) 
	{ return( PEISCHAR( base ) ); }
static gboolean pe_islist( Reduce *rc, PElement *base ) 
	{ return( PEISLIST( base ) ); }
static gboolean pe_isflist( Reduce *rc, PElement *base ) 
	{ return( PEISFLIST( base ) ); }
static gboolean pe_isclass( Reduce *rc, PElement *base )
        { return( PEISCLASS( base ) ); }

/* The types we might want to spot for builtins.
 */
static BuiltinTypeSpot vimage_spot = { "vips_image", pe_isimage };
static BuiltinTypeSpot real_spot = { "real", pe_isreal };
static BuiltinTypeSpot bool_spot = { "bool", pe_isbool };
static BuiltinTypeSpot complex_spot = { "complex|image", iscomplexarg };
static BuiltinTypeSpot flist_spot = { "non-empty list", pe_isflist };
static BuiltinTypeSpot realvec_spot = { "[real]", reduce_isrealvec };
static BuiltinTypeSpot matrix_spot = { "[[real]]", reduce_ismatrix };
static BuiltinTypeSpot string_spot = { "[char]", reduce_isfinitestring };
static BuiltinTypeSpot math_spot = { "image|real|complex", ismatharg };
static BuiltinTypeSpot instance_spot = { "class instance", pe_isclass };
static BuiltinTypeSpot any_spot = { "any", isany };

/* Args for "_".
 */
static BuiltinTypeSpot *underscore_args[] = {
        &string_spot,
        &any_spot
};

/* Do a _ call. Args already spotted.
 */
/*ARGSUSED*/
static void
apply_underscore_call( Reduce *rc, 
	const char *name, HeapNode **arg, PElement *out )
{
        PElement rhs;
	char text[MAX_STRSIZE];

	PEPOINTRIGHT( arg[0], &rhs );
	(void) reduce_get_string( rc, &rhs, text, MAX_STRSIZE );

	/* Pump though gettext.
	 */
	heap_string_new( rc->heap, _( text ), out );
}

/* Args for "get_member".
 */
static BuiltinTypeSpot *get_member_args[] = {
        &string_spot,
        &any_spot
};

/* Do a get_member call. Args already spotted.
 */
/*ARGSUSED*/
static void
apply_get_member_call( Reduce *rc, 
	const char *name, HeapNode **arg, PElement *out )
{
        PElement rhs;
	char mname[MAX_STRSIZE];
        PElement member;

	PEPOINTRIGHT( arg[1], &rhs );
	(void) reduce_get_string( rc, &rhs, mname, MAX_STRSIZE );
        PEPOINTRIGHT( arg[0], &rhs );
	if( !class_get_member( &rhs, mname, NULL, &member ) )
		reduce_throw( rc );

	PEPUTPE( out, &member );
}

/* Args for "has_member".
 */
static BuiltinTypeSpot *has_member_args[] = {
        &string_spot,
        &any_spot
};

/* Do a has_member call. Args already spotted.
 */
/*ARGSUSED*/
static void
apply_has_member_call( Reduce *rc, 
	const char *name, HeapNode **arg, PElement *out )
{
        PElement rhs;
	char mname[MAX_STRSIZE];
        PElement member;

	PEPOINTRIGHT( arg[1], &rhs );
	(void) reduce_get_string( rc, &rhs, mname, MAX_STRSIZE );
        PEPOINTRIGHT( arg[0], &rhs );
	PEPUTP( out, ELEMENT_BOOL, 
		class_get_member( &rhs, mname, NULL, &member ) );
}

/* Args for "is_instanceof".
 */
static BuiltinTypeSpot *is_instanceof_args[] = {
        &string_spot,
        &any_spot
};

/* Do an is_instance call. Args already spotted.
 */
/*ARGSUSED*/
static void
apply_is_instanceof_call( Reduce *rc, 
	const char *name, HeapNode **arg, PElement *out )
{
        PElement rhs;
	char kname[MAX_STRSIZE];

	PEPOINTRIGHT( arg[1], &rhs );
	(void) reduce_get_string( rc, &rhs, kname, MAX_STRSIZE );
        PEPOINTRIGHT( arg[0], &rhs );
	PEPUTP( out, ELEMENT_BOOL, reduce_isinstanceof( rc, kname, &rhs ) );
}

/* Args for builtin on complex.
 */
static BuiltinTypeSpot *complex_args[] = {
        &complex_spot
};

/* Do a complex op. Args already spotted.
 */
/*ARGSUSED*/
static void
apply_complex_call( Reduce *rc, 
	const char *name, HeapNode **arg, PElement *out )
{
        PElement rhs;

	PEPOINTRIGHT( arg[0], &rhs );

	if( PEISIMAGE( &rhs ) ) {
		if( strcmp( name, "re" ) == 0 ) 
			vips_spine( rc, "im_c2real", arg, out );
		else if( strcmp( name, "im" ) == 0 ) 
			vips_spine( rc, "im_c2imag", arg, out );
	}
	else if( PEISCOMPLEX( &rhs ) ) {
		if( strcmp( name, "re" ) == 0 ) {
			PEPUTP( out, 
				ELEMENT_NODE, GETLEFT( PEGETVAL( &rhs ) ) );
		}
		else if( strcmp( name, "im" ) == 0 ) {
			PEPUTP( out, 
				ELEMENT_NODE, GETRIGHT( PEGETVAL( &rhs ) ) );
		}
	}
	else
		error( "internal error #98743698437639487" );
}

/* Args for builtin on list.
 */
static BuiltinTypeSpot *flist_args[] = {
        &flist_spot
};

/* Do a complex op. Args already spotted.
 */
/*ARGSUSED*/
static void
apply_list_call( Reduce *rc, 
	const char *name, HeapNode **arg, PElement *out )
{
        PElement rhs;
        PElement a;

	PEPOINTRIGHT( arg[0], &rhs );
	assert( PEISFLIST( &rhs ) );

	if( strcmp( name, "hd" ) == 0 ) {
		PEGETHD( &a, &rhs );
		PEPUTPE( out, &a );
	}
	else if( strcmp( name, "tl" ) == 0 ) {
		PEGETTL( &a, &rhs );
		PEPUTPE( out, &a );
	}
	else
		error( "internal error #098734953" );
}

/* Args for "vips_image". 
 */
static BuiltinTypeSpot *image_args[] = { 
	&string_spot 
};

/* Do a image call.
 */
/*ARGSUSED*/
static void
apply_image_call( Reduce *rc, 
	const char *name, HeapNode **arg, PElement *out )
{
	Heap *hi = rc->heap;
	PElement rhs;
	char buf[FILENAME_MAX];
	char *fn;
	Imageinfo *ii;

	/* Get string. 
	 */
	PEPOINTRIGHT( arg[0], &rhs );
	if( reduce_get_string( rc, &rhs, buf, FILENAME_MAX ) < 0 )
		error( "internal error #29753946" );

	/* Try to load image from given string.
	 */
	if( !(fn = path_find_file( PATH_SEARCH, buf )) )
		reduce_throw( rc );
	if( !(ii = imageinfo_new_input( main_imageinfogroup, NULL, hi, fn )) ) {
		IM_FREE( fn );
		reduce_throw( rc );
	}
	IM_FREE( fn );

	PEPUTP( out, ELEMENT_IMAGE, ii );
	imageinfo_destroy_nonheap( ii );
}

/* Args for "math". 
 */
static BuiltinTypeSpot *math_args[] = { 
	&math_spot 
};

/* A math function ... name, number implementation, image implementation.
 */
typedef struct {
	const char *name;		/* ip name */
	double (*rfn)( double );	/* Number implementation */
	const char *ifn;		/* VIPS name */
} MathFn;

static double ip_sin( double a ) { return( sin( IM_RAD( a ) ) ); }
static double ip_cos( double a ) { return( cos( IM_RAD( a ) ) ); }
static double ip_tan( double a ) { return( tan( IM_RAD( a ) ) ); }
static double ip_asin( double a ) { return( IM_DEG( asin( a ) ) ); }
static double ip_acos( double a ) { return( IM_DEG( acos( a ) ) ); }
static double ip_atan( double a ) { return( IM_DEG( atan( a ) ) ); }
static double ip_exp10( double a ) { return( pow( 10.0, a ) ); }
static double ip_ceil( double a ) { return( ceil( a ) ); }
static double ip_floor( double a ) { return( floor( a ) ); }

/* Table of math functions ... number implementations, image implementations.
 */
static MathFn math_fn[] = {
	{ "sin", &ip_sin, "im_sintra" },
	{ "cos", &ip_cos, "im_costra" },
	{ "tan", &ip_tan, "im_tantra" },
	{ "asin", &ip_asin, "im_asintra" },
	{ "acos", &ip_acos, "im_acostra" },
	{ "atan", &ip_atan, "im_atantra" },
	{ "log", &log, "im_logtra" },
	{ "log10", &log10, "im_log10tra" },
	{ "exp", &exp, "im_exptra" },
	{ "exp10", &ip_exp10, "im_exp10tra" },
	{ "ceil", &ip_ceil, "im_ceil" },
	{ "floor", &ip_floor, "im_floor" }
};

/* Do a math function (eg. sin, cos, tan).
 */
/*ARGSUSED*/
static void
apply_math_call( Reduce *rc, 
	const char *name, HeapNode **arg, PElement *out )
{
	PElement rhs;
	int i;

	/* Find implementation.
	 */
	for( i = 0; i < IM_NUMBER( math_fn ); i++ )
		if( strcmp( name, math_fn[i].name ) == 0 )
			break;
	if( i == IM_NUMBER( math_fn ) )
		error( "internal error #928456936" );

	/* Get arg type ... real/complex/image
	 */
	PEPOINTRIGHT( arg[0], &rhs );
	if( PEISIMAGE( &rhs ) ) {
		/* Easy ... pass to VIPS.
		 */
		vips_spine( rc, math_fn[i].ifn, arg, out );
	}
	else if( PEISREAL( &rhs ) ) {
		double a = PEGETREAL( &rhs );
		double b = math_fn[i].rfn( a );

		heap_real_new( rc->heap, b, out );
	}
	else if( PEISCOMPLEX( &rhs ) ) {
		error_top( _( "Not implemented." ) );
		error_sub( _( "Complex math ops not implemented." ) );
		reduce_throw( rc );
	}
	else
		error( "internal error #92870653" );
}

/* Args for "predicate". 
 */
static BuiltinTypeSpot *pred_args[] = { 
	&any_spot 
};

/* A predicate function ... name, implementation.
 */
typedef struct {
	const char *name;				/* ip name */
	gboolean (*fn)( Reduce *, PElement * );	/* Implementation */
} PredicateFn;

/* Table of predicate functions ... name and implementation.
 */
static PredicateFn predicate_fn[] = {
	{ "is_image", &pe_isimage },
	{ "is_bool", &pe_isbool },
	{ "is_real", &pe_isreal },
	{ "is_char", &pe_ischar },
	{ "is_class", &pe_isclass },
	{ "is_list", &pe_islist },
	{ "is_complex", &pe_iscomplex }
};

/* Do a predicate function (eg. is_bool)
 */
/*ARGSUSED*/
static void
apply_pred_call( Reduce *rc, const char *name, HeapNode **arg, PElement *out )
{
	PElement rhs;
	gboolean res;
	int i;

	/* Find implementation.
	 */
	for( i = 0; i < IM_NUMBER( predicate_fn ); i++ )
		if( strcmp( name, predicate_fn[i].name ) == 0 )
			break;
	if( i == IM_NUMBER( predicate_fn ) )
		error( "internal error #928456936" );

	/* Call!
	 */
	PEPOINTRIGHT( arg[0], &rhs );
	res = predicate_fn[i].fn( rc, &rhs );
	PEPUTP( out, ELEMENT_BOOL, res );
}

/* Args for "error". 
 */
static BuiltinTypeSpot *error_args[] = { 
	&string_spot 
};

/* Do "error".
 */
/*ARGSUSED*/
static void
apply_error_call( Reduce *rc, const char *name, HeapNode **arg, PElement *out )
{
	char buf[MAX_STRSIZE];
	PElement rhs;

	/* Get string. 
	 */
	PEPOINTRIGHT( arg[0], &rhs );
	(void) reduce_get_string( rc, &rhs, buf, MAX_STRSIZE );

	error_top( _( "Macro error." ) );
	error_sub( "%s", buf );
	reduce_throw( rc );
}

/* Args for "print". 
 */
static BuiltinTypeSpot *print_args[] = { 
	&any_spot 
};

/* Do "print".
 */
/*ARGSUSED*/
static void
apply_print_call( Reduce *rc, const char *name, HeapNode **arg, PElement *out )
{
	PElement rhs;
	BufInfo buf;
	char txt[MAX_STRSIZE];

	buf_init_static( &buf, txt, MAX_STRSIZE );
	PEPOINTRIGHT( arg[0], &rhs );
	itext_value_ev( rc, &buf, &rhs );

	heap_string_new( rc->heap, buf_all( &buf ), out );
}

/* Args for "expand". 
 */
static BuiltinTypeSpot *expand_args[] = { 
	&string_spot 
};

/* Do "expand".
 */
/*ARGSUSED*/
static void
apply_expand_call( Reduce *rc, const char *name, HeapNode **arg, PElement *out )
{
	PElement rhs;
	char txt[FILENAME_MAX];
	char txt2[FILENAME_MAX];

	PEPOINTRIGHT( arg[0], &rhs );
	heap_get_string( &rhs, txt, FILENAME_MAX );
	expand_variables( txt, txt2 );

	heap_string_new( rc->heap, txt2, out );
}

/* All ip's builtin functions. 
 */
static BuiltinInfo builtin_table[] = {
	/* Other.
	 */
	{ "error", FALSE, 1, &error_args[0], &apply_error_call },
	{ "print", FALSE, 1, &print_args[0], &apply_print_call },
	{ "expand", FALSE, 1, &expand_args[0], &apply_expand_call },
	{ "_", FALSE, 1, &underscore_args[0], &apply_underscore_call },
	
	/* Predicates.
	 */
	{ "is_image", FALSE, IM_NUMBER( pred_args ), 
		&pred_args[0], apply_pred_call },
	{ "is_bool", FALSE, IM_NUMBER( pred_args ), 
		&pred_args[0], apply_pred_call },
	{ "is_real", FALSE, IM_NUMBER( pred_args ), 
		&pred_args[0], apply_pred_call },
	{ "is_class", FALSE, IM_NUMBER( pred_args ), 
		&pred_args[0], apply_pred_call },
	{ "is_char", FALSE, IM_NUMBER( pred_args ), 
		&pred_args[0], apply_pred_call },
	{ "is_list", FALSE, IM_NUMBER( pred_args ), 
		&pred_args[0], apply_pred_call },
	{ "is_complex", FALSE, IM_NUMBER( pred_args ), 
		&pred_args[0], apply_pred_call },
	{ "is_instanceof", FALSE, IM_NUMBER( is_instanceof_args ), 
		&is_instanceof_args[0], apply_is_instanceof_call },
	{ "has_member", FALSE, IM_NUMBER( has_member_args ), 
		&has_member_args[0], apply_has_member_call },
	{ "get_member", FALSE, IM_NUMBER( get_member_args ), 
		&get_member_args[0], apply_get_member_call },

	/* List and complex projections.
	 */
	{ "re", TRUE, IM_NUMBER( complex_args ), 
		&complex_args[0], apply_complex_call },
	{ "im", TRUE, IM_NUMBER( complex_args ), 
		&complex_args[0], apply_complex_call },
	{ "hd", TRUE, IM_NUMBER( flist_args ), 
		&flist_args[0], apply_list_call },
	{ "tl", TRUE, IM_NUMBER( flist_args ), 
		&flist_args[0], apply_list_call },

	/* Math.
	 */
	{ "sin", TRUE, IM_NUMBER( math_args ), 
		&math_args[0], apply_math_call },
	{ "cos", TRUE, IM_NUMBER( math_args ), 
		&math_args[0], apply_math_call },
	{ "tan", TRUE, IM_NUMBER( math_args ), 
		&math_args[0], apply_math_call }, 
	{ "asin", TRUE, IM_NUMBER( math_args ), 
		&math_args[0], apply_math_call },
	{ "acos", TRUE, IM_NUMBER( math_args ), 
		&math_args[0], apply_math_call },
	{ "atan", TRUE, IM_NUMBER( math_args ), 
		&math_args[0], apply_math_call }, 
	{ "log", TRUE, IM_NUMBER( math_args ), 
		&math_args[0], apply_math_call },
	{ "log10", TRUE, IM_NUMBER( math_args ), 
		&math_args[0], apply_math_call },
	{ "exp", TRUE, IM_NUMBER( math_args ), 
		&math_args[0], apply_math_call },
	{ "exp10", TRUE, IM_NUMBER( math_args ), 
		&math_args[0], apply_math_call },
	{ "ceil", TRUE, IM_NUMBER( math_args ), 
		&math_args[0], apply_math_call },
	{ "floor", TRUE, IM_NUMBER( math_args ), 
		&math_args[0], apply_math_call },

	/* Constructors.
	 */
	{ "vips_image", FALSE,  
		IM_NUMBER( image_args ), &image_args[0], apply_image_call },
};

/* Make the _builtin toolkit and populate.
 */
void
builtin_init( void )
{
	Toolkit *kit;
        int i;

	kit = toolkit_new( main_toolkitgroup, "_builtin" );

        for( i = 0; i < IM_NUMBER( builtin_table ); i++ ) {
		Symbol *sym;

		sym = symbol_new( symbol_root->expr->compile,
			builtin_table[i].name );
		assert( sym->type == SYM_ZOMBIE );
		sym->type = SYM_BUILTIN;
		sym->builtin = &builtin_table[i];
		(void) tool_new_sym( kit, -1, sym );
		symbol_made( sym );
	}

	filemodel_set_auto_load( FILEMODEL( kit ) );
	filemodel_set_modified( FILEMODEL( kit ), FALSE );
	kit->pseudo = TRUE;
}

/* Make a usage error.
 */
void
builtin_usage( BufInfo *buf, BuiltinInfo *builtin )
{
	int i;

	buf_appendf( buf, 
		ngettext( "Builtin \"%s\" takes %d argument.", 
			"Builtin \"%s\" takes %d arguments.", 
			builtin->nargs ),
		builtin->name, builtin->nargs );
        buf_appends( buf, "\n" );

	for( i = 0; i < builtin->nargs; i++ )
		buf_appendf( buf, "    %d - %s\n", 
		i + 1,
		builtin->args[i]->name );
}

#ifdef DEBUG
static void
builtin_trace_args( Heap *hi, const char *name, int n, HeapNode **arg )
{
	int i;
	BufInfo buf;
	char txt[100];

	buf_init_static( &buf, txt, 100 );

	for( i = 0; i < n; i++ ) {
		PElement t;

		PEPOINTRIGHT( arg[n - i - 1], &t );
		buf_appends( &buf, "(" );
		graph_pelement( hi, &buf, &t, FALSE );
		buf_appends( &buf, ") " );
	}

	printf( "builtin: %s %s\n", name, buf_all( &buf ) );
}
#endif /*DEBUG*/

/* Execute the internal implementation of a builtin function.
 */
void
builtin_run( Reduce *rc, Compile *compile,
	int op, const char *name, HeapNode **arg, PElement *out,
	BuiltinInfo *builtin )
{
	int i;

	/* Typecheck args.
	 */
	for( i = 0; i < builtin->nargs; i++ ) {
		BuiltinTypeSpot *ts = builtin->args[i];
		PElement base;

		PEPOINTRIGHT( arg[builtin->nargs - i - 1], &base );
		if( !ts->pred( rc, &base ) ) {
			BufInfo buf;
			char txt[100];

			buf_init_static( &buf, txt, 100 );
			itext_value_ev( rc, &buf, &base );
			error_top( _( "Bad argument." ) );
			error_sub( _( "Argument %d to builtin \"%s\" should "
				"be \"%s\", you passed:\n  %s" ),
				i + 1, name, ts->name,
				buf_all( &buf ) );
			reduce_throw( rc );
		}
	}

#ifdef DEBUG
	builtin_trace_args( rc->heap, name, builtin->nargs, arg );
#endif /*DEBUG*/

	builtin->fn( rc, name, arg, out );
}
