/* ip: display VASARI format picture files.
 *
 * Types for displaying things
 */

/* Image display preferences.
 */
struct display_info {
	Bool show_status;		/* Display status bar */
	Bool show_convert;		/* Display convert bar */
	Bool big_windows;		/* Large initial windows */
	Bool undoenabled;		/* Undo enabled for paintbox */
	Bool drag_locators;		/* Dragging of locators */
	Bool sculpt;			/* Sculpted region edges */
	Bool sfit;			/* Shrink to fit window */
	Bool paint_through;		/* Update children on paint actions */
};

/* The states the event routine for the canvas can be in. WAIT means we're
 * doing nothing but wait for something to happen, DRAW means we're rubber
 * banding a Rect, MOVE means we're moving a locator box.
 */
enum input_state {
	STATE_MOVE,		/* Wait for middle up */
	STATE_WAIT,		/* Waiting for left|IM_RECT_RIGHT/middle button down */
	STATE_RESIZE,		/* Waiting for IM_RECT_RIGHT button up */
	STATE_PAINT,		/* Wait for left up, painting */
	STATE_ADJUST,		/* Waiting for left button up */
	STATE_GRAB,		/* Waiting for either move or up */
	STATE_SUCK,		/* Sucking up ink */
	STATE_SCROLL		/* Scrolling canvas about with the mouse */
};

/* Our cursor shapes. Also states: some of these share the same physical
 * shape.
 */
typedef enum pointer_shape {
	/* Paintbox shapes.
	 */
	POINTER_DROPPER = 0,
	POINTER_PEN,
	POINTER_SMUDGE,
	POINTER_SMEAR,
	POINTER_TEXT,
	POINTER_RECT,
	POINTER_FLOOD,

	/* Resize shapes.
	 */
	POINTER_TOP,
	POINTER_BOTTOM,
	POINTER_LEFT,
	POINTER_RIGHT,
	POINTER_TOPRIGHT,
	POINTER_TOPLEFT,
	POINTER_BOTTOMRIGHT,
	POINTER_BOTTOMLEFT,

	/* Move, kill and edit.
	 */
	POINTER_MOVE,
	POINTER_KILL,
	POINTER_EDIT,

	/* Watch positions.
	 */
	POINTER_HGLASS1,
	POINTER_HGLASS2,
	POINTER_HGLASS3,
	POINTER_HGLASS4,
	POINTER_HGLASS5,
	POINTER_HGLASS6,
	POINTER_HGLASS7,
	POINTER_HGLASS8,

	/* No shape set (shape we inherit).
	 */
	POINTER_NONE
} PointerShape;
#define NUM_POINTERS ((int) POINTER_NONE + 1)

/* Our GCs.
 */
typedef enum gc_list {
	GC_PLAIN,		/* GC to paint image */
	GC_BACK,		/* paint background */
	GC_INVERT,		/* invert stuff */
	GC_TOP,			/* top shadow GC */
	GC_BOTTOM,		/* IM_RECT_BOTTOM shadow */
	GC_FG,			/* foreground paint */
	GC_BG,			/* background paint */
	GC_SELECTED,		/* selected regions */
	GC_LAST
} GcList;

/* GCs we use in image window.
 */
extern GdkGC *imagedisplay_gcs[ GC_LAST ];

/* The number of ToggleButton widgets we have in the Magnification menu.
 * These have to all be stored so that we can mark the current mag on a change
 * of scale.
 */
#define NUM_MAGS 8

/* The maximum number of pending repaints we allow.
 */
#define MAX_REPAINT (3000)

/* The bits that go together to make up a screen display. 
 */
typedef struct display {
	/* Have we seen our first expose yet.
	 */
	Bool firstexpose;

	/* Whether we are active: set on first expose event, cleared on kill.
	 */
	Bool enabled;

	/* Handle from run_win.
	 */
	void *handle;		

	/* Widgets we need to track for window.
	 */
	Widget tframe;		/* Frame for image area */
	Widget dframe;		/* Frame for draw widget */
	Widget menu;		/* Menu bar at top of window form */
	Widget draw;		/* The widget we are drawing the picture on */
	Widget form;		/* The form we resize */
	Widget mags[ NUM_MAGS ]; /* Toggle widgets for mags */
	Widget cmag;		/* Currently set mag widget */
	Widget panel;		/* Image panel toggle */
	GdkPixmap *icon;	/* Pixmap we use for the icon */
	Widget undo, redo;	/* Menu items for undo, redo */

	/* Stuff for gdk.
	 */
	GdkWindow *gdraw;	/* Window for gdb_rgb stuff */

	/* Scrollbars. The widgets, and whether they are currently visible.
	 */
	Widget hsb;		/* Horizontal bar */
	Bool ehsb;		/* hbar enabled */
	Widget vsb;		/* Vertical bar */
	Bool evsb;		/* vbar enabled */
	int sbw, sbh;		/* Minor axis sb size */

	/* Information for window in tree structure.
	 */
	struct _Symbol *sym;	/* Symbol for this screen */
	struct display *parent; /* Display above this one in tree */
	GSList *child;		/* Displays below this one */
	GSList *boxes; 		/* All box_info on this screen */
	GSList *regions; 	/* box_info for all the region boxes */
	GSList *pboxes; 	/* Locators we have on other screens */

	/* Info for image repaints.
	 */
	Rect vis;		/* Visible region of virtual canvas */
	int sw;			/* Size of DrawingArea widget */
	int sh;		
	int width;		/* Size of virtual canvas */
	int height;
	int scale;		/* Current scale factor */
	IMAGE *im; 		/* Picture data */
	REGION *ireg;		/* Region we use for PIO reading from image */
	im_threadgroup_t *tg;	/* Threads we eval with */

	IMAGE *shim; 		/* "Real" image behind display image */
	REGION *shireg;		/* Region we use for PIO reading from shim */

	/* Option box information.
	 */
	Bool drag;		/* Whether locators drag or move on this scr */
	Bool demo;		/* Drag mode enabled */
	int u, v;		/* Drag scroll direction + speed */
	XtIntervalId timer;	/* Timer for drag mode */

	/* Info for mouse operations on image.
	 */
	XtIntervalId click;	/* Timer for single/double click */
	int cx, cy;		/* Event posn for previous click */
	enum input_state state;	/* Input state */
	int drx, dry;		/* Offsets into box for drag posn */
	struct box_info *drbox;	/* Box being dragged */
	int ox, oy;		/* Mouse out of window posn */
	XtIntervalId tsuck;	/* Suck timer */

	/* Info for rulers.
	 */
	struct ruler_info *hrule;/* Horizontal and vertical */
	struct ruler_info *vrule;

	/* Status bar.
	 */
	struct status_bar *bar;

	/* Convert bar.
	 */
	struct convert_bar *conv;

	/* Cursor change tracking.
	 */
	PointerShape shape;	/* The tracking shape we have now */
	PointerShape pcursor;	/* The paint shape we have now */

	/* Full screen button - full screened now, old position and size.
	 */
	Bool fscr;		/* True for at full screen now */
	Position fsx, fsy;	/* Saved position and size */
	Dimension fsw, fsh;

	/* Calibrated display conversion tables.
	 */
	CalibrateInfo *cal;

	/* Asynch repaint stuff.
	 */
	GSList *pending;	/* Pending repaint list */
	int sheight;		/* Current repaint strip height */
} ImageDisplay;

/* Display preferences. Set globally, inherited by instances of display
 * windows.
 */
extern struct display_info display_preferences;

void *remove_display( ImageDisplay *scr );
void *break_region_display( Symbol *rsym, Symbol *sym );
void *break_display_region( Symbol *sym, Symbol *rsym );
void *pop_region_on_sym( Symbol *rsym, Symbol *sym );

void make_image( Symbol *sym, char *fl );
Bool reload_symbol( Symbol *sym );
Symbol *add_image( char *fl );
Symbol *add_image_load( char *fl );
Bool test_finished( void );




